/*
 * Handling of derivation key and appropriate data structures
 *
 * EXPORT:
 *         (DepQ*) new_DepQ ()
 *         (void) append_dependent (*Af_key, *DepQ)
 *         (void) drop_dependent_queue (*DepQ)
 *         (char*) make_derivation_key (*struct rules, *DepQ)
 *
 * IMPORT:
 *         AtFS
 *         (char*) expandmacro (*char)
 *         (char*) at_getfilename (*Af_attrs)
 *         (int) at_isDerived (*Af_attrs)
 *         (char*) at_revision (*Af_attrs)
 *         (char**) implicit_heritage (*char)
 *         (int) hashval (*char)
 *         (void) logerr (*char)
 *         (void) warning (int, *char)
 *
 *         struct rules *ruletab[]
 *         int expflg
 *         int noexpflg
 */

#include "shape.h"

extern char *expandmacro(), *at_getfilename(), *at_revision();
extern int hashval(), at_isDerived(), logerr(), expflg, noexpflg;
extern void warning();
extern struct rules *ruletab[];

#define QUEUESIZE 64

DepQ *new_DepQ () {
  DepQ *q;

  if ((q = (DepQ *)malloc (sizeof (DepQ))) == NULL) return (DepQ *)NULL;
  q->dependent_queue_initialized = FALSE;
  q->queue_changed = TRUE;
  return q;
}

void drop_dependent_queue (q) DepQ *q; {
  if (!q) return;
  if (q->dependent_queue_initialized) {
    while (q->head <= q->tail) af_dropkey (q->head++);
    free (q->dependent_queue);
    free (q);
  }
}

void append_dependent (key, q) Af_key *key; DepQ *q; {
  if (!q) return;
  q->queue_changed = TRUE;
  if (!q->dependent_queue_initialized && !init_dependent_queue(q)) {
    logerr ("Internal error: can't initialize dependent_queue");
    return;
  }
  if ((q->queued_dependents < q->current_max_dependents) || 
      (grow_dependent_queue (q))) {
    if (q->queued_dependents) 
      /* "tail" points to last element, exept when Q is empty */
      *++(q->tail) = *key;
    else
      *q->tail = *key;
    q->queued_dependents++;
    return;
  }
  else {
    logerr ("Internal error: can't expand dependent_queue");
    return;
  }
}

#define KVERS "AV(1.0)"

char *make_derivation_key (current_rule, q) 
     struct rules *current_rule; 
     DepQ *q; 
{
  register char *buf_end, *peek, *buf_bound;
  register int i;
  int hasht, len;
  char edit_buffer[MAXATTRLEN], *dis, *dependent_ids();
  static char *mts = "";
  short is_implicit_rule;
  struct rules *implicit_target();
  char **heritage;
  
  /*
   * This function finally builds the derivation key for the
   * current derivation. The produced key has the general 
   * form <magic opt-ingredients ; production ; opt-sequence-of-word >.
   * The key is returned in static memory. The variable is "keybuffer".
   */
  
  if (!q || !current_rule) return "";
  if (!q->dependent_queue_initialized) return "";
  if (!q->queue_changed) return q->keybuffer;
  
  q->keybuffer[0] = '\0';
  buf_bound = &q->keybuffer[MAXATTRLEN - 1];
  heritage = current_rule->heritage;

  /* keybuffer <- magic */
  (void) strcat (q->keybuffer, KVERS);
  
  /* keybuffer <- opt-ingredients */
  dis = dependent_ids (q, &len);
  if ((len += strlen (KVERS)) < MAXATTRLEN)
    (void) strcat (q->keybuffer, dis);
  else
    return mts;

  buf_end = q->keybuffer + len;

  if (buf_end < buf_bound) {
    (void) strcat (q->keybuffer, ";");
    buf_end++;
  }
  else
    return mts;
  
  /* production is made up of <production-key> "{" opt-macro-defs "}">. */
  /* first the production-key */
  is_implicit_rule = (index (current_rule->name, '%') != NIL);
  if (!is_implicit_rule) {
    if (!current_rule->cmdlist || !current_rule->cmdlist->command ||
	!*current_rule->cmdlist->command) {
      current_rule = implicit_target (current_rule->name);
      is_implicit_rule = TRUE;
    }
    if (!current_rule) 
      return mts;
    heritage = current_rule->heritage;
  }

  if (is_implicit_rule) {
    if ((buf_end += 2) < buf_bound)
      (void) strcat (q->keybuffer, "@(");
    else
      return mts;
    for ( i = 0; current_rule->targetlist[i]; i++) {
      peek = current_rule->targetlist[i];
      while (*peek && (buf_end < buf_bound)) {
	if (*peek == '%') { peek++; continue; }
	*buf_end = *peek;
	buf_end++;
	peek++;
      }
      if (buf_end >= buf_bound)
	return mts;
      *buf_end = '\0';
    }

    *buf_end = ':';
    *++buf_end = '\0';
    peek = current_rule->deplist;
    while (*peek && (buf_end < buf_bound)) {
      if ((*peek == '%') || (*peek == ' ')) {++peek; continue; }
      *buf_end = *peek;
      buf_end++;
      peek++;
    }
    if (buf_end >= buf_bound)
      return mts;

    *buf_end = '\0';
  }
  else { /* explicit rule */
    struct rules *this_ruletab_entry;
    int hasht_ext = 0;
    
    hasht = hashval(current_rule->name);
    this_ruletab_entry = ruletab[hasht];
    while (this_ruletab_entry && (this_ruletab_entry != current_rule)) {
      this_ruletab_entry = this_ruletab_entry->nextrule;
      hasht_ext++;
    }
    if (this_ruletab_entry == NULL)
      warning (100, "make_derivation_key: inconsistent rule parameters");
    if (hasht_ext)
      (void) sprintf (edit_buffer, "!(%d.%d", hasht, hasht_ext);
    else
      (void) sprintf (edit_buffer, "!(%d", hasht);
    if ((buf_end += (strlen (edit_buffer) + 2)) >= buf_bound)
      return mts;
    (void) strcat (q->keybuffer, edit_buffer);
  }
  
  /* ... now the opt-macro-defs */
  if ((buf_end += 2) < buf_bound)
    (void) strcat (q->keybuffer, "){");
  else
    return mts;

  if (heritage) {
    for (i = 0; heritage[i]; i++) {
      peek = heritage[i];
      while (*peek && (*peek != ')') && (buf_end < buf_bound)) {
	if ((*peek == '+') || (*peek == '(')) { ++peek; continue; }
	*buf_end = *peek;
	buf_end++; peek++;
      }
      if ((buf_end +1) >= buf_bound)
	return mts;

      /* let's assume we've seen the closing paren */
      *buf_end++ = '{';
      *buf_end = '\0';
      (void) strcpy (edit_buffer, heritage[i]);
      if (edit_buffer[0] == '+') {
	edit_buffer[0] = '$';
	dis = expandmacro (edit_buffer);
	if ((buf_end += strlen (dis) + 1) >= buf_bound)
	  return mts;
	(void) strcat (q->keybuffer, dis);
      }
      else {
	/* f*ck it !!! */
      }
      (void) strcat (q->keybuffer, "}");
    }
  }
  (void) strcat (q->keybuffer, "};");
  
  /* 
   * opt-sequence-of-word is used to add additional parameters
   * to the derivation key.
   */
  (void) strcpy (edit_buffer, expandmacro ("$(HOSTTYPE)"));
  if (!*edit_buffer)
    (void) strcpy (edit_buffer, expandmacro ("$(hosttype)"));
  if (!*edit_buffer)
    (void) strcpy (edit_buffer, "hostname.domain.dom");
  if ((buf_end += strlen (edit_buffer) +5) >= buf_bound)
    return mts;
  (void) strcat (q->keybuffer, edit_buffer);
  (void) strcat (q->keybuffer, ";");
  if (expflg)
    (void) strcat (q->keybuffer, "xpon");
  else if (noexpflg)
    (void) strcat (q->keybuffer, "xpoff");
  
  q->queue_changed = FALSE;
  return q->keybuffer;
}

static char *object_id (key) Af_key *key; {
  static char buffer[MAXPATHLEN];
  char datebuf[12], *s;
  Af_attrs attrbuf;

  buffer[0] = '\0';
  if (af_gattrs (key, &attrbuf) < 0) {
    af_perror ("object_id");
    return buffer;
  }
  (void) strcat (buffer, at_getfilename (&attrbuf));
  (void) strcat (buffer, "[");
  if (s = at_rcachekey (key)) {
    (void) strcat (buffer, "#");
    (void) strcat (buffer, s);
    free (s);
  }
  else {
    (void) strcat (buffer, at_revision (&attrbuf));
    (void) strcat (buffer, "(");
    (void) sprintf (datebuf, "%d", attrbuf.af_mtime);
    (void) strcat (buffer, datebuf);
    (void) strcat (buffer, ")");
  }
  (void) strcat (buffer, "]");
  udafree (&attrbuf);
  return buffer;
}


static char *dependent_ids (q, disl) DepQ *q; int *disl; {
  static char buffer[MAXATTRLEN];
  Af_key *this = q->head;
  int curlen = 0;
  char *s;

  buffer[0] = '\0';
  while (this && (this <= q->tail)) {
    s = object_id (this++);
    curlen += strlen (s);
    if (curlen < MAXATTRLEN)
      (void) strcat (buffer, s);
    else
      break;
  }
  *disl = curlen;
  return buffer;
}

static int init_dependent_queue (q) DepQ *q; {
  static Af_key nullkey;

  if (q->dependent_queue_initialized) return TRUE;
  if ((q->dependent_queue = (Af_key *)malloc (QUEUESIZE * sizeof (Af_key)))
      == (Af_key *)NULL) {
    logerr ("malloc");
    return FALSE;
  }
  q->head = q->tail = q->dependent_queue;
  *q->head = nullkey;
  q->current_max_dependents = QUEUESIZE;
  q->queued_dependents = 0;
  q->dependent_queue_initialized = TRUE;
  return TRUE;
}

static int grow_dependent_queue (q) DepQ *q; {
  unsigned int tail_offset = q->tail - q->dependent_queue;
  Af_key *queue_anchor = q->dependent_queue;

  if (!q->dependent_queue_initialized) {
    return init_dependent_queue(q);
  }

  if ((q->dependent_queue = (Af_key *)realloc ((char *)queue_anchor, 
			    2 * q->current_max_dependents * sizeof (Af_key)))
      == (Af_key *)NULL) {
    q->dependent_queue = queue_anchor;
    return FALSE;
  }
  if (q->dependent_queue != queue_anchor) {
    q->head = q->dependent_queue;
    q->tail = q->dependent_queue + tail_offset;
  }
  q->current_max_dependents = q->current_max_dependents * 2;
  return TRUE;
}
