/* Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 * at_mktime.c 
 *
 * $Header: at_mktime.c[1.5] Wed Apr 22 20:26:46 1992 axel@cs.tu-berlin.de accessed $
 *
 * EXPORT:
 *        time_t at_mktime (char *date_string) 
 *               -- convert date/time strings in various formats
 *               -- to Unix time-values. The following formats are
 *               -- supported:
 *
 *		     Mon Jan 19 22:34:47 1992
 *		     Mon Jan 19 22:34:47 MET 1992
 *		     Jan 31 04:09:59 1992
 *		     Jan 31 04:09
 *		     1992/01/31 04:09:59  -- statt ' ' auch '/' moeglich
 *		     1992/01/31 04:09
 *		     1992/01/31
 *		     92/01/31 04:09:59
 *		     92/01/31 04:09
 *		     92/01/31             
 */

#ifndef lint
static char *ATFSid = "$Header: at_mktime.c[1.5] Wed Apr 22 20:26:46 1992 axel@cs.tu-berlin.de accessed $";
#ifdef CFFLGS
static char *ConfFlg = CFFLGS;
	/* should be defined from within Makefile */
#endif
#endif

/*LINTLIBRARY*/

#include <sys/types.h>
#include <sys/time.h>
#include <stdio.h>
#include <strings.h>

#define TRUE 1
#define FALSE 0

#define BOTIME 70        /* Beginning Of TIME for Unix */
#define DINYEAR 365      /* No. of days in a non-leapyear */
#define DINMONTH 31      /* No. of days in a 'canonical' month */
#define SINDAY 86400     /* No. of seconds in a day (24 * 60 * 60) */
#define SINHR 3600       /* No. of seconds in an hour */
#define SINMIN 60        /* ... guess! ... */

#define RE1 "[MTWFS][oherau][nudeit] [JFMASOND][aepuco][nbrylgptvc] [ 123][0-9] [0-2][0-9]:[0-5][0-9]:[0-5][0-9] 19[7-9][0-9]"
/* Mon Jan 19 22:34:47 1992 */
#define RE2 "[MTWFS][oherau][nudeit] [JFMASOND][aepuco][nbrylgptvc] [ 123][0-9] [0-2][0-9]:[0-5][0-9]:[0-5][0-9] ... 19[7-9][0-9]"
/* Mon Jan 19 22:34:47 MET 1992 */
#define RE3 "[JFMASOND][aepuco][nbrylgptvc] [ 123][0-9] [0-2][0-9]:[0-5][0-9]:[0-5][0-9] 19[7-9][0-9]"
/* Jan 31 04:09:59 1992 */
#define RE4 "[JFMASOND][aepuco][nbrylgptvc] [ 123][0-9] [0-2][0-9]:[0-5][0-9]"
/* Jan 31 04:09 */
#define RE5 "[12][0-9][0-9][0-9]/[0-9][0-9]/[0-9][0-9][ /][0-2][0-9]:[0-5][0-9]:[0-5][0-9]"
/* 1992/01/31 04:09:59 */
#define RE6 "[12][0-9][0-9][0-9]/[0-9][0-9]/[0-9][0-9][ /][0-2][0-9]:[0-5][0-9]"
/* 1992/01/31 04:09 */
#define RE7 "[12][0-9][0-9][0-9]/[0-9][0-9]/[0-9][0-9]"
/* 1992/01/31 */
#define RE8 "[0-9][0-9]/[0-9][0-9]/[0-9][0-9][ /][0-2][0-9]:[0-5][0-9]:[0-5][0-9]"
/* 92/01/31 04:09:59 */
#define RE9 "[0-9][0-9]/[0-9][0-9]/[0-9][0-9][ /][0-2][0-9]:[0-5][0-9]"
/* 92/01/31 04:09 */
#define RE10 "[0-9][0-9]/[0-9][0-9]/[0-9][0-9]"
/* 92/01/31 */
#define RE11 "[1-9]\.[1-9]\.[0-9][0-9][ /][0-9]:[0-5][0-9]:[0-5][0-9]"
/* 1.5.92 7:03:01 */
#define RE12 "[1-3][0-9]\.[1-9]\.[0-9][0-9][ /][0-9]:[0-5][0-9]:[0-5][0-9]"
/* 10.5.92 7:03:01 */
#define RE13 "[1-9]\.1[012]\.[0-9][0-9][ /][0-9]:[0-5][0-9]:[0-5][0-9]"
/* 1.10.92 7:03:01 */
#define RE14 "[1-3][0-9]\.1[012]\.[0-9][0-9][ /][0-9]:[0-5][0-9]:[0-5][0-9]"
/* 12.10.92 7:03:01 */
#define RE15 "[1-9]\.[1-9]\.[0-9][0-9][ /][12][0-9]:[0-5][0-9]:[0-5][0-9]"
/* 1.5.92 17:03:01 */
#define RE16 "[1-3][0-9]\.[1-9]\.[0-9][0-9][ /][12][0-9]:[0-5][0-9]:[0-5][0-9]"
/* 10.5.92 17:03:01 */
#define RE17 "[1-9]\.1[012]\.[0-9][0-9][ /][12][0-9]:[0-5][0-9]:[0-5][0-9]"
/* 1.10.92 17:03:01 */
#define RE18 "[1-3][0-9]\.1[012]\.[0-9][0-9][ /][12][0-9]:[0-5][0-9]:[0-5][0-9]"
/* 12.10.92 17:03:01 */

static struct df_map {
  char *pat, *fmt;
} date_fmts[] = {
  { RE1, ""},
  { RE2, ""},
  { RE3, ""},
  { RE4, ""},
  { RE5, ""},
  { RE6, ""},
  { RE7, ""},
  { RE8, ""},
  { RE9, ""},
  { RE10, ""},
  { RE11, ""},
  { RE12, ""},
  { RE13, ""},
  { RE14, ""},
  { RE15, ""},
  { RE16, ""},
  { RE17, ""},
  { RE18, ""},
  { (char *)0, (char *)0}
};

time_t at_mktime (s) char *s; {
  time_t at_tval();
  return at_tval (s);
}

static time_t at_tval (s) char *s; {
  /* if s is a known format datestring, it is converted to a time_t */
  register int i=0;
  char *emsg, *re_comp();
  struct tm timeval;
  time_t at_str2tval();
  
  if (emsg = re_comp (date_fmts[0].pat)) {
    logerr (emsg);
    exit (1);
  }
  while (date_fmts[i].pat) {
    if (re_exec (s)) {
      return at_str2tval (s, i);
    }
    i++;
    if (emsg = re_comp (date_fmts[i].pat)) {
      logerr (emsg);
      exit (1);
    }
  }
  return 0;
}

static time_t dd_tval (s) char *s; {
  /* if argument represents a number, it is interpreted as days and
     converted into seconds (to be consistent with time_t)
   */
  char *ds_segs[4], *index();
  long res;
  register int i;

  for (i = 0; i < 5; i++)
    ds_segs[i] = (char *)0;

  ds_segs[0] = s;
  for (i = 1; i < 4; i++)
    if (ds_segs[i-1]) {
      if (ds_segs[i] = index (ds_segs[i-1], '.')) {
	*ds_segs[i] = '\0';
	ds_segs[i] += 1;
      }
      else break;
    }
  res = atol (ds_segs[0]) * SINDAY + 
    (ds_segs[1] ? atol (ds_segs[1]) * 60 * 60 : 0) +
      (ds_segs[2] ? atol (ds_segs[2]) * 60 : 0) + 
	(ds_segs[3] ? atol (ds_segs[3]) : 0);
  return res;
}

static char *year2 (s) char *s; {
  /* yields two-digit year; assume correct input */
  return (char *)s+strlen(s)-2;
}

static char *mon_tab[] = {
  "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug",
  "Sep", "Oct", "Nov", "Dec", (char *)0 
};

static char *mon12(s) char *s; {
  /* converts "Jan", "Feb" etc. to "01", "02" etc. */
  register int i; 
  static char outstr[3];

  for (i = 0; mon_tab[i]; i++) {
    if (strcmp (mon_tab[i], s))
      continue;
    (void)sprintf (outstr, "%2d", i+1);
    break;
  }
  return outstr;
}

static char *day2(s) char *s; {
  if ((strlen (s) == 2) && (s[0] == ' '))
    s[0] = '0';
  return s;
}

static time_t at_str2tval (str, format) char *str; int format; {
  char junk[80], mstr[10], yr4[5], norm_form[80], *thisyear();
  int yr, mon, day, hr, min, sec;
  time_t _at_mktime();

  switch (format) {
  case 0:   /* e.g. "Mon Jan 19 22:34:47 1992" */
    (void) sscanf (str, "%s %s %d %d:%d:%d %s", junk, mstr, &day, &hr, 
		   &min, &sec, yr4);
    (void) sprintf (norm_form, "%s/%s/%2d/%2d:%2d:%2d", year2(yr4), 
		    mon12(mstr), day, hr, min, sec);
    break;
  case 1:   /* e.g. "Mon Jan 19 22:34:47 MET 1992" */
    (void) sscanf (str, "%s %s %d %d:%d:%d %s %s", junk, mstr, &day, &hr, 
		   &min, &sec, junk, yr4);
    (void) sprintf (norm_form, "%s/%s/%2d/%2d:%2d:%2d", year2(yr4), 
		    mon12(mstr), day, hr, min, sec);
    break;
  case 2:   /* e.g. "Jan 19 22:34:47 1992" */
    (void) sscanf (str, "%s %d %d:%d:%d %s", mstr, &day, &hr, 
		   &min, &sec, yr4);
    (void) sprintf (norm_form, "%s/%s/%2d/%2d:%2d:%2d", year2(yr4), 
		    mon12(mstr), day, hr, min, sec);
    break;
  case 3:   /* e.g. "Jan 19 22:34" */
    (void) sscanf (str, "%s %d %d:%d", mstr, &day, &hr, 
		   &min);
    (void) strcpy (yr4, thisyear());
    (void) sprintf (norm_form, "%s/%s/%2d/%2d:%2d:%2d", year2(yr4), 
		    mon12(mstr), day, hr, min, 0);
    break;
  case 4:   /* e.g. "1992/01/31 04:09:59" */
  case 7:
    {
      char *c, *s;
      if ((s = index (str, '/')) && (c = index (s, ' ')) && 
	(index (c, ':'))) /* date and time delimited by ' ' */
	(void) sscanf (str, "%d/%d/%d %d:%d:%d", &yr, &mon, &day, 
		   &hr, &min, &sec);
      else /* date and time delimited by '/' */
	(void) sscanf (str, "%d/%d/%d/%d:%d:%d", &yr, &mon, &day, 
		   &hr, &min, &sec);
    (void) sprintf (norm_form, "%2d/%2d/%2d/%2d:%2d:%2d", yr%100, 
		    mon, day, hr, min, sec);
    }
    break;
  case 5:   /* e.g. "1992/01/31 04:09" */
  case 8:
    {
      char *c, *s;
      if ((s = index (str, '/')) && (c = index (s, ' ')) && 
	(index (c, ':'))) /* date and time delimited by ' ' */
	(void) sscanf (str, "%d/%d/%d %d:%d", &yr, &mon, &day, 
		       &hr, &min);
      else /* date and time delimited by '/' */
	(void) sscanf (str, "%d/%d/%d/%d:%d", &yr, &mon, &day, 
		       &hr, &min);
    (void) sprintf (norm_form, "%2d/%2d/%2d/%2d:%2d:%2d", yr%100, 
		    mon, day, hr, min, 0);
    }
    break;
  case 6:   /* e.g. "1992/01/31" */
  case 9:
    (void) sscanf (str, "%d/%d/%d %d:%d", &yr, &mon, &day);
    (void) sprintf (norm_form, "%2d/%2d/%2d/%2d:%2d:%2d", yr%100, 
		    mon, day, 0, 0, 0);
    break;
  case 10:  /* e.g. "7.12.93 0:12:04" */
  case 11:
  case 12:
  case 13:
  case 14:
  case 15:
  case 16:
  case 17:
    {
      char *s = index (str, '/');
      if (s) 
	*s = ' ';
      (void) sscanf (str, "%d.%d.%d %d:%d:%d", &day, &mon, &yr, &hr, &min,
		     &sec);
      (void) sprintf (norm_form, "%2d/%2d/%2d/%2d:%2d:%2d", yr%100, 
		      mon, day, hr, min, sec);
      if (s)
	*s = '/';
    }
    break;
  default:
    return 0;
    break;
  }
  return _at_mktime (norm_form);
}

static time_t _at_mktime (tstring) char *tstring; {
  /*
   *  converts a YY/MM/DD/HH:MM:SS string into a system time representation
   *  which by convention is the number of seconds since Jan. 1st 1970.
   */
  struct tm *time_info;
  struct timeval tv;
  struct timezone tz;

  int yr, mo, dy, hr = 0, mint = 0, sec = 0, nfyrs, ndays, nsecs;

  (void)gettimeofday (&tv, &tz);
  time_info = localtime (&tv.tv_sec);
  
  (void)sscanf (tstring, "%d/%d/%d/%d:%d:%d", &yr, &mo, &dy, &hr, &mint, &sec);
  if ((chkyr (yr)) && (chkmo (yr, mo)) && (chkdy (yr, mo, dy)) && 
      (chkhr (dy, hr)) && (chkmin (hr, mint))) {
    nfyrs = yr - BOTIME;
    ndays = nfyrs * DINYEAR + nlyrs (yr, BOTIME) + 
      (mo-1)*DINMONTH - cordays (yr, mo-1) + (dy-1);
    nsecs = ndays * SINDAY + hr * SINHR + mint * SINMIN + sec;
    nsecs -= time_info->tm_gmtoff;
    return nsecs;
  }
  else return 0;
}

static char *thisyear () {
  struct tm *now, *localtime();
  struct timeval t;
  struct timezone tzone; 
  static char yearstr[5];

  (void)gettimeofday (&t, &tzone);
  now = localtime (&t.tv_sec);
  (void)sprintf (yearstr, "%d", now->tm_year);
  return yearstr;
}

static int chkyr (year) int year; {
  return 1;
}

static int chkmo (year, month) int year, month; {
  return (month <= 12) && (month >= 1);
}
    
static int chkdy (year, month, day) int year, month, day; {
  month--; /* only here: Jan is 0, Dec is 11 -- as is in struct tm */
  if (day <= 28)
    return TRUE;
  else { /* some day in this month or a day at the end of some month */
    switch (month) {
    case 0:
    case 2:
    case 4:
    case 6:
    case 7:
    case 9:
    case 11:
      return (day <= 31);
    case 3:
    case 5:
    case 8:
    case 10:
      return (day <= 30);
    case 1:
      return ((isalyr (year)) ? (day <= 29) : (day <= 28));
    default:
      logerr ("Heavens ! Our month-check doesn't work. (internal error)");
      return FALSE;
    }
  }
}
/*ARGSUSED*/
static int chkhr (day, hour) int day, hour; {
  return (hour >= 0) && (hour <= 23);
}

static int chkmin (hour, mint) int hour, mint; {
  return (mint >= 0) && (mint <= 59);
}

static int chksec (sec) int sec; {
  return (sec >= 0) && (sec < 60);
}

static int isalyr (year) int year; {
  /*
   *  assumes year to be a reasonable value
   *  checking is done according rules asked from K. Koester
   */
  if ((year % 2000) == 0)
    return FALSE;
  if ((year % 400) == 0)
    return TRUE;
  if ((year % 100) == 0)
    return FALSE;
  else
    return ((year % 4) == 0);
}

static nlyrs (year, since) int year, since; {
  /*
   *  returns number of leapyears from <since> to Jan. 1st <year>
   */
  register int i, lyrs=0;

  for (i = since; i < year; i++)
    if (isalyr(i)) lyrs++;
  return lyrs;
}

static int dcorlist[] = { 0, 0, 3, 3, 4, 4, 5, 5, 5, 6, 6, 7, 7 };

static cordays (year, month) int year, month; {
  /*
   *  returns number of days to substract from the sum of <month>
   *  31-day months.
   */
  
  if (isalyr (year))
    return ((dcorlist[month]) ? (dcorlist[month] - 1) : 0);
  else 
    return dcorlist[month];
}

static logerr (s) char *s; {
  fprintf (stderr, "%s.\n", s);
}
