/*LINTLIBRARY*/
/*$__copyright$ */
/*
 *	Shape/AtFS
 *
 *	afenviron.c -- communication with the UNIX-Environment
 *
 *	Author: Andreas Lampen, TU-Berlin (andy@coma.UUCP)
 *					  (andy@db0tui62.BITNET)
 *
 *	$Header: afenviron.c[1.20] Fri Apr 24 17:56:05 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *	af_garown -- get owner of archive file
 *      af_gmaxbpsize -- get max. number of files in binary pool
 *	af_getuid -- returns uid of user if from local host
 *	af_getgid -- returns gid of user if from local host
 *	af_afuser -- returns name and host of caller
 *      af_checkread -- check read permissions of AF-file
 *      af_checkperm -- check access permissions for AF-file
 *      af_gethostname -- get host name
 *      af_getdomain -- get domain name
 */

#include <stdio.h>

#include "afsys.h"
#include "atfs.h"
#include "afarchive.h"

#include <sys/file.h>

char  *malloc();


/*================================================================
 *	af_garown
 *================================================================*/

EXPORT Af_user *af_garown (archname, writeok, gid)
     char    *archname;
     bool    *writeok; /* out */
     Gid_t   *gid; /* out */
{
  register char *namptr;
  char          ardirname[MAXPATHLEN];
  struct stat   ibuf;

  /* build name of directory, where the archive is located */
  (void) strcpy (ardirname, archname);

  /* cut name */
  namptr = rindex (ardirname, '/');
  *namptr = '\0';

  *writeok = FALSE;
  if (stat (ardirname, &ibuf) == ERROR)
    return ((Af_user *)0);
  else
#ifdef STATISTICS
    _regfileaccess (ardirname);
#endif
    if (!access (ardirname, W_OK))
      *writeok = TRUE;

  *gid = ibuf.st_gid;
  return (af_afuser (ibuf.st_uid));
} /* af_garown */ 


/*================================================================
 *      af_gmaxbpsize -- get max. number of files in binary pool
 *================================================================*/

EXPORT int af_gmaxbpsize (name)
     /*ARGSUSED*/
     char *name; /* unused up to now */
{
  register char *envval;
  char          *getenv();

  if (envval = getenv (AF_ENVBPSIZE))
    return (atoi (envval));
  else
    return AF_MAXBPSIZE;
}

/*========================================================================
 *	af_getuid - returns uid of user if from local host
 *                  AF_ERROR if user is unknown
 *                  AF_REMOTE if user is not local
 *========================================================================*/

EXPORT Uid_t af_getuid (name, host, domain)
/*ARGSUSED*/
     char *name, *host, *domain;
{
  register struct passwd *pwent;

  if (name == (char *)0) /* in this case, name and host are null pointers */
    return ((Uid_t) ERROR);

  if (strcmp (af_getdomain(), NOTNIL(domain)))
    return ((Uid_t) AF_REMOTE);

  if ((pwent = getpwnam (name)) == (struct passwd *)0)
    FAIL ("getuid", "cannot get user ID", AF_EINTERNAL, (Uid_t) ERROR);

  return (pwent->pw_uid);
}


/*========================================================================
 *	af_getgid - returns gid of user if from local host
 *                  AF_ERROR if user is unknown
 *                  AF_REMOTE if user is not local
 *========================================================================*/

EXPORT Gid_t af_getgid (name, host, domain)
/*ARGSUSED*/
     char *name, *host, *domain;
{
  register struct passwd *pwent;

  if (name == (char *)0) /* in this case, name and host are null pointers */
    return ((Gid_t) ERROR);

  if (strcmp (af_getdomain(), NOTNIL(domain)))
    return ((Gid_t) AF_REMOTE);

  if ((pwent = getpwnam (name)) == (struct passwd *)0)
    FAIL ("getgid", "cannot get group ID", AF_EINTERNAL, (Gid_t) ERROR);

  return (pwent->pw_gid);
}


/*========================================================================
 *	af_afuser - returns name, host, and domain of caller
 *========================================================================*/

static Uid_t   calleruid;
static Af_user caller;
static bool    initcaller = FALSE;

EXPORT Af_user *af_afuser (uid)
     Uid_t uid;
{
  static Af_user result;
  register struct passwd *pwent;
 
  if (!initcaller) /* if caller struct is not yet initialized */
    {
      calleruid = geteuid();
      if ((pwent = getpwuid ((int) calleruid)) == (struct passwd *)0)
	SFAIL ("getuser", "", AF_EINVUSER, (Af_user *)0);
      (void) strcpy (caller.af_username, pwent->pw_name);
      (void) strcpy (caller.af_userhost, af_gethostname ()); 
      (void) strcpy (caller.af_userdomain, af_getdomain ()); 
      initcaller = TRUE;
    }
  if (uid == calleruid)
    return (&caller);

  if ((pwent = getpwuid ((int) uid)) == (struct passwd *)0)
    SFAIL ("getuser", "", AF_EINVUSER, (Af_user *)0);
  (void) strcpy (result.af_username, pwent->pw_name);
  (void) strcpy (result.af_userhost, af_gethostname ()); 
  (void) strcpy (result.af_userdomain, af_getdomain ()); 

  return (&result);
}

/*====================================================================
 *   af_checkread -- see if AF-file is readable
 *====================================================================*/

EXPORT af_checkread (key)
     Af_key *key;
{
  Uid_t        uid, auuid, ownuid;
  Gid_t        augid, owngid;
  register int i, ngroups;
  int          gidset[NGROUPS];

  if ((VATTR(key).af_mode & 0004) == 0004) /* readable for world */
    return (AF_OK);

  if ((VATTR(key).af_mode & 0040) == 0040) /* readable for group */
    {
      /* this then part is BSD specific */
      ngroups = getgroups (NGROUPS, gidset);
      augid = af_getgid (VATTR(key).af_auname, VATTR(key).af_auhost, VATTR(key).af_audomain);
      owngid = af_getgid (CATTR(key).af_ownname, CATTR(key).af_ownhost, CATTR(key).af_owndomain);
      for (i=0; i < ngroups; i++)
	{
	  if ((augid == (Gid_t)gidset[i]) || (owngid == (Gid_t)gidset[i]))
	    return (AF_OK);
	}
    }

  if ((VATTR(key).af_mode & 0400) == 0400) /* readable by owner */
    {
      uid = geteuid();
      auuid = af_getuid (VATTR(key).af_auname, VATTR(key).af_auhost, VATTR(key).af_audomain);
      ownuid = af_getuid (CATTR(key).af_ownname, CATTR(key).af_ownhost, CATTR(key).af_owndomain);
      if ((auuid == uid) || (ownuid == uid))
	return (AF_OK);
    }

  return (ERROR);
}


/*====================================================================
 *   af_checkperm -- check access permissions for AF-file
 *====================================================================*/

EXPORT af_checkperm (key, mode)
     Af_key *key;
     int    mode;
{
  Uid_t uid = geteuid(), lockeruid;
  bool ok = FALSE;

  if (mode & AF_OWNER)
    {
      if (uid == af_getuid (CATTR(key).af_ownname, CATTR(key).af_ownhost, CATTR(key).af_owndomain))
	ok = TRUE;
    }
  if (!ok && (mode & AF_LOCKHOLDER))
    {
      if ((lockeruid = af_getuid (VATTR(key).af_lckname, VATTR(key).af_lckhost, VATTR(key).af_lckdomain)) == uid)
	ok = TRUE;
      else
	{
	  /* if object is locked by someone else */
	  if (lockeruid != (Uid_t) ERROR) 
	    goto exit;
	}
    }
  if (!ok && (mode & AF_AUTHOR))
    {
      if (uid == af_getuid (VATTR(key).af_auname, VATTR(key).af_auhost, VATTR(key).af_audomain))
	ok = TRUE;
    }
  if (!ok && (mode & AF_WORLD))
    {
      ok = TRUE;
    }
  
 exit:
  /* if access is not ok, or AtFS subdir is not writable */
  if (!ok)
    SFAIL ("checkperm", "", AF_EACCES, ERROR);
  if (!(key->af_ldes->af_extent & AF_UXWRITE))
    SFAIL ("checkperm", "", AF_ENOATFSDIR, ERROR);
  return (AF_OK);
}

/*================================================================
 *	af_gethostname
 *================================================================*/

static char *hostsym = (char *)0;

EXPORT char *af_gethostname ()
{
  char hostname[MAXHOSTNAMELEN];

  if (!hostsym)
    {
      (void) gethostname (hostname, MAXHOSTNAMELEN);
      hostsym = af_entersym (hostname);
    }
  return (hostsym);
} /* af_uchmod */

/*================================================================
 *	af_getdomain
 *================================================================*/

static char *domainsym = (char *)0;

EXPORT char *af_getdomain ()
{
  char domain[MAXDOMAIN+1];

  domain[0] = '\0';

  if (!domainsym)
    {
      getdomainname (domain, MAXDOMAIN);
      if (domain[0] == '\0')
	domainsym = af_gethostname ();
      else
	{
	  domain[MAXDOMAIN] = '\0';
	  if (domain[0] == '.')
	    domainsym = af_entersym (&domain[1]);
	  else
	    domainsym = af_entersym (domain);
	}
    }
  return (domainsym);
} /* af_uchmod */
