/*LINTLIBRARY*/
/* Copyright (C) 1989,1990,1991,1992 by
	Wilfried Koch, Andreas Lampen, Axel Mahler, Juergen Nickelsen,
	Wolfgang Obst and Ulrich Pralle
 
 This file is part of shapeTools.

 This software is published in the hope that it will be useful, but
 WITHOUT ANY WARRANTY for any part of this software to work correctly
 or as described in the manuals. See the ShapeTools Public License
 for details.

 Permission is granted to use, copy, modify, or distribute any part of
 this software but only under the conditions described in the ShapeTools 
 Public License. A copy of this license is supposed to have been given
 to you along with shapeTools in a file named LICENSE. Among other
 things, this copyright notice and the Public License must be
 preserved on all copies.
 */
/*
 *	Shape/AtFS
 *
 *	afdeltaproc.c -- generate and merge deltas
 *
 *	Author: Andreas Lampen, TU-Berlin (andy@coma.UUCP
 *					   andy@db0tui62.BITNET)
 *
 *	$Header: afdeltaproc.c[1.8] Fri Jan 31 18:05:08 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *      afReconsData -- reconstruct data from delta
 *      afMakeDelta  -- generate delta
 */

#include <stdio.h>

#include "atfs.h"
#include "afsys.h"
#include "afdelta.h"

char *malloc();
char *calloc();

LOCAL off_t getCmd (strPtr, cmdPtr)
     char          *strPtr;
     struct ed_cmd *cmdPtr;
{
  off_t i=1;

  if (*strPtr == '\0')
    cmdPtr->cmd = MV;
  else
    cmdPtr->cmd = AD;

  cmdPtr->length = 0;
  while (strPtr[i])
    {
      cmdPtr->length = cmdPtr->length << 3;
      cmdPtr->length |= ((strPtr[i++] - 48) & 07);
    }
  return (i+1);
}

LOCAL off_t getIndex (strPtr, indexPtr)
     char  *strPtr;
     off_t *indexPtr;
{
  off_t i=0;

  *indexPtr = 0;
  while (strPtr[i])
    {
      *indexPtr = *indexPtr << 3;
      *indexPtr |= ((strPtr[i++] - 48) & 07);
    }

  if (i==0) i++; /* correction for zero value */

  return (i+1);
}

LOCAL off_t putCmd (command, length, strPtr)
     int   command;
     off_t length;
     char  *strPtr;
{
  off_t i=1, j=0;
  unsigned tmpLen;	/* This should be "unsigned off_t" (and if
		         * "off_t" were not a typedef, we could do just
			 * that) */
  char  tmpStr[16];

  switch (command)
    {
    case MV:
      strPtr[0] = '\0';
      break;
    case AD:
      strPtr[0] = '\001';
      break;
    default:
      strPtr[0] = '?';
    }
  
  tmpLen = length;	/* The compiler should warn about any narrowing */
  while(tmpLen)
    {
      tmpStr[j++] = (tmpLen & 07) + 48;
      tmpLen >>= 3;	/* The result would be implementation-defined
			 * if "tmpLen" were negative */
    }

  while (j)
    strPtr[i++] = tmpStr[--j];

  if (!length)
    strPtr[i++] = '\0';
  strPtr[i++] = '\0';

  return (i);
}

LOCAL off_t putIndex (index, strPtr)
     off_t index;
     char  *strPtr;
{
  off_t i=0, j=0;
  unsigned tmpIdx;	/* This should be "unsigned off_t" (and if
		         * "off_t" were not a typedef, we could do just
			 * that) */
  char  tmpStr[16];

  tmpIdx = index;	/* The compiler should warn about any narrowing */
  while(tmpIdx)
    {
      tmpStr[j++] = (tmpIdx & 07) + 48;
      tmpIdx >>= 3;	/* The result would be implementation-defined
			 * if "tmpIdx" were negative */
    }

  while (j)
    strPtr[i++] = tmpStr[--j];

  if (!index)
    strPtr[i++] = '\0';
  strPtr[i++] = '\0';

  return (i);
}

/*===================================================================
 * afConvertDelta -- convert Deltas
 *    For sake of portability of archive files, numbers occuring
 *    in deltas ("command", "length" and "index") are stored in their
 *    ASCII representation rather than binary.
 *
 *===================================================================*/

EXPORT char *afConvertDelta (deltaStr, deltaSize, newDeltaSize)
     char  *deltaStr;
     off_t deltaSize, *newDeltaSize;
{
  struct ed_cmd cmd;
  off_t index;
  char *deltaPtr = deltaStr, *deltaEnd = deltaStr + deltaSize;
  char *newDelta, *newDeltaPtr;

  if (deltaSize < 16)
    deltaSize = 16;

  if ((newDelta = malloc ((unsigned) deltaSize*2)) == (char *)0)
    FAIL ("ConvertDelta", "malloc", AF_ESYSERR, (char *)0);
  newDeltaPtr = newDelta;
  /* newDeltaEnd = newDelta + deltaSize*2; */

  while (deltaPtr < deltaEnd)
    {
      bcopy (deltaPtr, (char *)&cmd, sizeof (cmd));
      deltaPtr += sizeof (cmd);

      newDeltaPtr += putCmd (cmd.cmd, cmd.length, newDeltaPtr); 

      switch (cmd.cmd)
	{
	case MV:
	  bcopy (deltaPtr, (char*)&index, sizeof (index));
	  deltaPtr += sizeof (index);
	  newDeltaPtr += putIndex (index, newDeltaPtr);
	  break;
	case AD:
	  bcopy (deltaPtr, newDeltaPtr, (Size_t) cmd.length);
	  deltaPtr += cmd.length;
	  newDeltaPtr += cmd.length;
	  break;
	}
    }
  *newDeltaSize = newDeltaPtr - newDelta;
  return (newDelta);
}

/*===================================================================
 *      afReconsData -- reconstruct data from delta
 *
 *===================================================================*/

EXPORT afReconsData (srcStr, deltaStr, srcSize, deltaSize, targetfn)
     char  *srcStr, *deltaStr;
     off_t srcSize, deltaSize;
     char  *targetfn;
{
  FILE *ftarg;
  struct ed_cmd cmd;
  off_t index;
  char *deltaPtr = deltaStr, *deltaEnd = deltaStr + deltaSize;

  if ((ftarg = fopen(targetfn, "w")) == (FILE *)0)
    return (ERROR);

  if (deltaSize == (off_t) 0)
    {
      (void) fwrite (srcStr, sizeof(char), (Size_t)srcSize, ftarg);
      (void) fclose (ftarg);
      return (AF_OK);
     }

  while (deltaPtr < deltaEnd)
    {
      deltaPtr += getCmd (deltaPtr, &cmd);

      switch (cmd.cmd)
	{
	case MV:
	  deltaPtr += getIndex (deltaPtr, &index);
	  (void) fwrite (srcStr + index, sizeof(char), (Size_t)cmd.length, ftarg);
	  break;
	case AD:
	  (void) fwrite (deltaPtr, sizeof(char), (Size_t)cmd.length, ftarg);
	  deltaPtr += cmd.length;
	  break;
	default:
	  (void) fclose(ftarg);
	  return (ERROR);
	}
    }
  (void) fclose(ftarg);
  return(AF_OK);
}
      
/*===================================================================
 *      afMakeDelta  -- generate delta
 *
 *===================================================================*/

LOCAL FILE *fdelta = (FILE *)0;

/*============================
 * manage suffix tree
 *
 *============================*/

LOCAL struct suffixes *suff_array = (struct suffixes *)0;
LOCAL struct indices *FreePtr;

#define HASH(STRING,OFFSET)\
        STRING[OFFSET] +\
        (STRING[OFFSET + 4] << 1) +\
        (STRING[OFFSET + 7] << 2) +\
        (STRING[OFFSET + 11] << 3) +\
        (STRING[OFFSET + 15] << 4) +\
        (STRING[OFFSET + 17] << 5)

LOCAL build_suffix_tree (string, length)
     register char *string;
     off_t         length;
{
  register struct indices *current;
  register off_t i = 0;
  register int hash;
  struct indices *LargeMemPtr;

  suff_array = (struct suffixes *) malloc(TREESIZE * sizeof (struct suffixes));
  if (suff_array == (struct suffixes *)0)
    return (ERROR);

  bzero((char *)suff_array, TREESIZE * sizeof(struct suffixes));

  if ((LargeMemPtr = (struct indices *) calloc ((unsigned) length, sizeof (struct indices))) == 0)
    return (ERROR);

  FreePtr = LargeMemPtr;

  hash = abs (HASH (string, 0));

  suff_array[hash].next = LargeMemPtr;
  LargeMemPtr++;

  suff_array[hash].last = suff_array[hash].next;
  suff_array[hash].next->index = 0;
  suff_array[hash].next->next = (struct indices *)0;

  for (i = 1; i < (length - (MINBLENG - 1)); i++)
    {
      hash = abs (HASH (string,i));
      if (suff_array[hash].last != (struct indices *)0 )
	current = suff_array[hash].last->next = LargeMemPtr;
      else
	current = suff_array[hash].next = LargeMemPtr;

      LargeMemPtr++;
      current->index = i;
      current->next = (struct indices *) 0;
      suff_array[hash].last = current;
    }
  return (AF_OK);
}

LOCAL void free_suffix_tree ()
{
  if (FreePtr)
    free ((char *)FreePtr);
  if (suff_array)
    free ((char *)suff_array);
}

LOCAL void find_max_bm(t_str, s_str, targetSize, srcSize, t_offset, s_offset, leng)
     register char *t_str, *s_str;
     off_t targetSize, srcSize;
     off_t t_offset;
     off_t *s_offset, *leng;
{
  register struct indices *current;
  register int i,j;
  register int hash;
  int off;
  int max;
  
  hash = abs (HASH (t_str, t_offset));
  if (suff_array[hash].next == (struct indices *)0)
    *leng = 0;
  else
    {
      max = 0;
      off = 0;
      current = suff_array[hash].next;
      while (current != (struct indices *)0)
	{
	  i = current->index;
	  j = t_offset;
	  while ((i < srcSize) && (j < targetSize) && 
		 (t_str[j] == s_str[i])) 
	    {
	      j++;
	      i++;
	    }
	  if ((i - current->index > max) &&
	      (t_str[t_offset] == s_str[current->index]) &&
	      (t_str[t_offset + 1] == s_str[current->index + 1]))
	    {
	      max = i - current->index;
	      off = current->index;
	    }
	  current = current->next;
	  if (current != (struct indices *)0)
	    {
	      while (((t_offset + max) < targetSize) &&
		     ((current->index + max) < srcSize) &&
		     (t_str[t_offset + max] !=
		      s_str[current->index + max]) &&
		     (current->next != (struct indices *)0))
		current = current->next;
	    }
	}
      *s_offset = off;
    if(max >= MINBLENG )
      *leng = max;
    else
      *leng = 0;
    }
  return;
}


LOCAL void write_cmd (command, length, index, targetString)
     int   command;
     off_t length;
     off_t index;
     char  *targetString;
{
  char cmdStr[MAXCMDLEN];
  int  cmdLength;

  cmdLength = putCmd (command, length, cmdStr);
  switch (command)
    {
    case MV:
      cmdLength += putIndex (index, &cmdStr[cmdLength]);
      (void) fwrite (cmdStr, sizeof(char), (Size_t)cmdLength, fdelta);
      break;
    case AD:
      (void) fwrite (cmdStr, sizeof(char), cmdLength, fdelta);
      (void) fwrite (targetString+index, sizeof(char), (Size_t)length, fdelta);
      break;
    }
}


EXPORT afMakeDelta (srcStr, targetStr, srcSize, targetSize, deltafn)
     char  *srcStr, *targetStr;
     off_t srcSize, targetSize;
     char  *deltafn;
{
  register off_t trg_index = 0;
  off_t          src_index, matchlength;
  register off_t nomtch_trg_index = 0, nomtchlength = 0;
  int nomatch = FALSE;

  fdelta  = fopen (deltafn, "w");

  /* if source and target are identical */
  if (srcSize == targetSize)
    {
      if ((bcmp (srcStr, targetStr, (int) srcSize)) == 0)
	{
	  (void) fclose (fdelta);
	  return (AF_OK);
	}
    }

  if ((srcSize <= MINBLENG) || (targetSize <= MINBLENG))
    {
      write_cmd (AD, targetSize, (off_t)0, targetStr);
      (void) fclose (fdelta);
      return(AF_OK);
    }

  if (build_suffix_tree (srcStr, srcSize) == ERROR)
    {
      (void) fclose (fdelta);
      return(ERROR);
    }

  while (trg_index < (targetSize - (MINBLENG - 1)))
    {
      find_max_bm (targetStr, srcStr, targetSize, srcSize, trg_index, &src_index, &matchlength);
      if (matchlength > 0)
	{
	  if (nomatch)
	    {
	      write_cmd (AD, nomtchlength, nomtch_trg_index, targetStr);
	      nomtch_trg_index = 0;
	      nomtchlength = 0;
	      nomatch = FALSE;
	    }
	  write_cmd (MV, matchlength, src_index, targetStr);
	  trg_index = trg_index + matchlength;
	}
      else
	{
	  if (nomatch)
	    nomtchlength++;
	  else
	    {
	      nomatch = TRUE;
	      nomtch_trg_index = trg_index;
	      nomtchlength = 1;
	    }
	  trg_index++;
	  if (trg_index >= targetSize)
	    write_cmd (AD, nomtchlength, nomtch_trg_index, targetStr);
	}
    }
  if (trg_index <= (targetSize - 1))
    {
      if (nomatch)
	write_cmd (AD, (nomtchlength + (targetSize - trg_index)), nomtch_trg_index, targetStr);
      else
	write_cmd (AD, (targetSize - trg_index), trg_index, targetStr);
    }
  (void) fclose (fdelta);

  return(AF_OK);
}
