/*$__copyright$ */
/*
 *	Shape/AtFS
 *
 *	afcache.c -- realize caching for archives in core
 *
 *	Author: Andreas Lampen, TU-Berlin (andy@coma.UUCP)
 *					  (andy@db0tui62.BITNET)
 *
 *	$Header: afcache.c[1.9] Wed Apr 22 20:31:05 1992 andy@cs.tu-berlin.de accessed $
 *
 *	EXPORT:
 *      afInitList -- initialize list of ASOs
 *	afDetachList -- remove list 
 *      afInitBpList -- initialize list of ASOs for Binary pool
 *      afDetBpool -- remove binary pool
 *	afTestList -- see if list is in core
 *	afRefreshList -- reread list if necessary
 *      afListSpace -- get space for new list
 */

#include <stdio.h>

#include "afsys.h"
#include "atfs.h"
#include "afarchive.h"

#ifdef MEMDEBUG
extern FILE *memprot;
#endif
extern int af_errno;

Uid_t   geteuid();
char    *malloc();

/*==========================================================================
 * List of list-descriptors + hash table for faster access
 *
 *==========================================================================*/

EXPORT Af_revlist *af_lists = (Af_revlist *)0; /* base address of all
						  list descriptors */
EXPORT Af_revlist *archFreelist = (Af_revlist *)0; /* beginning of freelist */

EXPORT int af_listmem = 0;                     /* # of bytes used for list */

LOCAL int archtabSize;

EXPORT int nameConflict = FALSE;

/*=========================================================================
 *     Hash stuff
 *
 *=========================================================================*/

LOCAL bool hinit = FALSE; /* indicate if hashtable is yet initialized */

typedef struct AfArnt AfArchent;

struct AfArnt { char       *symbol;
		Af_revlist *list;
		AfArchent  *next;
	      };

LOCAL AfArchent *afArchhash;

LOCAL char hashword[MAXPATHLEN], *hashsym;

/*=====================================================================
 * afHasharch -- put archive into hashtable
 *
 *=====================================================================*/ 

LOCAL afHasharch (symbol, list)
     char       *symbol;
     Af_revlist *list;
{
  register int where;
  AfArchent    *new, *curptr;

  where = afHashval (symbol, AF_MAXLISTS);
  if (!afArchhash[where].symbol)  /* entry is not used yet */
    {
      afArchhash[where].symbol = symbol;
      afArchhash[where].list = list;
      return (AF_OK);
    }
  else /* hash collision! */
    {
      if ((new = (AfArchent *)malloc ((unsigned) sizeof (AfArchent))) 
	  == (AfArchent *)0)
	FAIL ("Hasharch", "malloc", AF_ESYSERR, ERROR);
      archtabSize += sizeof (AfArchent);
#ifdef MEMDEBUG
      fprintf (memprot, "%x(archtab)-AL %d bytes\t\ttotal size: %d\n",
	       new, sizeof (AfArchent), archtabSize);
#endif
      new->symbol = symbol;
      new->list = list;
      new->next = NULL;
      curptr = &afArchhash[where];
      while (curptr->next)
	curptr = curptr->next;
      curptr->next = new;
      return (AF_OK);
    }
}

/*=====================================================================
 * afLookuparch -- search archive in hashtable
 *
 *=====================================================================*/ 

LOCAL Af_revlist *afLookuparch (symbol, path)
     char *symbol, *path;
{
  int where;
  AfArchent *entry;

  /* lookup symbol */
  where = afHashval (symbol, AF_MAXLISTS);
  if (afArchhash[where].symbol)  /* found something */
    {
      if ((afArchhash[where].symbol == symbol) &&
	  (afArchhash[where].list->af_cattrs.af_syspath == path))
	return (afArchhash[where].list);
      else /* maybe it is somewhere down the gully */ 
	{
	  entry = &afArchhash[where];
	  while (entry->next)
	    {
	      entry = entry->next;
	      if ((entry->symbol == symbol) &&
		  (entry->list->af_cattrs.af_syspath == path))
		return (entry->list);
	    }
	}
    }
  /* nothing found */
  return ((Af_revlist *)0);
}

/*=========================================================================
 *     afInitList
 *
 *=========================================================================*/

EXPORT Af_revlist *afInitList (pathsym, namesym, typesym)
     char *pathsym, *namesym, *typesym;
{
  register   int      i;
  register Af_revlist *list, *oldlist;
  Af_revlist          *afTestList();

  /* init hashtable if this is not yet done */
  if (!hinit)
    {
      archtabSize = AF_MAXLISTS * sizeof (AfArchent);
      if ((afArchhash = (AfArchent *)malloc ((unsigned) archtabSize)) == NULL)
	FAIL ("InitList", "malloc", AF_ESYSERR, (Af_revlist *)0);
#ifdef MEMDEBUG
      fprintf (memprot, "%x(archtab)-AL %d bytes\t\ttotal size: %d\n",
	       afArchhash, archtabSize, archtabSize);
#endif
      /* set hashlist all zeros */
      bzero ((char *)afArchhash, archtabSize);
      hinit = TRUE;
    }

#if (MAXNAMLEN < 128)
  /* perhaps the name should be shortened here, if it's too long */
#endif

  /* if list is already loaded */
  if (oldlist = afTestList (pathsym, namesym, typesym))
    {
      oldlist->af_extent |= AF_LISTBUSY;
      if (afReadAttrs (oldlist) == ERROR)
	{
	  oldlist->af_extent &= ~AF_LISTBUSY;
	  return ((Af_revlist *)0);
	}
      oldlist->af_extent &= ~AF_LISTBUSY;
      return (oldlist);
    }

  /* if there are no more descriptors available - allocate new space */
  if (archFreelist == (Af_revlist *)0)
    {
      if ((archFreelist = (Af_revlist *) malloc ((unsigned) (AF_LISTSEG * sizeof (Af_revlist)))) == (Af_revlist *)0)
	FAIL ("InitList", "malloc(i)", AF_ESYSERR, (Af_revlist *)0);
      archtabSize += AF_LISTSEG * sizeof (Af_revlist);
#ifdef MEMDEBUG
      fprintf (memprot, "%x(archtab)-AL %d bytes\t\ttotal size: %d\n",
	       archFreelist, AF_LISTSEG * sizeof (Af_revlist), archtabSize);
#endif

      /* build up new freelist */
      for (i=1; i<AF_LISTSEG; i++)
	archFreelist[i-1].af_next = &(archFreelist[i]);
      archFreelist[AF_LISTSEG-1].af_next = (Af_revlist *)0;
    }

  list = archFreelist;
  archFreelist = archFreelist->af_next;
  bzero ((char *)list, sizeof (*list));

  list->af_extent |= AF_LISTBUSY;
  list->af_mem = (char *)0;
  list->af_cattrs.af_host = af_gethostname ();
  list->af_cattrs.af_syspath = pathsym;
  list->af_cattrs.af_globname = namesym;
  list->af_cattrs.af_globtype = typesym;
  list->af_lastmod = (time_t) 0;

  if (afReadAttrs (list) == ERROR)
    {
      list->af_extent &= ~AF_LISTBUSY;
      list->af_next = archFreelist;
      archFreelist = list;
      return ((Af_revlist *)0);
    }
  list->af_extent &= ~AF_LISTBUSY;

  /* add list to chain */
  list->af_next = af_lists;
  af_lists = list;

  /* add list to hashtable */
  (void) afHasharch (hashsym, list);

  return (list);
}


/*==========================================================================
 *	afDetachList -- detach list data
 *
 *==========================================================================*/

EXPORT afDetachList (list)
     Af_revlist *list;
{
  if (nameConflict)
    {
      nameConflict = FALSE;
      return (AF_OK);
    }

  /* free all allocated memory */
  af_frmemlist (list);

  list->af_listlen = 0;
  list->af_nrevs = 0;
  list->af_datasize = 0;
  list->af_extent &= ~AF_SEGMASK;
  list->af_list = (Af_vattrs *)0;

  return (AF_OK);
}

/*==========================================================================
 *	definition of derived object cache list
 *      plus hashtable for faster access
 *
 *==========================================================================*/

     /* base address of all derived object cache list descriptors */
LOCAL Af_revlist *bplists = (Af_revlist *)0;  

/*=========================================================================
 *     afInitBpList
 *
 *=========================================================================*/

EXPORT Af_revlist *afInitBpList (pathsym)
     char *pathsym;
{
  register    int i;
  Af_revlist  *list, *oldlist;
  char        *arname;
  Af_user     *owner, *af_garown();
  bool        writeok;


  /* init hashtable if it is not yet done */
  if (!hinit)
    {
      archtabSize = AF_MAXLISTS * sizeof (AfArchent);
      if ((afArchhash = (AfArchent *)malloc ((unsigned) archtabSize)) == NULL)
	FAIL ("InitBpList", "malloc", AF_ESYSERR, (Af_revlist *)0);
#ifdef MEMDEBUG
      fprintf (memprot, "%x(archtab)-AL %d bytes\t\ttotal size: %d\n",
	       afArchhash, archtabSize, archtabSize);
#endif
      /* set hashlist all zeros */
      bzero ((char *)afArchhash, archtabSize);
      hinit = TRUE;
    }

  /* if there are open archives check if desired archive is loaded yet */
  if (bplists != (Af_revlist *)0)
    {
      if (oldlist = afLookuparch (pathsym, pathsym))
	{
	  oldlist->af_extent |= AF_LISTBUSY;
	  if (afRBpList (oldlist) == ERROR)
	    {
	      oldlist->af_extent &= ~AF_LISTBUSY;
	      return ((Af_revlist *)0);
	    }
	  oldlist->af_extent &= ~AF_LISTBUSY;
	  return (oldlist);
	}
    }
  
  /* check if AtFS subdirectory exists */
  if ((arname = af_gbpname (pathsym)) == (char *)0)
    SFAIL ("initlist", "", AF_ENOATFSDIR, (Af_revlist *)0);

  /* if there are no more descriptors available - allocate new space */
  if (archFreelist == (Af_revlist *)0)
    {
      if ((archFreelist = (Af_revlist *) malloc ((unsigned) (AF_LISTSEG * sizeof (Af_revlist)))) == (Af_revlist *)0)
	FAIL ("InitBpList", "malloc(i)", AF_ESYSERR, (Af_revlist *)0);
      archtabSize += AF_LISTSEG * sizeof (Af_revlist);
#ifdef MEMDEBUG
      fprintf (memprot, "%x(archtab)-AL %d bytes\t\ttotal size: %d\n",
	       archFreelist, AF_LISTSEG * sizeof (Af_revlist), archtabSize);
#endif

      /* build up new freelist */
      for (i=1; i<AF_LISTSEG; i++)
	archFreelist[i-1].af_next = &(archFreelist[i]);
      archFreelist[AF_LISTSEG-1].af_next = (Af_revlist *)0;
    }

  list = archFreelist;
  archFreelist = archFreelist->af_next;
  bzero ((char *)list, sizeof (*list));

  if ((owner = af_garown (arname, &writeok, &list->af_owngid)) == (Af_user *)0)
    list->af_owngid = getegid();

  list->af_extent |= AF_LISTBUSY;
  list->af_mem = (char *)0;
  list->af_cattrs.af_host = af_gethostname ();
  list->af_cattrs.af_syspath = pathsym;
  list->af_arfilename = arname;
  list->af_busyfilename = af_entersym ("");

  if (afRBpList (list) == ERROR)
    {
      list->af_extent &= ~AF_LISTBUSY;
      list->af_next = archFreelist;
      archFreelist = list;
      return ((Af_revlist *)0);
    }
  list->af_extent &= ~AF_LISTBUSY;

  /* add list to chain */
  list->af_next = bplists;
  bplists = list;

  /* add list to hashtable */
  (void) afHasharch (pathsym, list);

  return (list);
}

/*==========================================================================
 *	afDetBpool
 *
 *==========================================================================*/

EXPORT afDetBpool (list)
     Af_revlist *list;
{
  /* free all allocated memory */
  af_frmemlist (list);

  list->af_listlen = 0;
  list->af_datasize = 0;
  list->af_extent &= ~AF_COMPLETE;
  list->af_list = (Af_vattrs *)0;

  return (AF_OK);
}

/*==========================================================================
 *	afTestList -- see if list is in core
 *
 *==========================================================================*/

EXPORT Af_revlist *afTestList (pathsym, namesym, typesym)
     char *pathsym, *namesym, *typesym;
{
  if (!hinit)
    return ((Af_revlist *)0);

  (void) strcpy (hashword, namesym);
  if (typesym)
    if (typesym[0])
      {
	(void) strcat (hashword, ".");
	(void) strcat (hashword, typesym);
      }
  hashsym = af_entersym (hashword);

  /* if there are open archives check if desired archive is loaded yet */
  if (af_lists != (Af_revlist *)0)
    return (afLookuparch (hashsym, pathsym));

  return ((Af_revlist *)0);
}

/*==========================================================================
 *	afRefreshList -- re-read list (if necessary)
 *
 *==========================================================================*/

EXPORT afRefreshList (list, extent)
     Af_revlist *list;
     int extent;
{
  int retcode = AF_OK;

  list->af_extent |= AF_LISTBUSY;

  /* if list is a derived object pool */
  if ((list->af_extent & AF_BPOOL) == AF_BPOOL)
    retcode = afRBpList (list);
  else
    {
      if (extent & AF_ATTRS)
	retcode = afReadAttrs (list);
      if (extent & AF_DATA)
	retcode = afReadData (list);
    }

  list->af_extent &= ~AF_LISTBUSY;
  return (retcode);
}
/*==========================================================================
 *	afListSpace -- get space for new list
 *
 *==========================================================================*/

EXPORT afListSpace (size)
     int size;
{
  register Af_revlist *listptr;

  if (af_listmem + size < AF_MAXMEM)
    return (AF_OK);

  /* search oldest unused revision list */ /* not yet implemented */
  /* search first unused revision list and free memory */

  listptr = af_lists;
  while (listptr)
    {
      if (!(listptr->af_extent & AF_LISTBUSY) && (listptr->af_access < 1)
	  && (listptr->af_listlen > 0))
	(void) afDetachList (listptr);
      if (af_listmem + size < AF_MAXMEM)
	return (AF_OK);
      else
	listptr = listptr->af_next;
    }

  listptr = bplists;
  while (listptr)
    {
      if (!(listptr->af_extent & AF_LISTBUSY) && (listptr->af_access < 1)
	  && (listptr->af_listlen > 0))
	(void) afDetBpool (listptr);
      if (af_listmem + size < AF_MAXMEM)
	return (AF_OK);
      else
	listptr = listptr->af_next;
    }

  /* unable to free enough memory */
  return (ERROR);
}
