 /*
  * Khoros: $Id: ldfiltbp.c,v 1.3 1992/03/20 23:25:49 dkhoros Exp $
  */

#if !defined(lint) && !defined(SABER)
static char rcsid[] = "Khoros: $Id: ldfiltbp.c,v 1.3 1992/03/20 23:25:49 dkhoros Exp $";
#endif

 /*
  * $Log: ldfiltbp.c,v $
 * Revision 1.3  1992/03/20  23:25:49  dkhoros
 * VirtualPatch5
 *
  */

/*
 *----------------------------------------------------------------------
 *
 * Copyright 1992, University of New Mexico.  All rights reserved.
 * Permission to copy and modify this software and its documen-
 * tation only for internal use in your organization is hereby
 * granted, provided that this notice is retained thereon and
 * on all copies.  UNM makes no representations as to the sui-
 * tability and operability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 * 
 * UNM DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FIT-
 * NESS.  IN NO EVENT SHALL UNM BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY OTHER DAMAGES WHAT-
 * SOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER
 * IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PER-
 * FORMANCE OF THIS SOFTWARE.
 * 
 * No other rights, including, for example, the right to redis-
 * tribute this software and its documentation or the right to
 * prepare derivative works, are granted unless specifically
 * provided in a separate license agreement.
 *---------------------------------------------------------------------
 */

#include "unmcopyright.h"        /* Copyright 1992 by UNM */

/*>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>  <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<
 >>>>
 >>>>         File Name: ldfiltbp.c
 >>>>
 >>>>      Program Name: dfiltbp
 >>>>
 >>>> Date Last Updated: Mon Mar  9 20:16:27 1992 
 >>>>
 >>>>          Routines: ldfiltbp - the library call for dfiltbp
 >>>>
 >>>>>>>>>>>>>>>>>>>>>>>>>>>>>>   <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<*/


#include "vinclude.h"


/* -library_includes */
/* second order filters: six coefficients */
#define NCOEF 6
#define ARCSINH(x) (log((double)x + sqrt((double)x * (double)x + 1.0)))
#define ARCCOSH(x) (log((double)x + sqrt((double)x * (double)x - 1.0)))
#define MAXSTAGES 100
#include "vpoly.h"
static int chbiipass(),chbipass(),btwpass();

/* cleanup macros */
#define BTW_CLEANUP { \
      if(sreal!=NULL)free((char *)sreal); \
      if(simag!=NULL)free((char *)simag); \
      if(pr!=NULL)free((char *)pr); \
      if(pi!=NULL)free((char *)pi); \
      if(as!=NULL)free((char *)as); \
      if(bs!=NULL)free((char *)bs); \
      if(cs!=NULL)free((char *)cs); \
      if(ds!=NULL)free((char *)ds); \
      if(es!=NULL)free((char *)es); \
    }

/* -library_includes_end */


/****************************************************************
*
* Routine Name: ldfiltbp - library call for dfiltbp
*
* Purpose:
*    
*    designs bandpass filters
*    
*    

* Input:
*    
*    f1             lower digital rejection frequency in hertz.
*    
*    f2             lower digital cutoff frequency in hertz.
*    
*    f3             upper digital cutoff frequency in hertz.
*    
*    f4             upper digital rejection frequency in hertz.
*    
*    tolc           cutoff  tolerance.   specifies  the  gain  at  the
*                   passband  edge.   Also determines the magnitude of
*                   the ripple for Chebychev I filters.
*    
*    tolr           rejection tolerance.  specifies the  gain  at  the
*                   stopband  edge.   Also determines the magnitude of
*                   the ripple for Chebychev II filters.
*    
*    sfreq          sampling frequency of the system.
*    
*    class          specifies the filter class  to  be  used.   0  for
*                   Butterworth,  1  for Chebychev I, and 2 for Cheby-
*                   chev II.
*    
*    

* Output:
*    
*    poly           polynomial structure that will contain the  filter
*                   transfer function at exit.
*    
*    Return Value:  1 on success, 0 on failure.
*    
*    

*
* Written By: Jeremy Worley
*    
*    Jeremy Worley 09 Mar 1992 19:54 MST
*              Added checks of the return values of some internal rou-
*              tines.
*    
*    

****************************************************************/


/* -library_def */
int ldfiltbp(poly,f1,f2,f3,f4,tolc,tolr,sfreq,class)
    struct poly_struct *poly;
    int class;
    float f1,f2,f3,f4,tolc,tolr,sfreq;
/* -library_def_end */

/* -library_code */
{
    char *program = "ldfiltbp";
    float *a,*b,*c,*d,*e,wc,wr,w1,w2,w3,w4,epsilon,lambda;
    int i,offset=0,stages;
/*
** check for user stupidity
*/

    if(poly==NULL){
       fprintf(stderr,"%s:  polynomial structure must not be NULL\n",program);
       return(0);
    }

    if(f1>=f2 || f2>=f3 || f3>=f4){
       fprintf(stderr,"%s:  critical frequencies must progress \n",program);
       fprintf(stderr,"smallest to largest as follows: f1 f2 f f4.\n");
       return(0);
    }

    if(f1>0.5*sfreq || f2>0.5*sfreq || f3>0.5*sfreq || f4>0.5*sfreq){
       fprintf(stderr,"%s:  critical frequencies for the filter ",program);
       fprintf(stderr,"must be less than one half of the sampling \n");
       fprintf(stderr,"frequency.\n");
       return(0);
    }

    if(tolr>=1.0 || tolr<=0.0){
       fprintf(stderr,"%s:  rejection tolerance is out of range.\n",program);
       fprintf(stderr,"legal range is between 0.0 and 1.0\n");
       return(0);
    }

    if(tolc>=1.0 || tolc<=0.0){
       fprintf(stderr,"%s:  cutoff tolerance is out of range.\n",program);
       fprintf(stderr,"legal range is between 0.0 and 1.0\n");
       return(0);
    }

    if(tolr>=tolc){
       fprintf(stderr,"%s:  for a bandpass filter, the rejection tolerance.\n",
               program);
       fprintf(stderr,"must be less than the cutoff tolerance.\n");
       return(0);
    }

/*
** allocate memory for junk passed into low level filter routines 
*/

    a = (float *)malloc(MAXSTAGES*sizeof(float));
    if(a==NULL){
       fprintf(stderr,"%s:  [1] memory allocation failed.\n",program);
       return(0);
    }

    b = (float *)malloc(MAXSTAGES*sizeof(float));
    if(b==NULL){
       fprintf(stderr,"%s:  [2] memory allocation failed.\n",program);
       return(0);
    }

    c = (float *)malloc(MAXSTAGES*sizeof(float));
    if(c==NULL){
       fprintf(stderr,"%s:  [3] memory allocation failed.\n",program);
       return(0);
    }

    d = (float *)malloc(MAXSTAGES*sizeof(float));
    if(d==NULL){
       fprintf(stderr,"%s:  [4] memory allocation failed.\n",program);
       return(0);
    }

    e = (float *)malloc(MAXSTAGES*sizeof(float));
    if(e==NULL){
       fprintf(stderr,"%s:  [5] memory allocation failed.\n",program);
       return(0);
    }

/*
** prewarp the critical frequencies and calculate other goodies
*/

       w1 = tan((double)XV_PI*f1/sfreq);
       w2 = tan((double)XV_PI*f2/sfreq);
       w3 = tan((double)XV_PI*f3/sfreq);
       w4 = tan((double)XV_PI*f4/sfreq);

       wc = w3 - w2;
       wr = MIN((w2*w3-w1*w1)/w1,(w4*w4-w2*w3)/w4);

       epsilon = sqrt((1 - tolc*tolc)/(tolc*tolc));
       lambda  = sqrt((1 - tolr*tolr)/(tolr*tolr)); 

/*
** design the filter
*/

    switch(class){
       case 0 :  if(!btwpass(wc,wr,w2,w3,epsilon,lambda,a,b,c,d,e,&stages)){
                    fprintf(stderr,"%s: failed call to btwpass().\n",program);
                 }
                 break;
       case 1 :  if(!chbipass(wc,wr,w2,w3,epsilon,lambda,a,b,c,d,e,&stages)){
                    fprintf(stderr,"%s: failed call to chbipass().\n",program);
                 }
                 break;
       case 2 :  if(!chbiipass(wc,wr,w2,w3,epsilon,lambda,a,b,c,d,e,&stages)){
                    fprintf(stderr,"%s: failed call to chbiipass().\n",program);
                 }
                 break;
    }

/*
** now rearrange data into a format that writepoly() can understand
**
** the following code segment assumes that the polynomial for a stage is:
**
**    H(z)  = (a*z^2 + b*z + c)/(z^2 + d*z +e)
**
*/


    for(i=0;i<stages;i++){
        poly->terms[offset].coef = a[i]; poly->terms[offset].expon = 0.0;
        poly->terms[offset+1].coef = b[i]; poly->terms[offset+1].expon = -1.0;
        poly->terms[offset+2].coef = c[i]; poly->terms[offset+2].expon = -2.0;
        poly->terms[offset+3].coef = 1.0;  poly->terms[offset+3].expon = 0.0;
        poly->terms[offset+4].coef = d[i]; poly->terms[offset+4].expon = -1.0;
        poly->terms[offset+5].coef = e[i]; poly->terms[offset+5].expon = -2.0;
        poly->nterms[2*i] = 3; poly->nterms[2*i+1] = 3;
        offset += 6;
    }
    for(i=0;i<stages*6;i++){
        poly->terms[i].delay = 0;
        poly->terms[i].type = _STDTERM;
        if((poly->terms[i].varname = (char *)malloc(2*sizeof(char)))==NULL){
           fprintf(stderr,"%s:  malloc error after filter design\n",program);
           return(0);
        }
        strcpy(poly->terms[i].varname,"z");
    }
    strcpy(poly->indep_var,"z");
    strcpy(poly->func_name,"H");
    poly->stages = stages;
    return(1);
}

/***********************************************************************
*
*  Routine Name: btwpass() 
*
*          Date: Fri Sep  7 15:35:49 MDT 1990
*        
*       Purpose: computes coefficients for a band pass butterworth filter
*                made up of 'stages' stages of 2nd order filters. 
*                The function used for one stage is:  
*
*                       H(z) = (a*z^2 + b*z + c)/(z^2 + d*z + e)
*
*         Input: stages  - number of second order stages
*                wc      - analog cutoff frequency
*
*        Output: a - f   - coefficients as explained above
*
*    Written By: Jeremy Worley
* 
* Modifications:
*
***********************************************************************/

static int btwpass(wc,wr,w2,w3,epsilon,lambda,a,b,c,d,e,nstages)
    int *nstages;
    float wc,wr,w2,w3,*a,*b,*c,*d,*e,epsilon,lambda;
{
    float prodi = 0.0,prodr = 1.0,*sreal,*simag,poweri,scalar,
          ctempr, ctempi, numerr, numeri, *pr, *pi, *as, *bs, *cs,
          *ds, *es, denom, temp;
    int i,lowstages,stages,order;

/*
** calculate useful numbers
*/

    order = 2*ceil(log(lambda/epsilon)/log(wr/wc));
    order += order%4;
    stages = order/2;
    *nstages = stages;

/*
** allocate temporary memory
*/

   if(((sreal = (float *)malloc(sizeof(float)*stages))==NULL) ||
      ((simag = (float *)malloc(sizeof(float)*stages))==NULL) ||
      ((pr = (float *)malloc(sizeof(float)*2*stages))==NULL) ||
      ((pi = (float *)malloc(sizeof(float)*2*stages))==NULL) ||
      ((as = (float *)malloc(sizeof(float)*stages))==NULL) ||
      ((bs = (float *)malloc(sizeof(float)*stages))==NULL) ||
      ((cs = (float *)malloc(sizeof(float)*stages))==NULL) ||
      ((ds = (float *)malloc(sizeof(float)*stages))==NULL) ||
      ((es = (float *)malloc(sizeof(float)*stages))==NULL)){
      BTW_CLEANUP;
      return(0);
   }

/*
** compute the s-plane lowpass poles
*/

    lowstages = stages; /* MAX(stages/2,1); */

    for(i=0;i<lowstages;i++){
        poweri = XV_PI*(2.0*(i+1)+(float)lowstages-1)/(2.0*lowstages);
        scalar = wc*(pow((double)epsilon,(double)(-1.0/lowstages)));
        sreal[i] = scalar*cos(poweri);
        simag[i] = scalar*sin(poweri);
        cmul(&prodr,&prodi,sreal[i],simag[i],prodr,prodi);
    }

/*
** compute corresponding bandpass poles.
*/

    for(i=0;i<lowstages;i++){
        cmul(&ctempr,&ctempi,sreal[i],simag[i],sreal[i],simag[i]);
        csub(&ctempr,&ctempi,ctempr,ctempi,4.0*w2*w3,0.0);
        csqrt(&ctempr,&ctempi,ctempr,ctempi);

        csub(&numerr,&numeri,sreal[i],simag[i],ctempr,ctempi);
        pr[2*i] = numerr / 2.0;
        pi[2*i] = numeri / 2.0;

        cadd(&numerr,&numeri,sreal[i],simag[i],ctempr,ctempi); 
        pr[2*i+1] = numerr / 2.0;
        pi[2*i+1] = numeri / 2.0;
    }

/*
** compute the s-plane bandpass filter coefficients; use these values to 
** compute the digital coefficients.  if it is the first stage that we are
** calculating coefficients for, normalize the numerator's coefficients.
*/

    for(i=0;i<stages;i++){
        as[i] = 0.0;
        bs[i] = 1.0;
        cs[i] = 0.0;
        ds[i] = -2.0*pr[i];
        es[i] = (float)pow((double)vcabs(pr[i],pi[i]),(double)2.0);
 
        denom = 1.0 + ds[i] + es[i];
        a[i] = (as[i] + bs[i] + cs[i])/denom;
        b[i] = (2.0*cs[i] - 2.0*as[i])/denom;
        c[i] = (as[i] - bs[i] + cs[i])/denom;
        d[i] = (2.0*es[i] - 2.0)/denom;
        e[i] = (1.0 - ds[i] + es[i])/denom;

        if(i==0){ 
           temp = vcabs(prodr,prodi);
           a[i] *= temp;
           b[i] *= temp;
           c[i] *= temp;
        } 
    }

    BTW_CLEANUP; 
    return(1);
}

/***********************************************************************
*
*  Routine Name: chbipass() 
*
*          Date: Fri Sep  7 15:35:49 MDT 1990
*        
*       Purpose: computes coefficients for a band pass chebychev I filter
*                made up of 'stages' stages of 2nd order filters. 
*                The function used for one stage is:  
*
*                       H(z) = (a*z^2 + b*z + c)/(z^2 + d*z + e)
*
*         Input: stages  - number of second order stages
*                wc      - analog cutoff frequency
*
*        Output: a - f   - coefficients as explained above
*
*    Written By: Jeremy Worley
* 
* Modifications:
*
***********************************************************************/

static int chbipass(wc,wr,w2,w3,epsilon,lambda,a,b,c,d,e,nstages)
    int *nstages;
    float wc,wr,w2,w3,*a,*b,*c,*d,*e,epsilon,lambda;
{
    int order,n,stages,lowstages;
    float alpha,bn,sr,si,tr,ti,c1,c2,bsqrdr,bsqrdi,*poler,*polei;
    float fltden,fourac,as,bs,cs,ds,es;
    float prodpr = 1.0, prodpi = 0.0;

/*
** calculate order and number of stages
*/

    order = 2*(int)ceil((ARCCOSH(lambda/epsilon)/ARCCOSH(wr/wc)));
    order += order%4;

    stages = order/2;
    *nstages = stages;

/*
** allocate memory for poles
*/

    poler = (float *)malloc(2*stages*sizeof(float)); 
    if(poler==NULL)return(0);

    polei = (float *)malloc(2*stages*sizeof(float)); 
    if(polei==NULL){
       free(poler);
       return(0);
    }

/*
** calculate some useful numbers
*/

    lowstages = stages;

    alpha = ARCSINH(1.0/epsilon)/stages;

    c1 = wc*sinh(alpha);
    c2 = wc*cosh(alpha);

    for(n=0;n<lowstages;n++){
        bn = XV_PI*(((float)(2.0*(n+1)-1.0+lowstages))/(2.0*(float)lowstages)); 

        sr = c1*cos(bn);
        si = c2*sin(bn);
/*
** compute the bandpass poles
*/

        cmul(&prodpr,&prodpi,prodpr,prodpi,sr,si);
        cmul(&bsqrdr,&bsqrdi,sr,si,sr,si);
        fourac = 4.0*w2*w3;
        csub(&tr,&ti,bsqrdr,bsqrdi,fourac,0.0);
        csqrt(&tr,&ti,tr,ti);
        csub(&tr,&ti,sr,si,tr,ti);
        cdiv(&poler[2*n],&polei[2*n],tr,ti,2.0,0.0);

        csub(&tr,&ti,bsqrdr,bsqrdi,fourac,0.0);
        csqrt(&tr,&ti,tr,ti);
        cadd(&tr,&ti,sr,si,tr,ti);
        cdiv(&(poler[2*n+1]),&(polei[2*n+1]),tr,ti,2.0,0.0);
   }

/*
** compute the s-plane coefficients; use these to compute the digital
** coefficients
*/

   for(n=0;n<stages;n++){
        as = 0.0;
        bs = 1.0;
        cs = 0.0;
        ds = -2.0*poler[n];
        es = pow((double)vcabs(poler[n],polei[n]),(double)2.0);

        fltden = 1.0 + ds + es;
        a[n] = (as + bs + cs)/fltden;
        b[n] = (2.0*cs - 2.0*as)/fltden;
        c[n] = (as - bs + cs)/fltden;
        d[n] = (2.0*es - 2.0)/fltden;
        e[n] = (1.0 - ds + es)/fltden;

        if(n==0){
           tr = vcabs(prodpr,prodpi)/sqrt(1.0+epsilon*epsilon);
           a[n] *= tr;
           b[n] *= tr;
           c[n] *= tr;
        } 
    } /* end for */ 

    free(poler);
    free(polei);
    return(1);
}

/***********************************************************************
*
*  Routine Name: chbiipass() 
*
*          Date: Fri Sep  7 15:35:49 MDT 1990
*        
*       Purpose: computes coefficients for a band pass chebychev II filter
*                made up of 'stages' stages of 2nd order filters. 
*                The function used for one stage is:  
*
*                       H(z) = (a*z^2 + b*z + c)/(z^2 + d*z + e)
*
*         Input: stages  - number of second order stages
*                wc      - analog cutoff frequency
*                wr      - analog rejection frequency
*
*        Output: a - f   - coefficients as explained above
*
*    Written By: Jeremy Worley
* 
* Modifications:
*
***********************************************************************/


static int chbiipass(wc,wr,w2,w3,epsilon,lambda,a,b,c,d,e,nstages)
    int *nstages;
    float wc,wr,w2,w3,*a,*b,*c,*d,*e,epsilon,lambda;
{
    int n,order,stages,lowstages;
    float alpha,bn,sr,si,tr,ti,c1,c2,bsqrdr,bsqrdi,
          *poler,*polei,*zeror,*zeroi;
    float fltden,as,bs,cs,ds,es, fourac, br, bi;
    float prodpr = 1.0, prodzr = 1.0, prodpi = 0.0, prodzi = 0.0, epslnhat;

/*
** calculate order and number of stages.
*/

    order = 2*(int)ceil((ARCCOSH(lambda/epsilon)/ARCCOSH(wr/wc)));
    order += order%4;

    stages = order/2;
    *nstages = stages;
/*
** allocate temporary workspace for pole and zero calculation
*/

    poler = (float *)malloc(stages*2*sizeof(float));
    if(poler==NULL)return(0);

    polei = (float *)malloc(stages*2*sizeof(float));
    if(polei==NULL){
       free(poler);
       return(0);
    }

    zeror = (float *)malloc(stages*2*sizeof(float));
    if(zeror==NULL){
       free(poler);
       free(polei);
       return(0);
    }

    zeroi = (float *)malloc(stages*2*sizeof(float));
    if(zeroi==NULL){
       free(poler);
       free(polei);
       free(zeror);
       return(0);
    }

/*
** calculate some useful numbers
*/

    lowstages = stages;

    epslnhat = 1.0/(epsilon* (cosh((double)stages*ARCCOSH(wr/wc))));
    alpha = ARCSINH(1.0/epslnhat)/stages;

    c1 = wc*sinh(alpha);
    c2 = wc*cosh(alpha);
    fourac = 4.0*w2*w3;

    for(n=0;n<lowstages;n++){
        bn = XV_PI*(((float)(2.0*(n+1)-1.0+lowstages))/(2.0*(float)lowstages));

        sr = c1*cos(bn);
        si = c2*sin(bn);

/*
** compute the bandpass poles
*/

        cdiv(&br,&bi,wr*wc,0.0,sr,si);
        cmul(&prodpr,&prodpi,prodpr,prodpi,br,bi);
        cmul(&bsqrdr,&bsqrdi,br,bi,br,bi);

        csub(&tr,&ti,bsqrdr,bsqrdi,fourac,0.0);
        csqrt(&tr,&ti,tr,ti);
        cadd(&tr,&ti,br,bi,tr,ti);
        cdiv(&poler[2*n],&polei[2*n],tr,ti,2.0,0.0);

        csub(&tr,&ti,bsqrdr,bsqrdi,fourac,0.0);
        csqrt(&tr,&ti,tr,ti);
        csub(&tr,&ti,br,bi,tr,ti);
        cdiv(&poler[2*n+1],&polei[2*n+1],tr,ti,2.0,0.0);

/*
** compute the bandpass zeros
*/

        cdiv(&br,&bi,wr,0.0,0.0,sin(bn));
        cmul(&prodzr,&prodzi,prodzr,prodzi,br,bi);
        cmul(&bsqrdr,&bsqrdi,br,bi,br,bi);

        csub(&tr,&ti,bsqrdr,bsqrdi,fourac,0.0);
        csqrt(&tr,&ti,tr,ti);
        cadd(&tr,&ti,br,bi,tr,ti);
        cdiv(&zeror[2*n],&zeroi[2*n],tr,ti,2.0,0.0);

        csub(&tr,&ti,bsqrdr,bsqrdi,fourac,0.0);
        csqrt(&tr,&ti,tr,ti);
        csub(&tr,&ti,br,bi,tr,ti);
        cdiv(&zeror[2*n+1],&zeroi[2*n+1],tr,ti,2.0,0.0);
    }

/*
** compute the s-plane coefficients; use these to compute the digital
** coefficients
*/

    for(n=0;n<stages;n++){
        as = 1.0;
        bs = -2.0*zeror[n];
        cs = pow((double)vcabs(zeror[n],zeroi[n]),(double)2.0);
        ds = -2.0*poler[n];
        es = pow((double)vcabs(poler[n],polei[n]),(double)2.0);

        fltden = 1.0 + ds + es;
        a[n] = (as + bs + cs)/fltden;
        b[n] = (2.0*cs - 2.0*as)/fltden;
        c[n] = (as - bs + cs)/fltden;
        d[n] = (2.0*es - 2.0)/fltden;
        e[n] = (1.0 - ds + es)/fltden;

        if(n==0){
           tr =  vcabs(prodpr,prodpi)/vcabs(prodzr,prodzi);
           a[n] *= tr;
           b[n] *= tr;
           c[n] *= tr;
        }
    } /* end for */ 

    free(poler);
    free(polei);
    free(zeror);
    free(zeroi);
    return(1);
}


/***********************************************************************
*
*  Routine Name: csqrt() 
*
*          Date: Fri Oct 19 14:04:56 MDT 1990
*        
*       Purpose:  
*
*         Input: 
*
*        Output: 
*
*    Written By:  
*
* Modifications:
*
***********************************************************************/

int csqrt(sreal,simag,areal,aimag)
    float *sreal, *simag, areal, aimag;
{
    float ar,ai,r,w;

    if(areal==0.0 && aimag == 0.0){
       *sreal = 0.0;
       *simag = 0.0;
    }else{
       ar = (float)fabs((double)areal);
       ai = (float)fabs((double)aimag);

       if(ar >= ai){
          r = ai/ar;
          w = sqrt(ar) * sqrt(0.5*(1.0+sqrt(1.0+r*r)));
       }else{
          r = ar/ai;
          w = sqrt(ai) * sqrt(0.5*(r+sqrt(1.0+r*r)));
       }
       if(areal >= 0.0){
          *sreal = w;
          *simag = aimag/(2.0*w);
       }else{
          *simag = (aimag >= 0) ? w : -w;
          *sreal = aimag/(2*(*simag));
       }
    }
    return(1);
} 

/* -library_code_end */
