
/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/**********************************************************************
*
*  File: userfunc.c
*
*  Purpose: Lets user compile own function definitions for use in
*           expressions.  Function passed pointer to coordinate array.
*           User should also provide derivative function.
*           After defining functions, add names to arrays at bottom
*           of file.  Names in this file don't mattter; first function
*           in array is usr1 in datafile or queries, etc. Use in 
*           datafile does NOT use arguments, they are implicit.
*           Example: (usr1 + usr3)/usr10.
*/

#include "include.h"

/* usr1 as defined here gives conformal metric for 3-sphere in 
   stereographic projection */

static double usr1(x)
double *x; /* incoming parameters */
{ double denom;
  denom = 4+x[0]*x[0]+x[1]*x[1]+x[2]*x[2]; 
  return 16/denom/denom;
}

static double usr1_deriv(x,partials)
double *x; /* incoming parameters */
double *partials; /* outgoing partial derivatives */
{ double denom,cube;
  int i;

  denom = 4+x[0]*x[0]+x[1]*x[1]+x[2]*x[2]; 
  cube = denom*denom*denom;
  for ( i = 0 ; i < web.sdim ; i++ )
    partials[i] = -64/cube*x[i]; 

  return 16/denom/denom;
}

/* Add your functions to these arrays; this is how they will be invoked! */
#ifdef NOPROTO
double (*userfunc[])() = {usr1};
double (*userfunc_deriv[])() = {usr1_deriv};
#else
double (*userfunc[])(REAL*) = {usr1};
double (*userfunc_deriv[])(REAL*,REAL*) = {usr1_deriv};
#endif





