/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/********************************************************************
*
*  File: storage.c
*
*  Purpose:  File defining details of storage implementation.
*            All machine-dependent gory details should be here
*            (purely private details) or in storage.h
*            (for inclusion in other source files that need to know).
*            
*      This version has element ids implemented as longs.
*/

#include "include.h"

/* individual dimension block pointer arrays, handy for debugging */
char *vbase;
char *ebase;
char *fbase;
char *bbase;
char *febase;

/* individual dimension block pointer arrays, handy for fast addressing */
struct vertex *vvbase;
struct edge *eebase;
struct facet *ffbase;
struct body *bbbase;
struct facetedge *fefebase;

/* unified block references, indexed by element type */
char *base[NUMELEMENTS];


struct web web = {
{ { VERTEX, NULL, 0, NULLID, NULLID, 0, NULLID },
 { EDGE  , NULL, 0, NULLID, NULLID, 0, NULLID },
 { FACET , NULL, 0, NULLID, NULLID, 0, NULLID },
 { BODY  , NULL, 0, NULLID, NULLID, 0, NULLID },
 { FACETEDGE, NULL, 0, NULLID, NULLID, 0, NULLID } },
 3,SOAPFILM,LINEAR
    };


element_id
  NULLVERTEX  =  VERTEX  << TYPESHIFT, 
  NULLEDGE    =  EDGE  << TYPESHIFT,
  NULLFACET   =  FACET  << TYPESHIFT,
  NULLBODY    =  BODY  << TYPESHIFT,  
  NULLFACETEDGE =  FACETEDGE << TYPESHIFT;


struct element *elptr(id)
element_id id;
{
  int type = id_type(id);
  return (struct element *)(base[type] + (id & OFFSETMASK));
}


void extend(type,number)
int type;
int number;  /* of new items to allocate */
{
   char *newblock = NULL;
   element_id id;
   struct element *new = NULL;
   int k;
   int offset;
   long allocsize;

   if ( number <= 0 ) return;

   if ( base[type] )
     { allocsize = (long)(web.skel[type].maxcount + number)*sizes[type];
       if ( allocsize < MAXALLOC )
         newblock = kb_realloc(base[type],(unsigned int)allocsize);
       else
         {
           sprintf(errmsg,"Trying to allocate %ld bytes! \n",allocsize);
           error(errmsg,UNRECOVERABLE);
         }
     }
   else
      newblock = mycalloc(number,sizes[type]);
   base[type] = newblock;
   web.skel[type].base = newblock;
   switch ( type )
     {
       case VERTEX: vbase = newblock;
                    vvbase = (struct vertex *)newblock;
                    break;
       case EDGE:   ebase = newblock;
                    eebase = (struct edge *)newblock;
                    break;
       case FACET:  fbase = newblock; 
                    ffbase = (struct facet *)newblock;
                    break;
       case BODY:   bbase = newblock; 
                    bbbase = (struct body *)newblock;
                    break;
       case FACETEDGE: febase = newblock; 
                    fefebase = (struct facetedge *)newblock;
                    break;
     }
  
   /* add to freelist */
   offset = web.skel[type].maxcount*sizes[type];
   memset(newblock+offset,0,number*sizes[type]);
   id = ((long)type << TYPESHIFT) + VALIDMASK + offset;
   web.skel[type].free = id;
   for ( k = 0 ; k < number ; k++ )
     { 
       new = (struct element *)(newblock + offset);
       offset += sizes[type];
       id = ((long)type << TYPESHIFT) + VALIDMASK + offset;
       new->forechain = id;
       new->ordinal = web.skel[type].max_ord++;  
     }
   new->forechain = NULLID;
   web.skel[type].maxcount += number;

}

element_id new_element(type)
int type;
{
  element_id newid;
  struct element *new,*last;
  ORDTYPE ord;
  
  newid = web.skel[type].free;
  
  if ( !valid_id(newid) ) /* free list empty */
   {
     extend(type,BATCHSIZE);
     newid = web.skel[type].free;
   }
  new = elptr(newid);

  /* remove from free chain */
  web.skel[type].free = new->forechain;

  /* clean out old info */
  ord = new->ordinal;
  memset((char *)new,0,sizes[type]);
  new->ordinal = ord;


  /* add to end of in-use chain */
  new->forechain = NULLID;
  new->backchain = web.skel[type].last;
  if ( valid_id(web.skel[type].last) )
    {
      last = elptr(web.skel[type].last);    /* find end of in-use chain */
      last->forechain = newid;
    }
  else
    {      
      web.skel[type].used = newid;
    }
  web.skel[type].last = newid;

  new->attr = ALLOCATED | NEWELEMENT;
  new->type = type;
  web.skel[type].count++;  
  return newid;
}

void free_element(id)
element_id id;
{
  struct element *ptr;
  int type = id_type(id);

/* printf("freeing %X  type %d ordinal %d\n",id,type,ordinal(id)); */

  if ( !valid_id(id) )
   {
     sprintf(errmsg,"Trying to free invalid id %08X \n",id);
     error(errmsg,RECOVERABLE);
   }
   
  ptr = elptr(id);
  if ( !(ptr->attr & ALLOCATED) )
   {
     sprintf(errmsg,"Trying to free unallocated element id %08X \n",id);
     error(errmsg,RECOVERABLE);
   }
  ptr->attr = 0;  /* clear attributes */

  /* remove from in-use list */
  if ( valid_id(ptr->forechain) )
    elptr(ptr->forechain)->backchain = ptr->backchain;
  else
    web.skel[type].last = ptr->backchain;

  if ( valid_id(ptr->backchain) )
    elptr(ptr->backchain)->forechain = ptr->forechain;
  else
    web.skel[type].used = ptr->forechain;

  /* add to discard list */
  /* save forechain for generators */
  ptr->backchain = web.skel[type].discard;
  web.skel[type].discard = id & (TYPEMASK | VALIDMASK | OFFSETMASK);
                         /* clear old bits, keep only needed */

  web.skel[type].count--; 

}

int ordinal(id)
element_id id;
{
  int size;
  int type = id_type(id);
  size = sizes[type];
  if ( !valid_id(id) ) return -1;
  return  (id&OFFSETMASK)/size;
}

int generate_all(type,idptr)  /* re-entrant */
int type;
element_id *idptr;
{
  struct element *ptr;

  if ( !valid_id(*idptr) ) /* first time */
    { *idptr = web.skel[type].used;
      if ( !valid_id(*idptr) ) return 0;
    }
  else
    { ptr = (struct element *)(base[type] + (*idptr&OFFSETMASK));
      do
    	{ /* may have to get off discard list */
          *idptr = ptr->forechain;
          if ( !valid_id(*idptr) ) return 0;
          ptr = (struct element *)(base[type] + (*idptr&OFFSETMASK));
         }
      while ( !(ptr->attr & ALLOCATED) );
    }

  return 1;
}

void memory_report()
{
   long mem = 0;
   int k;

   mem = 0;
   for ( k = 0 ; k < NUMELEMENTS ; k++ )
     mem += web.skel[k].count*sizes[k];

   sprintf(msg,"Vertices: %ld   Edges: %ld   Facets: %ld   Facetedges: %ld   Memory: %ld\n",
     web.skel[0].count,web.skel[1].count,web.skel[2].count,web.skel[4].count,
     mem);
   outstring(msg);

#ifdef XENIX386
  { struct mallinfo minfo;
    minfo = mallinfo();
    printf("arena:  %d   blocks: %d    memory in use: %d   free: %d\n",
       minfo.arena,minfo.ordblks,minfo.uordblks,minfo.fordblks);
  }
#endif
       
  
}

void reset_skeleton()
{
  int i,j;
  struct boundary *bdry;
  struct constraint *con;
  struct surf_energy *surfen;
  struct quantity *quan;

  if ( gauss1Dpt ) { free((char *)gauss1Dpt); gauss1Dpt = NULL; }
  if ( gauss1Dwt ) { free((char *)gauss1Dwt); gauss1Dwt = NULL; }

  /* deallocate boundary specs */
  for ( i = 0, bdry = web.boundaries ; i < BDRYMAX ; i++, bdry++ )
    if ( bdry->coordf[0] )
      {
        for ( j = 0 ; j < web.sdim ; j++ )
	  free_expr(bdry->coordf[j]);
	free((char *)bdry->coordf[0]);
      }

  /* deallocate constraint specs */
  for ( i = 0, con = web.constraints ; i < CONSTRMAX ; i++, con++ )
    constraint_free(con);

  /* deallocate surface energy specs */
  for ( i = 0, surfen = web.surfen ; i < web.surfen_count ; i++, surfen++ )
    { if ( surfen->envect[0] )
        { for ( j = 0 ; j < web.sdim ; j++ )
             free_expr(surfen->envect[j]);
          free((char *)surfen->envect[0]);
	}
    }

  /* deallocate quantity specs */
  for ( i = 0, quan = web.quants ; i < web.quantity_count ; i++, quan++ )
    { if ( quan->attr & IN_USE )
        { for ( j = 0 ; j < web.sdim ; j++ )
             free_expr(quan->quanvect[j]);
          free((char *)quan->quanvect[0]);
	}
    }

  /* deallocate metric expressions */
  if ( web.metric_flag )
    for ( i = 0 ; i < web.sdim ; i++ )
      for ( j = 0 ; j < web.sdim ; j++ )
	free_expr(&web.metric[i][j]);

  /* free parameters */
  if ( web.params ) free((char *)web.params);

  if ( web.skel[0].base )
   for ( i = 0 ; i < NUMELEMENTS ; i++ )
    {
#ifdef MAXBLOCK
      for ( j = 0 ; j < web.skel[i].blocks ; j++ )  
      free((char *)web.skel[i].base[j]);  
#endif
      if ( web.skel[i].base ) free((char *)web.skel[i].base);
    }
  memset((char *)base,0,sizeof(base));
  vbase = ebase = fbase = bbase = febase = NULL;
  vvbase = NULL;
  eebase = NULL;
  ffbase = NULL;
  bbbase = NULL;
  fefebase = NULL;

  if ( web.inverse_periods )  free_matrix(web.inverse_periods);

  memset((char *)&web,0,sizeof(web));  /* total clean out */

#ifdef MAXBLOCK
  list_alloc();
#endif
}

/*******************************************************************
*
*  function: vgrad_init()
*
*  purpose:  allocates storage for vertex volume gradients
*
*/

volgrad *vgradbase;   /* allocated list of structures */
volgrad **vgptrbase;   /* allocated list of chain start pointers */
int      vgradtop;    /* number of first free  structure */
long     vgradmax;    /* number allocated */

void vgrad_init(qfixed)
int qfixed; /* how many fixed quantities */
{
  long allocsize;

  allocsize = (long)web.skel[VERTEX].maxcount*sizeof(volgrad  *);
  if ( allocsize < MAXALLOC )
   vgptrbase = (volgrad **)temp_calloc(web.skel[VERTEX].maxcount,
                                                   sizeof(volgrad *));
  if ( vgptrbase == NULL )
    {
      sprintf(errmsg,"Cannot allocate %d volgrad pointers of size %d\n",
        web.skel[VERTEX].maxcount, sizeof(volgrad  *));
      error(errmsg,RECOVERABLE);
    }

  if ( web.simplex_flag )  /* may be gross overestimate, but safe */
    vgradmax = web.skel[VERTEX].count*web.skel[BODY].count;
  else if ( web.dimension == STRING )
    vgradmax = 2*web.skel[EDGE].count;
  else /* SOAPFILM */
    /* could have factor of 2 on facets if could guarantee bodies
       next to an edge are separated by a facet, which we can't
       if we permit incomplete bodies. 
     */
    vgradmax = web.skel[FACETEDGE].count - web.skel[FACET].count
             + web.skel[BODY].count + qfixed*web.skel[VERTEX].count + 10;

  if ( web.modeltype == QUADRATIC )
    vgradmax += web.skel[FACETEDGE].count;  /* for midpoints */

  allocsize = (long)vgradmax*sizeof(struct volgrad);
  if ( allocsize < MAXALLOC )
    vgradbase = (struct volgrad *)temp_calloc(vgradmax,sizeof(struct volgrad));
  if ( vgradbase == NULL )
    {
      sprintf(errmsg,"Cannot allocate %ld volgrad structures of size %d\n",
        vgradmax,sizeof(struct volgrad));
      vgrad_end();
      error(errmsg,RECOVERABLE);
    }
  
  vgradtop = 0;

}

void vgrad_end()
{
  if ( vgradbase ) { temp_free((char *)vgradbase); vgradbase = NULL; }
  if ( vgptrbase ) { temp_free((char *)vgptrbase); vgptrbase = NULL; }
  vgradmax = 0;
}

volgrad *get_vertex_vgrad(v_id)
vertex_id v_id;
{
  return vgptrbase[ordinal(v_id)];
}

volgrad  *new_vgrad()
{

  if ( vgradtop >= vgradmax ) 
  /* error */
    {
      vgrad_end();
      sprintf(errmsg,"Too many volgrad structures requested: %d\n", vgradtop);
      error(errmsg,RECOVERABLE);
    }

  return vgradbase + vgradtop++;
}

      
volgrad *get_bv_vgrad(b_id,v_id)
body_id b_id;
vertex_id v_id;
{
  volgrad  *vgptr;

  vgptr = get_vertex_vgrad(v_id);
  while ( vgptr )
   if ( vgptr->b_id == b_id ) break;
   else vgptr = vgptr->chain;

  return vgptr;   /* null if not found */
}
      
volgrad *get_bv_new_vgrad(b_id,v_id)
body_id b_id;
vertex_id v_id;
{
  volgrad  **ptrptr;  /* pointer to pointer so can update if need be */

  ptrptr = vgptrbase + ordinal(v_id);
  while ( *ptrptr )
   if ( (*ptrptr)->b_id == b_id ) return *ptrptr;
   else ptrptr = &(*ptrptr)->chain;

  /* need to get new structure */
  *ptrptr = new_vgrad();
  (*ptrptr)->b_id = b_id;

  return *ptrptr;
}

void free_discards()
{ int type;

  for ( type = 0 ; type < NUMELEMENTS ; type++ )
    { element_id id = web.skel[type].discard;
      while ( valid_id(id) )
	{ struct element *ptr = elptr(id);
	  web.skel[type].discard = ptr->backchain;
	  ptr->forechain = web.skel[type].free;
	  ptr->backchain = NULLID;
	  web.skel[type].free = id;
	  id = web.skel[type].discard;
	}
    }
}
