/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*****************************************************************
*
*  File: painter.c 
*
*  Contents:  Routines to accumulate and sort facets for display
*             back to front.  Not device specific, but calls
*             device specific routines  for actual display.
*             Also does transformations on edges.
*/

#include "include.h"

static int count;     /* number of facets */
static int maxcount;  /* number allocated */
struct tsort *trilist; /* list for depth sorting triangles */

static int tcompare();

static int tcompare(t1,t2)  /* depth comparison for sort */
struct tsort *t1,*t2;
{

  if ( t1->mins[2] > t2->mins[2] ) return 1;
  return -1;
}

void painter_start()
{
  long allocsize;

  (*init_graphics)();

  innerflag = outerflag = 1;

  if ( bare_edge_count == 0 ) bare_edge_count = 20;  /* allow a few */
  /* allocate space for depth sort list */
  if ( web.torus_flag )
    if ( web.torus_clip_flag )
      maxcount = 5*web.skel[FACET].count;
    else maxcount = 2*web.skel[FACET].count;
  else
    maxcount = web.skel[FACET].count + bare_edge_count;
  allocsize = (long)maxcount*sizeof(struct tsort);
  if ( allocsize >= MAXALLOC  )
    maxcount = MAXALLOC/sizeof(struct tsort);
  trilist = (struct tsort *)temp_calloc(maxcount,sizeof(struct tsort));
  count = 0;
}

void painter_facet(gdata,f_id)
struct graphdata *gdata;
facet_id f_id;
{
  int i,j,n;
  REAL a[HOMDIM],b[HOMDIM];
  struct tsort *t;
  REAL normal[MAXCOORD],normalv[MAXCOORD];

  if ( count > maxcount ) return;
  if ( count == maxcount )
   { sprintf(errmsg,"Maxcount facets exceeded.\n");
     outstring(errmsg);
     count++;
     return;
   }
  t = trilist + count;
  t->flag = 1;
  t->f_id = f_id;

  /* accumulate list of triangles to display */

  for ( i = 0 ; i < FACET_VERTS ; i++ )
    {
      for ( j = 0 ; (j < web.sdim) && (j < HOMDIM-1) ; j++ ) 
        a[j] = gdata[i].x[j];
      a[HOMDIM-1] = 1.0;
      matvec_mul(view,a,b,HOMDIM,HOMDIM);  /* transform */
      t->x[i][0] = (float)b[1];
      t->x[i][1] = (float)b[2];
      t->x[i][2] = (float)b[0];
    } 
  vnormal(gdata[0].x,gdata[1].x,gdata[2].x,normal); 
  matvec_mul(view,normal,normalv,web.sdim,web.sdim);  
  t->normal[0] = (float)normalv[1]; 
  t->normal[1] = (float)normalv[2]; 
  t->normal[2] = (float)normalv[0];
  if ( t->normal[2] > (float)0.0 )
    for ( i = 0 ; i < web.sdim ; i++ ) t->normal[i] = -t->normal[i];

  /* find extents */
  for ( n = 0 ; n < web.sdim ; n++ )
    {
      if ( t->x[0][n] < t->x[1][n] )
        {
          if  ( t->x[2][n] < t->x[0][n] )
            {
              t->maxs[n] = t->x[1][n] - 0.00001;
              t->mins[n] = t->x[2][n] + 0.00001;
            }
          else if ( t->x[1][n] < t->x[2][n] )
            {
              t->maxs[n] = t->x[2][n] - 0.00001;
              t->mins[n] = t->x[0][n] + 0.00001;
            }
          else
            {
              t->maxs[n] = t->x[1][n] - 0.00001;
              t->mins[n] = t->x[0][n] + 0.00001;
            }
         }
       else
        {
          if  ( t->x[2][n] < t->x[1][n] )
            {
              t->maxs[n] = t->x[0][n] - 0.00001;
              t->mins[n] = t->x[2][n] + 0.00001;
            }
          else if ( t->x[0][n] < t->x[2][n] )
            {
              t->maxs[n] = t->x[2][n] - 0.00001;
              t->mins[n] = t->x[1][n] + 0.00001;
            }
          else
            {
              t->maxs[n] = t->x[0][n] - 0.00001;
              t->mins[n] = t->x[1][n] + 0.00001;
            }
        }
    }

  count++;
}

void painter_end()
{
  int i,j,k;
  int looptest;
  facet_id loopf[20]; /* for tracking loops */

  if ( count > maxcount ) count = maxcount;   /* in case there was excess */

  /* now sort on max z */
  qsort((char *)trilist,count,sizeof(struct tsort),tcompare);

  /* display */
  for ( k = 0 ; k < count ;  )
    {
      if ( breakflag ) break;
      if ( !trilist[k].flag ) { k++; continue; }

      /* i is current back facet */
      i = k;
      looptest = 0;
      for ( j = k+1 ; j < count ; j++ )
        {
          if ( breakflag ) break;
          if ( i == j ) continue;
          if ( !trilist[j].flag ) continue;
          if ( trilist[i].maxs[2] < trilist[j].mins[2] ) break;
          if ( !in_back(trilist + j, trilist + i) ) continue;
          if ( in_back(trilist + i, trilist + j) )
	    continue; /* have actually crossing facets */

          /* now have j as new back facet */
          if ( j < i )
	   { loopf[looptest] = trilist[i].f_id;
    	      /* test for looping */
	          if ( looptest++ > 10 )
		    { int m;
/*      fprintf(stderr,"painter_end: Looping too much.\n");
		      for ( m = 0 ; m < 10 ; m++ ) facet_dump(loopf[m],stdout);
*/
		      break;
		    }
           }
          i = j;  /* make back facet the live one */
          j = k; /* start tests over again */
        }
      (*display_facet)(trilist + i);
      trilist[i].flag = 0;
    }
 
  temp_free((char *)trilist); trilist = NULL;
  (*finish_graphics)();
}

float tableau[7][3];  /* simplex tableau */

/* returns 1 if ta is obscured by tb, else 0 */ 
int in_back(ta,tb)
struct tsort *ta,*tb;
{
  double da,db,d1,d2;
  int i,j,k,n;
  int afar=0,aeq=0,anear=0;  /* count of ta vertices relative to tb plane */
  int bfar=0,beq=0,bnear=0;  /* count of tb vertices relative to ta plane */
  REAL d;

  /* test x and y extent overlap */
  for ( n = 0 ; n < 2 ; n++ )
    if ( (tb->maxs[n] < ta->mins[n]) || (tb->mins[n] > ta->maxs[n]) )
        return 0;  

  /* see where tb is with respect to ta plane */
  da = dotf(ta->normal,ta->x[0],web.sdim);
  for ( k = 0 ; k < FACET_VERTS ; k++ )
    {
      d = dotf(ta->normal,tb->x[k],web.sdim); 
      if ( d < da - 0.0001 ) { bnear++; continue; }
      if ( d < da + 0.0001 ) { beq++; continue; }
      bfar++;
    }
  if ( bnear == 0 ) return 0;  /* tb on far side */
  if ( beq == FACET_VERTS ) return 0; /* both in same plane */

  /* see where ta is with respect to tb plane */
  db = dotf(tb->normal,tb->x[0],web.sdim);
  for ( k = 0 ; k < FACET_VERTS ; k++ )
    {
      d = dotf(tb->normal,ta->x[k],web.sdim); 
      if ( d < db - 0.0001 ) { anear++; continue; }
      if ( d < db + 0.0001 ) { aeq++; continue; }
      afar++;
    }

  if ( afar == 0 ) return 0;  /* ta in front */

  /* common edge, so can check which way fold goes */
  if ( (aeq == 2) && (beq == 2) )
    if ( (afar == 1) && (bnear == 1) ) return 1;
    else return 0;

  /* now the nitty gritty check to see if they overlap */
  /* Phase I linear program to see  if there is a feasible
     region where the triangles overlap */

  /* Objective is difference in z depths at point, zb - za */
  if ( (fabs(ta->normal[2]) < 0.000001) || (fabs(tb->normal[2]) < 0.000001) )
    return 0;
  tableau[0][0] = tb->normal[0]/tb->normal[2] - ta->normal[0]/ta->normal[2];
  tableau[0][1] = tb->normal[1]/tb->normal[2] - ta->normal[1]/ta->normal[2];
  tableau[0][2] = (float)(db/tb->normal[2] - da/ta->normal[2]);

  /* constraint coefficients from triangle sides */
  for ( i = 0 ; i < FACET_VERTS ; i++ )
    { j = (i + 1) % FACET_VERTS; k = (i + 2) % FACET_VERTS;
      tableau[i+1][0] = ta->x[i][1] - ta->x[j][1];
      tableau[i+1][1] = ta->x[j][0] - ta->x[i][0];
      tableau[i+1][2] = (float)dotf(tableau[i+1],ta->x[i],2);
      if ( dotf(tableau[i+1],ta->x[k],2) > tableau[i+1][2] )
        for ( n = 0 ; n < FACET_VERTS ; n++ ) 
          tableau[i+1][n] = -tableau[i+1][n];
      tableau[i+4][0] = tb->x[i][1] - tb->x[j][1];
      tableau[i+4][1] = tb->x[j][0] - tb->x[i][0];
      tableau[i+4][2] = (float)dotf(tableau[i+4],tb->x[i],2);
      if ( dotf(tableau[i+4],tb->x[k],2) > tableau[i+4][2] )
        for ( n = 0 ; n < FACET_VERTS ; n++ )
          tableau[i+4][n] = -tableau[i+4][n];
    }
  /* shrink feasible regions a bit to avoid spurious overlaps */
  for ( i = 1 ; i < 7 ; i++ ) tableau[i][2] -= (float)0.000001;

  /* move origin to corner of a triangle */
  d1 = tableau[5][0]*tableau[6][1];
  d2 = tableau[5][1]*tableau[6][0];
  if ( fabs(d1 - d2) < 0.000001 ) return 0;  /* degenerate triangle */
  if ( fabs(d1) > fabs(d2) )
    {
      pivot(5,0,7);
      pivot(6,1,7);
    }
  else
    {
      pivot(5,1,7);
      pivot(6,0,7);
    }
  /* now do phase I steps on first four rows */
  for ( n = 0 ; n < 10 ; n++ )
    {
      /* find negative right side */
      for ( i = 1 ; i < 5 ; i++ )
        if ( tableau[i][2] < (float)0.0 ) break;
      if ( i == 5 )  /* have feasible point */
        {
          /* compare depths at feasible point a bit inward */
          if ( tableau[0][2] > (float)0.0001*(tableau[0][0] + tableau[0][1]) )
            return 1;
          else return 0;
        }
      /* find negative in row */
      for ( j = 0 ; j < 2 ; j++ )
        if ( tableau[i][j] < (float)0.0 ) break;
      if ( j == 2 ) return 0;  /* no feasible point */
      pivot(i,j,5);
    }

  /* unsuccessful tableau, so just return 0 */
/* fprintf(stderr,"Unsuccessful tableau in in_back(). \n"); */
/*facet_dump(ta->f_id,stdout); */
/*facet_dump(tb->f_id,stdout); */
  return 0;

}  

void pivot(i,j,rows)
int i,j,rows;
{
  int n,m;
  double p = tableau[i][j];

  for ( n = 0 ; n < rows  ; n++ )
   {
     if ( n == i ) continue;
     for ( m = 0 ; m < 3 ; m++ )
       {
         if ( m == j ) continue;
         tableau[n][m] -= tableau[n][j]*tableau[i][m]/p;
       }
     tableau[n][j] /= -p;
   }
  tableau[i][j] = (float)1.0;
  for ( m = 0 ; m < 3 ; m++ )
    tableau[i][m] /= p;
}

void painter_edge(gdata)
struct graphdata *gdata;
{
  struct tsort t;
  int i,j;
  REAL a[HOMDIM],b[HOMDIM];

  for ( i = 0 ; i < 2 ; i++ )
    {
      for ( j = 0 ; (j < web.sdim) && (j < HOMDIM-1) ; j++ ) 
         a[j] = gdata[i].x[j];
      a[HOMDIM-1] = 1.0;
      matvec_mul(view,a,b,HOMDIM,HOMDIM);  /* transform */
      for ( j = 0 ; j < 2 ; j++ ) t.x[i][j] = (float)b[j];
    }       
  t.f_id = gdata->id;
  (*display_edge)(&t);
}


