/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/******************************************************************
*
*  File: modify.c
*
*  Contents:  low-level triangulation modification routines:
*
*         edge_divide() - subdivide an edge
*         cross_cut() -  subdivide a facet
*
*/

#include "include.h"

/******************************************************************
*
*  Function: face_triangulate()
*
*  Purpose:  Triangulate a n-sided face by putting in central vertex.
*/

void face_triangulate(f_id,edgecount)
facet_id f_id;
int edgecount;
{
  int i;
  vertex_id center; 
  facetedge_id fe,pre_fe;
  vertex_id rimv;
  edge_id spoke;
  facet_id fe_in,fe_out,next_fe;
  REAL *centerx,*x;
  struct boundary *bdry;
  WRAPTYPE wrap = 0;

  /* put a new vertex in the center */
  center = new_vertex(NULL);
  if ( get_fattr(f_id) & FIXED )
    set_attr(center,FIXED);

  /* center coordinates are average of vertices */
  centerx = get_coord(center);
  for ( i = 0 ; i < web.sdim ; i++ ) centerx[i] = 0.0;
  generate_facet_fe_init();
  while ( generate_facet_fe(f_id,&fe) )
    { REAL w[MAXCOORD];

      x = get_coord(get_fe_tailv(fe));
      if ( web.symmetry_flag )
	{ (*sym_wrap)(x,w,wrap);
	  x = w;
	  wrap = (*sym_compose)(wrap,get_fe_wrap(fe));
	}
      for ( i = 0 ; i < web.sdim ; i++ ) 
        centerx[i] += x[i];
    }
  for ( i = 0 ; i < web.sdim ; i++ ) centerx[i] /= edgecount;

  /* find centerpoint parameters for facet on boundary */
  if ( get_fattr(f_id) & BOUNDARY )   /* not working for torus */
    {
      REAL defaultp[MAXCOORD];
      REAL *paramb,*parammid,*xb;
      vertex_id base_v;
      REAL s[MAXCOORD];

      set_attr(center,BOUNDARY);
      bdry = get_facet_boundary(f_id);
      set_boundary(center,bdry);

      /* center parameters extrapolate from a vertex */
      /* try to find a vertex on same boundary */
      base_v = NULLVERTEX;
      generate_facet_fe_init();
      while ( generate_facet_fe(f_id,&fe) )
        { 
          if ( bdry == get_boundary(get_fe_tailv(fe)) )
            base_v = get_fe_tailv(fe);
        }
      if ( valid_id(base_v) )
        { paramb = get_param(base_v);
          xb = get_coord(base_v);
          for ( i = 0 ; i < web.sdim ; i++ )
            s[i] = xb[i];  /* displacement vector */
        }
      else
        { paramb = defaultp;
          defaultp[0] = defaultp[1] = defaultp[2] = 0.1;
          for ( i = 0 ; i < web.sdim ; i++ )
            s[i] = eval(bdry->coordf[i],defaultp);
          sprintf(msg,"Could not find vertex on same boundary as facet %d.\n",
	     get_original(f_id));
        }

      parammid = get_param(center);
      b_extrapolate(bdry,s,centerx,centerx,paramb,parammid);
    }

  /* install edge from rim to center */
  fe = get_facet_fe(f_id);  /* canonical starting point */
  pre_fe = get_prev_edge(fe);
  rimv = get_fe_tailv(fe);
  spoke = new_edge(rimv,center);
  if ( get_fattr(f_id) & FIXED )
    set_attr(spoke,FIXED);
  if ( get_fattr(f_id) & BOUNDARY )   
    {
      set_attr(spoke,BOUNDARY);
      bdry = get_facet_boundary(f_id);
      set_edge_boundary(spoke,bdry);
    }
  else if ( get_fattr(f_id) & CONSTRAINT )   
    {
      ATTR attr = get_fattr(f_id) & (BDRY_ENERGY | BDRY_CONTENT | CONSTRAINT );
      MAP conmap = get_f_constraint_map(f_id);

      set_attr(spoke,attr);
      set_attr(center,attr);
      set_e_conmap(spoke,conmap);
      set_v_conmap(center,conmap);
      project_v_constr(center);
      if ( web.modeltype == QUADRATIC )
        {
          vertex_id mid = get_edge_midv(spoke);
          set_attr(mid,attr);
          set_v_conmap(mid,conmap);
          project_v_constr(mid);
        }
    }

  if ( web.symmetry_flag )
        set_edge_wrap(spoke,0);
  fe_in = new_facetedge(f_id,spoke);
  set_edge_fe(spoke,fe_in);
  set_prev_edge(fe_in,pre_fe);
  set_next_edge(pre_fe,fe_in);
  fe_out = new_facetedge(f_id,edge_inverse(spoke));
  set_vertex_fe(center,fe_out);
  set_prev_edge(fe_out,fe_in);
  set_next_edge(fe_in,fe_out);
  set_prev_edge(fe,fe_out);
  set_next_edge(fe_out,fe);
  set_next_facet(fe_in,fe_inverse(fe_out));
  set_prev_facet(fe_in,fe_inverse(fe_out));
  set_next_facet(fe_out,fe_inverse(fe_in));
  set_prev_facet(fe_out,fe_inverse(fe_in));

  /* now go around cutting off triangles */
  while ( !equal_id(get_next_edge(fe),fe_in) )
    { next_fe = get_next_edge(fe); 
      cross_cut(get_prev_edge(fe),fe);
      fe = next_fe;
    }   
}

/*********************************************************
*
*  edge_divide()
*
*  Purpose: Subdivide an edge in the first stage of 
*           refinement.  Marks the new vertex with
*           the NEWVERTEX attribute, and both new edges
*           with the NEWEDGE attribute, so they can be
*           properly skipped during refinement.
*           Also sets FIXED attribute bit of new vertex and
*           edge if the old edge is FIXED.
* 
*  Input:   edge_id e_id - the edge to subdivide
*
*  Output:  new legal configuration with new vertex
*
*
*/
 
void edge_divide(e_id)
edge_id e_id;
{
  REAL s[MAXCOORD],*t,*mu,*mv,*h,m[MAXCOORD],q1[MAXCOORD],q3[MAXCOORD];
  edge_id  new_e;
  vertex_id divider,old_mid,new_mid,headv,tailv;
  int i,k;
  facetedge_id new_fe,old_fe;
  int wrap = 0;
  REAL w[MAXCOORD];

  if ( !valid_element(e_id) ) return;

  headv = get_edge_headv(e_id);
  tailv = get_edge_tailv(e_id);
  t = get_coord(tailv);
  if ( web.symmetry_flag )
     { wrap = get_edge_wrap(e_id);
       (*sym_wrap)(get_coord(headv),w,wrap);
       h = w;
     }
  else h = get_coord(headv);
  if ( web.modeltype == LINEAR )
   { 
     for ( k = 0 ; k < web.sdim ; k++ ) m[k] = (t[k] + h[k])/2;
     divider = new_vertex(m);
     if ( get_eattr(e_id) & FIXED ) set_attr(divider,FIXED);
    }
  else /* QUADRATIC */
    { 
      divider = get_edge_midv(e_id);  /* use old midpoint */
      /* get coordinates of new midpoint(s) */
      mu = get_coord(get_edge_midv(e_id));
      for ( k = 0 ; k < web.sdim ; k++ )
        {
          q1[k] = 0.375*t[k] + 0.75*mu[k] - 0.125*h[k];
          m[k] = mu[k];
          q3[k] = 0.375*h[k] + 0.75*mu[k] - 0.125*t[k];
        }
    }


  /* make refine() work properly */
  set_attr(divider,NEWVERTEX);

  new_e = new_edge(divider,get_edge_headv(e_id));
  set_edge_headv(e_id,divider);
  set_attr(new_e,NEWEDGE | get_eattr(e_id));
  set_original(new_e,get_original(e_id));
  set_edge_density(new_e,get_edge_density(e_id));
  set_edge_color(new_e,get_edge_color(e_id));
  if ( web.symmetry_flag )
    { set_edge_wrap(e_id,0);  /* new vertex in same unit cell as tail */
      set_edge_wrap(new_e,wrap); 
    }
  if ( (web.modeltype == LINEAR) && web.metric_flag )
    { /* get midpoint in metric middle */
      REAL front,rear;  /* edge lengths */
      REAL *xm = get_coord(divider);
      for(;;)  /* binary search for reasonable endpoint */ 
      {
        calc_edge(e_id);
        rear = get_edge_length(e_id);
        calc_edge(new_e);
        front = get_edge_length(new_e);
        if ( rear < 0.8*front )
        { /* bisect high end */
	  t = m;  /* low end */
	  for ( i = 0 ; i < web.sdim ; i++ )
	  { t[i] = xm[i];
	    xm[i] = (t[i] + h[i])/2;
          }
        }
        else if ( rear > 1.2*front )
        { /* bisect low end */
    	  h = w;  /* high end */
	  for ( i = 0 ; i < web.sdim ; i++ )
	  { h[i] = xm[i];
	    xm[i] = (t[i] + h[i])/2;
          }
        }
        else break;
      } 
    }
    
  if ( web.modeltype == QUADRATIC )
    {
      /* new midpoint for old edge */
      old_mid = new_vertex(q1);
      set_edge_midv(e_id,old_mid);
      if ( get_eattr(e_id) & FIXED ) set_attr(old_mid,FIXED);

      /* set up midpoint of new edge */
      new_mid = get_edge_midv(new_e);
      if ( get_eattr(e_id) & FIXED ) set_attr(new_mid,FIXED);
      mv = get_coord(new_mid);
      for ( k = 0 ; k < web.sdim ; k++ ) mv[k] = q3[k];
    }


  /* for free boundary edges, cannot just interpolate parameters
     due to wrap-around of angular parameters. So tangent extrapolate
     from one endpoint.
   */
  if ( get_eattr(e_id) & BOUNDARY )
    { 
      struct boundary *bdry;
      REAL *parama,*paramb,*parammid;
      REAL  defaultp[MAXCOORD];

      bdry = get_edge_boundary(e_id);
      set_edge_boundary(new_e,bdry);
      if ( web.modeltype == LINEAR )
        { 
          set_attr(divider,BOUNDARY);
          set_boundary(divider,bdry);

          /* find parameters of new midpoint */
          mv = get_coord(divider);
          parammid = get_param(divider);
          if ( (get_boundary(headv) != bdry) && (get_boundary(tailv) != bdry) )
            { sprintf(errmsg,"Vertices %d and %d of edge %d are on different boundaries.\n",
                ordinal(headv)+1,ordinal(tailv)+1,ordinal(e_id)+1);
              error(errmsg,WARNING);
              paramb = defaultp;
              defaultp[0] = defaultp[1] = defaultp[2] = 0.1;
              for ( i = 0 ; i < web.sdim ; i++ )
                s[i] = eval(bdry->coordf[i],defaultp);
              mu = s;
              /* projecting on tangent */
              b_extrapolate(bdry,mu,mv,mv,paramb,parammid);
            }
	  else
#ifdef PARAMAVG
          if ( (get_boundary(headv) == bdry) && (get_boundary(tailv) == bdry) )
           {
             mu = get_coord(tailv);
             parama = get_param(tailv);
             paramb = get_param(headv);
             /* projecting on tangent */
             b_extrapolate(bdry,mu,mv,mv,parama,parammid);
	     /* if not wrapped, take average parameter */
	     for (  i = 0 ; i < bdry->pcount ; i++ )
	       { if ( ((parama[i] < parammid[i]) && (parammid[i] < paramb[i]))
	         || ((parama[i] > parammid[i]) && (parammid[i] > paramb[i])))
		 parammid[i] = (parama[i] + paramb[i])/2;
	       }

           }
	  else
#endif
          if ( get_boundary(headv) == bdry )
           {
             mu = get_coord(headv);
             paramb = get_param(headv);
             /* projecting on tangent */
             b_extrapolate(bdry,mu,mv,mv,paramb,parammid);
	   }
	  else
           {
             mu = get_coord(tailv);
             paramb = get_param(tailv);
             /* projecting on tangent */
             b_extrapolate(bdry,mu,mv,mv,paramb,parammid);
	   }


        }
      else if ( web.modeltype == QUADRATIC )
        { 
          set_attr(old_mid,BOUNDARY);
          set_boundary(old_mid,bdry);
          set_attr(new_mid,BOUNDARY);
          set_boundary(new_mid,bdry);

          paramb = get_param(divider);
          mu = get_coord(divider);

          /* find new parameters of old edge midpoint */
          /* projecting on tangent */
          parammid = get_param(old_mid);          
          mv = get_coord(old_mid);
          b_extrapolate(bdry,mu,mv,mv,paramb,parammid);

          /* find parameters of new edge midpoint */
          /* projecting on tangent */
          parammid = get_param(new_mid);          
          mv = get_coord(new_mid);
          b_extrapolate(bdry,mu,mv,mv,paramb,parammid);
        }
    }
  else
    { 
      ATTR attr = get_eattr(e_id) & (BDRY_ENERGY | BDRY_CONTENT | CONSTRAINT );
      MAP conmap = get_e_constraint_map(e_id) | web.con_global_map;

      set_attr(new_e,attr);
      set_attr(divider,attr);
      if ( conmap || attr )
	{
          set_e_conmap(new_e,conmap);
          set_v_conmap(divider,conmap);
          project_v_constr(divider);
          if ( web.modeltype == QUADRATIC )
            {
              set_attr(old_mid,attr);
              set_attr(new_mid,attr);
              set_v_conmap(old_mid,conmap);
              set_v_conmap(new_mid,conmap);
              project_v_constr(old_mid);
              project_v_constr(new_mid);
	    }
        }
    }

  generate_edge_fe_init();  
  new_fe = NULLID;
  while ( generate_edge_fe(e_id,&old_fe) )
    { /* create new facetedge and splice into edge net */
      new_fe = new_facetedge(get_fe_facet(old_fe),new_e);
      set_next_edge(new_fe,get_next_edge(old_fe));
      set_prev_edge(get_next_edge(old_fe),new_fe);
      set_next_edge(old_fe,new_fe);
      set_prev_edge(new_fe,old_fe);
    }
  set_edge_fe(new_e,new_fe);
  set_vertex_fe(divider,new_fe);
  set_vertex_fe(headv,inverse_id(new_fe)); /* vertex at tail of its fe */

  generate_edge_fe_init();  
  while ( generate_edge_fe(e_id,&old_fe) )
   { /* copy over facet chain */
     facetedge_id fe,nfe;
     fe = get_next_edge(old_fe);
     nfe = get_next_facet(old_fe);
     set_next_facet(fe,get_next_edge(nfe));
     nfe = get_prev_facet(old_fe);
     set_prev_facet(fe,get_next_edge(nfe));
   }
}

/************************************************************
*
*   cross_cut()
*
*   Purpose: subdivides a facet into two facets by introducing
*            a new edge and a new facet.
*
*   Inputs:  facetedge_id first_fe  - first in chain defining new facet
*            facetidge_id last_fe   - last in chain;
*
*   Output:  new facet created with boundary first_fe,chain,last_fe,newedge
*            also both facets marked with NEWFACET attribute, so they
*            can be left alone during a refinement.
*
*/

void cross_cut(first_fe,last_fe)
facetedge_id first_fe,last_fe;
{
  facet_id old_f,new_f;
  edge_id  new_e;
  facetedge_id fe,new_fe_new,new_fe_old;
  int wrap;
  ATTR attr;

  old_f = get_fe_facet(first_fe);
  attr = get_fattr(old_f);

  new_e = new_edge(get_fe_headv(last_fe),get_fe_tailv(first_fe));
  if ( attr & FIXED )
    set_attr(new_e,FIXED);

  /* for QUADRATIC model, midpoint of new edge is left as the 
     linear interpolation given by new_edge() since triagular
     facets may not yet be established yet */

  new_f = new_facet();
  if ( inverted(old_f) ) invert(new_f);
  set_attr(old_f,NEWFACET);
  set_attr(new_f,attr);
  set_tag(new_f,get_tag(old_f));
  set_original(new_f,get_original(old_f));
  set_facet_color(new_f,get_facet_color(old_f));
  set_facet_density(new_f,get_facet_density(old_f));
  set_f_surfmap(new_f,get_f_surfen_map(old_f));
  set_f_quantmap(new_f,get_f_quant_map(old_f));
  set_facet_boundary(new_f,get_facet_boundary(old_f));
  if ( attr & BOUNDARY )
    { set_attr(new_e,BOUNDARY);
      set_edge_boundary(new_e,get_facet_boundary(old_f));
    }
  if ( attr & CONSTRAINT )   
    {
      ATTR cattr = attr & (BDRY_ENERGY | BDRY_CONTENT | CONSTRAINT );
      MAP conmap = get_f_constraint_map(old_f);

      set_attr(new_e,cattr);
      set_attr(new_f,cattr);
      set_e_conmap(new_e,conmap);
      set_f_conmap(new_f,conmap);
      if ( web.modeltype == QUADRATIC )
        {
          vertex_id mid = get_edge_midv(new_e);
          set_attr(mid,cattr);
          set_v_conmap(mid,conmap);
          project_v_constr(mid);
        }
    }

  new_fe_new = new_facetedge(new_f,new_e);
  new_fe_old = new_facetedge(old_f,edge_inverse(new_e));
  set_edge_fe(new_e,new_fe_new);
  set_facet_body(new_f,get_facet_body(old_f));
  set_facet_body(facet_inverse(new_f),get_facet_body(facet_inverse(old_f)));
  if ( phase_flag ) set_f_phase_density(new_f);
  set_facet_fe(new_f,new_fe_new);

  set_facet_fe(old_f,new_fe_old);

  /* install new facet into its facet-edges */
  /* and set torus wrap flags if needed */
  fe = first_fe;
  wrap = 0;
  while ( 1 )
  {
    set_fe_facet(fe,new_f);
    if ( web.symmetry_flag )
      wrap = (*sym_compose)(wrap,get_fe_wrap(fe));
    if ( equal_id(fe,last_fe) ) break;
    fe = get_next_edge(fe);
  }
  if ( web.symmetry_flag )
      set_edge_wrap(new_e,(*sym_inverse)(wrap));

  /* link up facet-edges */
  set_next_edge(get_prev_edge(first_fe),new_fe_old);
  set_prev_edge(new_fe_old,get_prev_edge(first_fe));
  set_prev_edge(get_next_edge(last_fe),new_fe_old);
  set_next_edge(new_fe_old,get_next_edge(last_fe));
  set_prev_edge(first_fe,new_fe_new);
  set_next_edge(new_fe_new,first_fe);
  set_next_edge(last_fe,new_fe_new);
  set_prev_edge(new_fe_new,last_fe);
  set_next_facet(new_fe_new,fe_inverse(new_fe_old));
  set_prev_facet(new_fe_new,fe_inverse(new_fe_old));
  set_next_facet(new_fe_old,fe_inverse(new_fe_new));
  set_prev_facet(new_fe_old,fe_inverse(new_fe_new));

}
