
/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*************************************************************
*
*    file:      lexinit.c
*
*    Purpose:   Reads in ASCII initial data files.
*
*     lexical analyzer version, using lex defs in datafile.lex
*/

#include "include.h"
#include "lex.h"
#include "express.h"
#include "ytab.h"

/***********************************************************
*
*  Function: reset_web()
*
*  Purpose: Zero out and initialize web storage structures
*
*/

void reset_web()
{
  int j;
  int cflag = web.torus_clip_flag;
  int bflag = web.torus_body_flag;

  reset_skeleton();  /* cleans out web and resets to 0 */

#ifdef TC
if ( farheapcheck() < 0 )
  error("Corrupt heap.\n",UNRECOVERABLE);
#endif

  /* reset nonzero flags and stuff */
  old_area_flag = 0;
  sqgauss_flag = 0;
  square_curvature_flag = 0;
  approx_curve_flag = 0;
  kusner_flag = 0;
  effective_area_flag = 0;
  autochop_flag = 0; autopop_flag = 0;
  runge_kutta_flag = 0;
  web.dimension = SOAPFILM;
  star_fraction = web.dimension + 1.0;
  areaname = "area";
  web.sdim = 3;
  total_time = 0.0;
  fixed_volumes = 0;
  compcount = web.sdim;
  homothety_target = 1.0;
  web.modeltype = LINEAR;
  web.hide_flag = 1;
  web.torus_clip_flag = cflag;
  web.torus_body_flag = bflag;
  web.torus_flag = 0;
  web.full_flag = 0;
  web.meritfactor = 1.0;
  web.grav_const = 1.0;
  web.symmetric_content = 0;
  web.pressure_flag = 0;
  web.projection_flag = 0;
  web.area_norm_flag = 0;
  web.vol_flag = 0;
  web.jiggle_flag = 0;
  web.temperature = 0.05;
  web.total_area = 0.0;
  web.total_facets = 0;
  web.scale = 0.1;
  web.maxscale = 1.0;
  web.pressure = 0.0;
  web.bodycount = 0;
  web.wulff_count = web.wulff_flag = 0;
  web.min_area = 0.1;
  web.min_length = 0.25;
  web.max_len = 1.4;
  web.max_angle = 0.1;
  web.spring_constant = 1.0;
  web.gauss_order = 2;  /* can do cubics exactly */
  web.gauss2D_order = 7; 
  web.tolerance = 0.00001;
  level = 0;
  no_refine = 0;
  zoom_number = 1;
  web.zoom_v = NULLVERTEX;
  web.zoom_radius = 99999.0;
  for ( j = 0 ; j < web.sdim ; j++ )
    web.torus_period[j][j] = 1.0;
  if ( tang ) { free_matrix(tang); tang = NULL; }
}

/******************************************************************
*
*  Function: initialize()
*
*  Purpose:  read in data file and create initial triangulation.
*
*/


#define MAXLIST 100
#define LINESIZE 1000


/* temporary lists of elements */
  vertex_id *vlist = NULL;
  int vmaxlist;
  edge_id *elist = NULL;
  int emaxlist;
  facet_id *flist = NULL;
  int fmaxlist;
  int facecount; 
  body_id *blist = NULL;
  int bmaxlist;

void initialize()  /* initialize from data file */
{
  int f;
  facetedge_id fe;
  int i,j,k;

  line_no = 1;
  parse_error_flag = 0;
  parse_errors = 0;
  recovery_flag = 0;
  facecount = 0;
  bare_edge_count = 0;
  vlist = NULL;
  elist = NULL;
  flist = NULL;
  blist = NULL;

  /* read in any header information */
  tok = yylex();
  while ( tok != 0 )
   switch ( tok )
    { 
      case SPACE_DIMENSION_:
          tok = gettok(INTEGER_);
	  web.sdim = int_val;
	  if ( int_val > MAXCOORD )
	    error("Space dimension too high.\n",DATAFILE_ERROR);
	  tok = yylex();
	  break;

      case SURFACE_DIMENSION_:
          tok = gettok(INTEGER_);
	  web.dimension = int_val;
	  if ( (int_val > MAXCOORD-1) || (int_val > web.sdim) )
	    error("Surface dimension too high.\n",DATAFILE_ERROR);
          star_fraction = web.dimension + 1.0;
	  tok = yylex();
	  break;

      case SIMPLEX_REP_:
	  web.simplex_flag = 1;
	  tok = yylex();
	  break;
	
      case CONFORMAL_:  /* have background metric on domain */
	  web.metric_flag = 1;
	  web.conformal_flag = 1;
	  recovery_flag = 0;
          if (exparse(web.sdim,&web.metric[0][0].root) <= 0 )
            { sprintf(errmsg,"Bad metric g[0][0] definition.\n");
              error(errmsg,DATAFILE_ERROR);
	    }
	  metric = dmatrix(0,web.sdim-1,0,web.sdim-1);
	  metric_partial = dmatrix3(web.sdim,web.sdim,
	     web.sdim);
	  det_array = dmatrix(0,web.dimension-1,0,web.dimension-1);
	  tok = yylex();
	  break;
	  
	
      case METRIC_:  /* have background metric on domain */
	  web.metric_flag = 1;
	  recovery_flag = 0;
	  for ( i = 0 ; i < web.sdim ; i++ )
	    for ( j = 0 ; j < web.sdim ; j++ )
	      { int esize;
                esize = exparse(web.sdim,&web.metric[i][j].root);
                if ( esize <= 0 )
                  { sprintf(errmsg,
                       "Bad metric g[%d][%d] definition.\n",i,j);
                    error(errmsg,DATAFILE_ERROR);
		  }
	      }
	  metric = dmatrix(0,web.sdim-1,0,web.sdim-1);
	  metric_partial = dmatrix3(web.sdim,web.sdim,
	     web.sdim);
	  det_array = dmatrix(0,web.dimension-1,0,web.dimension-1);
	  tok = yylex();
	  break;
	  
      case SYMMETRY_GROUP_:
          web.symmetry_flag = 1;
          tok = yylex();
          if ( tok == QUOTATION_ )
            {  if ( strcmp(yytext,symmetry_name) != 0 )
		 error("Symmetry name does not match internal name.\n",
		    DATAFILE_ERROR);
               tok = yylex();
            }
          else
              error("Cannot find symmetry group name.\n",DATAFILE_ERROR);
	  sym_wrap = group_wrap;
	  sym_form_pullback = group_form_pullback;
	  sym_inverse = group_inverse;
	  sym_compose = group_compose;
          recovery_flag = 0;
          break;


      case TORUS_:
          web.torus_flag = 1;
	  web.symmetry_flag = 1;
	  sym_wrap = torus_wrap;
	  sym_form_pullback = torus_form_pullback;
	  sym_inverse = torus_inverse;
	  sym_compose = torus_compose;
          tok = yylex();
          recovery_flag = 0;
          break;

      case TORUS_FILLED_:
          web.torus_flag = 1;
	  web.symmetry_flag = 1;
	  sym_wrap = torus_wrap;
	  sym_form_pullback = torus_form_pullback;
	  sym_inverse = torus_inverse;
	  sym_compose = torus_compose;
          web.full_flag = 1;
          tok = yylex();
          recovery_flag = 0;
          break;

      case STRING_:
          web.dimension = STRING;
          star_fraction = web.dimension + 1.0;
          tok = yylex();
          recovery_flag = 0;
          break;

      case SOAPFILM_:
          web.dimension = SOAPFILM;
          star_fraction = web.dimension + 1.0;
          tok = yylex();
          recovery_flag = 0;
          break;

      case LINEAR_:
          web.modeltype = LINEAR;
          tok = yylex();
          recovery_flag = 0;
          break;
           
      case QUADRATIC_:
          web.modeltype = QUADRATIC;
          tok = yylex();
          recovery_flag = 0;
          break;
           
      case SYMMETRIC_CONTENT_:
          web.symmetric_content = 1;
          tok = yylex();
          recovery_flag = 0;
          break;
           
      case MEAN_CURV_:
          web.area_norm_flag = 1;
          tok = yylex();
          recovery_flag = 0;
          break;

      case EFFECTIVE_AREA_:
          effective_area_flag = 1;
          tok = yylex();
          recovery_flag = 0;
          break;
           
      case JIGGLE_:
          web.jiggle_flag = 1;
          tok = yylex();
          recovery_flag = 0;
          break;
           

      case WULFF_: 
          tok = yylex();
          if ( tok == QUOTATION_ )
            {  wulff_initialize(yytext);
               tok = yylex();
            }
          else
              error("Cannot find Wulff file name.\n",DATAFILE_ERROR);
          recovery_flag = 0;
          break;

      case PHASEFILE_: 
          tok = yylex();
          if ( tok == QUOTATION_ )
            {  phase_initialize(yytext);
               tok = yylex();
            }
          else
              error("Cannot find phasefile name.\n",DATAFILE_ERROR);
          recovery_flag = 0;
          break;

      case PERIODS_:  recovery_flag = 0; read_periods(); break;

      case PARAMETERS_:  recovery_flag = 0; read_parameter(); break;

      case BOUNDARY_:  recovery_flag = 0; read_boundary(); break;

      case CONSTRAINT_:  recovery_flag = 0; read_constraint(); break;

      case SURFACE_ENERGY_: recovery_flag = 0; read_surface_energy(); break;

      case QUANTITY_: recovery_flag = 0; read_quantity(); break;

      case DIFFUSION_:
          web.diffusion_flag = 1;
          web.diffusion_const = 0.0;
          if ( read_const(&web.diffusion_const) <= 0 )
            error("No DIFFUSION value.\n",WARNING);
          else tok = yylex();
          recovery_flag = 0;
          break;

      case AUTOPOP_:
          autopop_flag = 1;
          tok = yylex();
          recovery_flag = 0;
          break;
           
      case AUTOCHOP_:
          autochop_flag = 1;
          autochop_size = 1.0;
          if ( read_const(&autochop_size) <= 0 )
            error("No AUTOCHOP length.\n",WARNING);
          else tok = yylex();
          recovery_flag = 0;
          break;
	  
      case APPROX_CURV_:
	  approx_curve_flag = 1;
	  tok = yylex();
	  recovery_flag = 0;
	  break;

      case SQUARE_CURVATURE_:
          /* allocate structure */
          if ( web.params == NULL )
            web.params = (struct param *)mycalloc(1,sizeof(struct param));
          else
            web.params = (struct param *)kb_realloc((char *)web.params,
                      (web.paramcount+1)*sizeof(struct param));
	  square_curvature_flag = 1;
	  square_curvature_param = web.paramcount;
	  strncpy(web.params[web.paramcount].name,
	      "square curvature modulus",31);
          if ( read_const(&web.params[web.paramcount++].value) <= 0 )
            error("No square curvature modulus value.\n",WARNING);
          else tok = yylex();
          recovery_flag = 0;
          break;


      case SCALE_LIMIT_:
          if ( read_const(&web.maxscale) <= 0 )
            error("No SCALE LIMIT value.\n",WARNING);
          else tok = yylex();
          recovery_flag = 0;
          break;

      case TOTAL_TIME_:
          if ( read_const(&total_time) <= 0 )
            error("No TOTAL_TIME value.\n",WARNING);
          else tok = yylex();
          recovery_flag = 0;
          break;

      case ZOOM_RADIUS_:
          if ( read_const(&web.zoom_radius) <= 0 )
            error("No ZOOM RADIUS value.\n",WARNING);
          else tok = yylex();
          recovery_flag = 0;
          break;

      case ZOOM_VERTEX_:
	  tok = yylex();
	  if ( tok != INTEGER_ )
            error("No ZOOM VERTEX number.\n",WARNING);
          else { zoom_number = int_val; tok = yylex(); }
          recovery_flag = 0;
          break;

      case INTEGRAL_ORDER_:
	  tok = yylex();
	  if ( tok != INTEGER_ )
            error("No INTEGRAL_ORDER value.\n",WARNING);
	  if ( int_val < 1 )
           { sprintf(errmsg,"Invalid INTEGRAL_ORDER value %d.\n",int_val);
             error(errmsg,WARNING);
           }
	  else  web.gauss_order = int_val; 
          tok = yylex();
          recovery_flag = 0;
          break;

      case CONSTRAINT_TOLERANCE_:
          if ( read_const(&web.tolerance) <= 0 )
	    error("No CONSTRAINT_TOLERANCE value.\n",WARNING);
          else tok = yylex();
          recovery_flag = 0;
          break;

      case MERITFACTOR_:
          if ( read_const(&web.meritfactor) <= 0 )
            error("No MERIT FACTOR value.\n",WARNING);
          else tok = yylex();
          recovery_flag = 0;
          break;

      case GRAV_CONST_:
          if ( read_const(&web.grav_const) <= 0 )
            error("No GRAVITY_CONST value.\n",WARNING);
          else 
	    { if ( web.grav_const != 0.0 )  web.gravflag = 1;
              tok = yylex();
	    }
          recovery_flag = 0;
          break;

      case SPRING_CONSTANT_:
          if ( read_const(&web.spring_constant) <= 0 )
            error("No SPRING_CONSTANT value.\n",WARNING);
          else tok = yylex();
          recovery_flag = 0;
          break;

      case SCALE_:
          if ( read_const(&web.scale) <= 0 )
            error("No SCALE value.\n",WARNING);
          else tok = yylex();
          if ( tok == FIXED_ )
            { web.motion_flag = 1;
              tok = yylex();
            }
          recovery_flag = 0;
          break;

      case TEMPERATURE_:
          if ( read_const(&web.temperature) <= 0 )
            error("No TEMPERATURE value.\n",WARNING);
          else tok = yylex();
          recovery_flag = 0;
          break;

      case PRESSURE_:
          if ( read_const(&web.pressure) <= 0 )
            error("No PRESSURE value.\n",WARNING);
          else 
	    { tok = yylex(); web.pressure_flag = 1; }
          recovery_flag = 0;
          break;

      case VERTICES_: 
	  if ( (web.dimension > SOAPFILM) && !web.simplex_flag )
	    error("Must have simplex representation for surface dimension over 2.\n",UNRECOVERABLE);
          recovery_flag = 0; 
	  convert_all_expr(); /* so can project constraints */ 
          read_vertices();
          break;

      case EDGES_: 
          recovery_flag = 0; 
          read_edges();
          break;

      case FACES_: 
          recovery_flag = 0; 
          read_faces();
          break;

      case BODIES_: 
          recovery_flag = 0; 
          read_bodies();
          break;

      default: 
          if ( !recovery_flag )
            { sprintf(errmsg,"Illegal token '%s'.\n",yytext);
              error(errmsg,PARSE_ERROR);
            }
          tok = yylex();
          break;
    }
          
  /* set up gaussian quadrature */
  gauss_setup();

  if ( web.dimension == STRING ) areaname = "length";
  else areaname = "area";

  /* create initial triangulation of each face */
  if ( !web.simplex_flag && (web.dimension == SOAPFILM) )
   for ( f = 1 ; f <= facecount ; f++ )
    { 
      int edgecount = 0;

      if ( !valid_id(flist[f]) ) continue;

      /* see how many edges, and if we want to refine */
      generate_facet_fe_init();
      while ( generate_facet_fe(flist[f],&fe) )
         edgecount++;
      if ( edgecount > 3 ) face_triangulate(flist[f],edgecount);
    }

  if ( web.torus_flag && web.bodycount )
    fix_volconst();  /* adjust torus volume constants */

  /* straighten out facet order around edges */
  if ( !web.simplex_flag && (web.dimension == SOAPFILM) )
    fe_reorder();
     
  /* put facet-edges on string network if no facets defined */
  if ( web.dimension == STRING )
    string_fixup();

  /* phase boundary energies */
  if ( phase_flag && (web.dimension == STRING) )
    { edge_id e_id;
      FOR_ALL_EDGES(e_id)
	set_e_phase_density(e_id);
    }
  if ( phase_flag && (web.dimension != STRING) )
    { facet_id f_id;
      FOR_ALL_FACETS(f_id)
	set_f_phase_density(f_id);
    }

  /* run preliminary checks */
  if ( !web.simplex_flag )
    if ( facetedge_check(PRELIMCHECK) || facet_body_check() )
      error("Bad data file.\n",DATAFILE_ERROR);


  if ( vlist == NULL )
    error("No vertices found in datafile.\n",RECOVERABLE);
  if ( (elist == NULL) && !web.simplex_flag )
    error("No edges found in datafile.\n",RECOVERABLE);

  if ( vlist ) free((char *)vlist);
  if ( elist ) free((char *)elist);
  if ( flist ) free((char *)flist);
  if ( blist ) free((char *)blist);
       
  datafile_flag = 0;

  for ( k = 2, simplex_factorial = 1.0 ; k <= web.dimension ; k++ )
    simplex_factorial *= k;
  volume_factorial = simplex_factorial*web.sdim;
  if ( web.simplex_flag )
   {
     refine_simplex_init();
   }

  if ( autopop_flag )
	{ int n;
          if ( web.dimension == STRING )
            sprintf(msg,"Number of vertices popped: %d\n", n = verpop_str());
          else
            sprintf(msg,"Number of vertices popped: %d\n", n = edgepop_film());
          outstring(msg);
          if ( n > 0 ) update_display();
	}

}  /* end initialize */

/************************************************************************
*
*  Function: read_vertices()
*
*/

void read_vertices()
{
  int k;
  double c[MAXCOORD];  /* temporary number buffer */
  int cnum,bnum,pcount;
  struct boundary *bdry;
  struct constraint *constr;
  int more_attr;

  /* read in vertex coordinates */
  vmaxlist = MAXLIST;
  vlist = (vertex_id *)mycalloc(sizeof(vertex_id),vmaxlist);
  if ( tok != VERTICES_  ) 
         error("Cannot find VERTICES.\n",UNRECOVERABLE);
  tok = yylex();
  while ( tok == LEAD_INTEGER_ )
   { 
     k = int_val;
     for ( pcount = 0 ; pcount < web.sdim ; pcount++ )
         if ( read_const(&c[pcount]) <= 0 ) break;
     if ( pcount < 1 ) break;
     tok = yylex();

     while ( k >= vmaxlist )
       { int spot = vmaxlist; 
         vmaxlist = k + MAXLIST;
         vlist = (vertex_id *)kb_realloc((char *)vlist,
				    vmaxlist*sizeof(vertex_id));
         if ( vlist == NULL )
            error("Cannot allocate memory.\n",UNRECOVERABLE);
         for ( ; spot < vmaxlist ; spot ++ ) vlist[spot] = NULLID;
       }
     if ( valid_id(vlist[k]) )
       { sprintf(errmsg,"Duplicate vertex number %d\n",k);
         error(errmsg,DATAFILE_ERROR);
       }
     vlist[k] = new_vertex(c);
     set_original(vlist[k],k);

     /* attributes */
     if ( web.con_global_map )
       { set_v_global(vlist[k]);
       }
     for ( more_attr = 1 ; more_attr ; )
       switch ( tok )
          {
             case FIXED_:
               set_attr(vlist[k],FIXED);
               tok = yylex();
               break;

             case BOUNDARY_:
               {
                 REAL *x,*param;
                 int n;

                 tok = gettok(INTEGER_);

                 bnum = abs(int_val);
                 if ( tok != INTEGER_ )  
                   { error("Need boundary number.\n",DATAFILE_ERROR);
                     break;
                   }
                 if ( (bnum >= BDRYMAX) 
                                 || (web.boundaries[bnum].coordf[0] == NULL) )
                   {
                     sprintf(errmsg,
                        "Bad boundary number %d for vertex %d.\n",bnum,k);
                     error(errmsg,DATAFILE_ERROR);
                     yylex();
                     break;
                   }        
                 set_attr(vlist[k],BOUNDARY);
                 if ( int_val < 0 )  set_attr(vlist[k],NEGBOUNDARY); 
                 set_boundary_num(vlist[k],bnum);
                 bdry = get_boundary(vlist[k]); 
                 if ( pcount != bdry->pcount )
                  { sprintf(errmsg,
                      "Wrong number of parameters for vertex %d.\n",k);
                    error(errmsg,DATAFILE_ERROR);
                  }        
                 if ( (bdry->attr & BDRY_ENERGY) && (yytext[0] != '0') )
                   set_attr(vlist[k], BDRY_ENERGY);
                 if ( bdry->attr & BDRY_CONTENT )
                   set_attr(vlist[k], BDRY_CONTENT);
                 param = get_param(vlist[k]);
                 x = get_coord(vlist[k]);
                 for ( n = 0 ; n < MAXPARAM ; n++ )
                     param[n] = x[n];
                 for ( n = 0 ; n < web.sdim ; n++ )
                     x[n] = eval(bdry->coordf[n],param);
                 tok = yylex();
               }
            break;

     /* see if constraint point */
     case CONSTRAINT_:
         tok = gettok(INTEGER_);
         while  ( tok == INTEGER_ )
           {
             cnum = abs(int_val);
             if ( (cnum >= CONSTRMAX)
                                 || (web.constraints[cnum].formula == NULL) )
               {
                 sprintf(errmsg,
                    "Bad constraint number %d for vertex %d.\n",cnum,k);
                 error(errmsg,DATAFILE_ERROR);
                 tok = yylex();
                 break;
               }
             set_attr(vlist[k],CONSTRAINT);
             if ( int_val < 0 )  set_attr(vlist[k],NEGBOUNDARY); 
             set_v_constraint_map(vlist[k],cnum);
             constr = get_constraint(cnum); 
             if ( (constr->attr & BDRY_ENERGY) && (yytext[0] != '0') )
               set_attr(vlist[k], BDRY_ENERGY);
             if ( constr->attr & BDRY_CONTENT )
               set_attr(vlist[k], BDRY_CONTENT);
             tok = yylex();
           }
         project_v_constr(vlist[k]);
         break;

         default:  more_attr = 0;
        }

     if ((get_vattr(vlist[k])&(BOUNDARY|CONSTRAINT)) == (BOUNDARY|CONSTRAINT))
        error("Cannot have constraint and boundary.",WARNING);
     if ( !(get_vattr(vlist[k])&BOUNDARY) && (pcount != web.sdim) )
       { sprintf(errmsg,"Wrong number of coordinates for vertex %d.\n",k);
         error(errmsg,WARNING);
        }
   }
  web.zoom_v = vlist[zoom_number];  /* vertex for zooming in on */
}  /* end read_vertices() */


/************************************************************************
*
*  Function: read_edges()
*
*/

void read_edges()
{
  int i,k;
  int head,tail;
  int cnum;
  struct boundary *bdry;
  struct constraint *constr;
  int more_attr;
  REAL value;  /* for constant expression values */
  WRAPTYPE wrap;
  int compcount;  /* proper number of components for integrands */

  compcount = binom_coeff(web.sdim,web.dimension-1);

  /* read in edges */
  emaxlist = MAXLIST;
  elist = (edge_id *)mycalloc(sizeof(edge_id),emaxlist);
  while ( (tok != EDGES_) && (tok != 0 ) ) 
    tok = yylex();
  if ( tok != EDGES_ )
        { sprintf(errmsg,"EDGES should be here instead of '%s'\n",yytext);
          error(errmsg,RECOVERABLE);
        }
  tok = yylex();
  while ( tok == LEAD_INTEGER_ )
   { wrap = 0;

     /* check edge number */
     k = int_val;
     while ( k >= emaxlist )
       { int spot = emaxlist; 
         emaxlist= k + MAXLIST;
         elist = (edge_id *)kb_realloc((char *)elist,emaxlist*sizeof(edge_id));
         if ( elist == NULL )
           error("Cannot allocate memory.\n",UNRECOVERABLE);
         for ( ; spot < emaxlist ; spot ++ ) elist[spot] = NULLID;
       }
     if ( valid_id(elist[k]) )
       { sprintf(errmsg,"Duplicate edge number %d\n",k);
         error(errmsg,DATAFILE_ERROR);
       }

     if ( web.simplex_flag )
     { /* read vertex list */
       int vcount = 0;
       vertex_id *v;
       elist[k] = new_edge(NULLID,NULLID);
       v = get_edge_vertices(elist[k]);
       while ( (tok = gettok(INTEGER_)) == INTEGER_ )
        { if ( vcount++ > (web.dimension-1) )
	    error("Too many vertices for edge.\n",DATAFILE_ERROR);
          if ( !valid_id(vlist[int_val]) )
          { sprintf(errmsg," Edge %d: vertex %d is not defined.\n",k,int_val);
            error(errmsg,DATAFILE_ERROR);
          }
	  *(v++) = vlist[int_val];
	}
       if ( vcount < web.dimension  )
	 error("Too few vertices for edge.\n",DATAFILE_ERROR);
     }
     else  /* facet_edge representation */
     {
     if ( (tok = gettok(INTEGER_)) != INTEGER_ )
        error("Edge tail vertex missing.\n",RECOVERABLE);
     tail = int_val;
     if ( (tok = gettok(INTEGER_)) != INTEGER_ )
       error("Edge head vertex missing.\n",RECOVERABLE);
     head = int_val;
     tok = yylex();
     if ( (tail < 0) || (tail >= vmaxlist) || !valid_id(vlist[tail]) )
       { sprintf(errmsg,"Edge %d: tail vertex %d is not defined.\n",k,tail);
         tail = 0;
         error(errmsg,DATAFILE_ERROR);
       }
     if ( (head < 0 ) || (head >= vmaxlist) || !valid_id(vlist[head]) )
       { sprintf(errmsg,"Edge %d: head vertex %d is not defined.\n",k,head);
         head = 0;
         error(errmsg,DATAFILE_ERROR);
       }
     elist[k] = new_edge(vlist[tail],vlist[head]);

     } 
     if ( web.torus_flag )
       { 
         for ( i = 0 ; i < web.sdim ; i++, tok = yylex() )
           switch ( tok )
             {
               case '+':
			  wrap += POSWRAP << (i*TWRAPBITS);
			  break;

               case '*':  break;

               case '-':
	       case UMINUS_:
			  wrap += NEGWRAP << (i*TWRAPBITS);
			  break;

               default :
                          error("Edge missing torus wrap sign.\n",WARNING);
                          break;
            }
       }

     set_original(elist[k],k);

     if ( web.dimension == STRING )
       set_edge_density(elist[k],1.0);
     else
       set_edge_density(elist[k],0.0);
     /* check attributes */
     for ( more_attr = 1; more_attr ; )
       switch ( tok )
         {
	   case WRAP_:
		if ( !web.symmetry_flag )
		   error("Cannot do wraps without torus or symmetry group.\n",
		     DATAFILE_ERROR);
		if ( read_const(&value) < 0 ) 
		  error("Missing wrap value.\n",DATAFILE_ERROR);
		wrap = (WRAPTYPE)value;
		set_edge_wrap(elist[k],wrap);
                tok = yylex();
		break;

           case FIXED_:
                set_attr(elist[k],FIXED);  
                set_attr(vlist[tail],FIXED);
                set_attr(vlist[head],FIXED);
                tok = yylex();
                break;

           case COLOR_:
		tok = gettok(INTEGER_);
		set_edge_color(elist[k],int_val);
	        tok = yylex();
		break;

           case BOUNDARY_:
                tok = gettok(INTEGER_);
                cnum = abs(int_val);
                if ( (cnum >= BDRYMAX) || (tok != INTEGER_ )
                                 || (web.boundaries[cnum].coordf[0] == NULL) )
                  {
                    sprintf(errmsg,
                      "Bad boundary number %d for edge %d.\n",cnum,k);
                    error(errmsg,DATAFILE_ERROR);
                  }
                set_attr(elist[k],BOUNDARY);
                if ( int_val < 0 )  set_attr(vlist[k],NEGBOUNDARY); 
                set_edge_boundary_num(elist[k],cnum);
                bdry = get_edge_boundary(elist[k]); 
                if ( (bdry->attr & BDRY_ENERGY) && (yytext[0] == '0') )
                  set_attr(elist[k], BDRY_ENERGY);
                if ( bdry->attr & BDRY_CONTENT )
                  set_attr(elist[k], BDRY_CONTENT);
                tok = yylex();
                break;

         case CONSTRAINT_:
         for ( tok = gettok(INTEGER_) ; tok == INTEGER_ ;  tok = yylex() )
           { struct constraint *con;
             cnum = abs(int_val);
             if ( (cnum >= CONSTRMAX)
                                 || (web.constraints[cnum].formula == NULL) )
               {
                 sprintf(errmsg,
                    "Bad constraint number %d for edge %d.\n",cnum,k);
                 error(errmsg,DATAFILE_ERROR);
               }
             con = get_constraint(cnum);
             /* check consistency of number of components */
	     if ( con->attr & (CON_ENERGY | CON_CONTENT) )
 	       if ( web.constraints[cnum].compcount != compcount )
		 error("Inconsistent number of components in constraint integrands.\n",
                  WARNING);
             set_attr(elist[k],CONSTRAINT);
             if ( int_val < 0 )  set_attr(elist[k],NEGBOUNDARY); 
             set_e_constraint_map(elist[k],cnum);
             constr = get_constraint(cnum); 
             if ( (constr->attr & BDRY_ENERGY) && (yytext[0] != '0') )
               set_attr(elist[k], BDRY_ENERGY);
             if ( constr->quantity_map ) /* quantities done with energies */
               set_attr(elist[k], BDRY_ENERGY);
             if ( constr->attr & BDRY_CONTENT )
               set_attr(elist[k], BDRY_CONTENT);
           }
         break;

         case DENSITY_:
             if ( read_const(&value) <= 0 )
               error("No DENSITY value.\n",WARNING);
             set_attr(elist[k],DENSITY);
             set_edge_density(elist[k],value);
             tok = yylex();
             break;


        case QUANTITY_:
        /* see if quantity */
            tok = gettok(INTEGER_);
            for ( i = 0 ; i < QUANTMAX ; i++, tok = yylex() ) 
              {
                if ( tok != INTEGER_ ) break;
                cnum = abs(int_val);
                if ( cnum == 0 ) break;  /* no more numbers */
                if ( (cnum >= QUANTMAX)
                                 || (web.quants[cnum].quanvect[0] == NULL) )
                  {
                    sprintf(errmsg,
                       "Bad quantity number %d for face %d.\n",cnum,k);
                    error(errmsg,DATAFILE_ERROR);
                  }
                set_attr(elist[k],SURF_QUANTITY);
/*              set_e_quant_map(elist[k],cnum);  */
error("Implement edge integral energy with constraint.\n",WARNING);
              }
            break;


        case ENERGY_:
        /* see if surface energy */
            tok = gettok(INTEGER_);
            for ( i = 0 ; i < SURFENMAX ; i++, tok = yylex() ) 
              {
                if ( tok != INTEGER_ ) break;
                cnum = abs(int_val);
                if ( cnum == 0 ) break;  /* no more numbers */
                if ( (cnum >= SURFENMAX)
                                 || (web.surfen[cnum].envect[0] == NULL) )
                  {
                    sprintf(errmsg,
                       "Bad energy number %d for edge %d.\n",cnum,k);
                    error(errmsg,DATAFILE_ERROR);
                  }
                set_attr(elist[k],SURF_ENERGY);
/*                set_e_surfen_map(elist[k],cnum);  */
error("Implement edge integral energy with constraint.\n",WARNING);
              }
            break;

         default: more_attr = 0; break;
       }

     if ( web.symmetry_flag )
       {  set_edge_wrap(elist[k],wrap);
          if ( web.modeltype == QUADRATIC )
           { /* have to adjust midpoints */
             REAL *x = get_coord(get_edge_midv(elist[k]));
	     REAL *t = get_coord(get_edge_tailv(elist[k]));
	     REAL h[MAXCOORD];
	     (*sym_wrap)(get_coord(get_edge_headv(elist[k])),h,wrap);
             for ( i = 0 ; i < web.sdim ; i++ )
                   x[i] = 0.5*(t[i] + h[i]);
           }
        }
    }
} /* end read_edges() */

/************************************************************************
*
*  Function: read_faces()
*
*/

void read_faces()
{
  int i,k;
  int e;
  facetedge_id fe,old_fe;
  facet_id ff_id;
  int cnum,bnum;
  struct constraint *constr;
  REAL value;  /* for constant expression values */
  int compcount;  /* proper number of components for integrands */

  compcount = binom_coeff(web.sdim,web.dimension-1);

  if ( !web.simplex_flag && emaxlist == 0 ) 
    error("No edges. \n",RECOVERABLE);

  /* read in faces */
  fmaxlist = MAXLIST;
  flist = (facet_id *)mycalloc(sizeof(facet_id),fmaxlist);
  if ( tok != FACES_ ) 
        error("Cannot find FACES.\n",RECOVERABLE);
  tok = yylex();
  while ( tok == LEAD_INTEGER_ )
   {
     int more_attr;
     vertex_id hh,tt;
     int edge_count = 0;

     k = int_val;
     old_fe = NULLFACETEDGE;
     while ( k >= fmaxlist )
       { int spot = fmaxlist;

         fmaxlist = k + MAXLIST;
         flist = (facet_id*)kb_realloc((char *)flist,fmaxlist*sizeof(facet_id));
         if ( flist == NULL )
           error("Cannot allocate memory.\n",UNRECOVERABLE);
         for ( ; spot < fmaxlist ; spot++ ) flist[spot] = NULLID;
       }
     if ( valid_id(flist[k]) )
       { sprintf(errmsg,"Duplicate face number %d\n",k);
         error(errmsg,DATAFILE_ERROR);
       }
     flist[k] = new_facet();
     set_original(flist[k],k);
     if ( web.simplex_flag )
     { /* read vertex list */
       int vcount = 0;
       vertex_id *v = get_facet_vertices(flist[k]);
       while ( (tok = gettok(INTEGER_)) == INTEGER_ )
        { if ( vcount++ > web.dimension )
	    error("Too many vertices for facet.\n",DATAFILE_ERROR);
          if ( !valid_id(vlist[int_val]) )
          { sprintf(errmsg," Facet %d: vertex %d is not defined.\n",k,int_val);
            error(errmsg,DATAFILE_ERROR);
          }
	  *(v++) = vlist[int_val];
	}
       if ( vcount < web.dimension + 1 )
	 error("Too few vertices for  facet.\n",DATAFILE_ERROR);
     }
     else  /* facet_edge representation */
     {
      while ( (tok = gettok(INTEGER_)) == INTEGER_ )
       { 
         edge_id e_id;
         facetedge_id edge_fe;
         
         edge_count++;
         e = int_val;
         if ( e >= emaxlist )
          { sprintf(errmsg," Facet %d: edge %d is not defined.\n",k,e);
            e = 0;
            error(errmsg,DATAFILE_ERROR);
          }
         e_id =  e > 0 ? elist[e] : edge_inverse(elist[-e]);

         if ( !valid_id(e_id) )
          { sprintf(errmsg," Facet %d: edge %d is not defined.\n",k,e);
            error(errmsg,DATAFILE_ERROR);
          }

         fe = new_facetedge(flist[k],e_id);
         if ( !valid_id(get_facet_fe(flist[k])) )
           { set_facet_fe(flist[k],fe);}
         else
           {
             hh = get_fe_headv(old_fe);
             tt = get_fe_tailv(fe);
             if ( !equal_id(hh,tt) ) 
               fprintf(stderr,"Inconsistency in face %d, edge %d tail vertex disagrees with previous head.\n",
                   k,e);
             set_next_edge(old_fe,fe);
           }
         set_prev_edge(fe,old_fe);
         old_fe = fe;

         set_vertex_fe(get_edge_tailv(e_id),fe);   /* link in vertices */
         set_vertex_fe(get_edge_headv(e_id),fe_inverse(fe));

         /* add to edge facet list, not in geometric order */
         edge_fe = get_edge_fe(e_id);
         if ( valid_id(edge_fe) )
           { /* insert in chain */
             set_next_facet(fe,get_next_facet(edge_fe));
             set_prev_facet(fe,edge_fe);
             set_prev_facet(get_next_facet(fe),fe);
             set_next_facet(edge_fe,fe);
           }
         else
           { set_next_facet(fe,fe);
             set_prev_facet(fe,fe);
             set_edge_fe(e_id,fe);     /* link edge to rest of world */
           }

       }
     if ( ((web.dimension == STRING) && (edge_count < 1))
           || ((web.dimension == SOAPFILM) && (edge_count < 3)) )
       { sprintf(errmsg,"Face %d has too few edges.\n",k);
         error(errmsg,DATAFILE_ERROR);
       }
     ff_id = get_facet_fe(flist[k]);
     tt = get_fe_tailv(ff_id);
     hh = get_fe_headv(fe);
     if ( equal_id(tt,hh) ) 
       {
         set_next_edge(fe,get_facet_fe(flist[k]));  /* close up ring */
         set_prev_edge(get_facet_fe(flist[k]),fe);
       }
     else 
       { if ( web.dimension != STRING )
          { sprintf(errmsg,
              "Inconsistency in face %d first and last edges.\n",k);
            error(errmsg,DATAFILE_ERROR);
          }
       }
     } /* end facet-edge representation edge read */

     set_facet_density(flist[k],1.0);
     /* have attributes, maybe */
     for ( more_attr = 1 ; more_attr ; )
      switch ( tok )
       { 
         case DENSITY_:
             if ( read_const(&value) <= 0 )
               error("No DENSITY value.\n",WARNING);
             set_attr(flist[k],DENSITY);
             set_facet_density(flist[k],value);
             tok = yylex();
             break;

         case NODISPLAY_:
        /* see if want not to be displayed */
             set_attr(flist[k],NODISPLAY);
             tok = yylex();
             break;

        case FIXED_:
        /* see if fixed in place */
             set_attr(flist[k],FIXED);
             tok = yylex();
             break;

        case PHASE_: 
	    if ( !phase_flag )
	      error("Phases not in effect.\n",DATAFILE_ERROR);
            if ( web.dimension != STRING )
	      error("Phases on facets only in STRING model.\n",DATAFILE_ERROR);
	    gettok(INTEGER_);
	    if ( (int_val < 0) || (int_val > phasemax) )
	      error("Illegal phase value.\n",DATAFILE_ERROR);
	    set_f_phase(flist[k],int_val);
	    tok = yylex();
	    break;

        case COLOR_:
	    tok = gettok(INTEGER_);
	    set_facet_color(flist[k],int_val);
	    tok = yylex();
	    break;

        case TAG_:
	/* optional inheritable tag */
	    gettok(INTEGER_);
	    set_tag(flist[k],(tagtype)int_val);
	    tok = yylex();
	    break;

        case BOUNDARY_:
        /* see if boundary facet */
            gettok(INTEGER_);
            bnum = int_val;
            if ( (bnum >= BDRYMAX) || (tok != INTEGER_)
                                 || (web.boundaries[bnum].coordf[0] == NULL) )
             {
               sprintf(errmsg,"Bad boundary number %d for facet %d.\n",bnum,k);
               error(errmsg,DATAFILE_ERROR);
             }
            set_attr(flist[k],BOUNDARY);
            set_facet_boundary_num(flist[k],bnum);
	    tok = yylex();
            break;

        case CONSTRAINT_:
        /* see if constraint */
            for ( tok = gettok(INTEGER_) ; tok == INTEGER_ ; tok = yylex() ) 
              {
                cnum = abs(int_val);
                if ( cnum == 0 ) break;  /* no more numbers */
                if ( (cnum >= CONSTRMAX)
                                 || (web.constraints[cnum].formula == NULL) )
                  {
                    sprintf(errmsg,
                       "Bad constraint number %d for face %d.\n",cnum,k);
                    error(errmsg,DATAFILE_ERROR);
                  }
                set_attr(flist[k],CONSTRAINT);
/* ?? */        if ( int_val < 0 ) set_attr(flist[k],NEGBOUNDARY);
                set_f_constraint_map(flist[k],cnum);
                constr = get_constraint(cnum); 
                if ( constr->attr & BDRY_ENERGY )
                  set_attr(flist[k], BDRY_ENERGY);
	        if ( constr->attr & (CON_ENERGY | CON_CONTENT) )
 	         if ( web.constraints[cnum].compcount != compcount )
		  error("Inconsistent number of components in constraint integrands.\n",
                  WARNING);
              }
            break;

        case ENERGY_:
        /* see if surface energy */
            tok = gettok(INTEGER_);
            for ( i = 0 ; i < SURFENMAX ; i++, tok = yylex() ) 
              {
                if ( tok != INTEGER_ ) break;
                cnum = abs(int_val);
                if ( cnum == 0 ) break;  /* no more numbers */
                if ( (cnum >= SURFENMAX)
                                 || (web.surfen[cnum].envect[0] == NULL) )
                  {
                    sprintf(errmsg,
                       "Bad energy number %d for face %d.\n",cnum,k);
                    error(errmsg,DATAFILE_ERROR);
                  }
                set_attr(flist[k],SURF_ENERGY);
                set_f_surfen_map(flist[k],cnum);
              }
            break;

        case QUANTITY_:
        /* see if quantity */
            tok = gettok(INTEGER_);
            for ( i = 0 ; i < QUANTMAX ; i++, tok = yylex() ) 
              {
                if ( tok != INTEGER_ ) break;
                cnum = abs(int_val);
                if ( cnum == 0 ) break;  /* no more numbers */
                if ( (cnum >= QUANTMAX)
                                 || (web.quants[cnum].quanvect[0] == NULL) )
                  {
                    sprintf(errmsg,
                       "Bad quantity number %d for face %d.\n",cnum,k);
                    error(errmsg,DATAFILE_ERROR);
                  }
                set_attr(flist[k],SURF_QUANTITY);
                set_f_quant_map(flist[k],cnum);
              }
            break;

            default: more_attr = 0; break;
       }


     if ( k > facecount ) facecount = k;
   }
} /* end read_faces() */

/************************************************************************
*
*  Function: read_facet_edges()
*
*/

void read_facet_edges()
{
  int k;
  int f;
  facetedge_id fe=NULLID,old_fe,first_fe=NULLID;
  facet_id f_id;

  /* read in facet-edge facet chains (obsolete) */
   { tok = yylex();
     while ( tok == LEAD_INTEGER_ )
     { 
     k = int_val;
     if ( !valid_id(elist[k]) )
       { sprintf(errmsg,"Edge %d is not defined.\n",k);
         error(errmsg,DATAFILE_ERROR);
       }
     old_fe = NULLFACETEDGE;
     while ( (tok = gettok(INTEGER_)) == INTEGER_ )
       { int loopcount = 0;  /* for detecting errors in edge list */ 
         f = int_val;
         f_id = f > 0 ? flist[f] : facet_inverse(flist[-f]);
         if ( !valid_id(f_id) )
           { sprintf(errmsg,"Edge %d: face %d is not defined.\n",k,f);
             error(errmsg,DATAFILE_ERROR);
           }
         fe = get_facet_fe(f_id);
         while ( !equal_id(get_fe_edge(fe),elist[k]) ) 
           { fe = get_next_edge(fe);
             if ( loopcount++ > 1000 )
               { sprintf(errmsg,
                  "Facet-edge inconsistency for edge %d and facet %d.\n",k,f);
                 error(errmsg,DATAFILE_ERROR);
               }
           }
         if ( valid_id(old_fe) )
           { set_next_facet(old_fe,fe);
             set_prev_facet(fe,old_fe);
           }
         else first_fe = fe;
         old_fe = fe;
       }
     set_next_facet(fe,first_fe);  /* close up chain */
     set_prev_facet(first_fe,fe);
     }
   }
} /* end read_facet_edges() */

/************************************************************************
*
*  Function: read_bodies()
*
*/

void read_bodies()
{
  int k;
  int f=0;
  facetedge_id fe;
  facet_id f_id;
  int more_attr;
  REAL value;  /* for constant expression values */

  /* read in bodies */
  bmaxlist = MAXLIST;
  blist = (body_id *)mycalloc(sizeof(body_id),bmaxlist);

  tok = yylex();
  while ( tok == LEAD_INTEGER_ )
      { 
        REAL den,vol;
        int face_count = 0;
     
        k = int_val;
        while ( k >= bmaxlist )
          { int spot = bmaxlist; 
            bmaxlist = k + MAXLIST;
            blist=(body_id*)kb_realloc((char *)blist,bmaxlist*sizeof(body_id));
            if ( blist == NULL )
               error("Cannot allocate memory.\n",UNRECOVERABLE);
            for ( ; spot < bmaxlist ; spot ++ ) blist[spot] = NULLID;
          }
        if ( valid_id(blist[k]) )
         { sprintf(errmsg,"Duplicate body number %d\n",k);
           error(errmsg,DATAFILE_ERROR);
         }
        blist[k] = new_body();
        set_original(blist[k],k);
        while ( (tok = gettok(INTEGER_)) == INTEGER_ )
          { 
            face_count++;
            f = int_val;
            if ( f >= fmaxlist )
             { sprintf(errmsg,"Body %d: face %d is not defined.\n",k,f);
               error(errmsg,DATAFILE_ERROR);
             }
            f_id = f > 0 ? flist[f] : facet_inverse(flist[-f]);
            if ( !valid_id(f_id) )
             { sprintf(errmsg,"Body %d: face %d is not defined.\n",k,f);
               error(errmsg,DATAFILE_ERROR);
             }
            set_facet_body(f_id, blist[k]);
          }
        if ( (web.dimension == SOAPFILM) && (face_count < 1) )
          { sprintf(errmsg,"Body %d has no faces.\n",k);
            error(errmsg,WARNING);
          }
        fe = f > 0 ? get_facet_fe(flist[f]) :
                    fe_inverse(get_facet_fe(flist[-f]));
        set_body_fe(blist[k],fe);

        more_attr = 1;
        while ( more_attr )
         switch ( tok )
         {
           case VOLUME_:
               /* have a fixed volume constraint */
               if ( read_const(&vol) <= 0 )
                 error("No VOLUME value.\n",WARNING);
               set_attr(blist[k],FIXEDVOL);
               set_body_fixvol(blist[k],vol);
	       fixed_volumes++;
               tok = yylex();
               break;

           case VOLCONST_:
               /* have a body volume adjustment */
               if ( read_const(&vol) <= 0 )
                 error("No VOLCONST value.\n",WARNING);
               set_body_volconst(blist[k],vol); 
               tok = yylex();
               break;

           case DENSITY_:
               /* have density for gravity */
               if ( read_const(&den) <= 0 )
                 error("No DENSITY value.\n",WARNING);
               web.gravflag = 1;
               set_body_density(blist[k],den);
               set_attr(blist[k],DENSITY);
               tok = yylex();
               break;

           case PRESSURE_:
               /* have prescribed pressure */
               web.pressflag = 1;
               set_attr(blist[k],PRESSURE);
               if ( read_const(&value) <= 0 )
                 error("No DENSITY value.\n",WARNING);
               set_body_pressure(blist[k],value);
               tok = yylex();
               break;

           case PHASE_: 
	       if ( !phase_flag )
	         error("Phases not in effect.\n",DATAFILE_ERROR);
               if ( web.dimension == STRING )
	         error("Phases must be on facets in STRING model.\n",
		   DATAFILE_ERROR);
	       gettok(INTEGER_);
	       if ( (int_val < 0) || (int_val > phasemax) )
	         error("Illegal phase value.\n",DATAFILE_ERROR);
	       gettok(INTEGER_);
	       set_b_phase(blist[k],int_val);
	       tok = yylex();
	       break;

           default:  more_attr = 0;
         }

        /* can't have both pressure and volume */
        if ((get_battr(blist[k]) & (FIXEDVOL|PRESSURE)) == (FIXEDVOL|PRESSURE))
           error("Body can't have volume and pressure.\n",WARNING);
     }

  if ( web.bodycount > 0 )
   {     
     web.projection_flag = 1;
   }
} /* end read_bodies() */

/****************************************************************
*
*  Function: read_periods()
*
*  Purpose:  Reads torus periods.
*/

void read_periods()
{
  int i,j;
  REAL value;

  /* read in torus periods */
  web.inverse_periods = dmatrix(0,web.sdim-1,0,web.sdim-1);
  for ( i = 0 ; i < web.sdim ; i++ )
   { 
     for ( j = 0 ; j < web.sdim ; j++ ) 
       { 
         if ( read_const(&value) <= 0 )
            { error ("Not enough values for periods.\n",DATAFILE_ERROR);
              return;
             }
         web.inverse_periods[i][j] = web.torus_period[i][j] = value;
        }
   }
  if ( mat_inv(web.inverse_periods,web.sdim) < 0 )
     error("Degenerate torus unit cell.\n",DATAFILE_ERROR);
  tok = yylex(); /* lookahead */
}

/****************************************************************
*
*  Function: read_paramter()
*
*  Purpose:  Reads one adjustable parameter.
*/

void read_parameter()
{
  int n;
  struct param *p;

  tok = yylex(); /* dispose of PARAMETER */
  if ( tok != UNKNOWN )
    { error("Need PARAMETER identifier.\n",WARNING);
      return;
    }
  /* allocate structure */
  if ( web.params == NULL )
    web.params = (struct param *)mycalloc(1,sizeof(struct param));
  else
    { web.params = (struct param *)kb_realloc((char *)web.params,
                      (web.paramcount+1)*sizeof(struct param));
      memset((char*)(web.params+web.paramcount),0,sizeof(struct param));
    }
  if ( web.params == NULL )
    error("Cannot allocate adjustable parameter memory.\n",UNRECOVERABLE);

  p = web.params+web.paramcount;
  strncpy(p->name,yytext,31); /* 31 significant characters */
  p->name[31] = '\0';
  p->value = 0.0;  /* default */

  /* test for duplicate name */
  for ( n = 0 ; n < web.paramcount ; n++ )
    if ( strcmp(web.params[n].name,p->name) == 0 )
       error("Duplicate parameter name.\n",DATAFILE_ERROR);

  tok = yylex(); /* dispose of IDENT_ */
  if ( tok == '=' ) /* have initial value */
    { if ( read_const(&p->value) < 0 )
         error("Need constant expression for initial value.\n",DATAFILE_ERROR);
      tok = yylex();  /* get back token exparse pushed back */
    }
  else if ( tok == PARAMETER_FILE_ )
    {
          tok = yylex();
          if ( tok == QUOTATION_ )
            {  FILE *pfd = path_open(yytext);
	       int k;
               double val;

	       if ( pfd == NULL )
		 { error("Cannot open parameter file.\n",DATAFILE_ERROR);
		   return;
		 }
	       strncpy(p->value_file,yytext,sizeof(p->value_file));
               p->values = (double *)mycalloc(1000,sizeof(double));
	       while ( fgets(msg,sizeof(msg),pfd) )
		 { sscanf(msg,"%d %lf",&k,&val);
		   if ( k >= 1000 )
		     { error("Too high element number in parameter file.\n",WARNING);
		       break;
		     }
		   p->values[k] = val;
		 }
	       fclose(pfd);
               p->flags |= FILE_VALUES;
               tok = yylex();
            }
          else
              error("Parameter file name missing.\n",DATAFILE_ERROR);
    }

  recovery_flag = 0;
  web.paramcount++;
}
       
     
/*************************************************************************
*
* Reads and parses information for one free boundary specification.
* Current line starts with BOUNDARY keyword.
*/

void read_boundary()
{
  int bnum;  /* boundary number */
  int pcount; /* number of parameters */
  int i;
  int esize;
  struct boundary *bdry;

  tok = yylex();  /* eat BOUNDARY */
  if ( tok != INTEGER_ ) 
    { error("Need boundary number.\n",DATAFILE_ERROR);
      return;
    }
  bnum = int_val;    /* boundary number */
  if ( (bnum < 0) || (tok != INTEGER_) ) 
    { sprintf(errmsg,"Bad boundary number: %d.\n",bnum);
      error(errmsg,DATAFILE_ERROR);
      tok = yylex();
      return;
    }
  if ( bnum >= BDRYMAX )
    { sprintf(errmsg,"Boundary number exceeds maximum (%d): %d.\n",
                                      BDRYMAX-1,bnum);
      error(errmsg,DATAFILE_ERROR);
      tok = yylex();
      return;
    }
  bdry = web.boundaries + bnum;
  if ( bdry->pcount > 0 )
    { sprintf(errmsg,"Boundary number %d already defined.\n",bnum);
      error(errmsg,DATAFILE_ERROR);
    }

  tok = yylex();
  if ( tok != PARAMETERS_ )
    { sprintf(errmsg,"Expecting PARAMETERS keyword for boundary %d.\n",bnum);
      error(errmsg,DATAFILE_ERROR);
      return;
    }
  tok = gettok(INTEGER_);
  bdry->pcount = pcount = int_val;
  if ( (pcount < 0) || (tok != INTEGER_) ) 
    { sprintf(errmsg,"Bad parameter count for boundary %d.\n",bnum);
      error(errmsg,DATAFILE_ERROR);
      return;
    }
  if ( pcount > MAXPARAM )
    { sprintf(errmsg,"Parameter count for boundary %d exceeds %d.\n",
                                bnum,MAXPARAM);
      error(errmsg,DATAFILE_ERROR);
      pcount = 2;
    }

  tok = yylex();
  if ( tok == CONVEX_ ) 
    { web.convex_flag = 1;
      bdry->attr |= B_CONVEX;
      tok = yylex();
    }

  /* read and parse coordinate functions */
  bdry->coordf[0] = (struct expnode *)mycalloc(web.sdim,sizeof(struct expnode));

  for ( i = 0 ; i < web.sdim ; i++ )
    {
      bdry->coordf[i] = bdry->coordf[0] + i;
      if ( (tok != COORD_ ) || ( int_val != 1 + i ))
        { sprintf(errmsg,
            "Expected coordinate %d definition for boundary %d.\n",i+1,bnum);
          error(errmsg,DATAFILE_ERROR);
          return;
        }
      esize = exparse(pcount,&bdry->coordf[i]->root);
      tok = yylex();
      if ( esize <= 0 )
        { sprintf(errmsg,
             "Bad coordinate %d definition for boundary %d.\n",i+1,bnum);
          error(errmsg,DATAFILE_ERROR);
          return;
        }
    }

  /* no boundary energy or content integrands */

  return;

}

/*************************************************************************
*
* Reads and parses information for one free constraint specification.
* Current line starts with constraint keyword.
*/

void read_constraint()
{
  int cnum;  /* constraint number */
  int i;
  int esize;
  int more_attr;
  struct constraint *con;
  struct quantity *quan;
  int snum;

  tok = yylex();  /* eat CONSTRAINT */
  if ( tok != INTEGER_ ) 
    { error("Need constraint number.\n",DATAFILE_ERROR);
      return;
    }
  cnum = int_val;    /* constraint number */
  if ( cnum < 0 ) 
    { error("Bad constraint number.\n",DATAFILE_ERROR);
      tok = yylex();
      return;
    }
  if ( cnum >= CONSTRMAX )
    { sprintf(errmsg,"Constraint number exceeds maximum (%d): %d.\n",
                               CONSTRMAX-1,cnum);
      error(errmsg,DATAFILE_ERROR);
      tok = yylex();
      return;
    }
  con = web.constraints + cnum;
  if ( con->formula != NULL )
    { sprintf(errmsg,"Constraint number %d already defined.\n",cnum);
      error(errmsg,DATAFILE_ERROR);
    }
  if ( cnum >= web.concount ) web.concount = cnum+1;

  tok = yylex();
  for ( more_attr = 1 ; more_attr ; )
   switch ( tok )
   {
     case CONVEX_:
        web.convex_flag = 1;
        con->attr |= B_CONVEX;
        tok = yylex();
        break;

     case NONNEGATIVE_:
        web.constr_flag = 1;
        con->attr |= NONNEGATIVE;
        tok = yylex();
        break;

     case NONPOSITIVE_:
        web.constr_flag = 1;
        con->attr |= NONPOSITIVE;
        tok = yylex();
        break;

     case GLOBAL_:
        web.constr_flag = 1;
        con->attr |= GLOBAL;
        web.con_global_map |= (1 << cnum);
        tok = yylex();
        break;

     default: more_attr = 0;
   }
    

  /* read and parse defining function */
  constraint_init(con);
  if ( tok != FUNCTION_ )
    { sprintf(errmsg,
        "Expected function definition for constraint %d.\n",cnum);
      error(errmsg,DATAFILE_ERROR);
      return;
    }
  esize = exparse(web.sdim,&con->formula->root);
  tok = yylex();
  if ( esize <= 0 )
    { sprintf(errmsg,"Bad function definition for constraint %d.\n",cnum);
      error(errmsg,DATAFILE_ERROR);
      return;
    }

  /* various integrands */
  for (;;)
   switch ( tok )
   { 
    case ENERGY_:
     /* read and parse energy function */
     tok = yylex();
     con->attr &= CON_ENERGY;
     for ( i = 0 ; i < MAXCONCOMP ; i++ )
      {
       if ( (tok != ENVECT_) || (int_val != 1 + i ))
         break;
       esize = exparse(web.sdim,&(con->envect[i]->root));
       tok = yylex();  
       if ( esize <= 0 )
        { sprintf(errmsg,
            "Bad component %d definition for constraint %d.\n",i+1,cnum);
          error(errmsg,DATAFILE_ERROR);
          return;
        }
      }
     con->compcount = i;
     con->attr |= BDRY_ENERGY;
     break;

    case CONTENT_:
     /* read and parse content vector potential */
     tok = yylex();
     con->attr &= CON_CONTENT;
     for ( i = 0 ; i < web.sdim ; i++ )
      {
       if ( (tok != CONVECT_) || (int_val != 1 + i ))
         break;
       esize = exparse(web.sdim,&con->convect[i]->root);
       tok = yylex();
       if ( esize <= 0 )
        { sprintf(errmsg,
             "Bad coordinate %d definition for constraint %d.\n",i+1,cnum);
          error(errmsg,DATAFILE_ERROR);
          return;
        }
      }
      /* check consistency of number of components */
	  if ( con->compcount )
	    { if ( con->compcount != i )
		 error("Inconsistent number of components in integrands.\n",
                  WARNING);
	    }
	  else
	    { if ( (i != 1) && (i != web.sdim) )
		error("Illegal number of components in integrand.\n",
		   DATAFILE_ERROR);
	      con->compcount = i;
	    }
     con->attr |= BDRY_CONTENT;
     break;
   
    case QUANTITY_:
      /* read and parse a quantity integrand */
     tok = yylex();
     if ( tok != INTEGER_ ) 
       { error("Need quantity number.\n",DATAFILE_ERROR);
         return;
       }
      snum = int_val;    /* spec number */
      if ( snum >= QUANTMAX )
        { sprintf(errmsg,
            "Quantity number exceeds maximum (%d): %d.\n",QUANTMAX-1,snum);
          error(errmsg,DATAFILE_ERROR);
          tok = yylex();
          return;
        }
      quan = web.quants + snum;
      if ( !quan->quanvect[0] )
       { sprintf(errmsg,"Quantity number %d not defined.\n",snum);
         error(errmsg,DATAFILE_ERROR);
         return;
       }
      con->quanvect[snum][0] = 
        (struct expnode *)mycalloc(web.sdim,sizeof(struct expnode));
      tok = yylex();
      for ( i = 0 ; i < web.sdim ; i++ )
       {
        con->quanvect[snum][i] = con->quanvect[snum][0] + i;
        if ( (tok !=  QVECT_) || (int_val != 1 + i ))
          break;
        esize = exparse(web.sdim,&con->quanvect[snum][i]->root);
        tok = yylex();
        if ( esize <= 0 )
         { sprintf(errmsg,
             "Bad integrand %d definition for quantity %d.\n",i+1,snum);
           error(errmsg,DATAFILE_ERROR);
           return;
         }
       }
        /* check consistency of number of components */
	  if ( con->compcount )
	    { if ( con->compcount != i )
		 error("Inconsistent number of components in integrands.\n",
                  WARNING);
	    }
	  else 
	    { if ( (i != 1) && (i != web.sdim) )
		error("Illegal number of components in integrand.\n",
		   DATAFILE_ERROR);
	      con->compcount = i;
	    }
     con->quantity_map |= (1<<snum);
     break;

    default: return; 
   } /* end switch for integrands */
}


/*************************************************************************
*
* Reads and parses information for one surface energy  specification.
* Current line starts with SURFACE ENERGY keyword.
* Optional GLOBAL keyword.
*/

void read_surface_energy()
{
  int snum;  /* spec number */
  int i;
  int esize;
  int more_attr;
  struct surf_energy *surfen;

  tok = gettok(INTEGER_);
  if ( tok != INTEGER_ ) 
    { error("Need surface energy number.\n",DATAFILE_ERROR);
      return;
    }
  tok = yylex();
  snum = int_val;    /* spec number */
  if ( snum >= SURFENMAX )
    { sprintf(errmsg,
        "Surface energy number exceeds maximum (%d): %d.\n",SURFENMAX-1,snum);
      error(errmsg,DATAFILE_ERROR);
      return;
    }
  surfen = web.surfen + snum;
  if ( surfen->envect[0] )
    { sprintf(errmsg,"Surface energy number %d already defined.\n",snum);
      error(errmsg,DATAFILE_ERROR);
      return; 
    }
  if ( snum >= web.surfen_count ) web.surfen_count = snum+1;

  for ( more_attr = 1 ; more_attr ; )
   switch ( tok )
   {
     case GLOBAL_:
        surfen->attr |= GLOBAL;
        web.surf_global_map |= (1 << snum);
        tok = yylex();
        break;

     default: more_attr = 0;
   }
    


  /* read and parse energy integrand vector */
  surfen->envect[0]=(struct expnode *)mycalloc(web.sdim,sizeof(struct expnode));
  for ( i = 0 ; i < web.sdim ; i++ )
    {
      surfen->envect[i] = surfen->envect[0] + i;
      if ( (tok != ENVECT_) || (int_val != 1 + i ))
        { sprintf(errmsg,
            "Expected component %d definition for surface energy %d.\n",
                  i+1,snum);
          error(errmsg,DATAFILE_ERROR);
	  return;
        }
      esize = exparse(web.sdim,&surfen->envect[i]->root);
      tok = yylex();
      if ( esize <= 0 )
        { sprintf(errmsg,
            "Bad component %d definition for surface energy %d.\n",i+1,snum);
          error(errmsg,DATAFILE_ERROR);
        }
    }

}


/*************************************************************************
*
* Reads and parses information for one quantity specification, 
* in particular, facet integrands and fixed value.
* Current line starts with QUANTITY keyword.
* Optional GLOBAL keyword.
*/

void read_quantity()
{
  int snum;  /* spec number */
  int i;
  int esize;
  int more_attr;
  struct quantity *quan;

  tok = gettok(INTEGER_);
  if ( tok != INTEGER_ ) 
    { error("Need quantity number.\n",DATAFILE_ERROR);
      return;
    }
  snum = int_val;    /* spec number */
  if ( snum >= QUANTMAX )
    { sprintf(errmsg,
        "Quantity number exceeds maximum (%d): %d.\n",QUANTMAX-1,snum);
      error(errmsg,DATAFILE_ERROR);
      tok = yylex();
      return;
    }
  if  ( snum >= web.quantity_count ) web.quantity_count = snum+1;
  quan = web.quants + snum;
  if ( quan->quanvect[0] )
    { sprintf(errmsg,"Quantity number %d already defined.\n",snum);
      error(errmsg,PARSE_ERROR);
    }
  quan->attr |= IN_USE;
  if ( snum >= web.quantity_count ) web.quantity_count = snum+1;

  tok = yylex();
  for ( more_attr = 1 ; more_attr ; )
   switch ( tok )
   {
     case GLOBAL_:
        quan->attr |= GLOBAL;
        web.quant_global_map |= (1 << snum);
        tok = yylex();
        break;

     case FIXED_:
        quan->attr |= QFIXED;
        tok = yylex();
        if ( (tok != '=') || (read_const(&quan->target) <= 0) )
           error("No quantity value.\n",DATAFILE_ERROR);
        else tok = yylex();
        break;

     default: more_attr = 0;
   }

  /* read and parse energy integrand vector */
  quan->quanvect[0]=(struct expnode *)mycalloc(web.sdim,sizeof(struct expnode));
  for ( i = 0 ; i < web.sdim ; i++ )
    {
      quan->quanvect[i] = quan->quanvect[0] + i;
      if ( (tok != QVECT_) || (int_val != 1 + i ))
        { sprintf(errmsg,
            "Expected component %d definition for quantity %d.\n",i+1,snum);
          error(errmsg,DATAFILE_ERROR);
        }
      esize = exparse(web.sdim,&quan->quanvect[i]->root);
      tok = yylex();
      if ( esize <= 0 )
        { sprintf(errmsg,
            "Bad component %d definition for quantity %d.\n",i+1,snum);
          error(errmsg,DATAFILE_ERROR);
        }
    }

}
/*************************************************************
*
*  Function: add_outside()
*
*  Purpose:  Adds body outside all other bodies.  Used in
*            dynamic pressure calculations.
*/

void add_outside()
{
      body_id b_id;   /* both models */

      b_id = new_body();
      web.outside_body = b_id;
      set_attr(b_id,PRESSURE); /* since has ambient pressure */

      if ( web.dimension == STRING )
       { edge_id e_id;
         facet_id f_id;

         f_id = new_facet();   /* outside facet */
         set_facet_body(f_id,b_id);

         /* add to outside of every edge bordering just ONE cell */
         FOR_ALL_EDGES(e_id)
           {
             facetedge_id new_fe;
             facetedge_id fe_id;
             
             fe_id = get_edge_fe(e_id);
             if ( !valid_id(fe_id) ) continue;
             if ( !equal_id(fe_id,get_next_facet(fe_id)) ) continue;
             new_fe = new_facetedge(inverse_id(f_id),e_id);
             set_next_facet(fe_id,new_fe);
             set_prev_facet(fe_id,new_fe);
             set_next_facet(new_fe,fe_id);
             set_prev_facet(new_fe,fe_id);
             set_facet_fe(f_id,new_fe);

           }
       }
     else /* SOAPFILM */
       {
         facet_id f_id;

         /* add to outside of every facet bordering just ONE cell */
         FOR_ALL_FACETS(f_id)
           {
             body_id b1_id,b2_id;

             b1_id = get_facet_body(f_id);
             b2_id = get_facet_body(inverse_id(f_id));
             if ( valid_id(b1_id) == valid_id(b2_id) ) continue;

             if ( valid_id(b1_id) )
               set_facet_body(inverse_id(f_id),b_id);
             if ( valid_id(b2_id) )
               set_facet_body(f_id,b_id);
           }
      }
}

/*********************************************************************
*
*  Function: fix_volconst()
*
*  Purpose:  For torus, figure out constant volume adjustments, 
*            figuring that given volumes are within 1/12 of a 
*            torus volume of their true value
*/

void fix_volconst()
{
  double adjust;
  body_id b_id;

      /* get volume of piece of unit cell */
      if ( web.dimension == STRING )
       {
         web.torusv = (web.torus_period[0][0]*web.torus_period[1][1]
                  - web.torus_period[0][1]*web.torus_period[1][0]);
         adjust = web.torusv/2;
       }
      else /* web.dimension == SOAPFILM */
       {
         web.torusv = triple_prod(web.torus_period[0],web.torus_period[1],
                                                    web.torus_period[2]);
         adjust = web.torusv/6;
       }

      /* calculate raw volumes */
      FOR_ALL_BODIES(b_id)
          set_body_volconst(b_id,0.0);
      calc_content();

      /* adjust volconsts */
      FOR_ALL_BODIES(b_id)
        { REAL vol = get_body_volume(b_id);
          REAL fix = get_body_fixvol(b_id);
          set_body_volconst(b_id,adjust*floor(0.5+(fix-vol)/adjust));
        }
}

/*******************************************************************
*
*  Function: fe_reorder()
*
*  Purpose:  Order facets properly around edge.
*
*/

struct fsort { facetedge_id fe;
               REAL angle;
             };

static int fcompare(a,b)
struct fsort *a,*b;
{
  if ( a->angle < b->angle ) return -1;
  if ( a->angle > b->angle ) return 1;
  return 0;
}

void fe_reorder()
{
  edge_id e_id;
 
  FOR_ALL_EDGES(e_id)
    { int fcount = 0;
      facetedge_id fe;
      struct fsort fe_list[20];
      REAL side[MAXCOORD],norm_a[MAXCOORD],side_a[MAXCOORD];
      REAL side_b[MAXCOORD],norm_b[MAXCOORD],a_norm,b_norm;
      REAL angle;
      int i;

      /* see if we have 3 or more facets */
      generate_edge_fe_init();
      while ( generate_edge_fe(e_id,&fe) ) fcount++;
      if ( fcount <= 2 ) continue;

      if ( fcount > 20 )
        error("More than 20 facets on an edge.\n",WARNING);

      /* use first facet as reference facet */
      get_edge_side(e_id,side);    
      generate_edge_fe_init();
      generate_edge_fe(e_id,&fe);
      fe_list[0].fe = fe;
      get_fe_side(get_next_edge(fe),side_a);
      cross_prod(side,side_a,norm_a);
      a_norm = sqrt(dot(norm_a,norm_a,web.sdim));
      if ( a_norm == 0.0 )
        { error("Zero area triangle in initial configuration.\n",WARNING);
          continue;
        }
      
      /* now get angles to rest of facets */
      for ( i = 1 ; i < fcount ; i++ )
        {
          generate_edge_fe(e_id,&fe);
          fe_list[i].fe = fe;
          get_fe_side(get_next_edge(fe),side_b);
          cross_prod(side,side_b,norm_b);
          b_norm = sqrt(dot(norm_b,norm_b,web.sdim));
          if ( b_norm == 0.0 )
            { error("Zero area triangle in initial configuration.\n",WARNING);
              continue;
            }
          angle = acos(0.999999999*dot(norm_a,norm_b,web.sdim)/a_norm/b_norm);
          if ( dot(norm_a,side_b,web.sdim) < 0.0 )
            angle = 2*M_PI - angle;
          fe_list[i].angle = angle;
        }

      /* sort by angle */
      qsort((char *)&fe_list[1],fcount-1,sizeof(struct fsort),fcompare);

      /* check consistency for facets on bodies */
      for ( i = 0 ; i < fcount-1 ; i++ )
        { facet_id f1,f2;
          int j;

          f1 = facet_inverse(get_fe_facet(fe_list[i].fe));
          f2 = get_fe_facet(fe_list[i+1].fe);
          if ( !equal_id(get_facet_body(f1),get_facet_body(f2)) )
            /* search for one with proper body */
            { for ( j = i+2 ; j < fcount ; j++ )
                { f2 = get_fe_facet(fe_list[j].fe);
                  if ( equal_id(get_facet_body(f1),get_facet_body(f2)) )
                    { /* swap */
                      struct fsort tmp;
                      tmp = fe_list[j];
                      fe_list[j] = fe_list[i+1];
                      fe_list[i+1] = tmp;
                      break;
                    }
                 }
            }
        }


      /* relink in proper order */
      for ( i = 0 ; i < fcount ; i++ )
        { set_next_facet(fe_list[i].fe,fe_list[(i+1)%fcount].fe);
          set_prev_facet(fe_list[i].fe,fe_list[(i+fcount-1)%fcount].fe);
        }
    }
}



/**************************************************************
*
*  Function: gettok()
*
*  Purpose:  get next integer or real value, possibly
*            with '-' preceding.
*/

int gettok(kind)
int kind;
{
  int sign = 1;

  tok = yylex();
  if ( tok == '-' ) { sign = -1; tok = yylex(); }
  if ( tok != kind )
   if ( !((tok == INTEGER_) && (kind == REAL_)) )
     return tok;

  int_val *= sign;
  real_val *= sign;
  tok = kind;
  return tok;
}


/*******************************************************************
*
*  function: read_const()
* 
*  purpose: read constant expression from datafile
*           and return value.
*
*  input:   REAL *value  - address for value
*
*  return:  < 0 error
*             0 no expression
*           > 0 valid expression
*/

int read_const(value)
REAL *value;
{
  int retval;
  struct expnode node;  /* for getting constant expression */

  node.root = NULL;
  node.evallist = NULL;
  const_expr_flag = 1;
  if ( (retval = exparse(0,&node.root)) <= 0 )
    { const_expr_flag = 0;
      return retval;
    }
  expr_convert(&node);
  *value = eval(&node,NULL);
  free_expr(&node);
  const_expr_flag = 0;
  return 1;
}

/*********************************************************************
*
*  function: string_fixup()
*
*  purpose: put minimal facet-edges on string net, so can
*           do edge changes.  all facet-edges given the null facet.
*
*/

void string_fixup()
{
  edge_id e_id;
  vertex_id headv,tailv;
  facetedge_id fe,ffe,tfe,hfe,prev;

  /* give every edge two facet-edges */
  /* first, do all single-fe edges */
  FOR_ALL_EDGES(e_id)
   {
     headv = get_edge_headv(e_id);
     hfe = get_vertex_fe(headv);
     tailv = get_edge_tailv(e_id);
     tfe = get_vertex_fe(tailv);
     fe = get_edge_fe(e_id);
     if ( valid_id(fe) )
       {
	 ffe = get_next_facet(fe);
	 if ( !equal_id(fe,ffe) ) continue; /* edge has two fe's */

	 /* if here, has one fe */
         ffe = new_facetedge(NULLFACET,inverse_id(e_id));
         set_next_facet(fe,inverse_id(ffe));
         set_next_facet(ffe,inverse_id(fe));
         set_prev_facet(fe,inverse_id(ffe));
         set_prev_facet(ffe,inverse_id(fe));

         /* splice into head vertex */ 
         if ( valid_id(hfe) ) 
           { prev = get_prev_edge(hfe);
	     if ( !valid_id(prev) )
	       { /* can make into loop  */ 
	         set_prev_edge(hfe,inverse_id(ffe));
	         set_prev_edge(ffe,inverse_id(hfe));
               }
             /* else have to leave dangling to complete later */
           }
         set_vertex_fe(headv,ffe);

         /* splice into tail vertex */ 
         if ( valid_id(tfe) ) 
           { prev = get_prev_edge(tfe);
	     if ( !valid_id(prev) )
	       { /* can make into loop */
	         set_prev_edge(tfe,ffe);
	         set_next_edge(ffe,tfe);
               }
             /* else have to leave dangling to complete later */
           }
         set_vertex_fe(tailv,inverse_id(ffe));
       }
     else continue; /* has no fe */
   }

  /* now do all bare edges */
  FOR_ALL_EDGES(e_id)
   {
     headv = get_edge_headv(e_id);
     hfe = get_vertex_fe(headv);
     tailv = get_edge_tailv(e_id);
     tfe = get_vertex_fe(tailv);
     fe = get_edge_fe(e_id);
     if ( valid_id(fe) ) continue;

     /* if vertex on facet, previous stage should have put
	a null-facet fe there */
     if ( (valid_id(hfe) && valid_id(get_fe_facet(hfe))) ||
	  (valid_id(tfe) && valid_id(get_fe_facet(tfe))) )
       error("Bare edge breaking into vertex fe loop.\n",WARNING);

     fe = new_facetedge(NULLFACET,e_id);
     ffe = new_facetedge(NULLFACET,inverse_id(e_id));
     set_next_edge(fe,ffe);
     set_next_edge(ffe,fe);
     set_prev_edge(fe,ffe);
     set_prev_edge(ffe,fe);
     set_next_facet(fe,inverse_id(ffe));
     set_next_facet(ffe,inverse_id(fe));
     set_prev_facet(fe,inverse_id(ffe));
     set_prev_facet(ffe,inverse_id(fe));
     set_edge_fe(e_id,fe);

     /* splice into head vertex */ 
     if ( valid_id(hfe) ) 
       { prev = get_prev_edge(hfe);
	 set_next_edge(fe,hfe);
	 set_prev_edge(hfe,fe);
	 set_next_edge(prev,ffe);
	 set_prev_edge(ffe,prev);
       }
     set_vertex_fe(headv,ffe);

     /* splice into tail vertex */ 
     if ( valid_id(tfe) ) 
       { prev = get_prev_edge(tfe);
	 set_next_edge(ffe,tfe);
	 set_prev_edge(tfe,ffe);
	 set_next_edge(prev,fe);
	 set_prev_edge(fe,prev);
       }
     set_vertex_fe(tailv,fe);

   }
}


/*******************************************************
*
*  phase_initialize()
*
*  Purpose: Read in phase boundary energies from file  
*
*  Input:   Name of file with phase boundary energies
*           First line has number of phases
*           Succeeding lines have pair of phase numbers
*                 and boundary energy.
*
*  Output:  Array of Wulff vectors, wulff_flag set
*/

void phase_initialize(phasename)
char *phasename;
{
  FILE *pfd;
  int i,j;       /* which phases */
  double value;

  phase_flag = 1;

  /* save  name */
  strncpy(phase_file_name,phasename,sizeof(phase_file_name));

  pfd = path_open(phasename);
  if ( pfd == NULL )
    { error("No phase boundary energy file.\n",DATAFILE_ERROR);
      return;
    }

  fscanf(pfd,"%d",&phasemax);
  phase_data = dmatrix(0,phasemax,0,phasemax);
  for ( i = 0 ; i <= phasemax ; i++ )
    for ( j = 0 ; j <= phasemax ; j++ ) 
      phase_data[i][j] = 1.0;
  while ( !feof(pfd) )
    {
      if ( fscanf(pfd,"%d %d %lf",&i,&j,&value) == 3 ) 
       { if ( (i < 0) || (i > phasemax) || (j < 0) || (j > phasemax) )
	   { sprintf(errmsg,"Bad phase numbers: %d %d\n",i,j);
	     error(errmsg,DATAFILE_ERROR);
	   }
	  phase_data[i][j] = phase_data[j][i] = value;
       }
    }
  fclose(pfd);
}

