/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/******************************************************
*
*  File: fixvol.c
*
*  Contents: Routines for projecting force on vertices
*            to perpendicular of all body volume gradients.
*            Not for TORUS SOAPFILM model.
*            Does constrained quantities also.
*
*/

#include "include.h"


/*************************************************************************
*
*  Function:  vol_project()
*
*  Purpose:   Keeps constant volume of bodies by projecting force
*             perpendicular to gradients of body volumes of bodies
*             with prescribed volumes.  
*             Adds forces due to prescribed pressures before
*             said projection.
*
*
*            
*/

void vol_project(set_pressure_flag)
int set_pressure_flag; /* whether body pressure should be set */
{
  body_id bi_id;  /* identifier for body i */
  body_id b_id;
  vertex_id v_id;
  int i,k;
  int bi,bj;
  REAL *g,*f;
  int dimension = web.sdim;
  int needflag = 0;
  REAL **a;  
  struct boundary *bdry;
  int maxbody = ordinal(web.skel[BODY].last) + web.quantity_count;
  int qfixed = 0;
  edge_id e_id;
  REAL ***vg = NULL;  /* for approx curvature */
  REAL **vge = NULL;  /* for approx curvature */

  /* see if anything needs to be done */
   FOR_ALL_BODIES(bi_id)
      if ( get_battr(bi_id) & (FIXEDVOL|PRESSURE) ) needflag = 1;
  for ( i = 0 ; i < web.quantity_count ; i++ )
      if ( get_quant(i)->attr & QFIXED ) { needflag = 1; qfixed++; }
  if ( !needflag ) return;

  /* allocate space to hold vertex body volume gradients */
  vgrad_init(qfixed);

  /* calculate body volume gradients at all control points 
     due to free surfaces */
  if ( web.simplex_flag ) 
    simplex_grad_l();
  else if ( web.dimension == STRING )
    (*string_grad)();
  else /* web.dimension == SOAPFILM */
    (*film_grad)();

  /* project to parameter space for boundary points */
  a = dmatrix(0,2,0,2);
  FOR_ALL_VERTICES(v_id)
    {
      int pcount;
      volgrad *vgptri;

      if ( get_vattr(v_id) & FIXED ) continue;
      if ( !(get_vattr(v_id) & BOUNDARY) ) continue;
      bdry = get_boundary(v_id);
      pcount = bdry->pcount;
      b_proj(bdry,get_param(v_id),a,PARAMPROJ);
      vgptri = get_vertex_vgrad(v_id);
      while ( vgptri )
        { REAL tmp[MAXCOORD];
          int m;
          matvec_mul(a,vgptri->grad,tmp,pcount,web.sdim);
          for ( m = 0 ; m < pcount ; m++ ) vgptri->grad[m] = tmp[m];
          for ( m = pcount ; m < web.sdim ; m++ )
            vgptri->grad[m] = 0.0;
          vgptri = vgptri->chain;
        }
    }
  free_matrix(a);

  /* add on gradients due to boundary integrals */
  if ( web.dimension == STRING )
    { string_bdry_grad();
      string_constr_grad();
      film_constr_grad();  /* for quantities */
    }
  else /* web.dimension == SOAPFILM */
    { film_bdry_grad();
      film_constr_grad();
    }

  /* set up body normals */
  FOR_ALL_VERTICES(v_id)
    {
      volgrad *vgptri;
      for ( vgptri = get_vertex_vgrad(v_id); vgptri ; vgptri = vgptri->chain )
        { 
          for ( i = 0 ; i < web.sdim ; i++ )
            vgptri->normal[i] = vgptri->grad[i];   
        }
    }

  if ( (web.dimension == SOAPFILM) && (web.modeltype == QUADRATIC) )
    { edge_id e_id;

      FOR_ALL_EDGES(e_id)
        {
          volgrad *vgptri;
          vertex_id headv,tailv;

          headv = get_edge_headv(e_id);
          tailv = get_edge_tailv(e_id);
          vgptri = get_vertex_vgrad(get_edge_midv(e_id));
          for ( ; vgptri ; vgptri = vgptri->chain )
            { 
              volgrad *hvgptr = get_bv_vgrad(vgptri->b_id,headv);
              volgrad *tvgptr = get_bv_vgrad(vgptri->b_id,tailv);
              for ( i = 0 ; i < web.sdim ; i++ )
                { hvgptr->normal[i] += 0.5*vgptri->grad[i];
                  tvgptr->normal[i] += 0.5*vgptri->grad[i];
                }
            }
        }
     }

  /* do area normalization if wanted */
  if ( web.area_norm_flag )
    {
      FOR_ALL_VERTICES(v_id)
        {
          REAL area = get_vertex_star(v_id)/star_fraction;
          volgrad *vgptri;
          for ( vgptri = get_vertex_vgrad(v_id); vgptri ; vgptri = vgptri->chain )
            { 
              for ( i = 0 ; i < web.sdim ; i++ )
                vgptri->normal[i] /= area;   
            }
        }
    }

  /* project to constraint tangent space for boundary points */
  a = dmatrix(0,2,0,2);
  FOR_ALL_VERTICES(v_id)
    {
      MAP conmap = get_v_constraint_map(v_id);
      int oncount = 0;
      struct constraint *con[CONSTRMAX];
      volgrad *vgptri;
      int j;
      REAL *x;

      if ( get_vattr(v_id) & FIXED ) continue;
      if ( !(get_vattr(v_id) & CONSTRAINT) ) continue;

      for ( j = 0 ; j < web.concount ; j++,conmap>>=1 )
	{ if ( !(conmap & 1) ) continue;
  	  if ( get_v_constraint_status(v_id,j) == ON_CONSTRAINT )
             con[oncount++] = get_constraint(j);
	}

      x = get_coord(v_id);
      for ( vgptri = get_vertex_vgrad(v_id); vgptri ; vgptri = vgptri->chain )
        { 
          REAL perp[MAXCOORD];

          constr_proj(TANGPROJ,oncount,con,x,vgptri->normal,perp,NULL,NO_DETECT);
          for ( j = 0 ; j < web.sdim ; j++ )
            vgptri->normal[j] -= perp[j];
        }
    }
  free_matrix(a);


  if ( web.pressure_flag )
    {
      /* add forces due to dynamic pressure */
      FOR_ALL_VERTICES(v_id)
        {
          volgrad *vgptr;
    
          if ( get_vattr(v_id) & FIXED ) continue;
          f = get_force(v_id);
          for ( vgptr = get_vertex_vgrad(v_id) ; vgptr ; vgptr = vgptr->chain )
            {
              REAL p = get_body_pressure(vgptr->b_id);
              for ( k = 0 ; k < dimension ; k++ )
                  f[k] += p*vgptr->normal[k];
            }
    
        }
      vgrad_end();
      return;
    }

  /* add prescribed pressure forces */
  FOR_ALL_VERTICES(v_id)
    {
      volgrad *vgptr;

      if ( get_vattr(v_id) & FIXED ) continue;
      f = get_force(v_id);
      for ( vgptr = get_vertex_vgrad(v_id) ; vgptr ; vgptr = vgptr->chain )
        {
          if ( !valid_id(vgptr->b_id) ) continue; /* skip quantities */
          if ( get_battr(vgptr->b_id) & PRESSURE )
           for ( k = 0 ; k < dimension ; k++ )
            { 
              f[k] += get_body_pressure(vgptr->b_id)*vgptr->normal[k];
            }
        }

    }

  /* set up matrices for fixed volume case */
  fleftside = dmatrix(0,maxbody,0,maxbody);
  rleftside = dmatrix(0,maxbody,0,maxbody);
  for ( k = 0 ; k <= maxbody ; k++ )
    { fleftside[k][k] = 1.0; /* for invertibility */
      rleftside[k][k] = 1.0; /* for invertibility */
    }
  rightside = vector(0,maxbody);
  vol_deficit = vector(0,maxbody);
  vol_restore = vector(0,maxbody);
  pressures = vector(0,maxbody);

   FOR_ALL_BODIES(bi_id)
    {
      bi = ordinal(bi_id);

      if ( !(get_battr(bi_id) & FIXEDVOL) ) continue;
      vol_deficit[bi] = get_body_fixvol(bi_id) - get_body_volume(bi_id);
      fleftside[bi][bi] = 0.0;  /* to undo safety move above */
      rleftside[bi][bi] = 0.0;  /* to undo safety move above */
    }
  for ( k = 0 ; k < web.quantity_count ; k++ )
    { struct quantity *quan = get_quant(k);
      if ( !(quan->attr & QFIXED) ) continue;
      bi = web.bodycount + k;
      vol_deficit[bi] = quan->target - quan->value;
      fleftside[bi][bi] = 0.0;  /* to undo safety move above */
      rleftside[bi][bi] = 0.0;  /* to undo safety move above */
    }


  /* generate sides of matrix equation term */
  if ( !approx_curve_flag )
   FOR_ALL_VERTICES(v_id)
    {
      volgrad *vgptri,*vgptrj;
      ATTR attr = get_vattr(v_id);

      if ( attr & FIXED ) continue;
      f = get_force(v_id);

      for ( vgptri = get_vertex_vgrad(v_id); vgptri ; vgptri = vgptri->chain )
        { REAL tmp;
          if (valid_id(vgptri->b_id))
            { if ( !(get_battr(vgptri->b_id)&FIXEDVOL) ) continue;
              bi = ordinal(vgptri->b_id);
            }
          else bi = web.bodycount + vgptri->b_id; /* for quantities */
          tmp = dot(vgptri->normal,vgptri->grad,dimension);
          fleftside[bi][bi] += tmp;
/*          if ( !(attr & HIT_WALL) )   */
            { rleftside[bi][bi] += tmp;
              rightside[bi] += dot(f,vgptri->grad,dimension);
            }
          for ( vgptrj = vgptri->chain ; vgptrj ; vgptrj = vgptrj->chain )
            { tmp = dot(vgptri->grad,vgptrj->normal,dimension);
              if ( valid_id(vgptrj->b_id) )bj = ordinal(vgptrj->b_id);
              else bj = web.bodycount + vgptrj->b_id;
              fleftside[bi][bj] += tmp;
              fleftside[bj][bi] += tmp;
/*              if ( !(attr & HIT_WALL) ) */
                {
                  rleftside[bi][bj] += tmp;
                  rleftside[bj][bi] += tmp;
                }
            }
         }
     }

  if ( approx_curve_flag )
    { int N = web.sdim*web.skel[VERTEX].max_ord;
      REAL *tempg = (REAL *)temp_calloc(N,sizeof(REAL));
      int j;
      double tmp;

      vge = dmatrix(0,maxbody+1,0,N);
      /* load volume gradients and calculate right side */
      FOR_ALL_VERTICES(v_id)
       {
         volgrad *vgptri,*vgptrj;
         ATTR attr = get_vattr(v_id);
	 int ord = ordinal(v_id);

         if ( attr & FIXED ) continue;

         f = get_force(v_id);
         for ( vgptri=get_vertex_vgrad(v_id); vgptri ; vgptri = vgptri->chain )
           {
             if (valid_id(vgptri->b_id))
               { if ( !(get_battr(vgptri->b_id)&FIXEDVOL) ) continue;
                 bi = ordinal(vgptri->b_id);
               }
             else bi = web.bodycount + vgptri->b_id; /* for quantities */
	     for ( j = 0 ; j < web.sdim ; j++ )
	       vge[bi][web.sdim*ord+j] = vgptri->grad[j];
             rightside[bi] += dot(f,vgptri->grad,web.sdim);
           }
        }

     /* convert gradients to vectors  and get dot products */
     for ( bi = 0 ; bi <= maxbody ; bi++ )
       { memcpy((char*)tempg,(char*)vge[bi],N*sizeof(REAL));
         mobility_mult(vge[bi]);
	 rleftside[bi][bi] += dot(tempg,vge[bi],N);  /* self product */ 
	 for ( bj = bi+1 ; bj <= maxbody ; bj++ ) /* other products */
	   { tmp = dot(vge[bi],vge[bj],N);
	     rleftside[bi][bj] += tmp;
	     rleftside[bj][bi] += tmp;
           }
        }

    }  /* end approx_curve_flag */

  /* solve for coefficients */
  if ( web.full_flag ) maxbody--;   /* to prevent singular matrix */
  mat_inv(rleftside,maxbody+1);
  matvec_mul(rleftside,rightside,pressures,maxbody+1,maxbody+1);

  /* install pressures into body structures */
  if ( set_pressure_flag == SET_PRESSURE )
    {
      FOR_ALL_BODIES(b_id)
       {
         bi = ordinal(b_id);
         if ( get_battr(b_id) & FIXEDVOL )
           set_body_pressure(b_id,-pressures[bi]);
       }
      for ( k = 0 ; k < web.quantity_count ; k++ )
        { struct quantity *quan = get_quant(k);
          if ( !(quan->attr & QFIXED) ) continue;
          bi = web.bodycount + k;
          quan->pressure = -pressures[bi];
	}
    }


  
  /* solve for volume restoration coefficients */
  matvec_mul(rleftside,vol_deficit,vol_restore,maxbody+1,maxbody+1);
  
  /* subtract multiples of volume gradients from force */
  /* and combine multiples of gradients for restoring motion */

  /* fix up forces and set restoring forces */
  FOR_ALL_VERTICES(v_id)
    {
      volgrad *vgptr;
      int ord = ordinal(v_id);

      if ( get_vattr(v_id) & FIXED ) continue;
      g = get_restore(v_id);
      for ( i = 0 ; i < dimension ; i++ ) g[i] = 0.0;
      f = get_force(v_id);
      for ( vgptr = get_vertex_vgrad(v_id) ; vgptr ; vgptr = vgptr->chain )
        {
          if ( valid_id(vgptr->b_id) ) bi = ordinal(vgptr->b_id);
          else bi = web.bodycount + vgptr->b_id; /* for quantities */
          for ( k = 0 ; k < dimension ; k++ )
            { 
/*               if ( !(get_battr(vgptr->b_id) & HIT_WALL) )    */
               if ( approx_curve_flag )
		 {
                   g[k] += vol_restore[bi]*vge[bi][web.sdim*ord+k];
                   f[k] -= pressures[bi]*vge[bi][web.sdim*ord +k];
		 }
	       else
		{
                   g[k] += vol_restore[bi]*vgptr->normal[k];
                   f[k] -= pressures[bi]*vgptr->normal[k];
		 }
            }
        }

    }

  vgrad_end();
  free_matrix(fleftside);
  free_matrix(rleftside);
  free((char *)rightside);
  free((char *)vol_deficit);
  free((char *)vol_restore);
  free((char *)pressures);
  if ( vg ) free_matrix3(vg);
}
         

  
