%{C
/**********************************************************
*
*  File:  lex.yy.c
*
*  Contents: lexical analyzer for evolver data files.
*            Constructed from datafile.lex by lex.
*/

/* will have my own input() and output() */
#undef input
#undef unput

#include "include.h"
#include "lex.h"
#include "ytab.h"
#include "express.h"

#ifndef NOPROTO
int yyback(int *,int);
#define yyoutput yyout_unused
#define yyunput yyunput_unused
#endif

extern FILE *data_fd;
extern int int_val;
extern double real_val;
extern char *cmdptr;
int line_no = 1;

int macro_flag;

char *white = " \t\r,:;";   /* whitespace */

#define NAMESIZE 30
struct macro { char name[NAMESIZE];
		    int  offset;    /* start of substitute string */
		    int  subsize;   /* size of substitute string */
		  } *macros;  /* dynamically allocated */
int macro_count;  /* number of macros defined */
int macro_max;    /* number of macro structures allocated */
#define SUBMAX 500
char  *macro_subs;  /* string space for substitution strings */
		    /* dynamically allocated */
int  macro_subs_top;  /* index of top of string space */
int  macro_subs_max;   /* space allocated for strings */

struct dkey { char *name; int token; } datafile_keywords[] =
{
  {"approximate_curvature",APPROX_CURV_},
  {"approx_curvature",APPROX_CURV_},
  {"phasefile",PHASEFILE_},
  {"phase",PHASE_},
  {"autopop",AUTOPOP_},
  {"autochop",AUTOCHOP_},
  {"total_time",TOTAL_TIME_},
  {"effective_area",EFFECTIVE_AREA_},
  {"runge_kutta",RUNGE_KUTTA_},
  {"color",COLOR_},
  {"square_curvature",SQUARE_CURVATURE_},
  {"space_dimension",SPACE_DIMENSION_},
  {"surface_dimension",SURFACE_DIMENSION_},
  {"simplex_representation",SIMPLEX_REP_},
  {"metric",METRIC_},
  {"conformal_metric",CONFORMAL_},
  {"fixed",FIXED_},
  {"symmetry_group",SYMMETRY_GROUP_},
  {"wrap",WRAP_},
  {"torus",TORUS_},
  {"torus_filled",TORUS_FILLED_},
  {"periods",PERIODS_},
  {"string",STRING_},
  {"soapfilm",SOAPFILM_},
  {"wulff",WULFF_},
  {"boundary",BOUNDARY_},
  {"boundaries",BOUNDARY_},
  {"constraint",CONSTRAINT_},
  {"constraints",CONSTRAINT_},
  {"surface_energy",SURFACE_ENERGY_},
  {"formula",FUNCTION_},
  {"function",FUNCTION_},
  {"parameter",PARAMETERS_},
  {"parameters",PARAMETERS_},
  {"parameter_file",PARAMETER_FILE_},
  {"symmetric_content",SYMMETRIC_CONTENT_},
  {"integral_order",INTEGRAL_ORDER_},
  {"constraint_tolerance",CONSTRAINT_TOLERANCE_ },
  {"convex",CONVEX_},
  {"nonnegative",NONNEGATIVE_},
  {"nonpositive",NONPOSITIVE_},
  {"global",GLOBAL_},
  {"energy",ENERGY_},
  {"energies",ENERGY_},
  {"content",CONTENT_},
  {"quadratic",QUADRATIC_},
  {"linear",LINEAR_},
  {"area_normalization",MEAN_CURV_},
  {"jiggle",JIGGLE_},
  {"diffusion",DIFFUSION_},
  {"merit_factor",MERITFACTOR_},
  {"gravity_constant",GRAV_CONST_},
  {"spring_constant",SPRING_CONSTANT_},
  {"scale",SCALE_},
  {"temperature",TEMPERATURE_},
  {"pressure",PRESSURE_},
  {"volume",VOLUME_},
  {"density",DENSITY_},
  {"tension",DENSITY_},
  {"nodisplay",NODISPLAY_},
  {"scale_limit",SCALE_LIMIT_},
  {"zoom_vertex",ZOOM_VERTEX_},
  {"zoom_radius",ZOOM_RADIUS_},
  {"quantity",QUANTITY_ },
  {"volconst",VOLCONST_ },
  {"sin",SIN_},
  {"cos",COS_},
  {"log",LOG_},
  {"exp",EXP_},
  {"tan",TAN_},
  {"asin",ASIN_},
  {"acos",ACOS_},
  {"atan",ATAN_},
  {"sqrt",SQRT_},
  {"sqr",SQR_},
  {"abs",ABS_},
  {"pi",PI_},
  {"e",E_},
  {"g",G_},
  {"tag",TAG_},
  {"original",ORIGINAL_},
  {"vertices",VERTICES_},
  {"edges",EDGES_},
  {"faces",FACES_},
  {"facets",FACES_},
  {"bodies",BODIES_},
  {"facet_edges",FACETEDGES_}
};

struct ckey { char *name; int token; } const_expr_keywords[] =
{
  {"sin",SIN_},
  {"cos",COS_},
  {"log",LOG_},
  {"exp",EXP_},
  {"tan",TAN_},
  {"asin",ASIN_},
  {"acos",ACOS_},
  {"atan",ATAN_},
  {"sqrt",SQRT_},
  {"sqr",SQR_},
  {"pi",PI_},
  {"e",E_},
  {"g",G_}
};

struct ckey command_keywords[] =
{
  {"id",ID_},
  {"and",AND_},
  {"or",OR_},
  {"not",NOT_},
  {"sin",SIN_},
  {"cos",COS_},
  {"log",LOG_},
  {"exp",EXP_},
  {"tan",TAN_},
  {"asin",ASIN_},
  {"acos",ACOS_},
  {"atan",ATAN_},
  {"sqrt",SQRT_},
  {"sqr",SQR_},
  {"abs",ABS_},
  {"pi",PI_},
  {"e",E_},
  {"g",G_},
  {"tag",TAG_},
  {"original",ORIGINAL_},
  {"fixed",FIXED_},
  {"vertices",VERTICES_},
  {"vertex",VERTICES_},
  {"edges",EDGES_},
  {"edge",EDGES_},
  {"facets",FACETS_},
  {"facet",FACETS_},
  {"faces",FACETS_},
  {"face",FACETS_},
  {"facet_edges",FACETEDGES_},
  {"facetedges",FACETEDGES_},
  {"bodies",BODIES_},
  {"body",BODIES_},
  {"everything",EVERYTHING_},
  {"topinfo",TOPINFO_},
  {"length",LENGTH_ },
  {"valence",VALENCE_ },
  {"area",AREA_ },
  {"volume",VOLUME_ },
  {"where",WHERE_ },
  {"list",LIST_ },
  {"show",SHOW_},
  {"delete",DELETE_ },
  {"refine",REFINE_ },
  {"shell",SHELL_},
  {"fixed",FIXED_},
  {"constraint",CONSTRAINT_},
  {"pressure",PRESSURE_},
  {"color",COLOR_},
  {"opacity",OPACITY_},
  {"volume",VOLUME_},
  {"density",DENSITY_},
  {"tension",DENSITY_},
  {"set",SET_},
  {"read",READ_},
  {"unset",UNSET_},
  {"recalc",RECALC_},
  {"integral_order",INTEGRAL_ORDER_}
 }; 

#define MAXCOLOR 17 
struct ckey colornames[MAXCOLOR] = {
    {"clear",CLEAR},     /* transparent */
    {"black",BLACK},		    /* dark colors */
    {"blue",BLUE},
    {"green",GREEN},
    {"cyan",CYAN},
    {"red",RED},
    {"magenta",MAGENTA},
    {"brown",BROWN},
    {"lightgray",LIGHTGRAY},
    {"darkgray",DARKGRAY},		    /* light colors */
    {"lightblue",LIGHTBLUE},
    {"lightgreen",LIGHTGREEN},
    {"lightcyan",LIGHTCYAN},
    {"lightred",LIGHTRED},
    {"lightmagenta",LIGHTMAGENTA},
    {"yellow",YELLOW},
    {"white",WHITE}
    };
%}
D     [0-9]
E     [DEde][+-]?{D}+
W     [ \t\r,:;]

%%

^{W}*-?{D}+  { real_val = (REAL)(int_val = atoi(strtok(yytext,white)));
               return(tok=LEAD_INTEGER_); 
	      }
{W}+[+-]{D}+ |
{D}+        { real_val = (REAL)(int_val = atoi(strtok(yytext,white))); 
	      if (datafile_flag && (parens==0))
		 return(tok = INTEGER_); 
              else return (tok = SIGNED_NUMBER_);
	      }
0x{D}+      { sscanf(yytext+2,"%x",&int_val); real_val = (REAL)int_val;
	      return(tok = INTEGER_);}  /* hex */
[01]+b      { char *c = yytext;  /* binary */
	      int_val = 0;
	      while ( isdigit(*c) ) { int_val = 2*int_val + *c - '0'; c++;}
	    }

{W}+[+-]{D}+"."{D}*({E})?   |
{W}+[+-]{D}*"."{D}+({E})?   |
{W}+[+-]{D}+{E}     { real_val = atof(strtok(yytext,white));  
	      if (datafile_flag && (parens==0))
		 return(tok = REAL_); 
              else return (tok = SIGNED_NUMBER_);
	      }

{D}+"."{D}*({E})?   |
{D}*"."{D}+({E})?   |
{D}+{E}     { real_val = atof(yytext);  return(tok = REAL_); }

"/*".*"*/"   ;
"//"[^\n]*       ;  
\".*\"        {
             int k,len = strlen(yytext);
	     for ( k = 0 ; k < len-2 ; k++ ) yytext[k] = yytext[k+1];
	     yytext[len-2] = '\0'; 
	     return(tok = QUOTATION_);
	     }

{W}+-{W}+  return (tok = minus_type());   

{W}+-        { if (datafile_flag && (parens == 0) )  return(tok = UMINUS_);
               else return ( tok = minus_type() );
             }
^-          return(tok = UMINUS_);
-       return (tok = minus_type());  
[<>+*/=]          return(tok = yytext[0]);
"("         {  parens++; return(tok = yytext[0]); }
")"         { parens--; return(tok = yytext[0]);  }
\^\-?{D}+     return (tok = '^');
"**"       return(tok = POW_);
"=="       return(tok = EQ_);
"!="       return(tok = NE_);
"<="       return(tok = LE_);
">="       return(tok = GE_);
"||"       return(tok = OR_);
"&&"       return(tok = AND_);
"!"        return(tok = NOT_);
^{W}*e{D}+  { int_val = atoi(strtok(yytext,white)+1); return(tok = ENVECT_); }
^{W}*c{D}+  { int_val = atoi(strtok(yytext,white)+1); return(tok = CONVECT_); }
^{W}*x{D}+  { int_val = atoi(strtok(yytext,white)+1); return(tok = COORD_); }
^{W}*q{D}+  { int_val = atoi(strtok(yytext,white)+1); return(tok = QVECT_); }
p{D}+      { int_val = atoi(yytext+1); return(tok = PARAM_); }
q{D}+      { int_val = atoi(yytext+1); return(tok = QVECT_); }
x{D}+      { int_val = atoi(yytext+1); return(tok = COORD_); }
x          { int_val = 1; return(tok = COORD_); }
y          { int_val = 2; return(tok = COORD_); }
z          { int_val = 3; return(tok = COORD_); }
usr{D}+	   { int_val = atoi(yytext+3); return(tok = USERFUNC_); }

#define    macro_flag = 1;
[a-z_][a-z0-9_]*   { if ( macro_flag ) record_macro();
                     else if ( !macro() ) /* search keywords */
		       { int k;
			 /* search parameter names */
                         for ( k = 0 ; k < (NTYPE)web.paramcount ; k++ )
                           if ( strncmp(yytext,web.params[k].name,31) == 0 )
                            { int_val = k;
			      return (tok = IDENT_);
                            }
			 /* search color names */
			 for ( k = 0 ; k < MAXCOLOR ; k++ )
			   if ( strcmp(yytext,colornames[k].name) ==  0 )
			     { real_val = int_val = colornames[k].token;
			       return (tok = INTEGER_);
			     }
			 if ( const_expr_flag )
			  { for ( k = 0 ; k < sizeof(const_expr_keywords)/
			        sizeof(struct ckey) ; k++ )
			     if (strcmp(yytext,const_expr_keywords[k].name)==0)
			       return (tok = const_expr_keywords[k].token);
			  }
			 else if ( datafile_flag )
			  { for ( k = 0 ; k < sizeof(datafile_keywords)/
			        sizeof(struct dkey) ; k++ )
			     if (strcmp(yytext,datafile_keywords[k].name)==0)
			       return (tok = datafile_keywords[k].token);
			  }
			 else
			  { for ( k = 0 ; k < sizeof(command_keywords)/
			      sizeof(struct ckey) ; k++ )
			     if ( strcmp(yytext,command_keywords[k].name) == 0 )
			       return (tok = command_keywords[k].token);
			  }
                         /* if here, then not keyword */
		         return(tok = UNKNOWN) ;
		       }
		   }

\n     ;
{W}*      ;
.       { if ( isprint(yytext[0]) )
	    sprintf(errmsg,"Illegal character: %c\n",yytext[0]);
	  else
	    sprintf(errmsg,"Illegal character: 0x%02X\n",yytext[0]);
          yyerror(errmsg);
        }
%%

/* My own input() function to handle case insensitivity and macros */
/* and line counting */

#define BUFFSIZE 400
static char buff[BUFFSIZE];
static int  spot = 0;
#define ERRBUFFSIZE 80
static char errbuff[ERRBUFFSIZE];  /* for reporting spot of error */
static int  errspot;

int rawinput()
{
  int c;
  if ( cmdptr )
   {
     if ( !*cmdptr ) return 0;
     else c = *(cmdptr++);
   }
  else
   {
     if ( data_fd == NULL ) return 0;  /* in case read after close */
     c = getc(data_fd);
     if ( c == -1 ) c = 0;
   }
  if ( c == '\n' ) errspot = 0 ;  /* reset for new line */
  errbuff[errspot++] = (char)c;
  if ( errspot == ERRBUFFSIZE ) /* need partial reset */
    { memcpy(errbuff,errbuff+ERRBUFFSIZE/2,ERRBUFFSIZE/2);
      errspot = ERRBUFFSIZE/2;
    }
  return tolower(c);
}

int input()
{
  int c;

  if ( spot > 0 ) 
    c = buff[--spot];
  else
    { 
      c = rawinput();
      if ( c == '\\' )  /* line splicing */
	{ c = rawinput();
	  if ( c == '\n' ) { line_no++; c = ' '; }
	  if ( c == 0 )
	    { outstring("more> ");
	      getstring(msg);
	      cmdptr = msg;
	      c = ' ';  /* linesplicing ends token */
	    }
	}
      else if ( c == '/' ) /* possible comment */
	{ c = rawinput();
	  if ( c == '/' ) /* end-of-line comment */
	    { while ( (c = rawinput()) != '\n' )
	        if ( c == 0 ) { break; } 
	    }
	  else if ( c == '*' ) /* in-line comment */
	    { int state = 0;
	      do 
		{ c = rawinput();
		  if ( c == 0 ) { break; }
		  else  if ( c == '\n' ) { line_no++; state = 0; }
		  else if ( c == '*' ) state = '*';
                  else if ( (c == '/') && (state == '*') )
		    { c = ' '; break; }
		  else state = 0;
		} 
	      while ( state != 1 );
	    }
	  else /* false alarm */
	    { buff[spot++] = (char)c;  /* save char */
	      c = '/';
	    }
	} /* end comment */
	    
    } /* end new char */

  if ( c == '\n' ) line_no++;
  return tolower(c);
}

void unput(c)
int c;
{
  if ( spot >= BUFFSIZE - 1 ) 
    error("lex buffer overflow.\n",UNRECOVERABLE);

  if ( c == '\n' ) line_no--;
  buff[spot++] = (char)c;
}

void dump_buff() /* for error reporting */
{
  int k;
  fprintf(stderr,"  Input line so far: ");
  for ( k = 0 ; k < errspot ; k++ )
    fputc(errbuff[k],stderr);
  fputc('\n',stderr);
  fputc('\n',stderr);
}

int yywrap()
{
  if ( datafile_flag )
    {
      if ( data_fd ) fclose(data_fd);
      data_fd = NULL;
      if ( macro_subs ) free((char *)macro_subs);
      if ( macros ) free ((char *)macros);
      macro_subs = NULL;
      macros = NULL;
      macro_count = macro_subs_top = macro_max = macro_subs_max = 0;
    }
  spot = 0;
  return 1;  /* done */
}

void yylex_init()
{
  spot = 0;
  errspot = 0;
}

/*************************************************************
*
*  Function: record_macro()
*
*  Purpose:  Record macro definition.
*
*/

void record_macro()
{
  int len;
  char *spot;

  macro_flag = 0;

  if ( macro_count >= macro_max )
    { if ( macro_max == 0 )
	{ macro_max = 10; 
	  macros = (struct macro *)mycalloc(macro_max,sizeof(struct macro));
	}
      else
	{ macro_max += 10; 
	  macros = (struct macro *)realloc((char *)macros,
			   macro_max*sizeof(struct macro));
	}
    }

  strncpy(macros[macro_count].name,yytext,NAMESIZE);

  if ( macro_subs_top+SUBMAX >= macro_subs_max )
    { if ( macro_subs_max == 0 )
	{ macro_subs_max = 2*SUBMAX; 
	  macro_subs = (char *)mycalloc(macro_subs_max,1);
	}
      else
	{ macro_subs_max += 2*SUBMAX; 
	  macro_subs = (char *)realloc(macro_subs,macro_subs_max);
	}
    }

  spot = macro_subs+macro_subs_top;
  for ( len = 0 ; len < SUBMAX ; len++ )
    { int c = input();
      if ( (c == '\0') || (c == '\n') ) break;
      spot[len] = (char)c;
    }

  if ( len >= SUBMAX )
     error("Macro definition too long.\n",UNRECOVERABLE);

  /* strip terminal comment */
  if ( strstr(spot,"//") )
    { *strstr(spot,"//") = 0;
      len = strlen(spot);
    }
  else spot[len] = 0;
  macros[macro_count].subsize = len;
  macros[macro_count++].offset = macro_subs_top;
  macro_subs_top += len+1;
}

/*************************************************************
*
*  Function:  macro()
*
*  Purpose: See if yytext is a macro, and do substitution.
*
*  Return:  1 if macro, 0 if not.
*
*/

int macro()
{
  int n,k;
  char *c;

  /* find name, simple linear search */
  for ( n = 0 ; n < macro_count ; n++ )
     if ( strcmp(yytext,macros[n].name) == 0 ) break;

  if ( n == macro_count ) return 0;   /* not found */

  /* insert string into input in reverse order so reads ok */
  c = macro_subs+macros[n].offset;
  for ( k = macros[n].subsize - 1 ; k >= 0 ; k-- )
    unput(c[k]);

  return 1;
}

/* for deciding type of minus sign */
int minus_type()
{
	   { switch ( tok )
		 {  case LEAD_INTEGER_:
		    case SIGNED_NUMBER_:
		    case INTEGER_:
		    case REAL_:
		    case PI_:
		    case E_:
		    case G_:
		    case PARAM_:
		    case COORD_:
		    case IDENT_:
		    case '^': 
		    case ')': return '-'; 
		    default: return UMINUS_ ;
		 }
	     }
  return '-';  /* to satisfy picky compilers that want a guaranteed return */
}

