#include <iostream.h>  // DEBUG
#include <qrect.h>
#include <qpainter.h>
#include <qpaintd.h>

#include "QJScorePainter.h"
#include "QJDrawComponents.h"
#include "QJDraw.h"
#include "QJListDraw.h"
#include "QJPixDraw.h"
#include "QJStaveNoteDraw.h"
#include "QJStaveTieDraw.h"

#include "rep/JStave.h"
#include "jam/JBeatToX.h"
#include "jam/JStaveToY.h"
#include "rep/JStaveNoteInfo.h"
#include "jam/JNoteShredder.h"
#include "rep/JBeat.h"

QJScorePainter::QJScorePainter(QPaintDevice * paintDevice,
			       QPixmap *      pixmap,
			       const JStave * stave,
			       const JBeatToX  *   beatToX,
			       const JStaveToY  *  staveToY,
			       QJDrawComponents *components)
  :  JScorePainter(),
     _paintDevice(paintDevice),
     _pixmap(pixmap),
     _stave(stave),
     _staveToY(staveToY),
     _components(components),
     _leftAdjust(components->size()), 
     _rightAdjust(components->size()),
     _updateRect(0,0,-1,-1),
     _needUpdate(FALSE),
     _style(NORMAL),
     _painter(0),
     _cursorDraw(0),
     _cursorBeat(0),
     _cursorOn(FALSE),
     _cursorPitch(60),
     _lastPitch(70),
     _beatToX(beatToX)
{
  _cursorDraw = QJDrawComponents::the()->component(QJDrawComponents::CURSOR);
}

QJScorePainter::~QJScorePainter()
{
  if (_painter)  delete _painter;
}

  // Clear everything !!!
void 
QJScorePainter::clear()
{
  //  QPainter p(_pixmap);

  setStyle(NORMAL);

  _painter->eraseRect(_beatToX->x(),
		      _staveToY->y(),
		      _beatToX->width(),
		      _staveToY->height());

  _updateRect=QRect(_beatToX->x(),
		    _staveToY->y(),
		    _beatToX->width(),
		    _staveToY->height());

  _needUpdate=TRUE;

  drawStaff();
  drawName();
  //  p.end();
}



void
QJScorePainter::updateDone()
{
  _needUpdate=FALSE;
  _updateRect.setRect(0,0,0,0);
}

// Wipe area from --- till 
// If side effects then you may adjust from and till.
void 
QJScorePainter::clear(JLRRep lr)
{
#ifdef J_DEBUG
  //  cout << "clear" << from << till <<endl;
#endif 

  // Account for size of components

  setStyle(NORMAL);
  int left=_beatToX->x(lr.left()); //-_leftAdjust;
  if (left < _beatToX->x()) 
    left=_beatToX->x();

  int right=_beatToX->x(lr.right()); // +_rightAdjust;
  if (right > _beatToX->x()+_beatToX->width()) 
    right=_beatToX->x()+_beatToX->width();

  int y=_staveToY->y();
  int h=_staveToY->height();


  //  QPainter p(_pixmap);
  _painter->eraseRect(left,y,right-left,h);

  addToUpdate(QRect(left,y,right-left,h));

  //  debug ("Call staff redraw");

  drawStaff(left,right);
  drawName();
  //  p.end();

}

void
QJScorePainter::addToUpdate(const QRect &rect)
{
  if (!_needUpdate) 
    _updateRect=rect;
  else
    _updateRect.unite(rect);

  _needUpdate=TRUE;
}


void
QJScorePainter::setStyle(JStyle style)
{

  // create a painter if not already active

  if (!_painter) {
    _painter = new QPainter(_pixmap);
  } else {
    if (_style == style) return;
  }


  if (style == HILIGHT) {
    _painter->setPen(QColor(255,0x10,0x10));
  } else if ( style == DULL ) {
    _painter->setPen(QColor(180,180,180));
  } else {
    _painter->setPen(QColor(0,0,0));
  }

  _style = style;

}

bool
QJScorePainter::drawNote(JBeat beat,
			 JDuration duration,
			 int pitch,
			 JStyle  style,
			 JNoteStyle flags)

{

  _lastPitch = pitch;

  JStaveNoteInfo noteInfo=_stave->staveNoteInfo(pitch);

  int y  = _staveToY->y(noteInfo.line());
	

  setStyle(style);

  // TODO: factory


  JNoteShredder  scoreNotes(beat,duration);

  bool needTie=false;

  int xlast = 0;

  JBeat     b;
  JDuration dur;

  while( scoreNotes.next(b,dur) ) {

    int x  = _beatToX->x(b);
    
    if (needTie) { 
      QJStaveTieDraw tie(x-xlast);
      tie.draw(x,y,_painter);
      QRect urect = *tie.bBox();
      urect.moveBy(x,y);
      addToUpdate(urect);
    }


    QJStaveNoteDraw d(b,
		      dur,
		      noteInfo,flags);
    
    d.draw(x,y,_painter);
    
    QRect urect=*d.bBox();
    urect.moveBy(x,y);

    addToUpdate(urect);
  }
  return TRUE;
}



bool
QJScorePainter::drawEvent(JBeat  beat)
{

   int y=_staveToY->y(-4);


  //  setStyle(style);

  // TODO: factory

#ifdef J_DEBUG
  cout << " I would draw an event if I knew how !!! " << endl;
#endif

//   QJStaveNoteDraw d(b,
// 		    dur,
// 		    noteInfo,flags);
  
//   d.draw(x,y,_painter);
  
//   QRect urect=*d.bBox();
//   urect.moveBy(x,y);
  
//   addToUpdate(urect);

  return TRUE;
}



void 
QJScorePainter::update()
{

  if (_painter == 0 ) return;
  delete _painter;
  _painter=0;

  const QRect &rect=_updateRect;

  //  debug(" QJScorePainter paint %d %d %d %d ",rect.x(),rect.y(),rect.width(),rect.height());
  
  bitBlt(_paintDevice,rect.x(),rect.y(),_pixmap,rect.x(),rect.y(),rect.width(),rect.height(),CopyROP);

  updateDone();

}

bool
QJScorePainter::drawBar(JBeat beat,
			int num,
			JStyle  style )
{

  

  int ybot=_staveToY->y(_stave->firstLine());
  int ytop=_staveToY->y(_stave->firstLine()+ (_stave->nLines()-1)*2 );


  setStyle(style);


  int x  = _beatToX->x(beat)-20;  //TODO unhack me !!!

  _painter->drawLine(x,ybot,x,ytop);

  QString str;

  str.setNum(num);

  _painter->drawText(x,ytop-2,str);

  QRect urect(x,ytop-30,1,ybot-ytop+32);

  addToUpdate(urect);

  return TRUE;
}



bool
QJScorePainter::drawRest(JBeat beat,
			 JDuration duration,
			 int pitch,
			 JStyle  style)
{

  if (pitch == -1 ) pitch = _lastPitch;

  JStaveNoteInfo noteInfo = _stave->staveNoteInfo(pitch);
  int  y = _staveToY->y(noteInfo.line());

  setStyle(style);

  JNoteShredder  scoreNotes(beat,duration);

  JBeat     b;
  JDuration dur;

  while( scoreNotes.next(b,dur) ) {

    int x  = _beatToX->x(b);

    QJStaveRestDraw d(b,
		      dur,
		      noteInfo);
    
    d.draw(x,y,_painter);
    
    QRect urect=*d.bBox();
    urect.moveBy(x,y);

    addToUpdate(urect);
  }


  return TRUE;
}


void 
QJScorePainter::drawName()
{
  QRect rect(_beatToX->x(),_staveToY->y(),
	     _beatToX->x()+100,_staveToY->y()+20);
  
  setStyle(NORMAL);
  _painter->drawRect(rect);

  addToUpdate(rect);

  _painter->drawText(rect,AlignVCenter|AlignHCenter,_name);
}

void 
QJScorePainter::drawStaff(int left,int right)
{

  setStyle(NORMAL);

  int firstLine=_stave->firstLine();

  for (int i=0 ; i < _stave->nLines();i++)
    {
      int y=_staveToY->y(firstLine+i*2);
      _painter->drawLine(left,y,right,y);
    }
  
  drawClef();

}

void 
QJScorePainter::drawClef()
{
  const QJDraw *d=0;

  setStyle(NORMAL);

  switch(_stave->type()) {
  case TREBLE_STAVE:
    d=_components->component(QJDrawComponents::TREBLE);
    break;
  case BASS_STAVE: 
    d=_components->component(QJDrawComponents::BASS);
  case ALTO_STAVE:    //etc.
    d=_components->component(QJDrawComponents::ALTO);
  default:
    break;
  }

  if ( d != 0 ) {
    d->draw(_beatToX->x()+6,
	    _staveToY->y(_stave->firstLine()+_stave->nLines()*2)-3,
	    _painter);
  }
}

void 
QJScorePainter::drawStaff()
{
  int left=_beatToX->x();
  int right=left+_beatToX->width();
  drawStaff(left,right);
}

void
QJScorePainter::eraseCursor()
{

  if (_cursorDraw == 0) return;

  if (_painter) _painter->flush();

  QRect rect= *_cursorDraw->bBox();;
  rect.moveBy(_cursorX,_cursorY);
  
  bitBlt(_paintDevice,rect.x(),rect.y(),
	 _pixmap,rect.x(),rect.y(),rect.width(),rect.height(),
	 CopyROP);


}


void 
QJScorePainter::drawCursor()
{

  // cursor goes straight onto draw device
  if (!_cursorOn) return;

  QPainter p(_paintDevice);

  p.setPen(QColor(255,0x10,0x10));

  JStaveNoteInfo noteInfo=_stave->staveNoteInfo(_cursorPitch);

  _cursorX= _beatToX->x(_cursorBeat);
  _cursorY= _staveToY->y(noteInfo.line());

  _cursorDraw->draw(_cursorX,_cursorY,&p);

  p.end();
}


void
QJScorePainter::setCursorAt(JBeat beat,int pitch)
{

  if (_stave == 0 ) return;

  eraseCursor();

  _cursorBeat=beat;
  _cursorPitch=pitch;

  if (_cursorOn)
    drawCursor();
}


void 
QJScorePainter::assertCursor()
{
  cursorOn(_cursorOn);
}

void 
QJScorePainter::cursorOn(bool yes)
{
  _cursorOn=yes;

  if (yes)
    drawCursor();
  else
    eraseCursor();
}













