/* Conversion of files between different charsets and usages.
   Copyright (C) 1990 Free Software Foundation, Inc.
   Francois Pinard <pinard@iro.umontreal.ca>, 1988.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* The following declarations are included by the main program only.

   The purpose of this separate file is to keep the main program pure of any
   particular character set, while providing the linking mechanism between the
   general algorithm with the currently available conversion routines.

   Adding a new character set or usage requires a single file for the
   conversion tables and routines, and modifying this inclusion file.
*/

typedef enum			/* Codes for available charsets */
  {
    CODE_APPLEMAC,		/* ASCII 8 bits for Apple's Macintosh */
    CODE_ASCII,			/* ASCII 7 bits, <BS> to overstrike */
    CODE_BANGBANG,		/* ASCII "bang bang", escapes are ! and !! */
    CODE_CCCASCII,		/* ASCII 8 bits as seen by Perkin Elmer */
    CODE_CDCASCII,		/* ASCII 8 bits as seen by Control Data */
    CODE_CDCNOS,		/* ASCII 6/12 from NOS, escapes are ^ and @ */
    CODE_EBCDIC,		/* EBCDIC */
    CODE_FLAT,			/* ASCII without diacritics nor underline */
    CODE_IBMPC,			/* ASCII 8 bits for IBM's PC */
    CODE_ICONQNX,		/* ASCII with diacritics for Unisys'ICON */
    CODE_LATEX,			/* ASCII with LaTeX codes */
    CODE_LATIN1,		/* ASCII extended by Latin Alphabet 1 */
    CODE_TEXTE			/* ASCII with easy French conventions */
  }
TYPE_code;

#ifdef ENUM_INT_BUG
#define TYPE_code int
#endif

/* The following array provides strings to represent available codes, and is
   parallel to TYPE_code declaration.  Each entry contains two strings:
   first a short form containing four letters, the same four letters used to
   name the conversion routines of this package; then a long form giving a
   more explicit strings.  */

struct keyword_struct
  {
    const char *s;
    const char *l;
  }
code_keywords[] =
  {
    { "maci", "applemac" },
    { "asci", "ascii" },
    { "bang", "bangbang" },
    { "ccca", "cccascii" },
    { "cdca", "cdcascii" },
    { "cdcn", "cdcnos" },
    { "ebcd", "ebcdic" },
    { "flat", "flat" },
    { "ibmp", "ibmpc" },
    { "icon", "iconqnx" },
    { "ltex", "latex" },
    { "lat1", "latin1" },
    { "txte", "texte" },
  };

#define NUMBER_OF_KEYWORDS \
  (sizeof (code_keywords) / sizeof (struct keyword_struct))

/* Description of all available conversion routines. */

#ifndef __STDC__
#define PARAMLIST ()
#else
#define PARAMLIST (FILE *, FILE *)
#endif

extern void applemac_ibmpc	PARAMLIST;
extern void ascii_cdcnos	PARAMLIST;
extern void ascii_flat		PARAMLIST;
extern void ascii_latex		PARAMLIST;
extern void ascii_texte		PARAMLIST;
extern void bangbang_latin1	PARAMLIST;
extern void cccascii_ebcdic	PARAMLIST;
extern void ascii8_ascii7	PARAMLIST;
extern void cdcascii_ebcdic	PARAMLIST;
extern void cdcnos_ascii	PARAMLIST;
extern void ebcdic_cccascii	PARAMLIST;
extern void ebcdic_cdcascii	PARAMLIST;
extern void ibmpc_applemac	PARAMLIST;
extern void ibmpc_iconqnx	PARAMLIST;
extern void ibmpc_latin1	PARAMLIST;
extern void iconqnx_ibmpc	PARAMLIST;
extern void latex_ascii		PARAMLIST;
extern void latex_latin1	PARAMLIST;
extern void latin1_bangbang	PARAMLIST;
extern void latin1_ibmpc	PARAMLIST;
extern void latin1_latex	PARAMLIST;
extern void latin1_texte	PARAMLIST;
extern void texte_ascii		PARAMLIST;
extern void texte_latin1	PARAMLIST;

#undef PARAMLIST

/* Constants for conversion costs. */

#define NOWAY	10000		/* No way for this conversion */

#define STEP	100		/* Nominal value for each step */
#define ALREADY	0		/* The code injects, no conversion */

#define LOOSE	+10		/* Some characters are thrown away */
#define EXACT	-10		/* The conversion would be reversible */

#define SLOW	+1		/* Multiple characters to one */
#define FAST	-1		/* One character to one */

typedef struct 
  {
#ifndef __STDC__
    void (*routine) ();		/* Address of conversion routine */
#else
    void (*routine) (FILE *, FILE *); /* Address of conversion routine */
#endif
    TYPE_code code_before;	/* Code of charset before conversion */
    TYPE_code code_after;	/* Code of charset after conversion */
    int conversion_cost;	/* Bonuses and penalties */
  }
TYPE_of_step;

TYPE_of_step single_steps[] =	/* Table of elementary conversions */
  {
    { applemac_ibmpc,	CODE_APPLEMAC,	CODE_IBMPC,	STEP LOOSE FAST },
    { NULL,		CODE_ASCII,	CODE_CDCASCII,	ALREADY         },
    { ascii_cdcnos,	CODE_ASCII,	CODE_CDCNOS,	STEP EXACT      },
    { ascii_flat,	CODE_ASCII,	CODE_FLAT,	STEP LOOSE      },
    { ascii_latex,	CODE_ASCII,	CODE_LATEX,	STEP LOOSE SLOW },
    { ascii_texte,	CODE_ASCII,	CODE_TEXTE,	STEP       SLOW },
    { bangbang_latin1,	CODE_BANGBANG,	CODE_LATIN1,	STEP EXACT      },
    { cccascii_ebcdic,	CODE_CCCASCII,	CODE_EBCDIC,	STEP EXACT FAST },
    { ascii8_ascii7,	CODE_CDCASCII,	CODE_ASCII,	STEP LOOSE FAST },
    { cdcascii_ebcdic,	CODE_CDCASCII,	CODE_EBCDIC,	STEP EXACT FAST },
    { cdcnos_ascii,	CODE_CDCNOS,	CODE_ASCII,	STEP EXACT      },
    { ebcdic_cccascii,	CODE_EBCDIC,	CODE_CCCASCII,	STEP EXACT FAST },
    { ebcdic_cdcascii,	CODE_EBCDIC,	CODE_CDCASCII,	STEP EXACT FAST },
    { ibmpc_applemac,	CODE_IBMPC,	CODE_APPLEMAC,	STEP LOOSE FAST },
    { ibmpc_iconqnx,	CODE_IBMPC,	CODE_ICONQNX,	STEP            },
    { ibmpc_latin1,	CODE_IBMPC,	CODE_LATIN1,	STEP LOOSE      },
    { iconqnx_ibmpc,	CODE_ICONQNX,	CODE_IBMPC,	STEP LOOSE SLOW },
    { latex_ascii,	CODE_LATEX,	CODE_ASCII,	STEP LOOSE SLOW },
    { latex_latin1,	CODE_LATEX,	CODE_LATIN1,	STEP LOOSE SLOW },
    { latin1_bangbang,	CODE_LATIN1,	CODE_BANGBANG,	STEP LOOSE FAST },
    { latin1_ibmpc,	CODE_LATIN1,	CODE_IBMPC,	STEP LOOSE      },
    { latin1_latex,	CODE_LATIN1,	CODE_LATEX,	STEP LOOSE      },
    { latin1_texte,	CODE_LATIN1,	CODE_TEXTE,	STEP LOOSE	},
    { texte_ascii,	CODE_TEXTE,	CODE_ASCII,	STEP       SLOW },
    { texte_latin1,	CODE_TEXTE,	CODE_LATIN1,	STEP LOOSE SLOW }
  };

#define NUMBER_OF_SINGLE_STEPS (sizeof (single_steps) / sizeof (TYPE_of_step))
