/*
 FILE: 			AnnotatedExtractor.java
 
 Given a database connection, extracts schema and data information using JDBC calls into local data structures.
 
 DO NOT RUN OR EDIT THIS FILE.  USE EXTRACTORXSPEC.JAVA INSTEAD.  THIS SOURCE CODE IS PROVIDED FOR YOUR CONVENIENCE ONLY.
 */

package unity.extractor;

import java.util.*;
import java.io.*;
import java.sql.*;

import javax.swing.JTextArea;

import unity.annotation.*;


public class AnnotatedExtractor
{
	protected DatabaseMetaData dmd;
	protected String dbName;
	protected AnnotatedSourceDatabase db;
	protected Connection con;
	protected JTextArea outputBox = null;
	protected String dbProduct, dbVersion,url, driver;
	
	public void setOutputArea(JTextArea a)
	{	outputBox = a;
	}
	
	public void setDatabaseName(String n)
	{	db.setDatabaseName(n); }
	
	public void extract(Connection c) throws SQLException
	{
		boolean access=false;				// True if extracting information from an Access database
		
		con = c;
		dmd = con.getMetaData();

		// Extract AnnotatedDataSource details and create AnnotatedSourceDatabase Object (Database level detailed information)
		dbName = con.getCatalog();
		dbProduct = dmd.getDatabaseProductName();
		dbVersion = dmd.getDatabaseProductVersion();
		url = dmd.getURL();
		driver = dmd.getDriverName();
		int slashIndex = dbName.lastIndexOf("\\");
		if (slashIndex > 0)
			dbName = dbName.substring(slashIndex+1);
				
		db = new AnnotatedSourceDatabase(dbName, dbProduct, dbVersion, url, driver);
		
		HashMap tablesCreated;
		
		if (dbProduct.indexOf("CCESS")>-1)
			access = true;
		tablesCreated = createAnnotatedSourceTables(con, access);

		db.setSourceTables(tablesCreated);
	}


	public HashMap createAnnotatedSourceTables(Connection con, boolean access) throws SQLException
	{	// Extract table information (includes number of rows and cardinality of attributes)
		HashMap tables = new HashMap();
		String  tableName, schemaName, catalogName, comment;
		AnnotatedSourceTable st = null;
		
		// Extract ResultSet of tables for this database
		String[] tableTypes = {"TABLE"};  //Type of tables to extract // tableTypes = "TABLE", "VIEW", "SYSTEM TABLE", "GLOBAL TEMPORARY", "LOCAL TEMPORARY", "ALIAS", "SYNONYM"
		ResultSet rs1 = dmd.getTables(null,null, "%",tableTypes);                // Retrieve all (user) tables
		
		while (rs1.next()) //Table information
		{
			tableName = rs1.getString(3);
			schemaName = rs1.getString(2);
			catalogName = rs1.getString(1);
			comment = rs1.getString(5);
			String message = "Processing "+"Table: "+tableName+"  Schema: "+schemaName+"  Catalog: "+catalogName;
			if (outputBox == null)
				System.out.println("Table: "+tableName+"  Schema: "+schemaName+"  Catalog: "+catalogName);
			else
				outputBox.append(message+"\n");
			
			// if (schemaName != null && !schemaName.equals(""))
			// 	tableName = schemaName+"."+tableName;
			HashMap fields = createFields(schemaName,tableName);
			
			if (access)
			{	// Access has different ways of extracting primary and foreign keys. Not currently extracting.
				// AnnotatedSourceKey pkey = createPrimaryKey(tableName, fields);
				st = new AnnotatedSourceTable(catalogName, schemaName, tableName, comment, fields, null);
				if (schemaName != null && !schemaName.equals(""))
					st.setSemanticTableName(dbName+"."+schemaName+"."+tableName);
				else
					st.setSemanticTableName(dbName+"."+tableName);	// Set semantic table name to be exactly system name for now
				// pkey.setTable(st);
				// ArrayList fkeys = createForeignKeys(tableName, fields, st);
				// st.setForeignKeys(fkeys);
				tables.put(tableName, st);
			}	
			else
			{
				AnnotatedSourceKey pkey = createPrimaryKey(tableName, fields);
				st = new AnnotatedSourceTable(catalogName, schemaName, tableName, comment, fields, pkey);
				if (schemaName != null && !schemaName.equals(""))
					st.setSemanticTableName(dbName+"."+schemaName+"."+tableName);
				else
					st.setSemanticTableName(dbName+"."+tableName);	// Set semantic table name to be exactly system name for now
				pkey.setTable(st);
				ArrayList fkeys = createForeignKeys(tableName, fields, st);
				st.setForeignKeys(fkeys);
				tables.put(tableName, st);
			}
			
			// Build query to extract table and field information
			StringBuffer sql = new StringBuffer(500);
			int count = 0;
			AnnotatedSourceField []asfs = new AnnotatedSourceField[fields.size()];
			
			if (!access)
			{					
				sql.append("SELECT COUNT(*)");
				Iterator it = fields.values().iterator();
				
				while(it.hasNext())
				{	AnnotatedSourceField asf = (AnnotatedSourceField) it.next();
					if (!asf.isBlob())
					{	sql.append(",count(distinct "+asf.getSQLColumnName()+")");
						asfs[count++] = asf;
					}
				}
				
				sql.append("\nFROM "+st.getSQLTableNameWithSchema());
			}
			else
			{	// Access does not support count(distinct).  Access 97 does not support derived tables, so statistics are impossible.
				if (dbVersion.indexOf("3.50")>=0)		// Access 97 - cannot do anything - just return row count for all fields
				{	sql.append("SELECT count(*)");	
					Iterator it = fields.values().iterator();
				
					while(it.hasNext())
					{	AnnotatedSourceField asf = (AnnotatedSourceField) it.next();
						if (!asf.isBlob())
						{	sql.append(",count(*)");
							asfs[count++] = asf;
						}
					}
					sql.append("\nFROM "+st.getSQLTableNameWithSchema());
				}
				else
				{	// Newer than Access 97 supports subqueries in FROM.  General syntax:
					// select * from
					// (Select count(*) from(select distinct shipVia from orders) as a) as x1,
					// (Select count(*) from(select distinct orderDate from orders) as b) as x2,
					// (Select count(*) from(select distinct orderId from orders) as c) as x3 ...
					sql.append("SELECT * FROM");
					String tbName = st.getSQLTableNameWithSchema();
					sql.append("(SELECT COUNT(*) FROM (SELECT * FROM "+tbName+") AS C) AS X0");
					
					Iterator it = fields.values().iterator();					
					while(it.hasNext())
					{	AnnotatedSourceField asf = (AnnotatedSourceField) it.next();
						if (!asf.isBlob())
						{	sql.append(",(SELECT COUNT(*) FROM (SELECT DISTINCT "+asf.getSQLColumnName()+" FROM "+tbName+") AS T"+count+") AS Z"+count);
							asfs[count++] = asf;
						}
					}
				}
			}
	
			try
			{
				Statement stmt = con.createStatement();
				if (outputBox == null)
					System.out.println("  Computing table statistics...");
				else
					outputBox.append("  Computing table statistics...");
				//System.out.println(sql);
				ResultSet rst = stmt.executeQuery(sql.toString());
				if (rst.next())
				{	if (st != null)
						st.setNumTuples(rst.getInt(1));
					int i = 0;
					while (i < count)
					{	asfs[i].setNumDistinctValues(rst.getInt(i+2));
						i++;
					}
				}
			}
			catch (SQLException e)
			{	// Squelch exception and keep going  
				if (outputBox == null)
					System.out.println("  Error while computing table statistics..."+e);
				else
					outputBox.append("  Error while computing table statistics..."+e+"\n");				
			}
			if (outputBox == null)
				System.out.println("Done processing table.");
			else
				outputBox.append("Done processing table.\n");		
		}
		
		Iterator it = tables.values().iterator();
		while(it.hasNext())
		{
			fillForeignKeysAndCreateJoins((AnnotatedSourceTable)it.next(), tables);
		}
		return tables;
	}

	public HashMap createFields(String schemaName, String tableName) throws SQLException
	{
		HashMap fieldsCreated = new HashMap();
		ResultSet rs2 = dmd.getColumns(null,schemaName,tableName,"%");	// JDBC call to get all fields in a table
		
		while (rs2.next())
		{
			String f1,f2,f3,f4,f6,f12,f13,f18;
			int f5,f7,f9,f10,f11,f16,f17;
			f1 = rs2.getString(1);
			f2 = rs2.getString(2);
			f3 = rs2.getString(3);
			f4 = rs2.getString(4);
			f5 = rs2.getInt(5);
			f6 = rs2.getString(6);
			f7 = rs2.getInt(7);
			f9 = rs2.getInt(9);
			f10 = rs2.getInt(10);
			f11 = rs2.getInt(11);
			f12 = rs2.getString(12);
			f13 = rs2.getString(13);
			f16 = rs2.getInt(16);
			f17 = rs2.getInt(17);
			f18 = rs2.getString(18);

			AnnotatedSourceField sf = new AnnotatedSourceField(f1, f2, f3,f4,f5,f6,f7,f9,f10,f11,f12,f13,f16,f17,f18);
			fieldsCreated.put(f4,sf);
		}
		return fieldsCreated;
	}
	
	
	public AnnotatedSourceKey createPrimaryKey(String tableName, HashMap fields ) throws SQLException
	{
		ResultSet rs3 = dmd.getPrimaryKeys(null, null,tableName);	// JDBC call to obtain the Primary Key of a table
		String keyName = "";
		ArrayList newKey = new ArrayList();							
		while(rs3.next())
		{
			String key = rs3.getString(6);							// Each entry describes a column of the primary key
			if(keyName.length() <1 && !(key.equals(null)))			// This code used to build a primary key name
				keyName = key;
			else if (key.equals(keyName))
				keyName = key;
			else if (!key.equals(null))
				keyName = keyName+"##"+key;
			else
				keyName = "PK_"+tableName;
			newKey.add(fields.get(rs3.getString(4).trim()));
		}
		
		AnnotatedSourceKey pkey = new AnnotatedSourceKey(newKey, 1 , keyName);
		return pkey;
	}

	public ArrayList createForeignKeys(String tableName, HashMap fields, AnnotatedSourceTable st) throws SQLException
	{
		ArrayList fkey = new ArrayList();
		ResultSet rs3 = dmd.getImportedKeys(null, null,tableName);		//JDBC call to obtain foreign keys for table
		ArrayList key = new ArrayList();
		String keyName = "";
		String toTblName = "";
		ArrayList fkFieldNames = new ArrayList();

		while(rs3.next())
		{
			if(toTblName.equals(rs3.getString(3).trim()))
			{
				key.add(fields.get(rs3.getString(8).trim()));
				fkFieldNames.add(rs3.getString(4).trim());
			}
			else
			{
				if(!rs3.isFirst()) //more than one foreign key for table
				{	//Create new AnnotatedSourceForeignKey in fkey for data collected on key.
					fkey.add(new AnnotatedSourceForeignKey( st, key, fkFieldNames, keyName, toTblName));
					key = new ArrayList();
					keyName = "";
					toTblName = "";
					fkFieldNames = new ArrayList();
				}
				//Get data for AnnotatedSourceForeignKey
				key.add(fields.get(rs3.getString(8).trim()));
				keyName = rs3.getString(12).trim();
				toTblName = rs3.getString(3).trim();
				fkFieldNames.add(rs3.getString(4).trim());
			}
		}
		if (key.size()>0)  // Table has at least one key. Create ForeignKey
		{
			fkey.add(new AnnotatedSourceForeignKey( st, key, fkFieldNames, keyName, toTblName));
		}

		return fkey;
	}

	private void fillForeignKeysAndCreateJoins(AnnotatedSourceTable st, HashMap tables)
	{
		// Use complete JDBC obtained data to complete object references for foreign keys and joins
		ArrayList fkey = st.getForeignKeys();
		for(int i=0;i<fkey.size();i++)
		{
			AnnotatedSourceForeignKey fk = (AnnotatedSourceForeignKey)fkey.get(i);
			String toTableName = fk.getToTableName();
			ArrayList fieldNames = fk.getFieldNames();
			AnnotatedSourceTable table = (AnnotatedSourceTable)tables.get(toTableName);

			fk.setToSourceTable(table);  //Sets SourceTable in SourceForeignKey Object
			fk.setToKey(table.getPrimaryKey());
			ArrayList fields = new ArrayList();
			for(int j=0;j<fieldNames.size();j++)
			{
				fields.add(table.getField((String)fieldNames.get(j)));
			}
			String joinName = st.getTableName()+"->"+table.getTableName();
			String reverseJoinName = table.getTableName()+"->"+st.getTableName();
			AnnotatedSourceJoin join = new AnnotatedSourceJoin(fk,table.getPrimaryKey(),joinName,3);
			AnnotatedSourceJoin reverse = new AnnotatedSourceJoin(table.getPrimaryKey(),fk,reverseJoinName,2);
			join.setReverseJoin(reverse);
			st.addJoin(join);
			reverse.setReverseJoin(join);
			table.addJoin(reverse);
		}
	}

	public AnnotatedSourceKey createPrimaryKeyAccess(String tableName, HashMap fields ) throws SQLException
	{	// This code is not working..
		// Get foreign and primary keys (for Access only)
        Statement stmt = con.createStatement();
        ResultSet foreignKeys = stmt.executeQuery("SELECT  szRelationship, szObject, szReferencedObject, szColumn, szReferencedColumn FROM MSysRelationships WHERE szObject like '"+tableName+"'");
        while (foreignKeys.next())
        {
           String fkName = foreignKeys.getString(1);//FK_NAME
           // if FK has no name - make it up (use tablename instead)
           if (fkName == null)
           {
               fkName = foreignKeys.getString(2);//PKTABLE_NAME
           }
        //   String frkey = foreignKeys.getString(3); //local column //FKCOLUMN_NAME
         //  String prkey = foreignKeys.getString(4); //foreign column //PKCOLUMN_NAME
		}
        return null;
	}
                                       
	public void exportXML(File file) throws IOException
	{
		FileOutputStream to = null;
		PrintWriter pw = null;
		
		to = new FileOutputStream(file);
		pw = new PrintWriter(to, true);  //flush pw on println
		
		exportXML(pw);				
	}
	
	public void exportXML(PrintWriter pw) throws IOException
	{	// Export database instance to XML file for use with UnityDriver		
		
		pw.println("<?xml version = \"1.0\"?>");
		pw.println();
		pw.println("<XSPEC xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\" xsi:noNamespaceSchemaLocation=\"xspec.xsd\">");
		pw.println(db.toXML());

		TreeMap sourceTables = new TreeMap(db.getSourceTables());
		Iterator tables = (sourceTables.keySet()).iterator();
		AnnotatedSourceTable table;

		while(tables.hasNext())
		{
			pw.println("   <TABLE>");
			table = (AnnotatedSourceTable)sourceTables.get(tables.next());
			pw.println(table.toXML());
			
			//Output Fields
			TreeMap sourceFields = new TreeMap(table.getSourceFields());
			Iterator fields = (sourceFields.keySet()).iterator();
			AnnotatedSourceField field;
			while(fields.hasNext())
			{
				pw.println("     <FIELD>");
				field = (AnnotatedSourceField)sourceFields.get(fields.next());
				pw.println(field.toXML());
				pw.println("     </FIELD>");
				pw.println();
			}

			//Output PrimaryKey			
			AnnotatedSourceKey pKey = (AnnotatedSourceKey)table.getPrimaryKey();
			if (pKey != null)
				pw.println( pKey.toXML() +"\n");			

			ArrayList fkeys= table.getForeignKeys();
			for(int i= 0; i<fkeys.size(); i++)
			{	pw.println(((AnnotatedSourceForeignKey)fkeys.get(i)).toXML());
				pw.println();
			}

			ArrayList joins = table.getJoins();
			for(int i= 0; i<joins.size(); i++)
			{	pw.println("     <JOIN>");
				pw.println(((AnnotatedSourceJoin)joins.get(i)).toXML());	
				pw.println("     </JOIN>\n");
			}

			pw.println("   </TABLE>\n");
		}

		pw.println("</XSPEC>\n");
		pw.close();
	}
} //end of the class