#charset "us-ascii"
/* 
 *  Copyright (c) 2006 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the TADS 3 Relation Library Extension
 *
 *  relation_object.t
 *
 *--------------------------------------------------------------------
 *  THE TADS 3 RELATION OBJECT FILE
 *--------------------------------------------------------------------
 */

#include "relation.h"

/*
 *  A Relation is an abstract datatype that represents 
 *  attribution a given parameter value, or a relationship 
 *  between more than one parameter value.  
 */
class RelationObject: RelationBase
{
    relSet      = nil
    parms       = []

    /*
     *  An abstract method for the RelationObject class.
     *  Returns true if the parameter list belongs to a 
     *  relation within relationSet; otherwise returns nil.
     */
    isElementOf(parms, relationSet) 
    {
        return relationSet.hasElement(parms...);
    }

    /*
     *  An abstract method for the RelationObject class.
     *  Returns a list of all relations sets for which the
     *  parameter list forms a relation.
     */
    getMembershipList(parms)
    {
        local vec;

        vec = new Vector(10);
        for (local rs = firstObj(RelationSetObject, ObjAll); rs != nil; 
            rs = nextObj(rs, RelationSetObject, ObjAll))
            if (isElementOf(parms, rs))
                vec.append(rs);
    
        return vec.toList();
    }

    /*
     *  Two relations are said to be equal if their 
     *  corresponding parameters match.
     */
    equals(other) { return match(other.parms); }

    /*
     *  Returns true if this relation object's parms
     *  match the mask; otherwise returns nil.
     */
    match(mask)
    {
        /*
         *  Passing an empty mask list matches
         *  this member of the set.
         */
        if (mask.length() == 0)
            return true;

        if (arity != mask.length())
            return nil;

        for (local i = 1; i <= parms.length(); ++i)
        {
            if (parms[i] is in (any))
                continue;

            if (mask[i] is in (any))
                continue;

            if (parms[i] != mask[i])
                return nil;
        }

        return true;
    }

    /*
     *  Returns the parameter list of this relation object
     */
    toList() { return parms; }
    
    /*
     *  Sets the internal details for this relation 
     *  object. This is explicity called when setting
     *  up statically-defined relation objects.
     */
    initInstance()
    {
        /*
         *  Remove null from any parm
         */
        parms   -= null;
        arity   = parms.length();
        relSet.addElement(self);
        if (arity > relSet.arity)
            relSet.arity = arity;
        name    = relSet.name;
        name    = buildName(parms);
    }

    /*
     *  This method is called when using the Relate() macro
     *  for creating dynamically-defined relation objects
     */
    construct(relSet, [parms])
    {
        self.relSet     = relSet;
        self.parms      = parms;
        initInstance();
    }
}