/*
** WinTADS.C
*/

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include "wintads.h"
#include "winio.h"
#include "htads\os.h"

/* Declare the global variables */
#define extern
#include "global.h"
#undef extern

extern BOOL InitMessageBox(HANDLE hInstance);
extern BOOL CreateMessageBox(HWND hwndParent, HANDLE hInst);
extern BOOL AdjustMessageBox(HWND hwnd);
extern int MessageBoxHeight(HWND hwnd);
extern BOOL InitStatus(HANDLE hInstance);
extern BOOL CreateStatus(HANDLE hInstance);
extern VOID UpdateStatus(VOID);
extern BOOL SwapStatus(HANDLE hInstance);
extern VOID SetStatusFont(LOGFONT *plf);
extern VOID TweakMenus(VOID);
extern VOID TweakRecentMenu(VOID);
extern VOID AddRecent(char *new);
extern VOID TweakBindingMenu(VOID);
extern VOID SetupMacroMenus(VOID);
extern VOID MyShowScrollBar(BOOL fShow);
extern BOOL APIENTRY FindDialogProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
extern BOOL APIENTRY StoryWindowDialogProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
extern BOOL APIENTRY InterpreterDialogProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
extern BOOL APIENTRY OptionsDialogProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
extern BOOL APIENTRY AboutDialogProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
extern BOOL APIENTRY RegistryDialogProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
extern DWORD os_main_shell(LPDWORD pdw);

LRESULT EXPENTRY FrameWndProc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK TadsMainWndProc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK About(HWND, UINT, WPARAM, LPARAM);
BOOL	CenterWindow(HWND, HWND);
void	HandleMenuCommand(HWND hwnd, UINT menuItemID);
LONG CalcSeparatorWidth(HWND hWnd);
VOID DrawSeparator(HWND hwnd, HDC hdc, LONG lXStart, LONG lYStart);
SHORT UpdateFonts(HDC hdc, BOOL fUpdate);
SHORT UpdateInputFont(HDC hdc, BOOL fBold);
SHORT DefaultFont(VOID);
BOOL GetFont(LOGFONT *plf);
BOOL GetColor(COLORREF *pcr);
SHORT QueryFile(CHAR *buf, SHORT bufsize, BOOL fWriting, SHORT iType, const CHAR *szPrompt);
UINT APIENTRY DialogProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);

extern SHORT cmdkeylist[];		// Defined in winctls.c
static char szAppName[] = "WinTADS";
static BOOL fNoMouseHit = FALSE;	// Mouse bug kludge--see WM_COMMAND below

// Makes it easier to determine appropriate code paths:
#if defined (WIN32)
	#define IS_WIN32 TRUE
#else
	#define IS_WIN32 FALSE
#endif
#define IS_NT      IS_WIN32 && (BOOL)(GetVersion() < 0x80000000)
#define IS_WIN32S  IS_WIN32 && (BOOL)(!(IS_NT) && (LOBYTE(LOWORD(GetVersion()))<4))
#define IS_WIN95 (BOOL)(!(IS_NT) && !(IS_WIN32S)) && IS_WIN32

/*
** Preferences functions
*/

/* SavePrefs updates the main preferences and saves them to the profile. It
   returns TRUE on success, FALSE on failure. */
BOOL SavePrefs(void)
{
	FILE	*ofp;
	char	szProfileName[50],
			szProfilePath[CCHMAXPATH];
	int	i;

/* Get the name of the profile, the name of the preferences, and the name
   of the key, all from a string resource */
	LoadString(hInst, strix_PrefsFileName, szProfileName, sizeof(szProfileName));

/* If we're supposed to, save the macro data */
	if (prefsTADS.fStickyMacros) {
		for (i = 0; i < 12; i++) {
			if (keycmdargs[cmdkeylist[i] | keytype_macro] != NULL) {
				strncpy(prefsTADS.szMacroText[i],
					keycmdargs[cmdkeylist[i] |
					keytype_macro], 254);
				prefsTADS.szMacroText[i][254] = 0;
			}
		}
	}

/* Open the profile and write the data */
	strcpy(szProfilePath, pszHomePath);
	strcat(szProfilePath, szProfileName);
	ofp = fopen(szProfilePath, "wb");
	if (ofp == NULL)
		return FALSE;
	if (fwrite(&prefsTADS, 1, sizeof(PREFS), ofp) != sizeof(PREFS)) {
		fclose(ofp);
		return FALSE;
	}

/* Don't forget to close the profile */
	return (fclose(ofp) == 0);
}

/* RestorePrefs reads the preferences from the profile and applies them to
   the proper windows.  It returns TRUE on success, FALSE on failure. */
BOOL RestorePrefs(void)
{
	FILE	*ifp;
	char	szProfileName[50],
			szProfilePath[CCHMAXPATH];

/* Start by zeroing out the preferences data block */
	memset((void *)&prefsTADS, 0, sizeof(PREFS));

/* Get the name of the profile, the name of the preferences, and the name
   of the key, all from a string resource */
	LoadString(hInst, strix_PrefsFileName, szProfileName, sizeof(szProfileName));
	
/* Open the profile and read the data */
	strcpy(szProfilePath, pszHomePath);
	strcat(szProfilePath, szProfileName);
	ifp = fopen(szProfilePath, "rb");
	if (ifp == NULL)
		return FALSE;
	if (fread(&prefsTADS, 1, sizeof(PREFS), ifp) != sizeof(PREFS)) {
		fclose(ifp);
		return FALSE;
	}
	if (fclose(ifp) != 0)
		return FALSE;
	if (strcmp(prefsTADS.version, PREFS_VER) != 0)
		return FALSE;

	return TRUE;
}

/* DefaultPrefs sets the preferences to my preferred default. */
void DefaultPrefs(void)
{
/* Zero out the preferences data block */
	memset((void *)&prefsTADS, 0, sizeof(PREFS));

/* Write the current version of WinTADS & its preferences */
	strcpy(prefsTADS.version, PREFS_VER);

/* Let the system set the size of the window */
	GetWindowRect(hwndFrame, &prefsTADS.rectFrame);
	
/* Set the margins */
	prefsTADS.marginx = 8;
	prefsTADS.marginy = 4;
	
/* Set the buffer */
	prefsTADS.buffersize = 4000;
	prefsTADS.bufferslack = 1000;
	
/* Set the history length */
	prefsTADS.historylength = 20;
	
/* Set all of the flags */
	prefsTADS.fStatusSeparated = FALSE;
	prefsTADS.paging = TRUE;
	prefsTADS.fulljustify = TRUE;
	prefsTADS.clearbyscroll = FALSE;
	prefsTADS.doublespace = FALSE;
	prefsTADS.fBoldInput = TRUE;
	prefsTADS.fStickyPaths = TRUE;
	prefsTADS.fStickyMacros = TRUE;
	prefsTADS.fCloseOnEnd = TRUE;
	prefsTADS.fCloseAborts = FALSE;

/* Set the foreground and background colors */
	prefsTADS.ulFore = 0L;
	prefsTADS.ulBack = RGB(255, 255, 255);
	
/* Set the .gam/.sav paths to "" */
	prefsTADS.szGamePath[0] = 0;
	prefsTADS.szSavePath[0] = 0;
	
/* Set the fonts */
	DefaultFont();
}


/*
** Calculate the width of the separator bar bet. the client & status bar.
** It is the same width as the client, so we return the client's width
*/
LONG CalcSeparatorWidth(HWND hwnd)
{
	RECT rect;
	LONG lWidth = 0;

	if (GetClientRect(hwnd, &rect))
		lWidth = rect.right - rect.left;

	lWidth = (lWidth < 0) ? 0 : lWidth;
	return( lWidth );
}

/*
** Draw the separator bar, starting at the point lXStart, lYStart.
*/
VOID DrawSeparator(HWND hwnd, HDC hdc, LONG lXStart, LONG lYStart)
{
	USHORT	i;
	POINT	start[5],
			end[5];
	HPEN	hpen, hpenOld;

	DWORD color[5] = { COLOR_3DHILIGHT, COLOR_3DLIGHT,
                      COLOR_3DLIGHT, COLOR_3DLIGHT,
                      COLOR_3DSHADOW };

	/******************************************************************/
	/* Init the POINTL arrays for drawing the horizontal separator    */
	/******************************************************************/
	LONG lSeparatorWidth = CalcSeparatorWidth( hwnd );

	if (IS_NT) {
		color[0] = COLOR_BTNFACE;
		color[1] = COLOR_BTNFACE;
		color[2] = COLOR_BTNFACE;
		color[3] = COLOR_BTNFACE;
		color[4] = COLOR_BTNSHADOW;
	}

	start[0].x = start[1].x = start[2].x = start[3].x =
		start[4].x = lXStart;
	start[0].y = start[1].y = start[2].y = start[3].y =
		start[4].y = lYStart;
	start[1].y += 1;
	start[2].y += 2;
	start[3].y += 3;
	start[4].y += 4;
	end[0] = start[0];
	end[0].x += lSeparatorWidth;
	end[1] = start[1];
	end[1].x += lSeparatorWidth;
	end[2] = start[2];
	end[2].x += lSeparatorWidth;
	end[3] = start[3];
	end[3].x += lSeparatorWidth;
	end[4] = start[4];
	end[4].x += lSeparatorWidth;

	/******************************************************************/
	/* Draw the horizontal separator bar.                             */
	/******************************************************************/
	for (i=0; i< SEPARATOR_WIDTH; i++) {
		hpen = CreatePen(PS_SOLID, 0, GetSysColor(color[i]));
		hpenOld = SelectObject(hdc, hpen);
		MoveToEx(hdc, start[i].x, start[i].y, NULL);
		LineTo(hdc, end[i].x, end[i].y);
		SelectObject(hdc, hpenOld);
		DeleteObject(hpen);
	}
}


/*
** Font functions
*/

/* UpdateFonts takes the fattrs in prefsTADS and tries to create both a
   regular font and a bold font. It returns TRUE if everything is ok,
   FALSE if no fonts could be created, and -1 if no bold font was created. */
SHORT UpdateFonts(HDC hdc, BOOL fUpdate)
{
	TEXTMETRIC	tm;
	BOOL		fC;
	LOGFONT		lfTemp, lfTempBold;

	/* Create the regular font */
	hfNorm = CreateFontIndirect(&prefsTADS.lfNormal);
	if (hfNorm == NULL)
		return FALSE;
	
	/* Create the bold font */
	hfBold = CreateFontIndirect(&prefsTADS.lfBold);
	if (hfBold == NULL)
		hfBold = CreateFontIndirect(&prefsTADS.lfNormal);

	/* Create the input font */
	UpdateInputFont(hdc, prefsTADS.fBoldInput);

	/* Finally, create the eight fonts which I use to display HTML 4 chars */
	memcpy(&lfTemp, &prefsTADS.lfNormal, sizeof(LOGFONT));
	memcpy(&lfTempBold, &prefsTADS.lfBold, sizeof(LOGFONT));

	lfTemp.lfCharSet = DEFAULT_CHARSET;		// Step 1: the default character set
	lfTempBold.lfCharSet = DEFAULT_CHARSET;
	hfHTML4Norm[HTML4_DEFAULT] = CreateFontIndirect(&lfTemp);
	if (hfHTML4Norm[HTML4_DEFAULT] == NULL)
		hfHTML4Norm[HTML4_DEFAULT] = CreateFontIndirect(&prefsTADS.lfNormal);
	hfHTML4Bold[HTML4_DEFAULT] = CreateFontIndirect(&lfTempBold);
	if (hfHTML4Bold[HTML4_DEFAULT] == NULL)
		hfHTML4Bold[HTML4_DEFAULT] = CreateFontIndirect(&prefsTADS.lfNormal);

	lfTemp.lfCharSet = ANSI_CHARSET;		// Step 2: the ANSI character set
	lfTempBold.lfCharSet = ANSI_CHARSET;
	hfHTML4Norm[HTML4_ANSI] = CreateFontIndirect(&lfTemp);
	if (hfHTML4Norm[HTML4_ANSI] == NULL)
		hfHTML4Norm[HTML4_ANSI] = CreateFontIndirect(&prefsTADS.lfNormal);
	hfHTML4Bold[HTML4_ANSI] = CreateFontIndirect(&lfTempBold);
	if (hfHTML4Bold[HTML4_ANSI] == NULL)
		hfHTML4Bold[HTML4_ANSI] = CreateFontIndirect(&prefsTADS.lfNormal);

	lfTemp.lfCharSet = EASTEUROPE_CHARSET;	// Step 3: the Eastern European character set
	lfTempBold.lfCharSet = EASTEUROPE_CHARSET;
	hfHTML4Norm[HTML4_EASTEUROPE] = CreateFontIndirect(&lfTemp);
	if (hfHTML4Norm[HTML4_EASTEUROPE] == NULL)
		hfHTML4Norm[HTML4_EASTEUROPE] = CreateFontIndirect(&prefsTADS.lfNormal);
	hfHTML4Bold[HTML4_EASTEUROPE] = CreateFontIndirect(&lfTempBold);
	if (hfHTML4Bold[HTML4_EASTEUROPE] == NULL)
		hfHTML4Bold[HTML4_EASTEUROPE] = CreateFontIndirect(&prefsTADS.lfNormal);

	lfTemp.lfCharSet = SYMBOL_CHARSET;		// Step 4: the symbol character set
	lfTempBold.lfCharSet = SYMBOL_CHARSET;
	strcpy(lfTemp.lfFaceName, "Symbol");
	strcpy(lfTempBold.lfFaceName, "Symbol");
	hfHTML4Norm[HTML4_SYMBOL] = CreateFontIndirect(&lfTemp);
	if (hfHTML4Norm[HTML4_SYMBOL] == NULL)
		hfHTML4Norm[HTML4_SYMBOL] = CreateFontIndirect(&prefsTADS.lfNormal);
	hfHTML4Bold[HTML4_SYMBOL] = CreateFontIndirect(&lfTempBold);
	if (hfHTML4Bold[HTML4_SYMBOL] == NULL)
		hfHTML4Bold[HTML4_SYMBOL] = CreateFontIndirect(&prefsTADS.lfNormal);

	/* If updating, update the lineheight (i.e. the height of a line of
	   text) */
	if (fUpdate) {
		XSetFont(hdc, hfNorm);
		GetTextMetrics(hdc, &tm);
		lineheight_story = tm.tmHeight;
		lineheightoff_story = tm.tmAscent;

		/* Update the # of pixels a space takes */
		XTextExtents(hdc, hfBold, " ", 1, &iBoldSpace);
		XTextExtents(hdc, hfNorm, " ", 1, &iNormSpace);

		/* Update the cursor */
		fC = fCursorOn;
		if (fC)
			XShowDot(hdcClient, FALSE);
		cursorHeight = lineheightoff_story + 1;
		if (fC)
			XShowDot(hdcClient, TRUE);

		/* Update the client window */
	  	xtext_resize(0, 0, (SHORT)(clientRect.right - clientRect.left),
	  		(SHORT)lClientHeight);
		InvalidateRect(hwndClient, NULL, FALSE);
	}
	
	return TRUE;
}

/* UpdateInputFont updates the input font. */
SHORT UpdateInputFont(HDC hdc, BOOL fBold)
{
	if (fBold)
		hfInput = hfBold;
	else hfInput = hfNorm;

	return hfInput;
}

/* DefaultFont selects a default font for the program: Arial.
   Its return values are the same as for UpdateFonts. */
SHORT DefaultFont(VOID)
{
	memset((void *)&prefsTADS.lfNormal, 0, sizeof(LOGFONT));
	prefsTADS.lfNormal.lfHeight = 18;
	prefsTADS.lfNormal.lfWeight = FW_NORMAL;
	prefsTADS.lfNormal.lfOutPrecision = OUT_TT_PRECIS;
	prefsTADS.lfNormal.lfQuality = DEFAULT_QUALITY;
	prefsTADS.lfNormal.lfPitchAndFamily = VARIABLE_PITCH | FF_DONTCARE;
	strcpy(prefsTADS.lfNormal.lfFaceName, "Times New Roman");
	
	memcpy(&prefsTADS.lfBold, &prefsTADS.lfNormal, sizeof(LOGFONT));
	prefsTADS.lfBold.lfWeight = FW_BOLD;

	memset((void *)&prefsTADS.lfStatus, 0, sizeof(LOGFONT));
	prefsTADS.lfStatus.lfHeight = 10;
	prefsTADS.lfStatus.lfWeight = FW_BOLD;
	prefsTADS.lfStatus.lfOutPrecision = OUT_TT_PRECIS;
	prefsTADS.lfStatus.lfQuality = DEFAULT_QUALITY;
	prefsTADS.lfStatus.lfPitchAndFamily = VARIABLE_PITCH | FF_DONTCARE;
	strcpy(prefsTADS.lfStatus.lfFaceName, "System");

	return UpdateFonts(hdcClient, TRUE);
}


/*
** Window routines
*/

/* SetupMainWindow takes care of all the piddling little details, like sizing
   and showing the window. */
VOID SetupMainWindow(VOID)
{
	extern short statusmode;		/* Defined in winio.c */
	LONG	width, height, winWidth, winHeight;

	/* Prepare the text engine */
	xtext_init();
	XSetUpHDC(hdcClient);
	statusmode = 0;

/* Apply the preferences to the size of the frame window.  Make sure we won't
   be bigger than the entire screen */
	width = GetSystemMetrics(SM_CXFULLSCREEN) - prefsTADS.rectFrame.left;
	height = GetSystemMetrics(SM_CYFULLSCREEN) - prefsTADS.rectFrame.top;
	winWidth = prefsTADS.rectFrame.right - prefsTADS.rectFrame.left;
	winHeight = prefsTADS.rectFrame.bottom - prefsTADS.rectFrame.top;

	if (width > winWidth)
		width = winWidth;
	if (height > winHeight)
		height = winHeight;
	
	MoveWindow(hwndFrame, prefsTADS.rectFrame.left, prefsTADS.rectFrame.top,
		width, height, TRUE);
	if (prefsTADS.fMaximized)
		ShowWindow(hwndFrame, SW_SHOWMAXIMIZED);
	else ShowWindow(hwndFrame, SW_SHOW);
	UpdateWindow(hwndFrame);
}

// A quick little routine that will center one window over another; handy for dialog boxes.
BOOL CenterWindow (HWND hwndChild, HWND hwndParent)
{
    RECT    rChild, rParent;
    int     wChild, hChild, wParent, hParent;
    int     wScreen, hScreen, xNew, yNew;
    HDC     hdc;

    GetWindowRect (hwndChild, &rChild);
    wChild = rChild.right - rChild.left;
    hChild = rChild.bottom - rChild.top;

    GetWindowRect (hwndParent, &rParent);
    wParent = rParent.right - rParent.left;
    hParent = rParent.bottom - rParent.top;

    hdc = GetDC (hwndChild);
    wScreen = GetDeviceCaps (hdc, HORZRES);
    hScreen = GetDeviceCaps (hdc, VERTRES);
    ReleaseDC (hwndChild, hdc);

    xNew = rParent.left + ((wParent - wChild) /2);
    if (xNew < 0) {
        xNew = 0;
    } else if ((xNew+wChild) > wScreen) {
        xNew = wScreen - wChild;
    }

    yNew = rParent.top  + ((hParent - hChild) /2);
    if (yNew < 0) {
        yNew = 0;
    } else if ((yNew+hChild) > hScreen) {
        yNew = hScreen - hChild;
    }

    return SetWindowPos (hwndChild, NULL, xNew, yNew, 0, 0, SWP_NOSIZE | SWP_NOZORDER);
}


VOID SetGameRegistryKeys(VOID)
{
	CHAR		szValue0[] = ".gam",
				szValue1[100], szValue2[100], szValue3[100], szValue4[100],
				szValue5[1000];
	HKEY		key1, key2, key3, key4;
	DWORD		d;

/* Get the key values from a string resource */
	LoadString(hInst, strix_GameKey1, szValue1, sizeof(szValue1));
	LoadString(hInst, strix_GameKey2, szValue2, sizeof(szValue2));
	LoadString(hInst, strix_GameKey3, szValue3, sizeof(szValue3));
	LoadString(hInst, strix_GameKey4, szValue4, sizeof(szValue4));

	RegCreateKeyEx(HKEY_CLASSES_ROOT, szValue0, 0, NULL,
		REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &key1, &d);
	RegCreateKeyEx(HKEY_CLASSES_ROOT, szValue1, 0, NULL,
		REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &key2, &d);
	RegCreateKeyEx(HKEY_CLASSES_ROOT, szValue2, 0, NULL,
		REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &key3, &d);
	RegCreateKeyEx(HKEY_CLASSES_ROOT, szValue3, 0, NULL,
		REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &key4, &d);
	RegSetValueEx(key1, NULL, 0, REG_SZ,
		(const unsigned char *)szValue1, strlen(szValue1) + 1);
	RegSetValueEx(key2, NULL, 0, REG_SZ,
		(const unsigned char *)szValue4, strlen(szValue4) + 1);
	sprintf(szValue5, "%s,2", pszExecutable);
	RegSetValueEx(key3, NULL, 0, REG_SZ,
		(const unsigned char *)szValue5, strlen(szValue5) + 1);
	sprintf(szValue5, "%s \"%%1\"", pszExecutable);
	RegSetValueEx(key4, NULL, 0, REG_SZ,
		(const unsigned char *)szValue5, strlen(szValue5) + 1);
	RegCloseKey(key1);
	RegCloseKey(key2);
	RegCloseKey(key3);
	RegCloseKey(key4);
}

VOID SetSaveRegistryKeys(VOID)
{
	CHAR		szValue0[] = ".sav",
				szValue1[100], szValue2[100], szValue3[100], szValue4[100],
				szValue5[1000];
	HKEY		key1, key2, key3, key4;
	DWORD		d;

/* Get the key values from a string resource */
	LoadString(hInst, strix_SaveKey1, szValue1, sizeof(szValue1));
	LoadString(hInst, strix_SaveKey2, szValue2, sizeof(szValue2));
	LoadString(hInst, strix_SaveKey3, szValue3, sizeof(szValue3));
	LoadString(hInst, strix_SaveKey4, szValue4, sizeof(szValue4));

	RegCreateKeyEx(HKEY_CLASSES_ROOT, szValue0, 0, NULL,
		REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &key1, &d);
	RegCreateKeyEx(HKEY_CLASSES_ROOT, szValue1, 0, NULL,
		REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &key2, &d);
	RegCreateKeyEx(HKEY_CLASSES_ROOT, szValue2, 0, NULL,
		REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &key3, &d);
	RegCreateKeyEx(HKEY_CLASSES_ROOT, szValue3, 0, NULL,
		REG_OPTION_NON_VOLATILE, KEY_ALL_ACCESS, NULL, &key4, &d);
	RegSetValueEx(key1, NULL, 0, REG_SZ,
		(const unsigned char *)szValue1, strlen(szValue1) + 1);
	RegSetValueEx(key2, NULL, 0, REG_SZ,
		(const unsigned char *)szValue4, strlen(szValue4) + 1);
	sprintf(szValue5, "%s,3", pszExecutable);
	RegSetValueEx(key3, NULL, 0, REG_SZ,
		(const unsigned char *)szValue5, strlen(szValue5) + 1);
	sprintf(szValue5, "%s %%1", pszExecutable);
	RegSetValueEx(key4, NULL, 0, REG_SZ,
		(const unsigned char *)szValue5, strlen(szValue5) + 1);
	RegCloseKey(key1);
	RegCloseKey(key2);
	RegCloseKey(key3);
	RegCloseKey(key4);
}

/*
** Main section
*/

int APIENTRY WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
	int nCmdShow)
{
	MSG			msg;
	WNDCLASSEX	wcx;
	HANDLE		hAccelTable;
	PCHAR		p;
	
	if (!hPrevInstance) {
		wcx.cbSize = sizeof(WNDCLASSEX);
		wcx.style = CS_HREDRAW | CS_VREDRAW;
		wcx.lpfnWndProc = (WNDPROC)FrameWndProc;
		wcx.cbClsExtra = 0;
		wcx.cbWndExtra = 0;
		wcx.hInstance = hInstance;
		wcx.hIcon = LoadIcon(hInstance, szAppName);
		hptrArrow = LoadCursor(NULL, IDC_ARROW);
		wcx.hCursor = hptrArrow;
		wcx.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);
		wcx.lpszMenuName = szAppName;
		wcx.lpszClassName = szAppName;
		wcx.hIconSm = LoadIcon(hInstance, "WinTADSSmall");
		
		if (!RegisterClassEx(&wcx)) {
			DisplayError("Unable to register WinTADS frame");
			return FALSE;
		}

        wcx.style         = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
        wcx.lpfnWndProc   = (WNDPROC)TadsMainWndProc;
		wcx.hCursor = NULL;
        wcx.hbrBackground = NULL;
        wcx.lpszMenuName  = NULL;
        wcx.lpszClassName = "WinTADSMainClient";

        if (!RegisterClassEx(&wcx)) {
        	DisplayError("Unable to register WinTADS window");
            return FALSE;
        }
        
        if (!InitMessageBox(hInstance)) {
        	DisplayError("Unable to register Message Box");
        	return FALSE;
        }
        if (!InitStatus(hInstance)) {
        	DisplayError("Unable to register Status window");
        	return FALSE;
        }
	}

	hwndFrame = CreateWindow(szAppName,		// Class name
		szAppName,							// Window name
		WS_OVERLAPPEDWINDOW |				// Style flags
		WS_CLIPCHILDREN,
		CW_USEDEFAULT, CW_USEDEFAULT,		// Initial (x,y) position
		CW_USEDEFAULT, CW_USEDEFAULT,		// Initial (width, height)
		NULL,								// Parent handle
		NULL,								// Menu handle
		hInstance,							// Instance
		NULL);								// Other parameters
	if (!hwndFrame) {
		DisplayError("Unable to create WinTADS frame");
		return FALSE;
	}
	
	hwndClient = CreateWindow("WinTADSMainClient", "WinTADSMainClient",
		WS_CHILD | WS_CLIPSIBLINGS | WS_BORDER | WS_VSCROLL | WS_VISIBLE,
		CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
		hwndFrame, NULL, hInstance, NULL);
	if (!hwndClient) {
		DisplayError("Unable to create WinTADS window");
		return FALSE;
	}

	// Save the instance in a global variable
	hInst = hInstance;

	// Load the accelerators
	hAccelTable = LoadAccelerators(hInstance, szAppName);
	if (!hAccelTable)
		DisplayError("Unable to create keyboard accelerators");

	// Load the common controls
	//InitCommonControls();

	// Get the DC
	hdcClient = GetDC(hwndClient);

	/* Set up the cursor */
	cursorX = cursorY = 0;
	cursorWidth = 1;
	fCursorOn = FALSE;

	/* Set up the initial path & executable names */
	GetModuleFileName(NULL, pszExecutable, CCHMAXPATH);
	strcpy(pszHomePath, pszExecutable);
	p = pszHomePath + strlen(pszHomePath) - 2;
	while (p != pszHomePath && *p != '\\')
		p--;
	if (p != pszHomePath)
		*(p+1) = 0;

	/* Restore the preferences */
	if (!RestorePrefs()) {				// If there's an error,
		DefaultPrefs();					//  set the default prefs
	}
	else UpdateFonts(hdcClient, TRUE);

	/* Set the colors */
	XSetForeColor(prefsTADS.ulFore);
	XSetBackColor(prefsTADS.ulBack);

	/* Create the message box */
	CreateMessageBox(hwndFrame, hInstance);

	/* Hide the vertical scroll bar */
	MyShowScrollBar(FALSE);

	/* Save the system pointers (N.B. hptrArrow is already set up) */
	hptrText = LoadCursor(NULL, IDC_IBEAM);
	hptrWait = LoadCursor(NULL, IDC_WAIT);
	fWaitCursor = FALSE;		/* Start out not showing wait cursor */

	/* Set up the keyboard */
	{ extern VOID init_kbd(VOID); init_kbd(); }
	
	/* Now that the keyboard is set up, take care of the macro menu */
	SetupMacroMenus();
	
	/* Set up flags */
	fMinimized = FALSE;

	/* Test binding */
	zcodepos = TestBinding(pszExecutable);
	zcodename[0] = 0;
	validBinding = (zcodepos != 0);
	TweakBindingMenu();	// Adjust binding options accordingly
	if (validBinding) {
		PostMessage(hwndFrame, WM_COMMAND, CMD_STARTTADS, 0);
	}
	else if (strlen(lpCmdLine) > 0) {	// Drag & drop, bay-bee!
		strcpy(zcodename, lpCmdLine);
		// Get rid of quote marks in the dragged filename
		if (zcodename[0] == '"')
			strcpy(zcodename, zcodename+1);
		if (zcodename[strlen(zcodename)-1] == '"')
			zcodename[strlen(zcodename)-1] = 0;
		DisplayMessage(zcodename);
		AddRecent(zcodename);
		PostMessage(hwndFrame, WM_COMMAND, CMD_STARTTADS, 0);
	}
	strcpy(sShare.szTADS, pszExecutable);
	strcpy(sShare.szData, zcodename);
	sShare.szOutput[0] = 0;
	
	TweakRecentMenu();					// Set up the "recent .gam" menu

	/* Make us visible */
	SetupMainWindow();

	/* Create the status bar */
	if (!CreateStatus(hInstance))
		DisplayError("Unable to create Status window");
	
	/* Set up the status bar */
	SetStatusTextLeft("");
	SetStatusTextRight("");

	/* Prepare the semaphores & thread handles */
	fThreadRunning = FALSE;
	hMachine = NULL;
	fWantsExtendedKeys = FALSE;

	/* Clear the timeZero variable (for os_get_sys_clock_ms()) */
	timeZero = 0;
	
	/* If we haven't asked about Registry strings, do so */
	if (!prefsTADS.fAskedAboutRegistry && !validBinding) {
		switch (DialogBox(hInst, MAKEINTRESOURCE(DLG_REGISTRY), hwndFrame,
			RegistryDialogProc)) {
		  case IDOK:
			SetGameRegistryKeys();
			if (DialogBox(hInst, MAKEINTRESOURCE(DLG_SAVEREGISTRY),
				hwndFrame, RegistryDialogProc) == IDOK) {
				SetSaveRegistryKeys();
			}
		  case IDCANCEL:
			prefsTADS.fAskedAboutRegistry = TRUE;
			break;

		  case IDD_LATER:
			break;
		}
	}

	SetTimer(hwndFrame, ID_CURSORTIMER, 500, NULL);

	while (GetMessage(&msg, NULL, 0, 0)) {
		if (!TranslateAccelerator(msg.hwnd, hAccelTable, &msg)) {
			TranslateMessage(&msg);
			DispatchMessage(&msg);
		}
	}

	KillTimer(hwndFrame, ID_CURSORTIMER);
	xtext_end();

	if (hMachine != NULL)
		CloseHandle(hMachine);

	SavePrefs();
	ReleaseDC(hwndClient, hdcClient);

	DestroyWindow(hwndStatus);

	return (msg.wParam);
}

/*
** FrameWndProc processes the messages sent to the frame window, in order to
** handle the addition of the bottom status bar (the message box)
*/
LONG APIENTRY FrameWndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	RECT			rect;
	LPMINMAXINFO	mmi;
	int				wmId, wmEvent;
	DWORD			dw;
	
	switch(msg) {
	  case WM_SIZE:
		AdjustMessageBox(hwnd);
		if (!prefsTADS.fStatusSeparated)
			AdjustStatusBar(hwnd);
		GetClientRect(hwnd, &rect);
		rect.bottom -= MessageBoxHeight(hwnd);
		if (!prefsTADS.fStatusSeparated)
			rect.top += StatusBarHeight(hwnd) + SEPARATOR_WIDTH;
		SetWindowPos(hwndClient, NULL, rect.left, rect.top, rect.right - rect.left,
			rect.bottom - rect.top, SWP_NOZORDER);
		if (fScrollVisible)
			rect.right -= GetSystemMetrics(SM_CXVSCROLL);
		lClientHeight = rect.bottom - rect.top;
		xtext_resize(0, 0, (SHORT)(rect.right - rect.left),
			(SHORT)lClientHeight);
		memcpy(&clientRect, &rect, sizeof(RECT));
		break;

	  case WM_GETMINMAXINFO:
		mmi = (LPMINMAXINFO)lParam;
		mmi->ptMinTrackSize.x = MINIMUM_WIDTH + GetSystemMetrics(SM_CXFRAME)*2;
		mmi->ptMinTrackSize.y = MessageBoxHeight(hwnd) + GetSystemMetrics(SM_CYMENU) +
			GetSystemMetrics(SM_CYCAPTION) + GetSystemMetrics(SM_CYFRAME)*2;
		if (!prefsTADS.fStatusSeparated)
			mmi->ptMinTrackSize.y += StatusBarHeight(hwnd) + SEPARATOR_WIDTH;
		break;

	  case WM_QUERYOPEN:
		if (!(IsIconic(hwnd) || IsZoomed(hwnd)))
			GetWindowRect(hwndFrame, &prefsTADS.rectFrame);
		return TRUE;

	  case WM_PAINT:
		{
			// Allow default proc to draw all of the frame.
			LRESULT rc = DefWindowProc(hwnd, msg, wParam, lParam);

			// Get presentation space handle for drawing.
			HDC hdc = GetDC(hwnd);

			// Draw the horizontal separator bar.
			if (!prefsTADS.fStatusSeparated)
				DrawSeparator(hwnd, hdc, 0,	StatusBarHeight(hwnd));

			ReleaseDC(hwnd, hdc);
			return(rc);
		}

	  case WM_COMMAND:
#if defined (WIN32)
		wmId = LOWORD(wParam);
		wmEvent = HIWORD(wParam);
#else
		wmId = wParam;
		wmEvent = HIWORD(lParam);
#endif
		switch (wmId) {
	  	  case CMD_STARTTADS:
			if ((hMachine = CreateThread((LPSECURITY_ATTRIBUTES)NULL, 0,
				(LPTHREAD_START_ROUTINE)os_main_shell, &dw, 0, &tidMachine)) ==
				INVALID_HANDLE_VALUE)
				DisplayError("Unable to create TADS thread");
			else {
				extern stream keyboard_stream;	// From winkbd.c
				clear_stream(&keyboard_stream);
			}
			break;
			
		  case CMD_SHOWERROR: {
			PCHAR	c;
			
			c = (PCHAR)lParam;
			DisplayError(c);
			break;
		  }
		
		  default:
			HandleMenuCommand(hwnd, wmId);
			break;
		}
		break;

	  case WM_CLOSE:
		if (fThreadRunning) {
			if (prefsTADS.fCloseAborts)
				fBreakSignalled = TRUE;
			else {
				stuff_kbd_stream(VK_ESCAPE | keytype_virtual);
				stuff_kbd_stream_with_string("quit\n");
			}
			return 0;
	  	}
		/* Save the frame window's position */
		if (IsZoomed(hwnd))
			prefsTADS.fMaximized = TRUE;
		else {
			prefsTADS.fMaximized = FALSE;
			if (!IsIconic(hwnd))
				GetWindowRect(hwndFrame, &prefsTADS.rectFrame);
		}
		break;

	  case WM_DESTROY:
	  	if (hfNorm)
	  		DeleteObject(hfNorm);
	  	if (hfBold)
	  		DeleteObject(hfBold);
		if (hwnd == hwndFrame)
			PostQuitMessage(0);
		break;

	  case WM_TIMER:
		switch (wParam) {
		  case ID_MBTIMER: {
			extern BOOL bClearMB;	// Defined in WinCtls.c
			if (bClearMB)
				SetMessageBoxText(NULL, FALSE);
			KillTimer(hwnd, wParam);
			break;
		  }
		  case ID_CURSORTIMER:
			if (fCursorOn)
				XBlinkDot(hdcClient);
			break;
		  default:
			return DefWindowProc(hwnd, msg, wParam, lParam);
		}
		break;

/* Create & destroy the cursor when we gain & lose the focus */
	  case WM_SETFOCUS:
		mainwin_activate(TRUE);
		if (fThreadRunning)
			mainwin_caret_changed(TRUE);
		else mainwin_caret_changed(FALSE);
		TweakMenus();
		break;
		
	  case WM_KILLFOCUS:
		mainwin_activate(FALSE);
		mainwin_caret_changed(FALSE);
		break;

	  case WM_KEYDOWN: {
	  	SHORT	repeat, flags;
	  	
	  	repeat = LOWORD(lParam);
	  	flags = HIWORD(lParam) & 0xFF00;
	  	if ((flags & 0x100) &&
	  		(wParam != VK_CONTROL && wParam != VK_MENU &&
	  		wParam != VK_SHIFT)) {
		// To signal CTRL/ALT/SHIFT+virtual key, set the appropriate flags
			wParam |= keytype_virtual;
			if (GetKeyState(VK_CONTROL) < 0)
				wParam |= keytype_ctrl;
			if (GetKeyState(VK_SHIFT) < 0)
				wParam |= keytype_shift;
			if (GetKeyState(VK_MENU) < 0)
				wParam |= keytype_alt;
			stuff_kbd_stream((short)wParam);
			return 0;
	  	}
	  	// Hardwire in CTRL+ALT+Q for a forced quit
	  	if ((wParam == 'Q' || wParam == 'q') && GetKeyState(VK_MENU) < 0
	  		&& GetKeyState(VK_CONTROL) < 0) {
			HandleMenuCommand(hwnd, IDM_FORCEQUIT);
			return 0;
		}
		break;
	  }

	  case WM_CHAR: {
		stuff_kbd_stream((short)wParam);
		return DefWindowProc(hwnd, msg, wParam, lParam);
	  }

	  default:
		return(DefWindowProc(hwnd, msg, wParam, lParam));
	}
	return (DefWindowProc(hwnd, msg, wParam, lParam));
}


/*
** TadsMainWndProc processes the messages sent to the client (main) window
*/
LONG APIENTRY TadsMainWndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	static BOOL	bMouseMoved = FALSE, bDoubleClicking = FALSE;
	HMENU		hwndPop;
	
	switch(msg) {
	  case WM_PAINT: {
	  	RECT	rclPaint;
	  	PAINTSTRUCT	ps;
	  	HDC		hdcPaint = BeginPaint(hwnd, &ps);
	  	
		GetClientRect(hwnd, &rclPaint);
		XClearArea(hdcPaint, rclPaint.left, rclPaint.top,
			rclPaint.right - rclPaint.left, rclPaint.bottom - rclPaint.top);
/* Only redraw the text if the program thread is running */
		if (fThreadRunning) {
			XSetUpHDC(hdcPaint);
			XKnowIgnorance();
			xtext_redraw(hdcPaint);
			XKnowIgnorance();
		}
		EndPaint(hwnd, &ps);
		break;
	  }
		
	  case WM_VSCROLL:
		switch (LOWORD(wParam)) {
		  case SB_LINEUP:
		  case SB_LINEDOWN:
		  case SB_PAGEUP:
		  case SB_PAGEDOWN:
			scroll_splat(LOWORD(wParam));
			return 0;
			
		  case SB_THUMBPOSITION:
			scroll_to(HIWORD(wParam));
			return 0;
		}
		break;
		

	  case WM_CONTEXTMENU:
		hwndPop = GetSubMenu(GetMenu(hwndFrame), 1);
		if (hwndPop)
		TrackPopupMenu(hwndPop, TPM_LEFTALIGN | TPM_TOPALIGN |
			TPM_LEFTBUTTON | TPM_RIGHTBUTTON, LOWORD(lParam), HIWORD(lParam),
			0, hwndFrame, NULL);
		break;

	  case WM_LBUTTONDOWN:
		if (fThreadRunning) {
			bMouseMoved = FALSE;
			bDoubleClicking = FALSE;
			xtext_hitdown(LOWORD(lParam), HIWORD(lParam), 1, 0, 1);
			return 0;
		}
		break;

	  case WM_LBUTTONUP:
	  	if (fThreadRunning) {
	  		if (bDoubleClicking)	// Don't prevent the double-click
	  			bDoubleClicking = FALSE;
	  		else if (bMouseMoved) {
				bMouseMoved = FALSE;
	  			xtext_hitup(LOWORD(lParam), HIWORD(lParam), 1, 0, 1);
	  		}
	  		fNoMouseHit = FALSE;
	  		TweakMenus();
	  		return 0;
		}
		break;
		
	  case WM_LBUTTONDBLCLK:
	  	if (fThreadRunning && !fNoMouseHit) {
			xtext_hitdown(LOWORD(lParam), HIWORD(lParam), 1, 0, 2);
			bDoubleClicking = TRUE;
			TweakMenus();
			return 0;
		}
		break;
		
/* When the pointer's in the client window, draw a text bar. Also send msgs
to xtext_hitmove if the mouse is dragged w/the button down. */
	  case WM_MOUSEMOVE:
		if (fWaitCursor)
			SetCursor(hptrWait);
		else SetCursor(hptrText);
		if (fThreadRunning && !fNoMouseHit) {
			if (wParam & MK_LBUTTON) {
				xtext_hitmove(LOWORD(lParam), HIWORD(lParam), 1, 0, 1);
				bMouseMoved = TRUE;
				return 0;
			}
			else bMouseMoved = FALSE;
		}
		break;

	  default:
		return DefWindowProc(hwnd, msg, wParam, lParam);
	}
	return DefWindowProc(hwnd, msg, wParam, lParam);
}

/*
** HandleMenuCommand, given the menu item ID from TadsMainWndProc, performs
** whatever is necessary for the menu item.
*/
VOID HandleMenuCommand(HWND hwnd, UINT menuItemID)
{
	LOGFONT		lf;
	COLORREF	cr;
	DWORD		dw, tid;
	CHAR		message[256];
	
	switch (menuItemID) {
	  case IDM_LOAD:
		fNoMouseHit = TRUE;
		if (!QueryFile(zcodename, CCHMAXPATH, FALSE, 1, NULL)) {
			strcpy(sShare.szData, zcodename);
			PostMessage(hwndFrame, WM_COMMAND, CMD_STARTTADS, 0);
		}
		break;

	  case IDM_RG1:
	  case IDM_RG2:
	  case IDM_RG3:
	  case IDM_RG4:
	  case IDM_RG5:
	  case IDM_RG6:
	  case IDM_RG7:
	  case IDM_RG8:
	  case IDM_RG9:
	  case IDM_RG10:
		if (fThreadRunning) return;
		strcpy(zcodename, prefsTADS.szRecentGames[menuItemID - IDM_RG1]);
		strcpy(sShare.szData, zcodename);
		AddRecent(zcodename);
		PostMessage(hwndFrame, WM_COMMAND, CMD_STARTTADS, 0);
		break;

	  case IDM_SAVE:
		stuff_kbd_stream(VK_ESCAPE | keytype_virtual);
		stuff_kbd_stream_with_string("save\n");
		break;

	  case IDM_RESTORE:
		stuff_kbd_stream(VK_ESCAPE | keytype_virtual);
		stuff_kbd_stream_with_string("restore\n");
		break;
		
	  case IDM_BIND:
	  	if (!QueryFile(sShare.szOutput, CCHMAXPATH, TRUE, 3, NULL)) {
			if (!CreateThread((LPSECURITY_ATTRIBUTES)NULL, 8192,
				(LPTHREAD_START_ROUTINE)BindCode, &dw, 0, &tid))
				DisplayError("Unable to create BindCode thread");
	  	}
	  	break;

	  case IDM_UNBIND:
	  	if (!QueryFile(sShare.szOutput, CCHMAXPATH, TRUE, 4, NULL)) {
			if (!CreateThread((LPSECURITY_ATTRIBUTES)NULL, 8192,
				(LPTHREAD_START_ROUTINE)UnbindCode, &dw, 0, &tid))
				DisplayError("Unable to create BindCode thread");
	  	}
	  	break;

	  case IDM_QUIT:
		PostMessage(hwnd, WM_CLOSE, 0, 0L);
		break;
		
	  case IDM_FORCEQUIT:
		fThreadRunning = FALSE;
		PostMessage(hwnd, WM_CLOSE, 0, 0L);

	  case IDM_UNDO:
		stuff_kbd_stream(VK_ESCAPE | keytype_virtual);
		stuff_kbd_stream_with_string("undo\n");
		break;
		
	  case IDM_CUT:
	  	xted_cutbuf(op_Wipe);
		break;
		
	  case IDM_COPY:
		xted_cutbuf(op_Copy);
		break;
		
	  case IDM_PASTE:
	  	{
	  		static PSZ	pszLocal = NULL;

			pszLocal = copy_clip_to_string();
	  		if (pszLocal) {
				stuff_kbd_stream_with_string(pszLocal);
	  			free(pszLocal);
	  			pszLocal = NULL;
	  		}
		}
		break;

	  case IDM_FIND:
		DialogBox(hInst, MAKEINTRESOURCE(DLG_FIND), hwndFrame,
			FindDialogProc);
		if (fd.findText[0] != 0) {
			if (!xtext_find(!fd.fBack, fd.fCase, fd.findText)) {
				sprintf(message, "\"%s\" not found", fd.findText);
				DisplayError(message);
			}
		}
		break;

	  case IDM_FINDAGAIN:
	  	if (fd.findText[0] == 0)
	  		DisplayError("No previous find was executed");
	  	else if (!xtext_find(!fd.fBack, fd.fCase, fd.findText)) {
	  		sprintf(message, "\"%s\" not found", fd.findText);
	  		DisplayError(message);
	  	}
		break;
		
	  case IDM_STORY_WINDOW:
		DialogBox(hInst, MAKEINTRESOURCE(DLG_STORYWINDOW), hwndFrame,
			StoryWindowDialogProc);
		break;

	  case IDM_INTERPRETER:
		DialogBox(hInst, MAKEINTRESOURCE(DLG_INTERPRETER), hwndFrame,
			InterpreterDialogProc);
		break;

	  case IDM_PROGRAM:
		DialogBox(hInst, MAKEINTRESOURCE(DLG_OPTIONS), hwndFrame,
			OptionsDialogProc);
		break;

	  case IDM_SET_MAIN_FONT:
		memcpy(&lf, &prefsTADS.lfNormal, sizeof(LOGFONT));
		if (GetFont(&lf)) {
			lf.lfItalic = FALSE;
			lf.lfUnderline = FALSE;
			lf.lfStrikeOut = FALSE;
			lf.lfWeight = FW_NORMAL;
			memcpy(&prefsTADS.lfNormal, &lf, sizeof(LOGFONT));
			lf.lfWeight = FW_BOLD;
			memcpy(&prefsTADS.lfBold, &lf, sizeof(LOGFONT));
			UpdateFonts(hdcClient, TRUE);
		}
		break;
	
	  case IDM_SET_STATUS_FONT:
		memcpy(&lf, &prefsTADS.lfStatus, sizeof(LOGFONT));
		if (GetFont(&lf)) {
			SetStatusFont(&lf);
			UpdateStatus();
			memcpy(&prefsTADS.lfStatus, &lf, sizeof(LOGFONT));
		}
		break;
		
	  case IDM_SET_FORE_COLOR:
		cr = prefsTADS.ulFore;
		if (GetColor(&cr)) {
			prefsTADS.ulFore = cr;
			XSetForeColor(prefsTADS.ulFore);
			XKnowIgnorance();
			InvalidateRect(hwndClient, NULL, FALSE);
			InvalidateRect(hwndStatus, NULL, FALSE);
		}
		break;

	  case IDM_SET_BACK_COLOR:
		cr = prefsTADS.ulBack;
		if (GetColor(&cr)) {
			prefsTADS.ulBack = cr;
			XSetBackColor(prefsTADS.ulBack);
			XKnowIgnorance();
			InvalidateRect(hwndClient, NULL, FALSE);
			InvalidateRect(hwndStatus, NULL, FALSE);
		}
		break;

	  case IDM_SEPARATED_STATUS:
		SwapStatus(hInst);
		break;

	  case IDM_DEFINE_MACRO:
	  	if (xtexted_getmodifymode(FALSE) != op_DefineMacro)
	  		xtexted_meta(op_DefineMacro);
		else xtexted_meta(op_Cancel);
		break;
		
	  case IDM_MACRO1:
	  case IDM_MACRO2:
	  case IDM_MACRO3:
	  case IDM_MACRO4:
	  case IDM_MACRO5:
	  case IDM_MACRO6:
	  case IDM_MACRO7:
	  case IDM_MACRO8:
	  case IDM_MACRO9:
	  case IDM_MACRO10:
	  case IDM_MACRO11:
	  case IDM_MACRO12:
		stuff_kbd_stream((short)(cmdkeylist[menuItemID - IDM_MACRO1] |
			keytype_virtual));
		break;

	  case IDM_ABOUT:
		DialogBox(hInst, MAKEINTRESOURCE(DLG_ABOUT), hwndFrame,
			AboutDialogProc);
		break;
	}
}

/*
** GetFont prepares and creates a font dialog for the user to select a font. The
**  passed pointer to a LOGFONT both initializes the dialog and contains the result
**  on exit. GetFont returns TRUE if everything is okay or FALSE otherwise.
*/
BOOL GetFont(LOGFONT *plf)
{
	CHOOSEFONT	cf;

	memset((void *)&cf, 0, sizeof(CHOOSEFONT));
	cf.lStructSize = sizeof(CHOOSEFONT);
	cf.lpLogFont = plf;
	cf.Flags = CF_SCREENFONTS | CF_FORCEFONTEXIST |
		CF_NOVERTFONTS | CF_INITTOLOGFONTSTRUCT;
	
	return ChooseFont(&cf);
}

/*
** GetColor prepares and creates a color dialog for the user to select a color.
** The passed pointer to a COLORREF both initializes the dialog and contains
** the returned color on exit. GetColor returns TRUE if everything is okay or
** FALSE otherwise.
*/
BOOL GetColor(COLORREF *pcr)
{
	CHOOSECOLOR	cc;
	static COLORREF	CustomColors[16];
	
	memset((void *)&cc, 0, sizeof(CHOOSECOLOR));
	cc.lStructSize = sizeof(CHOOSECOLOR);
	cc.rgbResult = *pcr;
	cc.lpCustColors = CustomColors;
	cc.Flags = CC_ANYCOLOR | CC_RGBINIT;
	
	if (!ChooseColor(&cc))
		return FALSE;
	*pcr = cc.rgbResult;
	return TRUE;
}

/*
** QueryFile asks the user for a file. fWriting indicates whether a file is
** to be written or read; iType is 0 for a saved game file, 1 for a .gam file,
** 2 for a transcript, 3 for the WinTADS executable+.gam, or 4 for plain WinTADS.
** szPrompt is the title of the dialog.
** The function returns OS_AFE_SUCCESS if everything went ok (which is zero), a
** non-zero failure code otherwise.
*/
SHORT QueryFile(char *buf, short bufsize, BOOL fWriting, short iType, const char *szPrompt)
{
	CHAR		szTitle[500],
				szFilename[CCHMAXPATH] = "",
				szFileTitle[CCHMAXPATH] = "",
				szFilePath[CCHMAXPATH] = "",
				szDefExt[4] = "",
				*szDefPath = NULL, *p;
	static CHAR	szOpenOK[] = "Open",
				szSaveOK[] = "Save",
				szWriteOK[] = "Write",
				szBindOK[] = "Bind",
				szUnbindOK[] = "Unbind",
				szLastSave[CCHMAXPATH] = "",
				szLastSavePath[CCHMAXPATH] = "",
				szLastWrite[CCHMAXPATH] = "",
				szLastWritePath[CCHMAXPATH] = "",
				szLastRestore[CCHMAXPATH] = "",
				szLastRestorePath[CCHMAXPATH] = "",
				szFilters[] =
					"TADS game files\0*.gam\0Saved Games\0*.sav\0All Files\0*.*\0",
				szTextFilters[] = "Text files\0*.txt\0All files\0*.*\0",
				szExeFilters[] =
					"Executable files\0*.exe\0";
	OPENFILENAME	ofn;
	BOOL		rc;

	/* Set up the OPENFILENAME structure */
	memset((void *)&ofn, 0, sizeof(OPENFILENAME));
	ofn.lStructSize = sizeof(OPENFILENAME);
	ofn.hwndOwner = hwndFrame;
	ofn.lpstrFilter = szFilters;
	ofn.lpstrFile = szFilename;
	ofn.lpstrInitialDir = szFilePath;
	ofn.nMaxFile = CCHMAXPATH;
	ofn.lpstrTitle = szTitle;
	ofn.Flags = OFN_HIDEREADONLY | OFN_LONGNAMES;
	if (szPrompt != NULL)
		strcpy(szTitle, szPrompt);
	
	if (!fWriting && iType == 0) {	/* Restore a saved game */
		ofn.nFilterIndex = 2;
		if (szPrompt == NULL)
			LoadString(hInst, strix_RestoreGame, szTitle, sizeof(szTitle));
		if (szLastRestore[0] == 0) {
			if (prefsTADS.szSavePath[0] != 0 &&
				prefsTADS.fStickyPaths) {
				strcpy(szLastRestore, prefsTADS.szSavePath);
			}
			else {
				if (GetEnvironmentVariable("TADSSAVE", szDefPath,
					sizeof(szDefPath)) != 0) {
					strcpy(szLastRestore, szDefPath);
					if (szLastRestore[strlen(szDefPath)-1] != '\\')
						strcat(szLastRestore, "\\");
				}
			}
			strcpy(szFilename, "*.sav");
		}
		else strcpy(szFilename, szLastRestore);
		strcpy(szFilePath, szLastRestorePath);
		rc = GetOpenFileName(&ofn);
	} else if (fWriting && iType == 0) {	/* Write a saved game */
		ofn.nFilterIndex = 2;
		strcpy(szDefExt, "sav");
		ofn.lpstrDefExt = szDefExt;
		if (szPrompt == NULL)
			LoadString(hInst, strix_SaveGameAs, szTitle, sizeof(szTitle));
		if (szLastSavePath[0] == 0) {
			if (prefsTADS.szSavePath[0] != 0 &&
				prefsTADS.fStickyPaths) {
				strcpy(szLastSavePath, prefsTADS.szSavePath);
			}
			else {
				if (GetEnvironmentVariable("TADSSAVE", szDefPath,
					sizeof(szDefPath)) != 0) {
					strcpy(szLastSavePath, szDefPath);
					if (szLastSavePath[strlen(szDefPath)-1] != '\\')
						strcat(szLastSavePath, "\\");
				}
			}
		}
		strcpy(szFilePath, szLastSavePath);
		strcpy(szFilename, szLastSave);
		rc = GetSaveFileName(&ofn);
	} else if (!fWriting && iType == 1) {	/* Load a .gam file */
		ofn.nFilterIndex = 1;
		if (szPrompt == NULL)
			LoadString(hInst, strix_OpenGame, szTitle, sizeof(szTitle));
		if (prefsTADS.szGamePath[0] != 0 && prefsTADS.fStickyPaths) {
			strcpy(szFilePath, prefsTADS.szGamePath);
			strcpy(szFilename, "*.gam");
		}
		else {
			if (GetEnvironmentVariable("TADSGAME", szDefPath,
				sizeof(szDefPath)) != 0) {
				strcpy(szFilePath, szDefPath);
				if (szFilePath[strlen(szDefPath)-1] != '\\')
					strcat(szFilePath, "\\");
			}
			strcpy(szFilename, "*.gam");
		}
		rc = GetOpenFileName(&ofn);
	} else if (fWriting && iType == 2) {	/* Write a transcript */
		ofn.nFilterIndex = 1;
		ofn.Flags |= OFN_OVERWRITEPROMPT;
		if (szPrompt == NULL)
			LoadString(hInst, strix_WriteScript, szTitle, sizeof(szTitle));
		strcpy(szDefExt, "txt");
		ofn.lpstrDefExt = szDefExt;
		ofn.lpstrFilter = szTextFilters;
		strcpy(szFilePath, szLastWritePath);
		rc = GetSaveFileName(&ofn);
	} else if (fWriting && iType == 3) {	/* Bind .gam file to WinTADS */
		ofn.nFilterIndex = 1;
		strcpy(szDefExt, "exe");
		ofn.lpstrDefExt = szDefExt;
		ofn.lpstrFilter = szExeFilters;
		ofn.Flags |= OFN_OVERWRITEPROMPT;
		LoadString(hInst, strix_BindGame, szTitle, sizeof(szTitle));
		szFilename[0] = 0;
		rc = GetSaveFileName(&ofn);
	} else if (fWriting && iType == 4) {	/* Unbind WinTADS from .gam file */
		ofn.nFilterIndex = 1;
		strcpy(szDefExt, "exe");
		ofn.lpstrDefExt = szDefExt;
		ofn.lpstrFilter = szExeFilters;
		ofn.Flags |= OFN_OVERWRITEPROMPT;
		LoadString(hInst, strix_UnbindGame, szTitle, sizeof(szTitle));
		szFilename[0] = 0;
		rc = GetSaveFileName(&ofn);
	}
	
	if (rc == 0) {
		if (CommDlgExtendedError() == 0)
			return OS_AFE_CANCEL;
		return OS_AFE_FAILURE;
	}
	
	if (strlen(ofn.lpstrFile) >= (unsigned short)bufsize) {
		strncpy(buf, ofn.lpstrFile, bufsize-1);
		buf[bufsize-1] = '\0';
	}
	else strcpy(buf, ofn.lpstrFile);

	if (!fWriting && iType == 0) {	/* Restore a saved game */
		strcpy(prefsTADS.szSavePath, ofn.lpstrFile);
		p = prefsTADS.szSavePath + strlen(prefsTADS.szSavePath) - 2;
		while (p != prefsTADS.szSavePath && *p != '\\')
			p--;
		if (*p == '\\') {
			strcpy(szLastSave, p+1);
			*(p+1) = 0;
		}
		strcpy(szLastSavePath, prefsTADS.szSavePath);
	}
	else if (fWriting && iType == 0) {	/* Write a saved game */
		strcpy(szLastRestorePath, ofn.lpstrFile);
		p = szLastRestorePath + strlen(szLastRestorePath) - 2;
		while (p != szLastRestorePath && *p != '\\')
			p--;
		if (*p == '\\') {
			strcpy(szLastRestore, p+1);
			*(p+1) = 0;
		}
		strcpy(prefsTADS.szSavePath, szLastRestorePath);
	}
	else if (!fWriting && iType == 1) {	/* Load a .gam file */
		strcpy(prefsTADS.szGamePath, ofn.lpstrFile);
		AddRecent(ofn.lpstrFile);
		p = prefsTADS.szGamePath + strlen(prefsTADS.szGamePath) - 2;
		while (p != prefsTADS.szGamePath && *p != '\\')
			p--;
		if (*p == '\\')
			*(p+1) = 0;
	}
	else if (fWriting) {
		strcpy(szLastWritePath, ofn.lpstrFile);
		p = szLastWritePath + strlen(szLastWritePath) - 2;
		while (p != szLastWritePath && *p != '\\')
			p--;
		if (*p == '\\') {
			strcpy(szLastWrite, p+1);
			*(p+1) = 0;
		}
	}
	
	return OS_AFE_SUCCESS;
}

UINT APIENTRY DialogProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch (msg) {
	  case WM_INITDIALOG:
		CenterWindow(hwnd, hwndFrame);
		return FALSE;
	  case WM_COMMAND:
		if (wParam == IDCANCEL) {
			EndDialog(hwnd, FALSE);
			return TRUE;
		}
	}
	return FALSE;
}

VOID DisplayMessage(PSZ pszString)
{
	MessageBox(GetFocus(), pszString, "WinTADS Message",
		MB_OK);
}

VOID DisplayInformation(PSZ pszString)
{
	MessageBeep(MB_ICONASTERISK);
	MessageBox(GetFocus(), pszString, "WinTADS Information",
		MB_ICONINFORMATION | MB_OK);
}

VOID DisplayError(PSZ pszString)
{
	MessageBeep(MB_ICONEXCLAMATION);
	MessageBox(GetFocus(), pszString, "WinTADS Error",
		MB_ICONERROR | MB_OK);
}
