/*
    Mines

    Copyright (c) 1979, 2005 James L. Dean

    Version 1.0 released 6/16/2005

    This C program and associated data files may be distributed or used without
payment to its author.  Derivative works must credit its author.

    "Mines" lets you explore mines. The mine you explore is determined by a mine
number specified at the beginning of a game.

    The object of a game is to visit all of the rooms and return all of the
treasures to the entrance without making too many moves.

    In a mine, the passages are straight.  So, for example, if you go North to
leave a room, you can go South to reenter it.  The rooms are not evenly spaced.
However, the distance between adjacent rooms is always a multiple of the minimum
distance between adjacent rooms.
*/
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <stdlib.h>
#include <ctype.h>

#define TRUE  1
#define FALSE 0

#define DEBUG_MEMORY 0

#if DEBUG_MEMORY
#define Malloc(x) DebugMalloc(x,__LINE__)
#define Free(x) DebugFree((void *) x,__LINE__); x=NULL
#else
#define Malloc(x) malloc(x)
#define Free(x) free((void *) x)
#endif

typedef struct RoomNode
          {
            char  *szDescription;
            int   nChokepoint;
            int   bMined;
            int   bVisited;
            union
              {
                struct
                  {
                    struct PassageNode *pPassageNorth;
                    struct PassageNode *pPassageSouth;
                    struct PassageNode *pPassageEast;
                    struct PassageNode *pPassageWest;
                    struct PassageNode *pPassageUp;
                    struct PassageNode *pPassageDown;
                  }                Passage;
                struct PassageNode *pPassage [6];
              }   Adjacent;
           } *RoomNodePtr;

typedef struct TreasureNode
          {
            char        *szTreasure;
            RoomNodePtr pRoomTreasure;
            char        *szGuardian;
            int         bSlain;
            char        *szWeapon;
            RoomNodePtr pRoomWeapon;
          } *TreasureNodePtr;

typedef struct PassageNode
          {
            int             bBlocked;
            TreasureNodePtr pTreasureGuardian;
            RoomNodePtr     pRoom1;
            RoomNodePtr     pRoom2;
          } *PassageNodePtr;

typedef struct PathStackNode
          {
            RoomNodePtr          pRoom;
            PassageNodePtr       pPassageUsedToEnterRoom;
            int                  nDirectionUsedToEnterRoom;
            struct PathStackNode *pNext;
          } *PathStackNodePtr;

static void Carry(RoomNodePtr,int,TreasureNodePtr);
static int  Command(void);
#if DEBUG_MEMORY
static void DebugFree(void *,int);
static void *DebugMalloc(unsigned int,int);
#endif
static void DisplayHelp(FILE *,int,int);
static void DisplayText(FILE *,char *,int,int);
static void ExcavateMine(int,char **,RoomNodePtr,int,TreasureNodePtr,int *,
             RoomNodePtr *,int *);
static void FindPathToEntrance(RoomNodePtr,RoomNodePtr,PathStackNodePtr *,
             int,char **,int *);
static void FreeRoom(RoomNodePtr,int,int);
static void FreeRoomDescriptions(int,char ***);
static void FreeRoomsAndPassages(RoomNodePtr);
static void FreeTreasureDescriptions(int,TreasureNodePtr *);
static void GetRoomDescriptions(char *,int *,char ***,int *);
static void GetTreasureDescriptions(char *,int *,TreasureNodePtr *,int *);
static void HideTreasuresAndWeapons(int,int,RoomNodePtr *,int,TreasureNodePtr);
static void Inventory(int,int,TreasureNodePtr,int *);
static void Leave(RoomNodePtr,int,TreasureNodePtr);
static void ListCommands(FILE *pFile,int);
static void ListPassages(RoomNodePtr,int *);
static void ListTreasures(RoomNodePtr,int,TreasureNodePtr,int *);
static void ListWeapons(RoomNodePtr,int,TreasureNodePtr,int *);
       int  main(int,char **);
static void Move(int *,int,RoomNodePtr *,int *);
static void PlayGame(int,RoomNodePtr,RoomNodePtr *,int,int,TreasureNodePtr,
             int *);
static void Points(int,RoomNodePtr,RoomNodePtr *,int,TreasureNodePtr,int,int,
             int *);
static int  RandomNumber(void);
#if DEBUG_MEMORY
static void ReportMemoryLeaks(void);
#endif
static void SeedRandomNumberGenerator(char *);
static void ShuffleRoomDescriptions(int,char **);
static void WayOut(int,RoomNodePtr,RoomNodePtr,RoomNodePtr *,int,
             TreasureNodePtr,int *);

#if DEBUG_MEMORY
static int  bMemoryError;
#endif
static int  m_nAddIndex;
static int  m_nDirectionOpposite [6] = {1,0,3,2,5,4};
static int  m_nReplaceIndex;
static int  m_nRN [8];
static int  m_nRNPartialSum;
static char *m_szDirection[6]
             = {"north","south","east","west","up","down"};
#if DEBUG_MEMORY
static char *pMemoryHead;
#endif

#if DEBUG_MEMORY
static void *DebugMalloc(
 unsigned int uiBytes,
 int          nLine)
  {
    FILE *o;
    char *pResult;

    if (uiBytes)
      {
        pResult=(char *)
         malloc(uiBytes+2*sizeof(char *)+3*sizeof(unsigned int)+sizeof(char));
        if (pResult)
          {
            *((char **) pResult)=pMemoryHead; /* next node in stack */
            *((char **) (pResult+sizeof(char *)))
             =NULL; /* previous node in stack */
            *((unsigned int *) (pResult+2*sizeof(char *)))=nLine;
            *((unsigned int *) (pResult+2*sizeof(char *)+sizeof(unsigned int)))
             =uiBytes;
            *((unsigned int *)
             (pResult+2*sizeof(char *)+2*sizeof(unsigned int)))
             =FALSE; /* freed */
            *(pResult+2*sizeof(char *)+3*sizeof(unsigned int)+uiBytes)
             ='\01'; /* overflow detector */
            if (pMemoryHead)
              *((char **) (pMemoryHead+sizeof(char *)))=pResult;
            pMemoryHead=pResult;
            pResult+=2*sizeof(char *)+3*sizeof(unsigned int);
          }
        else
          {
            bMemoryError=TRUE;
            if (o=fopen("memory.txt","a"))
              {
                fprintf(o,"On line %u, memory was exhausted.\n",nLine);
                fclose(o);
              }
          }
      }
    else
      pResult=(char *) NULL;
    return (void *) pResult;
  }

static void DebugFree(
 void *p,
 int  nLine)
  {
    int   bCorrupted;
    FILE  *o;
    unsigned int *pFreed;
    char  *pOverflow;

    /* Nothing is actually freed so that attempts to free twice are detected. */
    if (p)
      {
        bCorrupted=FALSE;
        pFreed=(unsigned int *) (((char *) p)-sizeof(unsigned int));
        if (*pFreed == TRUE)
          {
            bMemoryError=TRUE;
            if (o=fopen("memory.txt","a"))
              {
                fprintf(o,"On line %u, an attempt was made to free freed "
                 "memory that was allocated on line %u.\n",nLine,
                 *((unsigned int *) (((char *) p)-3*sizeof(unsigned int))));
                fclose(o);
              }
          }
        else
          if (*pFreed == FALSE)
            *pFreed=TRUE;
          else
            {
              bMemoryError=TRUE;
              bCorrupted=TRUE;
              if (o=fopen("memory.txt","a"))
                {
                  fprintf(o,"On line %u, the bytes preceding the "
                   "memory being freed were found to be corrupted.\n",nLine);
                  fclose(o);
                }
              *pFreed=TRUE;
            }
        if (! bCorrupted)
          {
            pOverflow=((char *) p)+(*((unsigned int *) (((char *) p)
             -2*sizeof(unsigned int))));
            if (*pOverflow != '\01')
              {
                 bMemoryError=TRUE;
                 if (o=fopen("memory.txt","a"))
                   {
                     fprintf(o,"On line %u, the byte following the "
                      "memory that was allocated on line %u was found to be "
                      "corrupted.\n",nLine,*((unsigned int *) (((char *) p)
                      -3*sizeof(unsigned int))));
                     fclose(o);
                   }
              }
          }
      }
    return;
  }

static void ReportMemoryLeaks(
 void)
  {
    int   bFreed;
    FILE  *o;
    unsigned int *pFreed;
    char  *pMemory;
    char  *pNext;
    char  *pOverflow;

    bFreed=TRUE;
    pMemory=pMemoryHead;
    while ((! bMemoryError) && (pMemory))
      {
        pFreed=(unsigned int *)
         (((char *) pMemory)+2*sizeof(char *)+2*sizeof(unsigned int));
        if (*pFreed != TRUE)
          {
            if (*pFreed == FALSE)
              {
                bFreed=FALSE;
                if (o=fopen("memory.txt","a"))
                  {
                    fprintf(o,"Memory allocated on line %u was never freed.\n",
                     *((unsigned int *) (((char *) pMemory)+2*sizeof(char *))));
                    fclose(o);
                  }
              }
            else
              {
                bMemoryError=TRUE;
                if (o=fopen("memory.txt","a"))
                  {
                    fprintf(o,"The bytes preceding the "
                     "memory that was allocated on line %u were found to be "
                     "corrupted.\n",
                     *((unsigned int *) (((char *) pMemory)+2*sizeof(char *))));
                    fclose(o);
                  }
              }
          }
        if (! bMemoryError)
          {
            pOverflow=pMemory+2*sizeof(char *)+3*sizeof(unsigned int)
             +(*((unsigned int *)
             (pMemory+2*sizeof(char *)+sizeof(unsigned int))));
            if (*pOverflow != '\01')
              {
                 bMemoryError=TRUE;
                 if (o=fopen("memory.txt","a"))
                   {
                     fprintf(o,"The byte following memory that was allocated "
                      "on line %u was found to be corrupted.\n",
                      *((unsigned int *)
                      (((char *) pMemory)+2*sizeof(char *))));
                     fclose(o);
                   }
              }
          }
        if (! bMemoryError)
          {
            pNext=*((char **) pMemory);
            if (pNext)
              {
                if (pMemory != *((char **) (pNext+sizeof(char *))))
                  {
                    bMemoryError=TRUE;
                    if (o=fopen("memory.txt","a"))
                      {
                        fprintf(o,"The memory stack is corrupted.\n");
                        fclose(o);
                      }
                  }
              }
            free((void *) pMemory);
            pMemory=pNext;
          }
      }
    if (bFreed && (! bMemoryError))
      {
        if (o=fopen("memory.txt","a"))
          {
            fprintf(o,"No memory errors were detected.\n");
            fclose(o);
          }
      }
    return;
  }
#endif

static void DisplayText(
 FILE *pFile,
 char *szText,
 int  bPrintBlankLine,
 int  nMaxLineLen)
  {
    int nColumn;
    int nPendingSpaces;
    int nTokenLen;
    int nTokenStart;
    int nTokenStop;

    nColumn=1;
    nPendingSpaces=0;
    nTokenStart=0;
    while (szText[nTokenStart] && (szText[nTokenStart] == ' '))
      {
        fputc((int) ' ',pFile);
        if (++nColumn > nMaxLineLen)
          {
            fputc((int) '\n',pFile);
            nColumn=1;
          }
        ++nTokenStart;
      }
    while (szText[nTokenStart])
      {
        while (szText[nTokenStart] && (szText[nTokenStart] == ' '))
          ++nTokenStart;
        if (nColumn+nPendingSpaces > nMaxLineLen)
          {
            fputc((int) '\n',pFile);
            nPendingSpaces=0;
            nColumn=1;
          }
        else
          while (nPendingSpaces)
            {
              fputc((int) ' ',pFile);
              ++nColumn;
              --nPendingSpaces;
            }
        if (szText[nTokenStart])
          {
            nTokenStop=nTokenStart;
            while (szText[nTokenStop] && (szText[nTokenStop] != ' '))
              ++nTokenStop;
            nTokenLen=nTokenStop-nTokenStart;
            if (nColumn+nTokenLen > nMaxLineLen)
              if (nTokenLen > nMaxLineLen)
                {
                  while (nColumn < nMaxLineLen)
                    {
                      fputc((int) (szText[nTokenStart]),pFile);
                      ++nColumn;
                      ++nTokenStart;
                    }
                  fputc((int) '\n',pFile);
                  nColumn=1;
                  while (nTokenStart < nTokenStop)
                    {
                      fputc((int) (szText[nTokenStart]),pFile);
                      if (++nColumn > nMaxLineLen)
                        {
                          fputc((int) '\n',pFile);
                          nColumn=1;
                        }
                      ++nTokenStart;
                    }
                }
              else
                {
                  fputc((int) '\n',pFile);
                  nColumn=1;
                  while (nTokenStart < nTokenStop)
                    {
                      fputc((int) (szText[nTokenStart]),pFile);
                      ++nColumn;
                      ++nTokenStart;
                    }
                }
            else
              while (nTokenStart < nTokenStop)
                {
                  fputc((int) (szText[nTokenStart]),pFile);
                  ++nColumn;
                  ++nTokenStart;
                }
            switch (szText[nTokenStop-1])
              {
                case '.':
                  nPendingSpaces=2;
                  break;
                case '?':
                  nPendingSpaces=2;
                  break;
                case '!':
                  nPendingSpaces=2;
                  break;
                default:
                  nPendingSpaces=1;
                  break;
              }
          }
      }
    if (nColumn > 1)
      fputc((int) '\n',pFile);
    if (bPrintBlankLine)
      fputc((int) '\n',pFile);
    return;
  }

static void ListCommands(
 FILE *pFile,
 int  nMaxLineLen)
  {
    DisplayText(pFile,
     "Commands in the game are \"N\", \"E\", \"S\", \"W\", \"U\", and "
     "\"D\" to move North, East, South, West, Up, or Down, "
     "respectively.  Other commands are \"C\" to carry things, \"I\" to "
     "inventory what you are carrying, \"L\" to leave treasures, \"P\" to "
     "get the points you\'ve scored, \"O\" for help getting out of the "
     "mine, \"H\" for help, and \"Q\" to quit.",TRUE,nMaxLineLen);
    return;
  }

static void DisplayHelp(
 FILE *pFile,
 int  bPrintBlankLine,
 int  nMaxLineLen)
  {
    ListCommands(pFile,nMaxLineLen);
    DisplayText(pFile,
     "In a mine, the passages are straight.  So, for example, if "
     "you go North to leave a room, you can go South to reenter it.  "
     "The rooms are not evenly spaced.  However, the distance between "
     "adjacent rooms is always a multiple of the minimum distance "
     "between adjacent rooms.",bPrintBlankLine,nMaxLineLen);
    return;
  }

static int RandomNumber(
 void)
  {
    int nResult;
   /*
       Each pseudo-random number is the modulo sum of the
     previous eight pseudo-random numbers.  A prime modulus
     makes it likely that the pseudo-random numbers will be
     uniformly distributed.  To speed computation, a partial
     sum of 7 of the 8 previous pseudo-random numbers is maintained.
          For a given set of initial values m_nRN[i], i=0,1,2,...,7,
     this random number generator should produce the same sequence
     of random numbers, no matter what 32-bit C Compiler it is compiled
     under.
   */
    do
      {
        nResult=m_nRNPartialSum+m_nRN[m_nAddIndex];
        if (nResult >= 32771) nResult-=32771;
        m_nRNPartialSum=nResult-m_nRN[m_nReplaceIndex];
        if (m_nRNPartialSum < 0) m_nRNPartialSum+=32771;
        m_nRN[m_nReplaceIndex]=nResult;
        m_nAddIndex=m_nReplaceIndex;
        if (++m_nReplaceIndex >= 8)
          m_nReplaceIndex=0;
      }
    while (nResult > 32767);
    return nResult;
  }

static void SeedRandomNumberGenerator(
 char *szMine)
  {
    char *pc;
    int  nKeyIndex;
    int  nKeyStart;
    int  nKeyLength;

    nKeyLength=strlen(szMine);
    nKeyIndex=0;
    if (nKeyLength < 8)
      {
        for (nKeyStart=nKeyLength; nKeyStart < 8; ++nKeyStart)
          m_nRN[nKeyIndex++]=1+(int) '0';
        pc=szMine;
      }
    else
      pc=szMine+nKeyLength-8;
    while (*pc)
      m_nRN[nKeyIndex++]=1+(int) *pc++;
    m_nRNPartialSum=0;
    for (nKeyIndex=7; nKeyIndex > 0; --nKeyIndex)
      {
        m_nRNPartialSum+=m_nRN[nKeyIndex];
        if (m_nRNPartialSum >= 32771)
          m_nRNPartialSum-=32771;
      }
    m_nReplaceIndex=1;
    m_nAddIndex=0;
    for (nKeyIndex=256; nKeyIndex--;)
      RandomNumber();
    return;
  }

static void GetRoomDescriptions(
 char *szFileName,
 int  *nRooms,
 char ***ppszRoom,
 int  *bErr)
  {
    int    bDescTerminated;
    size_t nLen;
    int    nRoom;
    FILE   *pFile;
    char   szBuf [256];
    char   *szDesc;
    char   *szTem;

    if (pFile=fopen(szFileName,"r"))
      {
        if (fgets(&szBuf[0],256,pFile))
          {
            *nRooms=atoi(&szBuf[0]);
            if (*ppszRoom=(char **) Malloc((*nRooms)*sizeof(char *)))
              {
                for (nRoom=0; ((! *bErr) && (nRoom < *nRooms));
                 ++nRoom)
                  {
                    szDesc=NULL;
                    bDescTerminated=FALSE;
                    while ((! *bErr)
                    &&     (! bDescTerminated)
                    &&     fgets(&szBuf[0],256,pFile))
                      {
                        if (nLen=strlen(&szBuf[0]))
                          {
                            if (szBuf[nLen-1] == '\n')
                              {
                                szBuf[--nLen]='\0';
                                bDescTerminated=TRUE;
                              }
                            if (nLen)
                              {
                                if (szBuf[nLen-1] == '\r')
                                  szBuf[--nLen]='\0';
                              }
                            if (nLen)
                              {
                                if (szDesc)
                                  if (szTem=(char *) Malloc(strlen(szDesc)
                                   +nLen+1))
                                    {
                                      strcpy(szTem,szDesc);
                                      strcat(szTem,&szBuf[0]);
                                      Free(szDesc);
                                      szDesc=szTem;
                                    }
                                  else
                                    {
                                      *bErr=TRUE;
                                      fprintf(stderr,
                                       "Fatal error:  out of memory");
                                    }
                                else
                                  if (szDesc=(char *) Malloc(nLen+1))
                                    strcpy(szDesc,&szBuf[0]);
                                  else
                                    {
                                      *bErr=TRUE;
                                      fprintf(stderr,
                                       "Fatal error:  out of memory");
                                    }
                              }
                          }
                      }
                    if (szDesc && *szDesc)
                      if (szTem=(char *) Malloc(1+strlen("You\'re in ")
                       +strlen(szDesc)))
                        {
                          strcpy(szTem,"You\'re in ");
                          strcat(szTem,szDesc);
                          Free(szDesc);
                          (*ppszRoom)[nRoom]=szTem;
                        }
                      else
                        {
                          *bErr=TRUE;
                          fprintf(stderr,"Fatal error:  out of memory\n");
                        }
                    else
                      {
                        *bErr=TRUE;
                        fprintf(stderr,"Fatal error:  \"%s\" contains fewer "
                         "room descriptions that the %d its first line "
                         "indicates.\n",szFileName,*nRooms);
                      }
                  }
                if (! *bErr)
                  {
                    while ((! *bErr) && fgets(&szBuf[0],256,pFile))
                      {
                        if (nLen=strlen(&szBuf[0]))
                          {
                            if (szBuf[nLen-1] == '\n')
                              {
                                szBuf[--nLen]='\0';
                                bDescTerminated=TRUE;
                              }
                            if (nLen)
                              {
                                if (szBuf[nLen-1] == '\r')
                                  szBuf[--nLen]='\0';
                              }
                            if (nLen)
                              {
                                *bErr=TRUE;
                                fprintf(stderr,"Warning:  \"%s\" contains "
                                 "more room descriptions than the %d its "
                                 "first line indicates.\n",szFileName,
                                 *nRooms);
                              }
                          }
                      }
                    *bErr=FALSE;
                  }
              }
            else
              {
                *bErr=TRUE;
                fprintf(stderr,"Fatal error:  out of memory\n");
              }
          }
        else
          {
            *bErr=TRUE;
            fprintf(stderr,"Fatal error:  the first line of \"%s\" couldn\'t "
             "be read.\n",szFileName);
          }
        fclose(pFile);
      }
    else
      {
        *bErr=TRUE;
        fprintf(stderr,
         "Fatal error:  \"%s\" couldn\'t be opened for inputting.\n",
         szFileName);
      }
    if (! *bErr)
      {
        if (*nRooms < 10)
          {
            *bErr=TRUE;
            fprintf(stderr,
             "Fatal error:  \"%s\" contains fewer than 10 rooms.\n",
             szFileName);
          }
      }
    return;
  }

static void FreeRoomDescriptions(
 int  nRooms,
 char ***ppszRoom)
  {
    int nRoom;

    for (nRoom=nRooms; nRoom--;)
      Free(((*ppszRoom)[nRoom]));
    Free(*ppszRoom);
    *ppszRoom=NULL;
    return;
  }

static void ShuffleRoomDescriptions(
 int  nRooms,
 char **pszRoom)
  {
    int  nRoom1;
    int  nRoom2;
    char *sz;

    for (nRoom1=nRooms-1; nRoom1 > 0; -- nRoom1)
      {
        nRoom2=RandomNumber()%nRoom1;
        sz=pszRoom[nRoom1];
        pszRoom[nRoom1]=pszRoom[nRoom2];
        pszRoom[nRoom2]=sz;
      }
    return;
  }

static void GetTreasureDescriptions(
 char            *szFileName,
 int             *nTreasures,
 TreasureNodePtr *ppTreasure,
 int             *bErr)
  {
    int    bDescTerminated;
    size_t nLen;
    int    nTreasure;
    FILE   *pFile;
    char   szBuf [256];
    char   *szGuardian;
    char   *szTem;
    char   *szTreasure;
    char   *szWeapon;

    if (pFile=fopen(szFileName,"r"))
      {
        if (fgets(&szBuf[0],256,pFile))
          {
            *nTreasures=atoi(&szBuf[0]);
            if (*ppTreasure=(TreasureNodePtr)
             Malloc((*nTreasures)*sizeof(struct TreasureNode)))
              {
                for (nTreasure=0; ((! *bErr) && (nTreasure < *nTreasures));
                 ++nTreasure)
                  {
                    (*ppTreasure)[nTreasure].pRoomTreasure=NULL;
                    (*ppTreasure)[nTreasure].pRoomWeapon=NULL;
                    (*ppTreasure)[nTreasure].bSlain=FALSE;
                    szTreasure=NULL;
                    bDescTerminated=FALSE;
                    while ((! *bErr)
                    &&     (! bDescTerminated)
                    &&     fgets(&szBuf[0],256,pFile))
                      {
                        if (nLen=strlen(&szBuf[0]))
                          {
                            if (szBuf[nLen-1] == '\n')
                              {
                                szBuf[--nLen]='\0';
                                bDescTerminated=TRUE;
                              }
                            if (nLen)
                              {
                                if (szBuf[nLen-1] == '\r')
                                  szBuf[--nLen]='\0';
                              }
                            if (nLen)
                              {
                                if (szTreasure)
                                  if (szTem=(char *) Malloc(strlen(szTreasure)
                                   +nLen+1))
                                    {
                                      strcpy(szTem,szTreasure);
                                      strcat(szTem,&szBuf[0]);
                                      Free(szTreasure);
                                      szTreasure=szTem;
                                    }
                                  else
                                    {
                                      *bErr=TRUE;
                                      fprintf(stderr,
                                       "Fatal error:  out of memory");
                                    }
                                else
                                  if (szTreasure=(char *) Malloc(nLen+1))
                                    strcpy(szTreasure,&szBuf[0]);
                                  else
                                    {
                                      *bErr=TRUE;
                                      fprintf(stderr,
                                       "Fatal error:  out of memory");
                                    }
                              }
                          }
                      }
                    if (szTreasure && *szTreasure)
                      (*ppTreasure)[nTreasure].szTreasure=szTreasure;
                    else
                      {
                        *bErr=TRUE;
                        fprintf(stderr,"Fatal error:  \"%s\" contains fewer "
                         "treasure descriptions than the %d indicated by its "
                         "first line.\n",szFileName,*nTreasures);
                      }
                    if (! *bErr)
                      {
                        szGuardian=NULL;
                        bDescTerminated=FALSE;
                        while ((! *bErr)
                        &&     (! bDescTerminated)
                        &&     fgets(&szBuf[0],256,pFile))
                          {
                            if (nLen=strlen(&szBuf[0]))
                              {
                                if (szBuf[nLen-1] == '\n')
                                  {
                                    szBuf[--nLen]='\0';
                                    bDescTerminated=TRUE;
                                  }
                                if (nLen)
                                  {
                                    if (szBuf[nLen-1] == '\r')
                                      szBuf[--nLen]='\0';
                                  }
                                if (nLen)
                                  {
                                    if (szGuardian)
                                      if (szTem=(char *)
                                       Malloc(strlen(szGuardian)+nLen+1))
                                        {
                                          strcpy(szTem,szGuardian);
                                          strcat(szTem,&szBuf[0]);
                                          Free(szGuardian);
                                          szGuardian=szTem;
                                        }
                                      else
                                        {
                                          *bErr=TRUE;
                                          fprintf(stderr,
                                           "Fatal error:  out of memory");
                                        }
                                    else
                                      if (szGuardian=(char *) Malloc(nLen+1))
                                        strcpy(szGuardian,&szBuf[0]);
                                      else
                                        {
                                          *bErr=TRUE;
                                          fprintf(stderr,
                                           "Fatal error:  out of memory");
                                        }
                                  }
                              }
                          }
                        if (szGuardian && *szGuardian)
                          (*ppTreasure)[nTreasure].szGuardian=szGuardian;
                        else
                          {
                            *bErr=TRUE;
                            fprintf(stderr,"Fatal error:  \"%s\" contains "
                             "fewer treasure descriptions than the %d "
                             "indicated by its first line.\n",szFileName,
                             *nTreasures);
                          }
                      }
                    if (! *bErr)
                      {
                        szWeapon=NULL;
                        bDescTerminated=FALSE;
                        while ((! *bErr)
                        &&     (! bDescTerminated)
                        &&     fgets(&szBuf[0],256,pFile))
                          {
                            if (nLen=strlen(&szBuf[0]))
                              {
                                if (szBuf[nLen-1] == '\n')
                                  {
                                    szBuf[--nLen]='\0';
                                    bDescTerminated=TRUE;
                                  }
                                if (nLen)
                                  {
                                    if (szBuf[nLen-1] == '\r')
                                      szBuf[--nLen]='\0';
                                  }
                                if (nLen)
                                  {
                                    if (szWeapon)
                                      if (szTem=(char *)
                                       Malloc(strlen(szWeapon)+nLen+1))
                                        {
                                          strcpy(szTem,szWeapon);
                                          strcat(szTem,&szBuf[0]);
                                          Free(szWeapon);
                                          szWeapon=szTem;
                                        }
                                      else
                                        {
                                          *bErr=TRUE;
                                          fprintf(stderr,
                                           "Fatal error:  out of memory");
                                        }
                                    else
                                      if (szWeapon=(char *) Malloc(nLen+1))
                                        strcpy(szWeapon,&szBuf[0]);
                                      else
                                        {
                                          *bErr=TRUE;
                                          fprintf(stderr,
                                           "Fatal error:  out of memory");
                                        }
                                  }
                              }
                          }
                        if (szWeapon && *szWeapon)
                          (*ppTreasure)[nTreasure].szWeapon=szWeapon;
                        else
                          {
                            *bErr=TRUE;
                            fprintf(stderr,"Fatal error:  \"%s\" contains "
                             "fewer treasure descriptions than the %d "
                             "indicated by its first line.\n",szFileName,
                             *nTreasures);
                          }
                      }
                  }
                if (! *bErr)
                  {
                    while ((! *bErr) && fgets(&szBuf[0],256,pFile))
                      {
                        if (nLen=strlen(&szBuf[0]))
                          {
                            if (szBuf[nLen-1] == '\n')
                              {
                                szBuf[--nLen]='\0';
                                bDescTerminated=TRUE;
                              }
                            if (nLen)
                              {
                                if (szBuf[nLen-1] == '\r')
                                  szBuf[--nLen]='\0';
                              }
                            if (nLen)
                              {
                                *bErr=TRUE;
                                fprintf(stderr,"Warning:  \"%s\" contains "
                                 "more treasure descriptions than the %d its "
                                 "first line indicates.\n",szFileName,
                                 *nTreasures);
                              }
                          }
                      }
                    *bErr=FALSE;
                  }
              }
            else
              {
                *bErr=TRUE;
                fprintf(stderr,"Fatal error:  out of memory\n");
              }
          }
        else
          {
            *bErr=TRUE;
            fprintf(stderr,"Fatal error:  the first line of \"%s\" couldn\'t "
             "be read.\n",szFileName);
          }
        fclose(pFile);
      }
    else
      {
        *bErr=TRUE;
        fprintf(stderr,
         "Fatal error:  \"%s\" couldn\'t be opened for inputting.\n",
         szFileName);
      }
    return;
  }

static void FreeTreasureDescriptions(
 int             nTreasures,
 TreasureNodePtr *ppTreasure)
  {
    int nTreasure;

    for (nTreasure=nTreasures; nTreasure--;)
      {
        Free(((*ppTreasure)[nTreasure].szWeapon));
        Free(((*ppTreasure)[nTreasure].szGuardian));
        Free(((*ppTreasure)[nTreasure].szTreasure));
      }
    Free(*ppTreasure);
    *ppTreasure=NULL;
  }

static void ExcavateMine(
 int             nRooms,
 char            **pszRoom,
 RoomNodePtr     pRoomEntrance,
 int             nTreasures,
 TreasureNodePtr pTreasure,
 int             *nChokepoints,
 RoomNodePtr     *ppRoomWithName,
 int             *bErr)
  {
    int            nChokepoint;
    int            nDirectionOpposite;
    int            nDirection1;
    int            nDirection2;
    int            nDistance;
    int            nHeight;
    int            nLen;
    int            nMax;
    int            nRoom;
    int            nStep;
    int            nTreasure;
    int            nUnblocked;
    int            nWidth;
    int            nX;
    int            nXFirstRoom;
    int            nY;
    int            nYFirstRoom;
    int            nZ;
    int            nZFirstRoom;
    PassageNodePtr pPassage;
    RoomNodePtr    pRoom;
    RoomNodePtr    pRoomDown;
    RoomNodePtr    pRoomNext;
    RoomNodePtr    pRoomStart;
    RoomNodePtr    pRoomSouth;
    RoomNodePtr    pRoomSouthBase;
    RoomNodePtr    pRoomWest;
    RoomNodePtr    pRoomWestBase;
    RoomNodePtr    pRoomWestBaseBase;

    nMax=(int) (10.0+exp(log((double) nRooms)/3.0));
    do
      {
        nLen=RandomNumber()%(nMax-3)+3;
        nWidth=RandomNumber()%(nMax-3)+3;
        nHeight=RandomNumber()%(nMax-3)+3;
      }
    while ((nLen*nWidth*nHeight < 2*(nRooms-1))
    ||     (nLen*nWidth*nHeight > 3*(nRooms-1)));
    nXFirstRoom=0;
    nYFirstRoom=RandomNumber()%nLen;
    nZFirstRoom=RandomNumber()%nHeight;
    pRoomWestBase=NULL;
    pRoomWestBaseBase=NULL;
    for (nX=0; ((! *bErr) && (nX < nWidth)); ++nX)
      {
        pRoomSouthBase=NULL;
        for (nY=0; ((! *bErr) && (nY < nLen)); ++nY)
          {
            pRoomWest=pRoomWestBase;
            pRoomSouth=pRoomSouthBase;
            pRoomDown=NULL;
            for (nZ=0; ((! *bErr) && (nZ < nHeight)); ++nZ)
              {
                if (pRoom=(RoomNodePtr) Malloc(sizeof(struct RoomNode)))
                  {
                    pRoom->szDescription=NULL;
                    pRoom->bMined=FALSE;
                    pRoom->nChokepoint=-1;
                    pRoom->bVisited=FALSE;
                    pRoom->Adjacent.Passage.pPassageNorth=NULL;
                    pRoom->Adjacent.Passage.pPassageSouth=NULL;
                    pRoom->Adjacent.Passage.pPassageEast=NULL;
                    pRoom->Adjacent.Passage.pPassageWest=NULL;
                    pRoom->Adjacent.Passage.pPassageUp=NULL;
                    pRoom->Adjacent.Passage.pPassageDown=NULL;
/*
(X,Y,Z) gets linked to (X-1,Y,Z); (X,Y-1,Z); and (X,Y,Z-1)
                       pRoomWest  pRoomSouth     pRoomDown
*/
                    if (nZ == 0)
                      {
                        pRoomSouthBase=pRoom;
                        if (nY == 0)
                          pRoomWestBaseBase=pRoom;
                      }
                    if (pRoomDown)
                      {
                      /* link room to one below it */
                        if (pPassage=(PassageNodePtr)
                         Malloc(sizeof(struct PassageNode)))
                          {
                            pPassage->bBlocked=TRUE;
                            pPassage->pTreasureGuardian=NULL;
                            pPassage->pRoom1=pRoom;
                            pPassage->pRoom2=pRoomDown;
                            pRoom->Adjacent.Passage.pPassageDown=pPassage;
                            pRoomDown->Adjacent.Passage.pPassageUp=pPassage;
                          }
                        else
                          {
                            *bErr=TRUE;
                            fprintf(stderr,"Fatal error:  out of memory\n");
                          }
                      }
                    if (pRoomSouth)
                      {
                        /* link to room to the south */
                        if (pPassage=(PassageNodePtr)
                         Malloc(sizeof(struct PassageNode)))
                          {
                            pPassage->bBlocked=TRUE;
                            pPassage->pTreasureGuardian=NULL;
                            pPassage->pRoom1=pRoom;
                            pPassage->pRoom2=pRoomSouth;
                            pRoom->Adjacent.Passage.pPassageSouth=pPassage;
                            pRoomSouth->Adjacent.Passage.pPassageNorth=pPassage;
                          }
                        else
                          {
                            *bErr=TRUE;
                            fprintf(stderr,"Fatal error:  out of memory\n");
                          }
                        if (pRoomSouth->Adjacent.Passage.pPassageUp)
                          if (pRoomSouth->Adjacent.Passage.pPassageUp->pRoom1
                           == pRoomSouth)
                            pRoomSouth
                             =pRoomSouth->Adjacent.Passage.pPassageUp->pRoom2;
                          else
                            pRoomSouth
                             =pRoomSouth->Adjacent.Passage.pPassageUp->pRoom1;
                        else
                          pRoomSouth=NULL;
                      }
                    if (pRoomWest)
                      {
                      /* link to the room to the west */
                        if (pPassage=(PassageNodePtr)
                         Malloc(sizeof(struct PassageNode)))
                          {
                            pPassage->bBlocked=TRUE;
                            pPassage->pTreasureGuardian=NULL;
                            pPassage->pRoom1=pRoom;
                            pPassage->pRoom2=pRoomWest;
                            pRoom->Adjacent.Passage.pPassageWest=pPassage;
                            pRoomWest->Adjacent.Passage.pPassageEast=pPassage;
                          }
                        else
                          {
                            *bErr=TRUE;
                            fprintf(stderr,"Fatal error:  out of memory\n");
                          }
                        if (pRoomWest->Adjacent.Passage.pPassageUp)
                          if (pRoomWest->Adjacent.Passage.pPassageUp->pRoom1
                           == pRoomWest)
                            pRoomWest=pRoomWest->Adjacent.Passage.pPassageUp->
                             pRoom2;
                          else
                            pRoomWest=pRoomWest->Adjacent.Passage.pPassageUp->
                             pRoom1;
                        else
                          pRoomWest=NULL;

                      }
                    /* If this is the first room, connect it to the entrance. */
                    if ((nX == nXFirstRoom)
                    &&  (nY == nYFirstRoom)
                    &&  (nZ == nZFirstRoom))
                      {
                        if (pPassage=(PassageNodePtr)
                         Malloc(sizeof(struct PassageNode)))
                          {
                            pPassage->bBlocked=FALSE;
                            pPassage->pTreasureGuardian=NULL;
                            pPassage->pRoom1=pRoom;
                            pPassage->pRoom2=pRoomEntrance;
                            pRoom->Adjacent.Passage.pPassageWest=pPassage;
                            pRoomEntrance->Adjacent.Passage.pPassageEast
                             =pPassage;
                          }
                        else
                          {
                            *bErr=TRUE;
                            fprintf(stderr,"Fatal error:  out of memory\n");
                          }
                      }
                    pRoomDown=pRoom;
                  }
                else
                  {
                    *bErr=TRUE;
                    fprintf(stderr,"Fatal error:  out of memory\n");
                  }
              }
            if (pRoomWestBase)
              {
                if (pRoomWestBase->Adjacent.Passage.pPassageNorth)
                  {
                    if (pRoomWestBase->Adjacent.Passage.pPassageNorth->pRoom1
                     == pRoomWestBase)
                      pRoomWestBase
                       =pRoomWestBase->Adjacent.Passage.pPassageNorth->pRoom2;
                    else
                      pRoomWestBase
                       =pRoomWestBase->Adjacent.Passage.pPassageNorth->pRoom1;
                  }
              }
          }
        pRoomWestBase=pRoomWestBaseBase;
      }
    nChokepoint=-1;
    nTreasure=0;
    pRoom=pRoomEntrance;
    nRoom=0;
    while (nRoom < nRooms)
      {
        pRoomStart=pRoom;
        nDirection1=RandomNumber()%6;
        nDirectionOpposite=m_nDirectionOpposite[nDirection1];
        nDistance=RandomNumber()%3+1;
        nStep=0;
        while ((pRoom->Adjacent.pPassage[nDirection1])
        &&     (nStep < nDistance))
          {
            pPassage=pRoom->Adjacent.pPassage[nDirection1];
            if (pPassage->pRoom1 == pRoom)
              pRoomNext=pPassage->pRoom2;
            else
              pRoomNext=pPassage->pRoom1;
            if (pPassage->bBlocked)
              if ((pRoomNext->nChokepoint == nChokepoint)
              ||  (! pRoomNext->bMined))
                {
                  pRoom=pRoomNext;
                  pRoom->bMined=TRUE;
                  pRoom->nChokepoint=nChokepoint;
                  pPassage->bBlocked=FALSE;
                  if (pRoom->szDescription)
                    ++nStep;
                  else
                    {
                      nUnblocked=0;
                      for (nDirection2=6;
                       ((nUnblocked < 2) && (nDirection2--));)
                        if (nDirection2 != nDirectionOpposite)
                          if (pRoom->Adjacent.pPassage[nDirection2])
                            if (! pRoom->Adjacent.pPassage[nDirection2]->bBlocked)
                              ++nUnblocked;
                      if (nUnblocked < 2)
                        ++nStep;
                      else
                        nStep=nDistance;
                    }
                }
              else
                nStep=nDistance;
            else
              {
                ++nStep;
                pRoom=pRoomNext;
                nChokepoint=pRoom->nChokepoint;
              }
          }
        if (nStep)
          {
            if (pRoom->szDescription == NULL)
              {
                pRoom->szDescription=pszRoom[nRoom];
                nDirection1=0;
                nUnblocked=0;
                for (nDirection2=6;
                 ((nUnblocked < 2) && (nDirection2--));)
                  if (pRoom->Adjacent.pPassage[nDirection2])
                    if (! pRoom->Adjacent.pPassage[nDirection2]->bBlocked)
                      {
                        nDirection1=nDirection2;
                        ++nUnblocked;
                      }
                if (nUnblocked == 1)
                  {
                    if ((nRooms-nRoom)*RandomNumber()
                     < 32768*(nTreasures-nTreasure))
                      {
                        nChokepoint=nTreasure;
                        pRoom->Adjacent.pPassage[nDirection1]->
                         pTreasureGuardian=pTreasure+nTreasure;
                        ++nTreasure;
                      }
                  }
                pRoom->nChokepoint=nChokepoint;
                ppRoomWithName[nRoom]=pRoom;
                ++nRoom;
              }
          }
      }
    ppRoomWithName[nRoom]=pRoomEntrance;
    *nChokepoints=nTreasure;
    return;
  }

static void FindPathToEntrance(
 RoomNodePtr      pRoomEntrance,
 RoomNodePtr      pRoom,
 PathStackNodePtr *ppPathStackHead,
 int              nDirectionUsedToEnterRoom,
 char             **pszWayOut,
 int              *bErr)
  {
    int              bRoomAlreadyInStack;
    int              nDirection1;
    int              nDirection2;
    int              nDirection3;
    int              nDirectionRandom [6];
    int              nPathLen;
    PathStackNodePtr pPathStack;

    if (pRoom == pRoomEntrance)
      {
        nPathLen=0;
        pPathStack=*ppPathStackHead;
        while (pPathStack)
          {
            if (pPathStack->pRoom->szDescription)
              ++nPathLen;
            pPathStack=pPathStack->pNext;
          }
        if (*pszWayOut=(char *) Malloc(1+nPathLen))
          {
            (*pszWayOut)[nPathLen]='\0';
            (*pszWayOut)[--nPathLen]=(char) toupper((int)
             *m_szDirection[nDirectionUsedToEnterRoom]);
            pPathStack=*ppPathStackHead;
            while (pPathStack)
              {
                if (pPathStack->pNext)
                  {
                    if (pPathStack->pRoom->szDescription)
                      (*pszWayOut)[--nPathLen]=(char) toupper((int)
                       *m_szDirection[pPathStack->nDirectionUsedToEnterRoom]);
                  }
                pPathStack=pPathStack->pNext;
              }
          }
        else
          {
            *bErr=TRUE;
            fprintf(stderr,"Fatal error:  out of memory\n");
          }
      }
    else
      {
        bRoomAlreadyInStack=FALSE;
        pPathStack=*ppPathStackHead;
        while (pPathStack && (! bRoomAlreadyInStack))
          {
            bRoomAlreadyInStack=(pPathStack->pRoom == pRoom);
            pPathStack=pPathStack->pNext;
          }
        if (! bRoomAlreadyInStack)
          {
            if (pPathStack=(PathStackNodePtr)
             Malloc(sizeof(struct PathStackNode)))
              {
                pPathStack->pRoom=pRoom;
                pPathStack->nDirectionUsedToEnterRoom=nDirectionUsedToEnterRoom;
                pPathStack->pNext=*ppPathStackHead;
                *ppPathStackHead=pPathStack;
                for (nDirection1=6; nDirection1--;)
                  nDirectionRandom[nDirection1]=nDirection1;
                for (nDirection1=5; nDirection1 > 0; --nDirection1)
                  {
                    nDirection2=RandomNumber()%nDirection1;
                    nDirection3=nDirectionRandom[nDirection1];
                    nDirectionRandom[nDirection1]=nDirectionRandom[nDirection2];
                    nDirectionRandom[nDirection2]=nDirection3;
                  }
                for (nDirection1=6;
                 ((! *bErr) && (! *pszWayOut) && (nDirection1--));)
                  {
                    nDirection2=nDirectionRandom[nDirection1];
                    if (pRoom->Adjacent.pPassage[nDirection2])
                      {
                        if (! pRoom->Adjacent.pPassage[nDirection2]->bBlocked)
                          {
                            if (pRoom->Adjacent.pPassage[nDirection2]->pRoom1
                             == pRoom)
                              FindPathToEntrance(pRoomEntrance,
                               pRoom->Adjacent.pPassage[nDirection2]->pRoom2,
                               ppPathStackHead,
                               nDirection2,pszWayOut,bErr);
                            else
                              FindPathToEntrance(pRoomEntrance,
                               pRoom->Adjacent.pPassage[nDirection2]->pRoom1,
                               ppPathStackHead,
                               nDirection2,pszWayOut,bErr);
                          }
                      }
                  }
                *ppPathStackHead=pPathStack->pNext;
                Free(pPathStack);
              }
            else
              {
                *bErr=TRUE;
                fprintf(stderr,"Fatal error:  out of memory\n");
              }
          }
      }
    return;
  }

static void WayOut(
 int             nRooms,
 RoomNodePtr     pRoom,
 RoomNodePtr     pRoomEntrance,
 RoomNodePtr     *ppRoomWithName,
 int             nTreasures,
 TreasureNodePtr pTreasure,
 int             *bErr)
  {
    int              bCarryingTreasure;
    int              nRoom;
    int              nTreasure;
    PathStackNodePtr pPathStackHead;
    char             *szText;
    char             *szWayOut;

    if (pRoom == pRoomEntrance)
      DisplayText(stdout,"You\'re already at the entrance.",TRUE,79);
    else
      {
        bCarryingTreasure=FALSE;
        for (nTreasure=nTreasures; ((! bCarryingTreasure) && (nTreasure--));)
          bCarryingTreasure=(pTreasure[nTreasure].pRoomTreasure == NULL);
        if (bCarryingTreasure)
          {
            pPathStackHead=NULL;
            szWayOut=NULL;
            FindPathToEntrance(pRoomEntrance,pRoom,&pPathStackHead,0,&szWayOut,
             bErr);
            do
              nRoom=RandomNumber()%nRooms;
            while (ppRoomWithName[nRoom] == pRoom);
            pTreasure[nTreasure].pRoomTreasure=ppRoomWithName[nRoom];
            if (szText=(char *) Malloc(1
             +strlen("The pirate takes one of your treasures.  "
             "  As he leaves, he shouts the letters \"")
             +strlen(szWayOut)+strlen("\".")))
              {
                strcpy(szText,"The pirate takes one of your treasures.  ");
                if (strlen(szWayOut) == 1)
                  strcat(szText,"  As he leaves, he shouts the letter \"");
                else
                  strcat(szText,"  As he leaves, he shouts the letters \"");
                strcat(szText,szWayOut);
                strcat(szText,"\".");
                DisplayText(stdout,szText,TRUE,79);
                Free(szText);
              }
            else
              {
                *bErr=TRUE;
                fprintf(stderr,"Fatal error:  out of memory\n");
              }
            Free(szWayOut);
          }
        else
          DisplayText(stdout,"Nothing happens.",TRUE,79);
      }
    return;
  }

static void HideTreasuresAndWeapons(
 int             nChokepoints,
 int             nRooms,
 RoomNodePtr     *ppRoomWithName,
 int             nTreasures,
 TreasureNodePtr pTreasure)
  {
    int nRoom;
    int nTreasure;

    for (nTreasure=0; nTreasure < nChokepoints; ++nTreasure)
      {
        do
          nRoom=RandomNumber()%nRooms;
        while (ppRoomWithName[nRoom]->nChokepoint != nTreasure);
        pTreasure[nTreasure].pRoomTreasure=ppRoomWithName[nRoom];
        do
          nRoom=RandomNumber()%(nRooms+1);
        while (ppRoomWithName[nRoom]->nChokepoint >= nTreasure);
        pTreasure[nTreasure].pRoomWeapon=ppRoomWithName[nRoom];
      }
    for (nTreasure=nChokepoints; nTreasure < nTreasures; ++nTreasure)
      {
        nRoom=RandomNumber()%nRooms;
        pTreasure[nTreasure].pRoomTreasure=ppRoomWithName[nRoom];
      }
    return;
  }

static void FreeRoom(
 RoomNodePtr pRoom,
 int         bDown,
 int         bSouth)
  {
    Free((pRoom->Adjacent.Passage.pPassageWest));
    if (bSouth)
      {
        Free((pRoom->Adjacent.Passage.pPassageSouth));
      }
    else
      {
        Free((pRoom->Adjacent.Passage.pPassageNorth));
      }
    if (bDown)
      {
        Free((pRoom->Adjacent.Passage.pPassageDown));
      }
    else
      {
        Free((pRoom->Adjacent.Passage.pPassageUp));
      }
    Free(pRoom);
    return;
  }

static void FreeRoomsAndPassages(
 RoomNodePtr pRoomEntrance)
  {
    int            bDown;
    int            bRoomFreed;
    int            bSouth;
    PassageNodePtr pPassage;
    RoomNodePtr    pRoom;
    RoomNodePtr    pRoomNext;

    pRoom=pRoomEntrance;
    pPassage=pRoom->Adjacent.Passage.pPassageEast;
    if (pPassage->pRoom1 == pRoom)
      pRoomNext=pPassage->pRoom2;
    else
      pRoomNext=pPassage->pRoom1;
    if (pRoom->szDescription)
      Free((pRoom->szDescription));
    Free(pRoom);
    pRoom=pRoomNext;
    Free(pPassage);
    pRoom->Adjacent.Passage.pPassageWest=NULL;
    while (pRoom->Adjacent.Passage.pPassageEast)
      if (pRoom->Adjacent.Passage.pPassageEast->pRoom1 == pRoom)
        pRoom=pRoom->Adjacent.Passage.pPassageEast->pRoom2;
      else
        pRoom=pRoom->Adjacent.Passage.pPassageEast->pRoom1;
    while (pRoom->Adjacent.Passage.pPassageNorth)
      if (pRoom->Adjacent.Passage.pPassageNorth->pRoom1 == pRoom)
        pRoom=pRoom->Adjacent.Passage.pPassageNorth->pRoom2;
      else
        pRoom=pRoom->Adjacent.Passage.pPassageNorth->pRoom1;
    while (pRoom->Adjacent.Passage.pPassageUp)
      if (pRoom->Adjacent.Passage.pPassageUp->pRoom1 == pRoom)
        pRoom=pRoom->Adjacent.Passage.pPassageUp->pRoom2;
      else
        pRoom=pRoom->Adjacent.Passage.pPassageUp->pRoom1;
    bDown=TRUE;
    bSouth=TRUE;
    bRoomFreed=TRUE;
    while (bRoomFreed)
      {
        bRoomFreed=FALSE;
        if (bDown)
          if (pRoom->Adjacent.Passage.pPassageDown)
            {
              if (pRoom->Adjacent.Passage.pPassageDown->pRoom1 == pRoom)
                pRoomNext=pRoom->Adjacent.Passage.pPassageDown->pRoom2;
              else
                pRoomNext=pRoom->Adjacent.Passage.pPassageDown->pRoom1;
              FreeRoom(pRoom,bDown,bSouth);
              pRoom=pRoomNext;
              bRoomFreed=TRUE;
            }
          else
            bDown=FALSE;
        else
          if (pRoom->Adjacent.Passage.pPassageUp)
            {
              if (pRoom->Adjacent.Passage.pPassageUp->pRoom1 == pRoom)
                pRoomNext=pRoom->Adjacent.Passage.pPassageUp->pRoom2;
              else
                pRoomNext=pRoom->Adjacent.Passage.pPassageUp->pRoom1;
              FreeRoom(pRoom,bDown,bSouth);
              pRoom=pRoomNext;
              bRoomFreed=TRUE;
            }
          else
            bDown=TRUE;
        if (! bRoomFreed)
          {
            if (bSouth)
              if (pRoom->Adjacent.Passage.pPassageSouth)
                {
                  if (pRoom->Adjacent.Passage.pPassageSouth->pRoom1 == pRoom)
                    pRoomNext=pRoom->Adjacent.Passage.pPassageSouth->pRoom2;
                  else
                    pRoomNext=pRoom->Adjacent.Passage.pPassageSouth->pRoom1;
                  FreeRoom(pRoom,!bDown,bSouth);
                  pRoom=pRoomNext;
                  bRoomFreed=TRUE;
                }
              else
                bSouth=FALSE;
            else
              if (pRoom->Adjacent.Passage.pPassageNorth)
                {
                  if (pRoom->Adjacent.Passage.pPassageNorth->pRoom1 == pRoom)
                    pRoomNext=pRoom->Adjacent.Passage.pPassageNorth->pRoom2;
                  else
                    pRoomNext=pRoom->Adjacent.Passage.pPassageNorth->pRoom1;
                  FreeRoom(pRoom,!bDown,bSouth);
                  pRoom=pRoomNext;
                  bRoomFreed=TRUE;
                }
              else
                bSouth=TRUE;
          }
        if (! bRoomFreed)
          {
            if (pRoom->Adjacent.Passage.pPassageWest)
              {
                if (pRoom->Adjacent.Passage.pPassageWest->pRoom1 == pRoom)
                  pRoomNext=pRoom->Adjacent.Passage.pPassageWest->pRoom2;
                else
                  pRoomNext=pRoom->Adjacent.Passage.pPassageWest->pRoom1;
                FreeRoom(pRoom,!bDown,!bSouth);
                pRoom=pRoomNext;
                bRoomFreed=TRUE;
              }
          }
      }
    FreeRoom(pRoom,!bDown,!bSouth);
    return;
  }

static int Command(
 void)
  {
    int nResult;
    int nTem;

    fprintf(stdout,"Command? ");
    nResult=toupper(fgetc(stdin));
    nTem=nResult;
    while ((nTem != EOF) && (nTem != (int) '\n'))
      nTem=fgetc(stdin);
    fputc((int) '\n',stdout);
    return nResult;
  }

static void ListPassages(
 RoomNodePtr pRoom,
 int         *bErr)
  {
    int             nDirection;
    int             nFirstChar;
    int             nPassage;
    int             nPassages;
    RoomNodePtr     pRoomAlongPassage;
    TreasureNodePtr pTreasure;
    char            *szDescription;
    char            szPassages []
                     = "There are passages North, South, East, West, Up, and "
                     "Down.";
    char            *szText;

    nPassages=0;
    for (nPassage=6; nPassage--;)
      if (pRoom->Adjacent.pPassage[nPassage])
        ++nPassages;
    if (nPassages > 1)
      strcpy(&szPassages[0],"There are passages");
    else
      strcpy(&szPassages[0],"There is a passage");
    nPassage=0;
    for (nDirection=0; nDirection < 6; ++nDirection)
      if (pRoom->Adjacent.pPassage[nDirection]
      &&  (! pRoom->Adjacent.pPassage[nDirection]->bBlocked))
        {
          if (++nPassage > 1)
            if (nPassages != 2)
              strcat(&szPassages[0],",");
          strcat(&szPassages[0]," ");
          if (nPassage == nPassages)
            if (nPassages > 1)
              strcat(&szPassages[0],"and ");
          strcat(&szPassages[0],m_szDirection[nDirection]);
        }
    strcat(&szPassages[0],".");
    DisplayText(stdout,&szPassages[0],TRUE,79);
    for (nDirection=0; nDirection < 6; ++nDirection)
      {
        szDescription=NULL;
        pRoomAlongPassage=pRoom;
        while ((pRoomAlongPassage->Adjacent.pPassage[nDirection])
        &&     (! pRoomAlongPassage->Adjacent.pPassage[nDirection]->bBlocked)
        &&     (szDescription == NULL))
          {
            if (pTreasure=pRoomAlongPassage->Adjacent.pPassage[
             nDirection]->pTreasureGuardian)
              {
                nFirstChar=toupper((int) *(pTreasure->szGuardian));
                if ((nFirstChar == 'A')
                ||  (nFirstChar == 'E')
                ||  (nFirstChar == 'I')
                ||  (nFirstChar == 'O')
                ||  (nFirstChar == 'U'))
                  if (szText=(char *)
                   Malloc(1+strlen("The passage ")
                   +strlen(m_szDirection[nDirection])
                   +strlen(" is guarded by an ")
                   +strlen(pTreasure->szGuardian)+strlen(".")))
                    {
                      strcpy(szText,"The passage ");
                      strcat(szText,m_szDirection[nDirection]);
                      strcat(szText," is guarded by an ");
                      strcat(szText,pTreasure->szGuardian);
                      strcat(szText,".");
                      DisplayText(stdout,szText,TRUE,79);
                      Free(szText);
                    }
                  else
                    {
                      *bErr=TRUE;
                      fprintf(stderr,"Fatal error:  out of memory\n");
                    }
                else
                  if (szText=(char *)
                   Malloc(1+strlen("The passage ")
                   +strlen(m_szDirection[nDirection])
                   +strlen(" is guarded by a ")
                   +strlen(pTreasure->szGuardian)+strlen(".")))
                    {
                      strcpy(szText,"The passage ");
                      strcat(szText,m_szDirection[nDirection]);
                      strcat(szText," is guarded by a ");
                      strcat(szText,pTreasure->szGuardian);
                      strcat(szText,".");
                      DisplayText(stdout,szText,TRUE,79);
                      Free(szText);
                    }
                  else
                    {
                      *bErr=TRUE;
                      fprintf(stderr,"Fatal error:  out of memory\n");
                    }
              }
            if (pRoomAlongPassage->Adjacent.pPassage[nDirection]->pRoom1
             == pRoomAlongPassage)
              pRoomAlongPassage
               =pRoomAlongPassage->Adjacent.pPassage[nDirection]->pRoom2;
            else
              pRoomAlongPassage
               =pRoomAlongPassage->Adjacent.pPassage[nDirection]->pRoom1;
            szDescription=pRoomAlongPassage->szDescription;
          }
      }
    return;
  }

static void ListTreasures(
 RoomNodePtr     pRoom,
 int             nTreasures,
 TreasureNodePtr pTreasure,
 int             *bErr)
  {
    int  nFirstChar;
    int  nTreasure;
    char *szText;

    for (nTreasure=nTreasures; nTreasure--;)
      if (pTreasure[nTreasure].pRoomTreasure == pRoom)
        {
          nFirstChar=toupper((int) *(pTreasure[nTreasure].szTreasure));
          if ((nFirstChar == (int) 'A')
          ||  (nFirstChar == (int) 'E')
          ||  (nFirstChar == (int) 'I')
          ||  (nFirstChar == (int) 'O')
          ||  (nFirstChar == (int) 'U'))
            if (szText=(char *) Malloc(1+strlen("There is an ")
             +strlen(pTreasure[nTreasure].szTreasure)+strlen(" here.")))
              {
                strcpy(szText,"There is an ");
                strcat(szText,pTreasure[nTreasure].szTreasure);
                strcat(szText," here.");
                DisplayText(stdout,szText,TRUE,79);
                Free(szText);
              }
            else
              {
                *bErr=TRUE;
                fprintf(stderr,"Fatal error:  out of memory\n");
              }
          else
            if (szText=(char *) Malloc(1+strlen("There is a ")
             +strlen(pTreasure[nTreasure].szTreasure)+strlen(" here.")))
              {
                strcpy(szText,"There is a ");
                strcat(szText,pTreasure[nTreasure].szTreasure);
                strcat(szText," here.");
                DisplayText(stdout,szText,TRUE,79);
                Free(szText);
              }
            else
              {
                *bErr=TRUE;
                fprintf(stderr,"Fatal error:  out of memory\n");
              }
        }
    return;
  }

static void ListWeapons(
 RoomNodePtr     pRoom,
 int             nTreasures,
 TreasureNodePtr pTreasure,
 int             *bErr)
  {
    int  nFirstChar;
    int  nTreasure;
    char *szText;

    for (nTreasure=nTreasures; nTreasure--;)
      if (pTreasure[nTreasure].pRoomWeapon == pRoom)
        {
          nFirstChar=toupper((int) *(pTreasure[nTreasure].szWeapon));
          if ((nFirstChar == (int) 'A')
          ||  (nFirstChar == (int) 'E')
          ||  (nFirstChar == (int) 'I')
          ||  (nFirstChar == (int) 'O')
          ||  (nFirstChar == (int) 'U'))
            if (szText=(char *) Malloc(1+strlen("There is an ")
             +strlen(pTreasure[nTreasure].szWeapon)+strlen(" here.")))
              {
                strcpy(szText,"There is an ");
                strcat(szText,pTreasure[nTreasure].szWeapon);
                strcat(szText," here.");
                DisplayText(stdout,szText,TRUE,79);
                Free(szText);
              }
            else
              {
                *bErr=TRUE;
                fprintf(stderr,"Fatal error:  out of memory\n");
              }
          else
            if (szText=(char *) Malloc(1+strlen("There is a ")
             +strlen(pTreasure[nTreasure].szWeapon)+strlen(" here.")))
              {
                strcpy(szText,"There is a ");
                strcat(szText,pTreasure[nTreasure].szWeapon);
                strcat(szText," here.");
                DisplayText(stdout,szText,TRUE,79);
                Free(szText);
              }
            else
              {
                *bErr=TRUE;
                fprintf(stderr,"Fatal error:  out of memory\n");
              }
        }
    return;
  }

static void Carry(
 RoomNodePtr     pRoom,
 int             nTreasures,
 TreasureNodePtr pTreasure)
  {
    int nPickedUp;
    int nTreasure;

    nPickedUp=0;
    for (nTreasure=nTreasures; nTreasure--;)
      {
        if (pTreasure[nTreasure].pRoomWeapon == pRoom)
          {
            ++nPickedUp;
            pTreasure[nTreasure].pRoomWeapon=NULL;
          }
        if (pTreasure[nTreasure].pRoomTreasure == pRoom)
          {
            ++nPickedUp;
            pTreasure[nTreasure].pRoomTreasure=NULL;
          }
      }
    if (nPickedUp == 0)
      DisplayText(stdout,"There is nothing to carry.",TRUE,79);
    return;
  }

static void Leave(
 RoomNodePtr     pRoom,
 int             nTreasures,
 TreasureNodePtr pTreasure)
  {
    int nLeft;
    int nTreasure;

    nLeft=0;
    for (nTreasure=nTreasures; nTreasure--;)
      if (pTreasure[nTreasure].pRoomTreasure == NULL)
        {
          ++nLeft;
          pTreasure[nTreasure].pRoomTreasure=pRoom;
        }
    if (nLeft == 0)
      DisplayText(stdout,"You aren\'t carrying anything to leave.",TRUE,79);
    return;
  }

static void Inventory(
 int             nChokepoints,
 int             nTreasures,
 TreasureNodePtr pTreasure,
 int             *bErr)
  {
    int  bFirstLine;
    int  nTreasure;
    char *szText;

    bFirstLine=TRUE;
    for (nTreasure=nTreasures; nTreasure--;)
      if (pTreasure[nTreasure].pRoomTreasure == NULL)
        {
          if (bFirstLine)
            {
              DisplayText(stdout,"You are carrying the following:",TRUE,79);
              bFirstLine=FALSE;
            }
          if (szText=(char *) Malloc(1+strlen("     ")
           +strlen(pTreasure[nTreasure].szTreasure)))
            {
              strcpy(szText,"     ");
              strcat(szText,pTreasure[nTreasure].szTreasure);
              DisplayText(stdout,szText,FALSE,79);
              Free(szText);
            }
          else
            {
              *bErr=TRUE;
              fprintf(stderr,"Fatal error:  out of memory\n");
            }
        }
    for (nTreasure=nChokepoints; nTreasure--;)
      if ((pTreasure[nTreasure].pRoomWeapon == NULL)
      &&  (! pTreasure[nTreasure].bSlain))
        {
          if (bFirstLine)
            {
              DisplayText(stdout,"You are carrying the following:",TRUE,79);
              bFirstLine=FALSE;
            }
          if (szText=(char *) Malloc(1+strlen("     ")
           +strlen(pTreasure[nTreasure].szWeapon)))
            {
              strcpy(szText,"     ");
              strcat(szText,pTreasure[nTreasure].szWeapon);
              DisplayText(stdout,szText,FALSE,79);
              Free(szText);
            }
          else
            {
              *bErr=TRUE;
              fprintf(stderr,"Fatal error:  out of memory\n");
            }
        }
    if (bFirstLine)
      DisplayText(stdout,"You aren\'t carrying anything.",TRUE,79);
    else
      DisplayText(stdout," ",FALSE,79);
    return;
  }

static void Move(
 int         *nMoves,
 int         nDirection,
 RoomNodePtr *pRoom,
 int         *bErr)
  {
    int         bGuarded;
    RoomNodePtr pRoomAlongPassage;
    char        *szDescription;
    char        *szText;

    if (((*pRoom)->Adjacent.pPassage[nDirection])
    &&  (! (*pRoom)->Adjacent.pPassage[nDirection]->bBlocked))
      {
        bGuarded=FALSE;
        szDescription=NULL;
        pRoomAlongPassage=*pRoom;
        while((pRoomAlongPassage->Adjacent.pPassage[nDirection])
        &&    (szDescription == NULL))
          {
            if (pRoomAlongPassage->Adjacent.pPassage[
             nDirection]->pTreasureGuardian)
              if (pRoomAlongPassage->Adjacent.pPassage[
               nDirection]->pTreasureGuardian->pRoomWeapon)
                if (szText=(char *)
                 Malloc(1+strlen("You carry nothing to overcome the ")
                 +strlen(pRoomAlongPassage->Adjacent.pPassage[
                 nDirection]->pTreasureGuardian->szGuardian)+strlen(".")))
                  {
                    bGuarded=TRUE;
                    strcpy(szText,"You carry nothing to overcome the ");
                    strcat(szText,pRoomAlongPassage->Adjacent.pPassage[
                     nDirection]->pTreasureGuardian->szGuardian);
                    strcat(szText,".");
                    DisplayText(stdout,szText,TRUE,79);
                    Free(szText);
                  }
                else
                  {
                    *bErr=TRUE;
                    fprintf(stderr,"Fatal error:  out of memory\n");
                  }
              else
                if (szText=(char *) Malloc(1+strlen("Your ")
                 +strlen(pRoomAlongPassage->Adjacent.pPassage[
                 nDirection]->pTreasureGuardian->szWeapon)
                 +strlen(" overcomes the ")
                 +strlen(pRoomAlongPassage->Adjacent.pPassage[
                 nDirection]->pTreasureGuardian->szGuardian)+strlen(".")))
                  {
                    strcpy(szText,"Your ");
                    strcat(szText,pRoomAlongPassage->Adjacent.pPassage[
                     nDirection]->pTreasureGuardian->szWeapon);
                    strcat(szText," overcomes the ");
                    strcat(szText,pRoomAlongPassage->Adjacent.pPassage[
                     nDirection]->pTreasureGuardian->szGuardian);
                    strcat(szText,".");
                    DisplayText(stdout,szText,TRUE,79);
                    Free(szText);
                    pRoomAlongPassage->Adjacent.pPassage[
                     nDirection]->pTreasureGuardian->bSlain=TRUE;
                    pRoomAlongPassage->Adjacent.pPassage[
                     nDirection]->pTreasureGuardian=NULL;
                  }
                else
                  {
                    *bErr=TRUE;
                    fprintf(stderr,"Fatal error:  out of memory\n");
                  }
            if (pRoomAlongPassage->Adjacent.pPassage[nDirection]->pRoom1
             == pRoomAlongPassage)
              pRoomAlongPassage
               =pRoomAlongPassage->Adjacent.pPassage[nDirection]->pRoom2;
            else
              pRoomAlongPassage
               =pRoomAlongPassage->Adjacent.pPassage[nDirection]->pRoom1;
            szDescription=pRoomAlongPassage->szDescription;
          }
        if (! bGuarded)
          {
            ++(*nMoves);
            pRoomAlongPassage->bVisited=TRUE;
            *pRoom=pRoomAlongPassage;
          }
      }
    else
      DisplayText(stdout,"You can\'t go that way.",TRUE,79);
    return;
  }

static void Points(
 int             nRooms,
 RoomNodePtr     pRoomEntrance,
 RoomNodePtr     *ppRoomWithName,
 int             nTreasures,
 TreasureNodePtr pTreasure,
 int             nMoves,
 int             bRank,
 int             *bErr)
  {
    int  nRoom;
    int  nRoomsVisited;
    int  nScore;
    char szRooms [12];
    char szRoomsVisited [12];
    int  nTreasure;
    int  nTreasuresCarried;
    int  nTreasuresRecovered;
    char szMoves [12];
    char szScore [12];
    char *szText;
    char szTreasures [12];
    char szTreasuresCarried [12];
    char szTreasuresRecovered [12];

    sprintf(&szRooms[0],"%d",nRooms+1);
    nScore=0;
    nTreasuresRecovered=0;
    nTreasuresCarried=0;
    nRoomsVisited=0;
    for (nRoom=0; nRoom <= nRooms; ++nRoom)
      if (ppRoomWithName[nRoom]->bVisited)
        ++nRoomsVisited;
    sprintf(&szRoomsVisited[0],"%d",nRoomsVisited);
    sprintf(&szMoves[0],"%d",nMoves);
    if (szText=(char *) Malloc(1+strlen("You have moved ")
     +strlen(&szMoves[0])+strlen(" times to visit ")
     +strlen(&szRoomsVisited[0])+strlen(" of ")+strlen(&szRooms[0])
     +strlen(" locations.")))
      {
        strcpy(szText,"You have moved ");
        strcat(szText,&szMoves[0]);
        strcat(szText," times to visit ");
        strcat(szText,&szRoomsVisited[0]);
        strcat(szText," of ");
        strcat(szText,&szRooms[0]);
        strcat(szText," locations.");
        DisplayText(stdout,szText,FALSE,79);
        Free(szText);
      }
    else
      {
        *bErr=TRUE;
        fprintf(stderr,"Fatal error:  out of memory\n");
      }
    if (! *bErr)
      {
        sprintf(&szTreasures[0],"%d",nTreasures);
        for (nTreasure=nTreasures; nTreasure--;)
          if (pTreasure[nTreasure].pRoomTreasure == NULL)
            ++nTreasuresCarried;
        sprintf(&szTreasuresCarried[0],"%d",nTreasuresCarried);
        if (szText=(char *) Malloc(1+strlen("You hold ")
         +strlen(&szTreasuresCarried[0])+strlen(" of ")+strlen(&szTreasures[0])
         +strlen(" treasures.")))
          {
            strcpy(szText,"You hold ");
            strcat(szText,&szTreasuresCarried[0]);
            strcat(szText," of ");
            strcat(szText,&szTreasures[0]);
            strcat(szText," treasures.");
            DisplayText(stdout,szText,FALSE,79);
            Free(szText);
          }
        else
          {
            *bErr=TRUE;
            fprintf(stderr,"Fatal error:  out of memory\n");
          }
      }
    if (! *bErr)
      {
        for (nTreasure=nTreasures; nTreasure--;)
          if (pTreasure[nTreasure].pRoomTreasure == pRoomEntrance)
            ++nTreasuresRecovered;
        sprintf(&szTreasuresRecovered[0],"%d",nTreasuresRecovered);
        if (szText=(char *) Malloc(1+strlen("You have returned ")
         +strlen(&szTreasuresRecovered[0])
         +strlen(" of ")+strlen(&szTreasures[0])
         +strlen(" treasures to the entrance of the mine.")))
          {
            strcpy(szText,"You have returned ");
            strcat(szText,&szTreasuresRecovered[0]);
            strcat(szText," of ");
            strcat(szText,&szTreasures[0]);
            strcat(szText," treasures to the entrance of the mine.");
            DisplayText(stdout,szText,FALSE,79);
            Free(szText);
          }
        else
          {
            *bErr=TRUE;
            fprintf(stderr,"Fatal error:  out of memory\n");
          }
      }
    if (! *bErr)
      {
        DisplayText(stdout," ",FALSE,79);
        nScore=25*nRoomsVisited/(nRooms+1)+75*nTreasuresRecovered/nTreasures
         +45*nTreasuresCarried/nTreasures;
        if (nRoomsVisited > 5*nRooms)
          {
            nScore=nScore-nRoomsVisited/(5*nRooms);
            if (nScore < 0)
              nScore=0;
          }
        sprintf(&szScore[0],"%d",nScore);
        if (szText=(char *) Malloc(1+strlen("You have scored ")
         +strlen(&szTreasuresRecovered[0])
         +strlen(" of 100 points.")))
          {
            strcpy(szText,"You have scored ");
            strcat(szText,&szScore[0]);
            strcat(szText," of 100 points.");
            DisplayText(stdout,szText,TRUE,79);
            Free(szText);
          }
        else
          {
            *bErr=TRUE;
            fprintf(stderr,"Fatal error:  out of memory\n");
          }
      }
    if (! *bErr)
      {
        if (bRank)
          {
            if (nScore < 25)
              DisplayText(stdout,
               "Your score ranks you as a beginner.",FALSE,79);
            else if (nScore < 50)
              DisplayText(stdout,
               "Your score ranks you as a novice adventurer.",FALSE,79);
            else if (nScore < 75)
              DisplayText(stdout,
               "Your score ranks you as a seasoned explorer.",FALSE,79);
            else if (nScore < 100)
              DisplayText(stdout,
               "Your score ranks you as a grissly old prospector.",FALSE,79);
            else
              DisplayText(stdout,
               "Your score ranks you as an expert treasure hunter; "
               "there is no higher rating.",FALSE,79);
          }
      }
    return;
  }

static void PlayGame(
 int             nRooms,
 RoomNodePtr     pRoomEntrance,
 RoomNodePtr     *ppRoomWithName,
 int             nChokepoints,
 int             nTreasures,
 TreasureNodePtr pTreasure,
 int             *bErr)
  {
    int         nCommand;
    int         nMoves;
    RoomNodePtr pRoom;

    nMoves=0;
    pRoom=pRoomEntrance;
    nCommand=(int) 'Q';
    do
      {
        DisplayText(stdout,pRoom->szDescription,TRUE,79);
        ListTreasures(pRoom,nTreasures,pTreasure,bErr);
        if (! *bErr)
          ListWeapons(pRoom,nTreasures,pTreasure,bErr);
        if (! *bErr)
          ListPassages(pRoom,bErr);
        if (! *bErr)
          {
            nCommand=Command();
            switch (nCommand)
              {
                case (int) 'N':
                  Move(&nMoves,0,&pRoom,bErr);
                  break;
                case (int) 'S':
                  Move(&nMoves,1,&pRoom,bErr);
                  break;
                case (int) 'E':
                  Move(&nMoves,2,&pRoom,bErr);
                  break;
                case (int) 'W':
                  Move(&nMoves,3,&pRoom,bErr);
                  break;
                case (int) 'U':
                  Move(&nMoves,4,&pRoom,bErr);
                  break;
                case (int) 'D':
                  Move(&nMoves,5,&pRoom,bErr);
                  break;
                case (int) 'C':
                  Carry(pRoom,nTreasures,pTreasure);
                  break;
                case (int) 'I':
                  Inventory(nChokepoints,nTreasures,pTreasure,bErr);
                  break;
                case (int) 'L':
                  Leave(pRoom,nTreasures,pTreasure);
                  break;
                case (int) 'P':
                  Points(nRooms,pRoomEntrance,ppRoomWithName,nTreasures,
                   pTreasure,nMoves,FALSE,bErr);
                  break;
                case (int) 'O':
                  WayOut(nRooms,pRoom,pRoomEntrance,ppRoomWithName,nTreasures,
                   pTreasure,bErr);
                  break;
                case (int) 'Q':
                  Points(nRooms,pRoomEntrance,ppRoomWithName,nTreasures,
                   pTreasure,nMoves,TRUE,bErr);
                  break;
                case (int) 'H':
                  DisplayHelp(stdout,TRUE,79);
                  break;
                default:
                  fprintf(stdout,"I don\'t recognize that command.\n\n");
                  ListCommands(stdout,79);
                  break;
              }
          }
      }
    while ((! *bErr) && (nCommand != (int) 'Q'));
    return;
  }

int main(
 int  nArgs,
 char **pszArg)
  {
    int             bErr;
    int             nChokepoints;
    int             nRooms;
    int             nTreasures;
    char            **pszRoom;
    TreasureNodePtr pTreasure;
    RoomNodePtr     pRoomEntrance;
    RoomNodePtr     *ppRoomWithName;

    bErr=FALSE;
#if DEBUG_MEMORY
    bMemoryError=FALSE;
    pMemoryHead=NULL;
    remove("memory.txt");
#endif
    if (nArgs == 2)
      {
        pszRoom=NULL;
        pTreasure=NULL;
        if (pRoomEntrance=(RoomNodePtr) Malloc(sizeof(struct RoomNode)))
          {
            if (pRoomEntrance->szDescription=(char *)
             Malloc(1+strlen("You\'re in the entrance to the mine.")))
              {
                strcpy(pRoomEntrance->szDescription,
                 "You\'re in the entrance to the mine.");
                pRoomEntrance->bMined=TRUE;
                pRoomEntrance->nChokepoint=-1;
                pRoomEntrance->bVisited=TRUE;
                pRoomEntrance->Adjacent.Passage.pPassageNorth=NULL;
                pRoomEntrance->Adjacent.Passage.pPassageSouth=NULL;
                pRoomEntrance->Adjacent.Passage.pPassageEast=NULL;
                pRoomEntrance->Adjacent.Passage.pPassageWest=NULL;
                pRoomEntrance->Adjacent.Passage.pPassageUp=NULL;
                pRoomEntrance->Adjacent.Passage.pPassageDown=NULL;
                SeedRandomNumberGenerator(*(pszArg+1));
                nRooms=0;
                GetRoomDescriptions("rooms.dat",&nRooms,&pszRoom,&bErr);
                if (! bErr)
                  ShuffleRoomDescriptions(nRooms,pszRoom);
                nTreasures=0;
                if (! bErr)
                  GetTreasureDescriptions("treasure.dat",&nTreasures,&pTreasure,
                   &bErr);
                nChokepoints=0;
                if (! bErr)
                  {
                    if (ppRoomWithName=(RoomNodePtr *)
                     Malloc((nRooms+1)*sizeof(RoomNodePtr)))
                      {
                        ExcavateMine(nRooms,pszRoom,pRoomEntrance,nTreasures,
                         pTreasure,&nChokepoints,ppRoomWithName,&bErr);
                        if (! bErr)
                          HideTreasuresAndWeapons(nChokepoints,nRooms,
                           ppRoomWithName,nTreasures,pTreasure);
                        if (! bErr)
                          PlayGame(nRooms,pRoomEntrance,ppRoomWithName,
                           nChokepoints,nTreasures,pTreasure,&bErr);
                        Free(ppRoomWithName);
                      }
                    else
                      {
                        bErr=TRUE;
                        fprintf(stderr,"Fatal error:  out of memory\n");
                      }
                  }
                if (pRoomEntrance)
                  FreeRoomsAndPassages(pRoomEntrance);
                if (pTreasure)
                  FreeTreasureDescriptions(nTreasures,&pTreasure);
                if (pszRoom)
                  FreeRoomDescriptions(nRooms,&pszRoom);
              }
            else
              {
                bErr=TRUE;
                Free(pRoomEntrance);
                fprintf(stderr,"Fatal error:  out of memory\n");
              }
          }
        else
          {
            bErr=TRUE;
            fprintf(stderr,"Fatal error:  out of memory\n");
          }
      }
    else
      {
        bErr=TRUE;
        fprintf(stderr,
         "%s lets you explore a mine.  The object of the game is to\n"
         "  visit all of the rooms and return all the treasures to the\n"
         "  entrance without making too many moves.\n\n"
         "  Usage:  %s <MineNumber>\n"
         "    where a each mine number gives a different mine.\n\n"
         "    The program reads descriptions of the rooms from the file\n"
         "    \"rooms.dat\" and descriptions of treasures, guardians, and\n"
         "    weapons from \"treasure.dat\"; see \"notes.txt\" for details.\n\n"
         "Example:  %s 1\n\n",*pszArg,*pszArg,*pszArg);
        DisplayHelp(stderr,FALSE,61);
      }
#if DEBUG_MEMORY
    ReportMemoryLeaks();
#endif
    return bErr;
  }

