#include <minix/config.h>
#include <stdio.h>
#include <sys/types.h>
#include <fcntl.h>
#undef NULL
#include <unistd.h>
#include <sgtty.h>
#include <errno.h>
#include "/usr/src/kernel/etherframe.h"
#include "/usr/src/kernel/etherproto.h"

#define le2be(s) ((((unsigned short)(s)&0x00ff)<<8)|((unsigned short)(s)>>8))
#define be2le(s) ((((unsigned short)(s)&0x00ff)<<8)|((unsigned short)(s)>>8))

void eth_send();
void eth_receive();
void eth_ioctl();
void print_ethaddr();
void get_ethaddr();
void eth_dump();
void myflush();

#define DEVNAME "/dev/ether"

Eth_frame frame;
Eth_addr myaddr;

void main(argc, argv)
int argc;
char *argv[];
{
  char c, *devname=DEVNAME;
  int fd, ret;
  
  if(argc>1)
    devname=argv[1];

  printf("Ethernet frames sender/receiver is using file %s\n", devname);
  fd=open(devname, O_RDONLY);
  if(fd==-1)
  {
    printf("could not open %s for reading\n", devname);
    exit(1);
  }
  ret=ioctl(fd, EIOCGETADDR, &myaddr);
  if (ret<0)
  {
    perror("IOCTL call (EIOCGETADDR) failed");
    exit(1);
  }
  else
  {
    printf("My ethernet address is ");
    print_ethaddr(&myaddr);
    putchar('\n');
  }
  close(fd);

  /* main loop */
  do
  {
    printf("(s)end, (r)eceive, (i)octl, (q)uit ? ");
    c=getchar(); 
    myflush();
    switch(c)
    {
      case 's': eth_send(devname); break;
      case 'r': eth_receive(devname); break;
      case 'i': eth_ioctl(devname); break;
      case 'q': break;
      default: break;
    }
  } while( c!='q' );
  exit(0);
}


void eth_send(devname)
char *devname;
{
  int fd, ret, size, count;
  unsigned proto;

  fd=open(devname, O_WRONLY);
  if(fd==-1)
  {
    printf("could not open %s for writing\n", devname);
    exit(1);
  }
/*
  printf("Enter source ");
  get_ethaddr(&frame.ef_header.eh_srcaddr);
*/
  frame.ef_header.eh_srcaddr = myaddr;	/* source address (structure copy) */
  printf("Enter destination ");
  get_ethaddr(&frame.ef_header.eh_dstaddr);
  printf("Enter protocol (in little endian): ");
  scanf("%x", &proto);
  myflush();
  frame.ef_header.eh_proto=le2be(proto);

  print_ethaddr(&frame.ef_header.eh_srcaddr);
  printf(" -> ");
  print_ethaddr(&frame.ef_header.eh_dstaddr);
  printf(", proto %04x\n", proto);

  for(;;)
  {
    printf("Now enter data to send, DEL to stop:\n");
    gets(frame.ef_data);
    count=strlen(frame.ef_data);
    if(count==0)
      continue;		/* no empty lines, please */
    count++;
    size=sizeof(Eth_header)+count;
    printf("Sending %d bytes of data, (total %d): ", count, size);
    ret=write(fd, &frame, size);
    if(ret<size)
      printf("error, returned %d", ret);
    else
      printf("OK, %d bytes sent", ret);
    printf("\n\n");
  }
  /* NEVER */
  close(fd);
}


void eth_receive(devname)
char *devname;
{
  int fd, ret, display_data, omode, loopcnt=0;

  omode=O_RDONLY;
  printf("(b)locking or (n)on-blocking read ? ");
  ret=getchar();
  myflush();
  if(ret=='n')
    omode |= O_NONBLOCK;

  fd=open(devname, omode);
  if(fd==-1)
  {
    printf("could not open %s for reading\n", devname);
    exit(1);
  }

  printf("display data ? ");
  ret=getchar();
  myflush();
  display_data=(ret=='y');

  printf("Receiving frames, press DEL to stop\n");
  for(;;)
  {
    ret=read(fd, &frame, sizeof(Eth_frame));
    if(ret<0)
    {
      if ((omode & O_NONBLOCK) && (errno == EAGAIN))
      {
	printf("\rno frame, looping: %d", ++loopcnt);
	continue;
      }
      printf("error reading %s\n", devname);
      break;
    }
    if ((omode & O_NONBLOCK) && loopcnt>0)
      printf("\n");
    if(ret==0)
    {
      printf("strange: ret==0\n");
      continue;
    }
    else if(ret<=sizeof(Eth_header))
      printf("strange: received %d bytes\n", ret);
    else
      eth_dump(&frame, ret, display_data);
    loopcnt = 0;
    printf("\n");
  }
  /* NEVER */
  close(fd);
}


void eth_ioctl(devname)
char *devname;
{
  int fd, ret, c, i;
  struct ethfilter ef;
  unsigned filters[NR_FILTERS];

  fd=open(devname, O_RDONLY);
  if(fd==-1)
  {
    printf("could not open %s for reading\n", devname);
    exit(1);
  }

  printf("(1) get address, (2) get filter, (3) set filter ? ");
  c=getchar(); 
  myflush();

  switch(c)
  {
     case '1':
       ret=ioctl(fd, EIOCGETADDR, &myaddr);
       if (ret<0)
       {
         perror("IOCTL call (EIOCGETADDR) failed");
       }
       else
       {
         printf("my ethernet address is ");
         print_ethaddr(&myaddr);
	 putchar('\n');
       }
       break;

     case '2':
       ef.ef_flags=FF_ACCEPT;	/* just for order */
       ef.ef_count=sizeof(filters)/sizeof(filters[0]);
       ef.ef_ptr=filters;
       ret=ioctl(fd, EIOCGETFILTER, &ef);
       if (ret<0)
         perror("IOCTL call (EIOCGETFILTER) failed");
       else
       {
         printf("number of filters = %d, mode = %02x   ", 
		ef.ef_count, ef.ef_flags);
	 for(i=0 ; i<ef.ef_count ; i++)
	   printf("%04x ", filters[i]);
	 putchar('\n');
       }
       break;

     case '3':
       printf("(n)ormal (i.e. accept) or (r)everse (i.e. reject) mode ? ");
       c=getchar();
       myflush();
       ef.ef_flags=(c=='r') ? FF_REJECT : FF_ACCEPT;
       printf("number of filters (0..%d) ? ", NR_FILTERS);
       scanf("%d", &i);
       myflush();
       if (i<0) i=0;
       if (i>NR_FILTERS) i=NR_FILTERS;
       ef.ef_count=i;
       for (i=0 ; i<ef.ef_count ; i++)
       {
	 printf("filters[%d] = ", i);
	 scanf("%x", filters+i);
	 myflush();
       }
       ef.ef_ptr=filters;
       ret=ioctl(fd, EIOCSETFILTER, &ef);
       if (ret<0)
         perror("IOCTL call (EIOCSETFILTER) failed");
       else
       {
         printf("new filters set\n");
       }
       break;

     default: break;
  }

  close(fd);
}

/* print ethernet address */
void print_ethaddr(addr)
Eth_addr *addr;
{
 int i;

 for(i=0 ; i<=4 ; i++)
   printf("%02x:", addr->ea[i] & 0x00ff);
 printf("%02x", addr->ea[5] & 0x00ff);
}

/* get ethernet address */
void get_ethaddr(addr)
Eth_addr *addr;
{
  int ret=0, i;
  int buf[sizeof(Eth_addr)];

  while(ret!=6)
  {
    printf("ethernet address: ");
    ret=scanf("%02x:%02x:%02x:%02x:%02x:%02x",
	       &buf[0], &buf[1], &buf[2],
	       &buf[3], &buf[4], &buf[5]);
    myflush();
  }
  for(i=0 ; i<sizeof(Eth_addr) ; i++)
    addr->ea[i]=(unsigned char)buf[i];

#if 2*2 != 4	/* debug */
  printf("%02x:%02x:%02x:%02x:%02x:%02x",
	addr->ea[0], addr->ea[1], addr->ea[2],
	addr->ea[3], addr->ea[4], addr->ea[5]);
#endif
}


void eth_dump(frame, size, display_data)
Eth_frame *frame;
int size, display_data;
{
  int data_size;

  data_size=size-sizeof(Eth_header);
  print_ethaddr(&frame->ef_header.eh_srcaddr);
  printf(" -> ");
  print_ethaddr(&frame->ef_header.eh_dstaddr);
  printf(", proto %04x, data size %d\n",
	 be2le(frame->ef_header.eh_proto), data_size);
  if(display_data)
    printf("%s\n", frame->ef_data);
}

void myflush()
{
  while(getchar()!='\n')
   ;
}

