// SPDX-License-Identifier: ISC
/*
 * Copyright (c) 2012-2017 Qualcomm Atheros, Inc.
 * Copyright (c) 2018-2019, The Linux Foundation. All rights reserved.
 */

#include <linux/module.h>
#include <linux/debugfs.h>
#include <linux/seq_file.h>
#include <linux/pci.h>
#include <linux/rtnetlink.h>
#include <linux/power_supply.h>
#include "wil6210.h"
#include "wmi.h"
#include "txrx.h"
#include "pmc.h"

/* Nasty hack. Better have per device instances */
static u32 mem_addr;
static u32 dbg_txdesc_index;
static u32 dbg_ring_index; /* 24+ for Rx, 0..23 for Tx */
static u32 dbg_status_msg_index;
/* 0..wil->num_rx_status_rings-1 for Rx, wil->tx_sring_idx for Tx */
static u32 dbg_sring_index;

enum dbg_off_type {
	doff_u32 = 0,
	doff_x32 = 1,
	doff_ulong = 2,
	doff_io32 = 3,
	doff_u8 = 4
};

/* offset to "wil" */
struct dbg_off {
	const char *name;
	umode_t mode;
	ulong off;
	enum dbg_off_type type;
};

static void wil_print_desc_edma(struct seq_file *s, struct wil6210_priv *wil,
				struct wil_ring *ring,
				char _s, char _h, int idx)
{
	u8 num_of_descs;
	bool has_skb = false;

	if (ring->is_rx) {
		struct wil_rx_enhanced_desc *rx_d =
			(struct wil_rx_enhanced_desc *)
			&ring->va[idx].rx.enhanced;
		u16 buff_id = le16_to_cpu(rx_d->mac.buff_id);

		if (wil->rx_buff_mgmt.buff_arr &&
		    wil_val_in_range(buff_id, 0, wil->rx_buff_mgmt.size))
			has_skb = wil->rx_buff_mgmt.buff_arr[buff_id].skb;
		seq_printf(s, "%c", (has_skb) ? _h : _s);
	} else {
		struct wil_tx_enhanced_desc *d =
			(struct wil_tx_enhanced_desc *)
			&ring->va[idx].tx.enhanced;

		num_of_descs = (u8)d->mac.d[2];
		has_skb = ring->ctx && ring->ctx[idx].skb;
		if (num_of_descs >= 1)
			seq_printf(s, "%c", has_skb ? _h : _s);
		else
			/* num_of_descs == 0, it's a frag in a list of descs */
			seq_printf(s, "%c", has_skb ? 'h' : _s);
	}
}

static void wil_print_ring(struct seq_file *s, struct wil6210_priv *wil,
			   const char *name, struct wil_ring *ring,
			   char _s, char _h)
{
	void __iomem *x;
	u32 v;

	seq_printf(s, "RING %s = {\n", name);
	seq_printf(s, "  pa     = %pad\n", &ring->pa);
	seq_printf(s, "  va     = 0x%p\n", ring->va);
	seq_printf(s, "  size   = %d\n", ring->size);
	if (wil->use_enhanced_dma_hw && ring->is_rx)
		seq_printf(s, "  swtail = %u\n", *ring->edma_rx_swtail.va);
	else
		seq_printf(s, "  swtail = %d\n", ring->swtail);
	seq_printf(s, "  swhead = %d\n", ring->swhead);
	if (wil->use_enhanced_dma_hw) {
		int ring_id = ring->is_rx ?
			WIL_RX_DESC_RING_ID : ring - wil->ring_tx;
		/* SUBQ_CONS is a table of 32 entries, one for each Q pair.
		 * lower 16bits are for even ring_id and upper 16bits are for
		 * odd ring_id
		 */
		x = wmi_addr(wil, RGF_DMA_SCM_SUBQ_CONS + 4 * (ring_id / 2));
		v = readl_relaxed(x);

		v = (ring_id % 2 ? (v >> 16) : (v & 0xffff));
		seq_printf(s, "  hwhead = %u\n", v);
	}
	seq_printf(s, "  hwtail = [0x%08x] -> ", ring->hwtail);
	x = wmi_addr(wil, ring->hwtail);
	if (x) {
		v = readl(x);
		seq_printf(s, "0x%08x = %d\n", v, v);
	} else {
		seq_puts(s, "???\n");
	}

	if (ring->va && (ring->size <= (1 << WIL_RING_SIZE_ORDER_MAX))) {
		uint i;

		for (i = 0; i < ring->size; i++) {
			if ((i % 128) == 0 && i != 0)
				seq_puts(s, "\n");
			if (wil->use_enhanced_dma_hw) {
				wil_print_desc_edma(s, wil, ring, _s, _h, i);
			} else {
				volatile struct vring_tx_desc *d =
					&ring->va[i].tx.legacy;
				seq_printf(s, "%c", (d->dma.status & BIT(0)) ?
					   _s : (ring->ctx[i].skb ? _h : 'h'));
			}
		}
		seq_puts(s, "\n");
	}
	seq_puts(s, "}\n");
}

static int ring_show(struct seq_file *s, void *data)
{
	uint i;
	struct wil6210_priv *wil = s->private;

	wil_print_ring(s, wil, "rx", &wil->ring_rx, 'S', '_');

	for (i = 0; i < ARRAY_SIZE(wil->ring_tx); i++) {
		struct wil_ring *ring = &wil->ring_tx[i];
		struct wil_ring_tx_data *txdata = &wil->ring_tx_data[i];

		if (ring->va) {
			int cid = wil->ring2cid_tid[i][0];
			int tid = wil->ring2cid_tid[i][1];
			u32 swhead = ring->swhead;
			u32 swtail = ring->swtail;
			int used = (ring->size + swhead - swtail)
				   % ring->size;
			int avail = ring->size - used - 1;
			char name[10];
			char sidle[10];
			/* performance monitoring */
			cycles_t now = get_cycles();
			uint64_t idle = txdata->idle * 100;
			uint64_t total = now - txdata->begin;

			if (total != 0) {
				do_div(idle, total);
				snprintf(sidle, sizeof(sidle), "%3d%%",
					 (int)idle);
			} else {
				snprintf(sidle, sizeof(sidle), "N/A");
			}
			txdata->begin = now;
			txdata->idle = 0ULL;

			snprintf(name, sizeof(name), "tx_%2d", i);

			if (cid < wil->max_assoc_sta)
				seq_printf(s,
					   "\n%pM CID %d TID %d 1x%s BACK([%u] %u TU A%s) [%3d|%3d] idle %s\n",
					   wil->sta[cid].addr, cid, tid,
					   txdata->dot1x_open ? "+" : "-",
					   txdata->agg_wsize,
					   txdata->agg_timeout,
					   txdata->agg_amsdu ? "+" : "-",
					   used, avail, sidle);
			else
				seq_printf(s,
					   "\nBroadcast 1x%s [%3d|%3d] idle %s\n",
					   txdata->dot1x_open ? "+" : "-",
					   used, avail, sidle);

			wil_print_ring(s, wil, name, ring, '_', 'H');
		}
	}

	return 0;
}
DEFINE_SHOW_ATTRIBUTE(ring);

static void wil_print_sring(struct seq_file *s, struct wil6210_priv *wil,
			    struct wil_status_ring *sring)
{
	void __iomem *x;
	int sring_idx = sring - wil->srings;
	u32 v;

	seq_printf(s, "Status Ring %s [ %d ] = {\n",
		   sring->is_rx ? "RX" : "TX", sring_idx);
	seq_printf(s, "  pa     = %pad\n", &sring->pa);
	seq_printf(s, "  va     = 0x%pK\n", sring->va);
	seq_printf(s, "  size   = %d\n", sring->size);
	seq_printf(s, "  elem_size   = %zu\n", sring->elem_size);
	seq_printf(s, "  swhead = %d\n", sring->swhead);
	if (wil->use_enhanced_dma_hw) {
		/* COMPQ_PROD is a table of 32 entries, one for each Q pair.
		 * lower 16bits are for even ring_id and upper 16bits are for
		 * odd ring_id
		 */
		x = wmi_addr(wil, RGF_DMA_SCM_COMPQ_PROD + 4 * (sring_idx / 2));
		v = readl_relaxed(x);

		v = (sring_idx % 2 ? (v >> 16) : (v & 0xffff));
		seq_printf(s, "  hwhead = %u\n", v);
	}
	seq_printf(s, "  hwtail = [0x%08x] -> ", sring->hwtail);
	x = wmi_addr(wil, sring->hwtail);
	if (x) {
		v = readl_relaxed(x);
		seq_printf(s, "0x%08x = %d\n", v, v);
	} else {
		seq_puts(s, "???\n");
	}
	seq_printf(s, "  desc_rdy_pol   = %d\n", sring->desc_rdy_pol);
	seq_printf(s, "  invalid_buff_id_cnt   = %d\n",
		   sring->invalid_buff_id_cnt);

	if (sring->va && (sring->size <= (1 << WIL_RING_SIZE_ORDER_MAX))) {
		uint i;

		for (i = 0; i < sring->size; i++) {
			u32 *sdword_0 =
				(u32 *)(sring->va + (sring->elem_size * i));

			if ((i % 128) == 0 && i != 0)
				seq_puts(s, "\n");
			if (i == sring->swhead)
				seq_printf(s, "%c", (*sdword_0 & BIT(31)) ?
					   'X' : 'x');
			else
				seq_printf(s, "%c", (*sdword_0 & BIT(31)) ?
					   '1' : '0');
		}
		seq_puts(s, "\n");
	}
	seq_puts(s, "}\n");
}

static int srings_show(struct seq_file *s, void *data)
{
	struct wil6210_priv *wil = s->private;
	int i = 0;

	for (i = 0; i < WIL6210_MAX_STATUS_RINGS; i++)
		if (wil->srings[i].va)
			wil_print_sring(s, wil, &wil->srings[i]);

	return 0;
}
DEFINE_SHOW_ATTRIBUTE(srings);

static void wil_seq_hexdump(struct seq_file *s, void *p, int len,
			    const char *prefix)
{
	seq_hex_dump(s, prefix, DUMP_PREFIX_NONE, 16, 1, p, len, false);
}

static void wil_print_mbox_ring(struct seq_file *s, const char *prefix,
				void __iomem *off)
{
	struct wil6210_priv *wil = s->private;
	struct wil6210_mbox_ring r;
	int rsize;
	uint i;

	wil_halp_vote(wil);

	if (wil_mem_access_lock(wil)) {
		wil_halp_unvote(wil);
		return;
	}

	wil_memcpy_fromio_32(&r, off, sizeof(r));
	wil_mbox_ring_le2cpus(&r);
	/*
	 * we just read memory block from NIC. This memory may be
	 * garbage. Check validity before using it.
	 */
	rsize = r.size / sizeof(struct wil6210_mbox_ring_desc);

	seq_printf(s, "ring %s = {\n", prefix);
	seq_printf(s, "  base = 0x%08x\n", r.base);
	seq_printf(s, "  size = 0x%04x bytes -> %d entries\n", r.size, rsize);
	seq_printf(s, "  tail = 0x%08x\n", r.tail);
	seq_printf(s, "  head = 0x%08x\n", r.head);
	seq_printf(s, "  entry size = %d\n", r.entry_size);

	if (r.size % sizeof(struct wil6210_mbox_ring_desc)) {
		seq_printf(s, "  ??? size is not multiple of %zd, garbage?\n",
			   sizeof(struct wil6210_mbox_ring_desc));
		goto out;
	}

	if (!wmi_addr(wil, r.base) ||
	    !wmi_addr(wil, r.tail) ||
	    !wmi_addr(wil, r.head)) {
		seq_puts(s, "  ??? pointers are garbage?\n");
		goto out;
	}

	for (i = 0; i < rsize; i++) {
		struct wil6210_mbox_ring_desc d;
		struct wil6210_mbox_hdr hdr;
		size_t delta = i * sizeof(d);
		void __iomem *x = wil->csr + HOSTADDR(r.base) + delta;

		wil_memcpy_fromio_32(&d, x, sizeof(d));

		seq_printf(s, "  [%2x] %s %s%s 0x%08x", i,
			   d.sync ? "F" : "E",
			   (r.tail - r.base == delta) ? "t" : " ",
			   (r.head - r.base == delta) ? "h" : " ",
			   le32_to_cpu(d.addr));
		if (0 == wmi_read_hdr(wil, d.addr, &hdr)) {
			u16 len = le16_to_cpu(hdr.len);

			seq_printf(s, " -> %04x %04x %04x %02x\n",
				   le16_to_cpu(hdr.seq), len,
				   le16_to_cpu(hdr.type), hdr.flags);
			if (len <= MAX_MBOXITEM_SIZE) {
				unsigned char databuf[MAX_MBOXITEM_SIZE];
				void __iomem *src = wmi_buffer(wil, d.addr) +
					sizeof(struct wil6210_mbox_hdr);
				/*
				 * No need to check @src for validity -
				 * we already validated @d.addr while
				 * reading header
				 */
				wil_memcpy_fromio_32(databuf, src, len);
				wil_seq_hexdump(s, databuf, len, "      : ");
			}
		} else {
			seq_puts(s, "\n");
		}
	}
 out:
	seq_puts(s, "}\n");
	wil_mem_access_unlock(wil);
	wil_halp_unvote(wil);
}

static int mbox_show(struct seq_file *s, void *data)
{
	struct wil6210_priv *wil = s->private;
	int ret;

	ret = wil_pm_runtime_get(wil);
	if (ret < 0)
		return ret;

	wil_print_mbox_ring(s, "tx", wil->csr + HOST_MBOX +
		       offsetof(struct wil6210_mbox_ctl, tx));
	wil_print_mbox_ring(s, "rx", wil->csr + HOST_MBOX +
		       offsetof(struct wil6210_mbox_ctl, rx));

	wil_pm_runtime_put(wil);

	return 0;
}
DEFINE_SHOW_ATTRIBUTE(mbox);

static int wil_debugfs_iomem_x32_set(void *data, u64 val)
{
	struct wil_debugfs_iomem_data *d = (struct
					    wil_debugfs_iomem_data *)data;
	struct wil6210_priv *wil = d->wil;
	int ret;

	ret = wil_pm_runtime_get(wil);
	if (ret < 0)
		return ret;

	writel_relaxed(val, (void __iomem *)d->offset);

	wmb(); /* make sure write propagated to HW */

	wil_pm_runtime_put(wil);

	return 0;
}

static int wil_debugfs_iomem_x32_get(void *data, u64 *val)
{
	struct wil_debugfs_iomem_data *d = (struct
					    wil_debugfs_iomem_data *)data;
	struct wil6210_priv *wil = d->wil;
	int ret;

	ret = wil_pm_runtime_get(wil);
	if (ret < 0)
		return ret;

	*val = readl((void __iomem *)d->offset);

	wil_pm_runtime_put(wil);

	return 0;
}

DEFINE_DEBUGFS_ATTRIBUTE(fops_iomem_x32, wil_debugfs_iomem_x32_get,
			 wil_debugfs_iomem_x32_set, "0x%08llx\n");

static void wil_debugfs_create_iomem_x32(const char *name, umode_t mode,
					 struct dentry *parent, void *value,
					 struct wil6210_priv *wil)
{
	struct wil_debugfs_iomem_data *data = &wil->dbg_data.data_arr[
					      wil->dbg_data.iomem_data_count];

	data->wil = wil;
	data->offset = value;

	debugfs_create_file_unsafe(name, mode, parent, data, &fops_iomem_x32);
	wil->dbg_data.iomem_data_count++;
}

static int wil_debugfs_ulong_set(void *data, u64 val)
{
	*(ulong *)data = val;
	return 0;
}

static int wil_debugfs_ulong_get(void *data, u64 *val)
{
	*val = *(ulong *)data;
	return 0;
}

DEFINE_DEBUGFS_ATTRIBUTE(wil_fops_ulong, wil_debugfs_ulong_get,
			 wil_debugfs_ulong_set, "0x%llx\n");

/**
 * wil6210_debugfs_init_offset - create set of debugfs files
 * @wil - driver's context, used for printing
 * @dbg - directory on the debugfs, where files will be created
 * @base - base address used in address calculation
 * @tbl - table with file descriptions. Should be terminated with empty element.
 *
 * Creates files accordingly to the @tbl.
 */
static void wil6210_debugfs_init_offset(struct wil6210_priv *wil,
					struct dentry *dbg, void *base,
					const struct dbg_off * const tbl)
{
	int i;

	for (i = 0; tbl[i].name; i++) {
		switch (tbl[i].type) {
		case doff_u32:
			debugfs_create_u32(tbl[i].name, tbl[i].mode, dbg,
					   base + tbl[i].off);
			break;
		case doff_x32:
			debugfs_create_x32(tbl[i].name, tbl[i].mode, 