/*
 * pmac-cons.c: Console support for PowerMac (PCI-based).
 *
 * Copyright (C) 1996 Paul Mackerras.
 * 7200/Platinum code hacked by Mark Abene.
 */
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/malloc.h>
#include <linux/tty.h>
#include <linux/mm.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/string.h>
#include <linux/kd.h>
#include <linux/version.h>
#include <asm/vc_ioctl.h>
#include <asm/processor.h>
#include <asm/prom.h>
#include <asm/pgtable.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <asm/adb.h>
#include <asm/cuda.h>
#define INCLUDE_LINUX_LOGO_DATA
#include <asm/linux_logo.h>
#include <asm/init.h>
#include <linux/selection.h>
#include <linux/console_struct.h>
#include <linux/vt_kern.h>
#include "../char/console_macros.h"
#include "pmac-cons.h"
#include "control.h"
#include "platinum.h"
#include "valkyrie.h"
#include "chips.h"
#ifdef CONFIG_ATY_VIDEO
#include "aty.h"
#endif
#ifdef CONFIG_IMSTT_VIDEO
#include "imstt.h"
#endif
#include <linux/console_compat.h>

int video_mode = VMODE_NVRAM;
int color_mode = CMODE_NVRAM;

/*
 * The format of "screen_info" is strange, and due to early
 * i386-setup code. This is just enough to make the console
 * code think we're on a EGA+ colour display.
 */
struct screen_info screen_info = {
	0, 0,			/* orig-x, orig-y */
	{0, 0},			/* unused1 */
	0,			/* orig-video-page */
	0,			/* orig-video-mode */
	80,			/* orig-video-cols */
	0,			/* unused [short] */
	0,			/* ega_bx */
	0,			/* unused [short] */
	25,			/* orig-video-lines */
	0,			/* isVGA */
	16			/* video points */
};

/*
 * We allocate enough character+attribute memory for the largest
 * screen resolution supported.
 */
#define MAX_TEXT_COLS	(1280/8)
#define MAX_TEXT_ROWS	(1024/16)

/*
 * We get a sense value from the monitor and use it to choose
 * what resolution to use.  This structure maps sense values
 * to display mode values (which determine the resolution and
 * frequencies).
 */
struct mon_map {
	int	sense;
	int	vmode;
} monitor_map [] = {
	{0x000, VMODE_1280_1024_75},	/* 21" RGB */
	{0x114, VMODE_640_870_75P},	/* Portrait Monochrome */
	{0x221, VMODE_512_384_60},	/* 12" RGB*/
	{0x331, VMODE_1280_1024_75},	/* 21" RGB (Radius) */
	{0x334, VMODE_1280_1024_75},	/* 21" mono (Radius) */
	{0x335, VMODE_1280_1024_75},	/* 21" mono */
	{0x40A, VMODE_640_480_60I},	/* NTSC */
	{0x51E, VMODE_640_870_75P},	/* Portrait RGB */
	{0x603, VMODE_832_624_75},	/* 12"-16" multiscan */
	{0x60b, VMODE_1024_768_70},	/* 13"-19" multiscan */
	{0x623, VMODE_1152_870_75},	/* 13"-21" multiscan */
	{0x62b, VMODE_640_480_67},	/* 13"/14" RGB */
	{0x700, VMODE_640_480_50I},	/* PAL */
	{0x714, VMODE_640_480_60I},	/* NTSC */
	{0x717, VMODE_800_600_75},	/* VGA */
	{0x72d, VMODE_832_624_75},	/* 16" RGB (Goldfish) */
	{0x730, VMODE_768_576_50I},	/* PAL (Alternate) */
	{0x73a, VMODE_1152_870_75},	/* 3rd party 19" */
	{0x73f, VMODE_640_480_67},	/* no sense lines connected at all */
	{-1,	VMODE_640_480_60},	/* catch-all, must be last */
};

int
map_monitor_sense(int sense)
{
	struct mon_map *map;

	for (map = monitor_map; map->sense >= 0; ++map)
		if (map->sense == sense)
			break;
	return map->vmode;
}

/*
 * Horizontal and vertical resolution for each mode.
 */
struct vmode_attr vmode_attrs[VMODE_MAX] = {
	{512, 384, 60, 1},
	{512, 384, 60},
	{640, 480, 50, 1},
	{640, 480, 60, 1},
	{640, 480, 60},
	{640, 480, 67},
	{640, 870, 75},
	{768, 576, 50, 1},
	{800, 600, 56},
	{800, 600, 60},
	{800, 600, 72},
	{800, 600, 75},
	{832, 624, 75},
	{1024, 768, 60},
	{1024, 768, 72},
	{1024, 768, 75},
	{1024, 768, 75},
	{1152, 870, 75},
	{1280, 960, 75},
	{1280, 1024, 75}
};

static void invert_cursor(int);
static int map_unknown(struct device_node *);
static void unknown_init(void);
static void unknown_set_palette(unsigned char red[], unsigned char green[],
				unsigned char blue[], int index, int ncolors);

struct display_interface {
	char *name;
	void (*map_interface)(struct device_node *);
	void (*init_interface)(void);
	int  (*setmode)(struct vc_mode *, int);
	void (*set_palette)(unsigned char red[], unsigned char green[],
			    unsigned char blue[], int index, int ncolors);
	void (*set_blanking)(int blank_mode);
} displays[] = {
#ifdef CONFIG_CONTROL_VIDEO
	{ "control", map_control_display, control_init,
	  control_setmode, control_set_palette, control_set_blanking },
#endif
#ifdef CONFIG_PLATINUM_VIDEO
	{ "platinum", map_platinum, platinum_init,
	  platinum_setmode, platinum_set_palette, platinum_set_blanking },
#endif
#ifdef CONFIG_VALKYRIE_VIDEO
	{ "valkyrie", map_valkyrie_display, valkyrie_init,
	  valkyrie_setmode, valkyrie_set_palette, valkyrie_set_blanking },
#endif
#ifdef CONFIG_CHIPS_VIDEO
	{ "chips65550", map_chips_display, chips_init,
	  chips_setmode, chips_set_palette, chips_set_blanking },
#endif
#ifdef CONFIG_ATY_VIDEO
	{ "ATY,mach64", map_aty_display, aty_init,
	  aty_setmode, aty_set_palette, aty_set_blanking },
	{ "ATY,XCLAIM", map_aty_display, aty_init,
	  aty_setmode, aty_set_palette, aty_set_blanking },
	{ "ATY,264VT", map_aty_display, aty_init,
	  aty_setmode, aty_set_palette, aty_set_blanking },
	{ "ATY,mach64ii", map_aty_display, aty_init,
	  aty_setmode, aty_set_palette, aty_set_blanking },
	{ "ATY,264GT-B", map_aty_display, aty_init,
	  aty_setmode, aty_set_palette, aty_set_blanking },
	{ "ATY,mach64_3D_pcc", map_aty_display, aty_init,
	  aty_setmode, aty_set_palette, aty_set_blanking },
 	{ "ATY,XCLAIM3D", map_aty_display, aty_init,
	  aty_setmode, aty_set_palette, aty_set_blanking },
 	{ "ATY,XCLAIMVR", map_aty_display, aty_init,
	  aty_setmode, aty_set_palette, aty_set_blanking },
#if 0 /* problematic */
 	{ "ATY,RAGEII_M", map_aty_display, aty_init,
	  aty_setmode, aty_set_palette, aty_set_blanking },
#endif
	{ "ATY,XCLAIMVRPro", map_aty_display, aty_init,
	  aty_setmode, aty_set_palette, aty_set_blanking },
 	{ "ATY,mach64_3DU", map_aty_display, aty_init,
	  aty_setmode, aty_set_palette, aty_set_blanking },
 	{ "ATY,XCLAIM3DPro", map_aty_display, aty_init,
	  aty_setmode, aty_set_palette, aty_set_blanking },
#endif
#ifdef CONFIG_IMSTT_VIDEO
	{ "IMS,tt128mb", map_imstt_display_ibm, imstt_init, 
	  imstt_setmode, imstt_set_palette_ibm, imstt_set_blanking },
	{ "IMS,tt128mb8", map_imstt_display_tvp, imstt_init, 
	  imstt_setmode, imstt_set_palette_tvp, imstt_set_blanking },
	{ "IMS,tt128mb8A", map_imstt_display_tvp, imstt_init, 
	  imstt_setmode, imstt_set_palette_tvp, imstt_set_blanking },
#endif
	{ NULL }
};

struct display_interface unknown_display = {
	"unknown", NULL, unknown_init, NULL, unknown_set_palette, NULL
};

static struct display_interface *current_display;

static int cursor_pos = -1;
static unsigned *cursor_fb;	/* address of cursor pos in frame buffer */
static unsigned cursor_bits;	/* bits changed to turn cursor on */

int pixel_size;			/* in bytes */
int n_scanlines;		/* # of scan lines */
int line_pitch;			/* # bytes in 1 scan line */
int row_pitch;			/* # bytes in 1 row of characters */
unsigned char *fb_start;	/* addr of top left pixel of top left char */
struct vc_mode display_info;

#define cmapsz	(16*256)
extern unsigned char vga_font[cmapsz];

__openfirmware

static inline unsigned pixel32(int currcons, int cidx)
{
	cidx *= 3;
	return (palette[cidx] << 16) | (palette[cidx + 1] << 8)
		| palette[cidx + 2];
}

static inline unsigned pixel16(int currcons, int cidx)
{
	unsigned p;

	p = ((cidx << 10) & 0x7c00) + ((cidx << 5) & 0x03e0)
		+ (cidx & 0x1f);
	return (p << 16) | p;
}

void
__set_origin(unsigned short offset)
{
}

static void
invert_cursor(int cpos)
{
	int row, col;
	int l, c, nw;
	unsigned *fb, mask;
	int currcons = fg_console;	/* for `color', which is a macro */

	if (cpos == -1) {
		/* turning cursor off */
		fb = cursor_fb;
		mask = cursor_bits;
	} else {
		row = cpos / video_num_columns;
		col = cpos - row * video_num_columns;
		fb = (unsigned *) (fb_start + row * row_pitch
				   + col * pixel_size * 8);
		switch (color_mode) {
		case CMODE_16:
			mask = pixel16(currcons, foreground)
				^ pixel16(currcons, background >> 4);
			break;
		default:
			mask = (color ^ (color >> 4)) & 0xf;
			mask |= mask << 8;
			mask |= mask << 16;
			break;
		}
		cursor_fb = fb;
		cursor_bits = mask;
	}
	nw = pixel_size * 2;	/* pixel_size * 8 (char width) / 4 */
	for (l = 0; l < 16; ++l) {
		for (c = 0; c < nw; ++c)
			fb[c] ^= mask;
		fb = (unsigned *) ((char *)fb + line_pitch);
	}
}

void
hide_cursor()
{
	unsigned long flags;

	save_flags(flags);
	cli();
	if (cursor_pos != -1) {
		invert_cursor(-1);
		cursor_pos = -1;
	}
	restore_flags(flags);
}

void
set_cursor(int currcons)
{
	unsigned long flags;
	int old_cursor;

	if (currcons != fg_console || console_blanked)
		return;

	save_flags(flags);
	cli();
	if (!deccm) {
		hide_cursor();
	} else {
		old_cursor = cursor_pos;
		cursor_pos = (pos - video_mem_start) >> 1;
		if (old_cursor != -1)
			invert_cursor(-1);
		invert_cursor(cursor_pos);
	}
	restore_flags(flags);
}

/*
 * NOTE: get_scrmem() and set_scrmem() are here only because
 * the VGA version of set_scrmem() has some direct VGA references.
 */
void
get_scrmem(int currcons)
{
	memcpyw((unsigned short *)vc_scrbuf[currcons],
		(unsigned short *)origin, video_screen_size);
	origin = video_mem_start = (unsigned long)vc_scrbuf[currcons];
	scr_end = video_mem_end = video_mem_start + video_screen_size;
	pos = origin + y*video_size_row + (x<<1);
}

void
set_scrmem(int currcons, long offset)
{
	if (video_mem_term - video_mem_base < offset + video_screen_size)
		offset = 0;
	memcpyw((unsigned short *)(video_mem_base + offset),
		(unsigned short *) origin, video_screen_size);
	video_mem_start = video_mem_base;
	video_mem_end = video_mem_term;
	origin = video_mem_base + offset;
	scr_end = origin + video_screen_size;
	pos = origin + y*video_size_row + (x<<1);
}

int
set_get_cmap(unsigned char *p, int set)
{
	int i, j, err;

	err = verify_area(set? VERIFY_READ: VERIFY_WRITE, p, 48);
	if (err)
		return err;

	for (i = 0; i < 16; ++i) {
		if (set) {
			get_user(default_red[i], p++);
			get_user(default_grn[i], p++);
			get_user(default_blu[i], p++);
		} else {
			put_user(default_red[i], p++);
			put_user(default_grn[i], p++);
			put_user(default_blu[i], p++);
		}
	}

	if (set) {
		for (j = 0; j < MAX_NR_CONSOLES; ++j) {
			if (!vc_cons_allocated(j))
				continue;
			for (i = 0; i < 16; ++i) {
				vc_cons[j].d->vc_palette[3*i+0] = default_red[i];
				vc_cons[j].d->vc_palette[3*i+1] = default_grn[i];
				vc_cons[j].d->vc_palette[3*i+2] = default_blu[i];
			}
		}
		set_palette();
	}

	return 0;
}

int
set_get_font(unsigned char *p, int set, int ch512)
{
	return 0;
}

void
set_palette()
{
	int i, j, n;
	unsigned char red[16], green[16], blue[16];

	if (console_blanked || current_display == NULL
	    || current_display->set_palette == NULL
	    || vt_cons[fg_console]->vc_mode == KD_GRAPHICS)
		return;

	for (i = j = 0; i < 16; ++i) {
		n = color_table[i] & 0xf;
		red[n] = vc_cons[fg_console].d->vc_palette[j++];
		green[n] = vc_cons[fg_console].d->vc_palette[j++];
		blue[n] = vc_cons[fg_console].d->vc_palette[j++];
	}
	(*current_display->set_palette)(red, green, blue, 0, 16);
}

void
pmac_init_palette()
{
	int i, n;
	unsigned char red[16], green[16], blue[16];

	for (i = 0; i < 16; ++i) {
		n = color_table[i] & 0xf;
		red[n] = default_red[i];
		green[n] = default_grn[i];
		blue[n] = default_blu[i];
	}
	(*current_display->set_palette)(red, green, blue, 0, 16);
}

static int vesa_blanking_mode;
static int vesa_blanked;

void
vesa_blank()
{
	if (vesa_blanking_mode == 0 || vesa_blanked
	    || current_display == NULL
	    || current_display->set_blanking == NULL)
		return;
	(*current_display->set_blanking)(vesa_blanking_mode);
	vesa_blanked = vesa_blanking_mode;
}

void
vesa_unblank()
{
	if (vesa_blanked == 0
	    || current_display == NULL
	    || current_display->set_blanking == NULL)
		return;
	(*current_display->set_blanking)(VESA_NO_BLANKING);
	vesa_blanked = VESA_NO_BLANKING;
}

void
set_vesa_blanking(const unsigned long arg)
{
	unsigned char *argp = (unsigned char *)(arg + 1);
	unsigned int mode;

	if (verify_area(VERIFY_READ, argp, 1))
		return;

	get_user(mode, argp);
	vesa_blanking_mode = (mode <= VESA_POWERDOWN)? mode: \
		DEFAULT_VESA_BLANKING_MODE;
}

void
vesa_powerdown()
{
	if (vesa_blanked == 0 || vesa_blanked == VESA_POWERDOWN
	    || current_display == NULL
	    || current_display->set_blanking == NULL)
		return;
	(*current_display->set_blanking)(VESA_POWERDOWN);
	vesa_blanked = VESA_POWERDOWN;
}

void
memsetw(unsigned short *p, unsigned short c, unsigned count)
{
	count /= 2;
	if ((unsigned long)(p + count) > video_mem_base
	    && (unsigned long)p < video_mem_term) {
		for (; p < (unsigned short *) video_mem_base && count != 0; --count)
			*p++ = c;
		for (; p < (unsigned short *) video_mem_term && count != 0; --count) {
			if (*p != c) {
				*p = c;
				pmac_blitc(c, (unsigned long)p);
			}
			++p;
		}
	}
	for (; count != 0; --count)
		*p++ = c;
}

void
memcpyw(unsigned short *to, unsigned short *from, unsigned count)
{
	unsigned short c;

	count /= 2;
	if ((unsigned long)(to + count) > video_mem_base
	    && (unsigned long)to < video_mem_term) {
		for (; to < (unsigned short *) video_mem_base && count != 0; --count)
			*to++ = *from++;
		for (; to < (unsigned short *) video_mem_term && count != 0; --count) {
			c = *from++;
			if (*to != c) {
				*to = c;
				pmac_blitc(c, (unsigned long)to);
			}
			++to;
		}
	}
	for (; count != 0; --count)
		*to++ = *from++;
}

void
pmac_find_display()
{
	struct display_interface *disp;
	struct device_node *dp;
	struct vmode_attr *ap;

	current_display = NULL;
	for (disp = displays; disp->name != NULL; ++disp) {
		dp = find_devices(disp->name);
		if (dp == 0)
			continue;
		current_display = disp;
		disp->map_interface(dp);
		break;
	}

	if (current_display == NULL) {
		/*
		 * We haven't found a display that we know about,
		 * but if there is a display with sufficient prom support,
		 * we may be able to use it in a limited fashion.
		 * If there is, it has already been opened in prom_init().
		 */
		int i;
		for (i = 0; i < prom_num_displays; ++i) {
			dp = find_path_device(prom_display_paths[i]);
			if (dp != 0 && map_unknown(dp)) {
				current_display = &unknown_display;
				break;
			} else {
				printk(KERN_INFO "Can't use %s for display\n",
				       prom_display_paths[i]);
			}
		}
	}

	if (current_display == NULL
	    || video_mode <= 0 || video_mode > VMODE_MAX) {
		printk(KERN_INFO "No usable display device found\n");
		current_display = NULL;
		return;
	}
	ap = &vmode_attrs[video_mode - 1];
	screen_info.orig_video_cols = ap->hres / 8;
	screen_info.orig_video_lines = ap->vres / 16;
	printk("using video mode %d (%dx%d at %dHz%s), %d bits/pixel\n",
	       video_mode, ap->hres, ap->vres, ap->vfreq,
	       ap->interlaced? " interlaced": "",
	       (color_mode + 1) * 8);
}

int
pmac_display_supported(const char *name)
{
	struct display_interface *disp;

	for (disp = displays; disp->name != NULL; ++disp)
		if (strcmp(name, disp->name) == 0)
			return 1;
	return 0;
}

int
console_getmode(struct vc_mode *mode)
{
	*mode = display_info;
	return 0;
}

int
console_setmode(struct vc_mode *mode, int doit)
{
	int err;

	if (current_display == NULL || current_display->setmode == NULL)
		return -EINVAL;
	err = (*current_display->setmode)(mode, doit);
	if (doit && err == 0)
		memset((void *)video_mem_base, 0, video_screen_size);
	return err;
}

int
console_setcmap(int n_entries, unsigned char *red,
		unsigned char *green, unsigned char *blue)
{
	if (current_display == NULL || current_display->set_palette == NULL)
		return -EOPNOTSUPP;
	(*current_display->set_palette)(red, green, blue, 0, n_entries);
	return 0;
}

int
console_powermode(int mode)
{
	if (mode == VC_POWERMODE_INQUIRY)
		return vesa_blanked;
	if (mode < VESA_NO_BLANKING || mode > VESA_POWERDOWN)
		return -EINVAL;
	if (current_display == NULL || current_display->set_blanking == NULL)
		return mode == VESA_NO_BLANKING? 0: -ENXIO;
	(*current_display->set_blanking)(mode);
	vesa_blanked = mode;
	return 0;
}

void
pmac_vmode_setup(char *str, int *ints)
{
	if (ints[0] >= 1)
		video_mode = ints[1];
	if (ints[0] >= 2)
		color_mode = ints[2];
}

void
con_type_init(const char **type_p)
{
	unsigned long nb = MAX_TEXT_COLS * MAX_TEXT_ROWS * 2;

	if (current_display == NULL)
		return;
	current_display->init_interface();
	can_do_color = 1;
	video_type = VIDEO_TYPE_PMAC;
	*type_p = display_info.name;
	video_mem_base = (unsigned long) kmalloc(nb, GFP_ATOMIC);
	video_mem_term = video_mem_base + nb;
	memset((char *) video_mem_base, 0, video_screen_size);
}

int
con_is_present(void)
{
	return current_display != NULL;
}

static __inline__ void
draw_logo_8(void)
{
	unsigned char *fb = fb_start;
	unsigned char *p = linux_logo;
	int yy;

	(*current_display->set_palette)
		(linux_logo_red, linux_logo_green, linux_logo_blue,
		 32, LINUX_LOGO_COLORS);

	for (yy = 0; yy < LINUX_LOGO_HEIGHT; ++yy) {
		memcpy(fb, p, LINUX_LOGO_WIDTH);
		fb += line_pitch;
		p += LINUX_LOGO_WIDTH;
	}
}

static __inline__ void
draw_logo_15(void)
{
	unsigned short *fb;
	unsigned char *row = fb_start;
	unsigned char *p = linux_logo;
	int i, xx, yy;
	unsigned char grey[16];

	/*
	 * For 15-bit mode, treat the screen as a 4/4/4 TrueColor.
	 */
	for (i = 0; i < 16; ++i)
		grey[i] = i << 4;
	(*current_display->set_palette)(grey, grey, grey, 16, 16);

	for (yy = 0; yy < LINUX_LOGO_HEIGHT; ++yy) {
		fb = (unsigned short *) row;
		for (xx = 0; xx < LINUX_LOGO_WIDTH; ++xx) {
			i = *p++ - 32;
			*fb++ = 0x4210
				+ ((linux_logo_red[i] & 0xf0) << 6)
				+ ((linux_logo_green[i] & 0xf0) << 1)
				+ (linux_logo_blue[i] >> 4);
		}
		row += line_pitch;
	}
}

static __inline__ void
draw_logo_24(void)
{
	unsigned long *fb;
	unsigned char *row = fb_start;
	unsigned char *p = linux_logo;
	int xx, yy, v;

	(*current_display->set_palette)
		(linux_logo_red, linux_logo_green, linux_logo_blue,
		 32, LINUX_LOGO_COLORS);

	for (yy = 0; yy < LINUX_LOGO_HEIGHT; ++yy) {
		fb = (unsigned long *) row;
		for (xx = 0; xx < LINUX_LOGO_WIDTH; ++xx) {
			v = *p++;
			v |= v << 8;
			v |= v << 16;
			*fb++ = v;
		}
		row += line_pitch;
	}
}

void
con_type_init_finish(void)
{
	char *p;
	int c;
	unsigned short *addr;
	char xy[2];
	int currcons = 0;	/* for `attr', which is a macro */

	if (current_display == NULL
	    || current_display->set_palette == NULL)
		return;

	switch (color_mode) {
	case CMODE_8:
		draw_logo_8();
		break;

	case CMODE_16:
		draw_logo_15();
		break;

	case CMODE_32:
		draw_logo_24();
		break;
	}
	xy[0] = 0;
	xy[1] = (LINUX_LOGO_HEIGHT + 16) / 16;
	putconsxy(0, xy);

	switch (_machine) {
	    case _MACH_Pmac:
		p = "PowerMac/Linux " UTS_RELEASE;
		break;
	    default:
		p = "Linux/PPC " UTS_RELEASE;
		break;
	}
	addr = (unsigned short *) video_mem_base + 2 * video_num_columns
		+ LINUX_LOGO_WIDTH / 8 + 8;
	for (; *p; ++p) {
		c = (attr << 8) + *p;
		scr_writew(c, addr);
		++addr;
	}
}

int
con_adjust_height(unsigned long height)
{
	return -EINVAL;
}

static unsigned long expand_bits_8[16] = {
	0x00000000,
	0x000000ff,
	0x0000ff00,
	0x0000ffff,
	0x00ff0000,
	0x00ff00ff,
	0x00ffff00,
	0x00ffffff,
	0xff000000,
	0xff0000ff,
	0xff00ff00,
	0xff00ffff,
	0xffff0000,
	0xffff00ff,
	0xffffff00,
	0xffffffff
};

static unsigned long expand_bits_16[4] = {
	0x00000000,
	0x0000ffff,
	0xffff0000,
	0xffffffff
};

void
pmac_blitc(unsigned charattr, unsigned long addr)
{
	int col, row, fg, bg, l, bits;
	unsigned char *fp;
	unsigned long *fb;
	static int cached_row_start, cached_row_end = -1;
	static unsigned char *cached_fb;
	static int cached_attr = -1;
	static unsigned long cached_fg, cached_bg;

	col = (addr - video_mem_base) / 2;
	if (cursor_pos == col)
		cursor_pos = -1;
	if (!(col >= cached_row_start && col < cached_row_end)) {
		row = col / video_num_columns;
		cached_row_start = row * video_num_columns;
		cached_row_end = cached_row_start + video_num_columns;
		cached_fb = fb_start + row * row_pitch;
	}
	fb = (unsigned long *)
		(cached_fb + (col - cached_row_start) * pixel_size * 8);

	if ((charattr & 0xff00) != cached_attr) {
		fg = (charattr >> 8) & 0xf;
		bg = (charattr >> 12) & 0xf;
		switch (color_mode) {
		case CMODE_16:
			fg = pixel16(fg_console, fg);
			bg = pixel16(fg_console, bg);
			break;
		default:
			fg += fg << 8;
			fg += fg << 16;
			bg += bg << 8;
			bg += bg << 16;
		}
		fg ^= bg;
		cached_fg = fg;
		cached_bg = bg;
	} else {
		fg = cached_fg;
		bg = cached_bg;
	}

	fp = &vga_font[(charattr & 0xff) * 16];
	switch (color_mode) {
	case CMODE_32:
		for (l = 0; l < 16; ++l) {
			bits = *fp++;
			fb[0] = (-(bits >> 7) & fg) ^ bg;
			fb[1] = (-((bits >> 6) & 1) & fg) ^ bg;
			fb[2] = (-((bits >> 5) & 1) & fg) ^ bg;
			fb[3] = (-((bits >> 4) & 1) & fg) ^ bg;
			fb[4] = (-((bits >> 3) & 1) & fg) ^ bg;
			fb[5] = (-((bits >> 2) & 1) & fg) ^ bg;
			fb[6] = (-((bits >> 1) & 1) & fg) ^ bg;
			fb[7] = (-(bits & 1) & fg) ^ bg;
			fb = (unsigned long *) ((char *)fb + line_pitch);
		}
		break;
	case CMODE_16:
		for (l = 0; l < 16; ++l) {
			bits = *fp++;
			fb[0] = (expand_bits_16[bits >> 6] & fg) ^ bg;
			fb[1] = (expand_bits_16[(bits >> 4) & 3] & fg) ^ bg;
			fb[2] = (expand_bits_16[(bits >> 2) & 3] & fg) ^ bg;
			fb[3] = (expand_bits_16[bits & 3] & fg) ^ bg;
			fb = (unsigned long *) ((char *)fb + line_pitch);
		}
		break;
	default:
		for (l = 0; l < 16; ++l) {
			bits = *fp++;
			fb[0] = (expand_bits_8[bits >> 4] & fg) ^ bg;
			fb[1] = (expand_bits_8[bits & 0xf] & fg) ^ bg;
			fb = (unsigned long *) ((char *)fb + line_pitch);
		}
	}
}


/*
 * The following provides a very basic screen driver for displays
 * that we basically don't know anything about, but which we can
 * initialize by using their Open Firmware "open" method.
 */

static int unknown_modes[] = {
	VMODE_512_384_60,
	VMODE_640_480_60,
	VMODE_800_600_60,
	VMODE_832_624_75,
	VMODE_1024_768_60,
	VMODE_1152_870_75,
	VMODE_1280_960_75,
	VMODE_1280_1024_75,
	0
};

static unsigned char *frame_buffer;
static unsigned char *unknown_cmap_adr;
static volatile unsigned char *unknown_cmap_data;
static unsigned long frame_buffer_phys;

static int map_unknown(struct device_node *dp)
{
	int i, mode;
	int width;
	int *pp, len;
	unsigned *up, address;

	printk("map_unknown(%p), name = %s\n", dp, dp->full_name);

	/* check the depth */
	if ((pp = (int *) get_property(dp, "depth", &len)) != NULL
	    && len == sizeof(int) && *pp != 8) {
		printk("%s: can't use depth = %d\n", dp->full_name, *pp);
		return 0;
	}

	width = 640;		/* default values */
	n_scanlines = 480;
	if ((pp = (int *) get_property(dp, "width", &len)) != NULL
	    && len == sizeof(int))
		width = *pp;
	if ((pp = (int *) get_property(dp, "height", &len)) != NULL
	    && len == sizeof(int))
		n_scanlines = *pp;
	line_pitch = width;
	if ((pp = (int *) get_property(dp, "linebytes", &len)) != NULL
	    && len == sizeof(int))
		line_pitch = *pp;
	printk(KERN_INFO "width=%d height=%d pitch=%d\n",
	       width, n_scanlines, line_pitch);

	len = n_scanlines * line_pitch;
	if ((up = (unsigned *) get_property(dp, "address", &len)) != NULL
	    && len == sizeof(unsigned)) {
		address = *up;
	} else {
		for (i = 0; i < dp->n_addrs; ++i)
			if (dp->addrs[i].size >= len)
				break;
		if (i >= dp->n_addrs) {
			printk("no framebuffer address found for %s\n",
			       dp->full_name);
			return 0;
		}
		address = dp->addrs[i].address;
	}
	printk(KERN_INFO "%s: using address %x\n", dp->full_name, address);
	frame_buffer_phys = address;
	frame_buffer = __ioremap(frame_buffer_phys, len, _PAGE_WRITETHRU);

	video_mode = 0;
	color_mode = CMODE_8;
	pixel_size = 1;

	for (i = 0; (mode = unknown_modes[i]) != 0; ++i) {
		if (vmode_attrs[mode-1].hres <= width
		    && vmode_attrs[mode-1].vres <= n_scanlines)
			video_mode = mode;
	}
	if (video_mode == 0) {
		printk(KERN_INFO "%s: no mode found for %d x %d\n",
		       dp->full_name, width, n_scanlines);
		return 0;
	}

	display_info.height = n_scanlines;
	display_info.width = width;
	display_info.depth = 8;
	display_info.pitch = line_pitch;
	display_info.mode = video_mode;
	strncpy(display_info.name, dp->name, sizeof(display_info.name));
	display_info.fb_address = frame_buffer_phys;
	display_info.cmap_adr_address = 0;
	display_info.cmap_data_address = 0;
	unknown_cmap_adr = 0;
	/* XXX kludge for ati */
	if (strncmp(dp->name, "ATY,", 4) == 0) {
		display_info.disp_reg_address = frame_buffer_phys + 0x7ffc00;
		display_info.cmap_adr_address = frame_buffer_phys + 0x7ffcc0;
		display_info.cmap_data_address = frame_buffer_phys + 0x7ffcc1;
		unknown_cmap_adr = ioremap(address + 0x7ff000, 0x1000) + 0xcc0;
		unknown_cmap_data = unknown_cmap_adr + 1;
	}

	return 1;
}

static void
unknown_init()
{
	unsigned *p;
	int i;

	row_pitch = line_pitch * 16;
	fb_start = frame_buffer;

	/* Clear screen */
	p = (unsigned *) frame_buffer;
	for (i = n_scanlines * line_pitch / sizeof(unsigned); i != 0; --i)
		*p++ = 0;
}

static void
unknown_set_palette(unsigned char red[], unsigned char green[],
		    unsigned char blue[], int index, int ncolors)
{
	volatile unsigned char *a, *d;
	int i;

	if (unknown_cmap_adr == 0)
		return;
	a = unknown_cmap_adr;
	d = unknown_cmap_data;
	for (i = 0; i < ncolors; ++i) {
		*a = index + i;	eieio();
		*d = red[i];	eieio();
		*d = green[i];	eieio();
		*d = blue[i];	eieio();
	}
}


unsigned char vga_font[cmapsz] = {
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x81, 0xa5, 0x81, 0x81, 0xbd, 
0x99, 0x81, 0x81, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xff, 
0xdb, 0xff, 0xff, 0xc3, 0xe7, 0xff, 0xff, 0x7e, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x6c, 0xfe, 0xfe, 0xfe, 0xfe, 0x7c, 0x38, 0x10, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x7c, 0xfe, 
0x7c, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 
0x3c, 0x3c, 0xe7, 0xe7, 0xe7, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x18, 0x3c, 0x7e, 0xff, 0xff, 0x7e, 0x18, 0x18, 0x3c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c, 
0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 
0xff, 0xff, 0xe7, 0xc3, 0xc3, 0xe7, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x42, 0x42, 0x66, 0x3c, 0x00, 
0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xc3, 0x99, 0xbd, 
0xbd, 0x99, 0xc3, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x1e, 0x0e, 
0x1a, 0x32, 0x78, 0xcc, 0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x3c, 0x66, 0x66, 0x66, 0x66, 0x3c, 0x18, 0x7e, 0x18, 0x18, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x33, 0x3f, 0x30, 0x30, 0x30, 
0x30, 0x70, 0xf0, 0xe0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0x63, 
0x7f, 0x63, 0x63, 0x63, 0x63, 0x67, 0xe7, 0xe6, 0xc0, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x18, 0x18, 0xdb, 0x3c, 0xe7, 0x3c, 0xdb, 0x18, 0x18, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0xc0, 0xe0, 0xf0, 0xf8, 0xfe, 0xf8, 
0xf0, 0xe0, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x06, 0x0e, 
0x1e, 0x3e, 0xfe, 0x3e, 0x1e, 0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 
0x66, 0x00, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7f, 0xdb, 
0xdb, 0xdb, 0x7b, 0x1b, 0x1b, 0x1b, 0x1b, 0x1b, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x7c, 0xc6, 0x60, 0x38, 0x6c, 0xc6, 0xc6, 0x6c, 0x38, 0x0c, 0xc6, 
0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0xfe, 0xfe, 0xfe, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c, 
0x7e, 0x18, 0x18, 0x18, 0x7e, 0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x18, 0x3c, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x7e, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x18, 0x0c, 0xfe, 0x0c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x60, 0xfe, 0x60, 0x30, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc0, 0xc0, 
0xc0, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x24, 0x66, 0xff, 0x66, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x38, 0x7c, 0x7c, 0xfe, 0xfe, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xfe, 0x7c, 0x7c, 
0x38, 0x38, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x18, 0x3c, 0x3c, 0x3c, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x66, 0x66, 0x24, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6c, 
0x6c, 0xfe, 0x6c, 0x6c, 0x6c, 0xfe, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00, 
0x18, 0x18, 0x7c, 0xc6, 0xc2, 0xc0, 0x7c, 0x06, 0x06, 0x86, 0xc6, 0x7c, 
0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc2, 0xc6, 0x0c, 0x18, 
0x30, 0x60, 0xc6, 0x86, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 
0x6c, 0x38, 0x76, 0xdc, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x30, 0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x30, 0x30, 0x30, 
0x30, 0x30, 0x18, 0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x18, 
0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x3c, 0xff, 0x3c, 0x66, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e, 
0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x02, 0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0, 0x80, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xce, 0xde, 0xf6, 0xe6, 0xc6, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x38, 0x78, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 
0x06, 0x0c, 0x18, 0x30, 0x60, 0xc0, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x7c, 0xc6, 0x06, 0x06, 0x3c, 0x06, 0x06, 0x06, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x1c, 0x3c, 0x6c, 0xcc, 0xfe, 
0x0c, 0x0c, 0x0c, 0x1e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc0, 
0xc0, 0xc0, 0xfc, 0x06, 0x06, 0x06, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x38, 0x60, 0xc0, 0xc0, 0xfc, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc6, 0x06, 0x06, 0x0c, 0x18, 
0x30, 0x30, 0x30, 0x30, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 
0xc6, 0xc6, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x06, 0x06, 0x0c, 0x78, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 
0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x18, 0x18, 0x00, 0x00, 0x00, 0x18, 0x18, 0x30, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x06, 0x0c, 0x18, 0x30, 0x60, 0x30, 0x18, 0x0c, 0x06, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x00, 0x00, 
0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 
0x30, 0x18, 0x0c, 0x06, 0x0c, 0x18, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x7c, 0xc6, 0xc6, 0x0c, 0x18, 0x18, 0x18, 0x00, 0x18, 0x18, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xde, 0xde, 
0xde, 0xdc, 0xc0, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 
0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x66, 0x66, 0x66, 0x66, 0xfc, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0xc2, 0xc0, 0xc0, 0xc0, 
0xc0, 0xc2, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x6c, 
0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x6c, 0xf8, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xfe, 0x66, 0x62, 0x68, 0x78, 0x68, 0x60, 0x62, 0x66, 0xfe, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x66, 0x62, 0x68, 0x78, 0x68, 
0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 
0xc2, 0xc0, 0xc0, 0xde, 0xc6, 0xc6, 0x66, 0x3a, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x0c, 
0x0c, 0x0c, 0x0c, 0x0c, 0xcc, 0xcc, 0xcc, 0x78, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xe6, 0x66, 0x66, 0x6c, 0x78, 0x78, 0x6c, 0x66, 0x66, 0xe6, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0, 0x60, 0x60, 0x60, 0x60, 0x60, 
0x60, 0x62, 0x66, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xe7, 
0xff, 0xff, 0xdb, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde, 0xce, 0xc6, 0xc6, 0xc6, 0xc6, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 
0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0x66, 
0x66, 0x66, 0x7c, 0x60, 0x60, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xd6, 0xde, 0x7c, 
0x0c, 0x0e, 0x00, 0x00, 0x00, 0x00, 0xfc, 0x66, 0x66, 0x66, 0x7c, 0x6c, 
0x66, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 
0xc6, 0x60, 0x38, 0x0c, 0x06, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xff, 0xdb, 0x99, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 
0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 
0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xc3, 0xc3, 0xc3, 0xc3, 0xc3, 0xdb, 0xdb, 0xff, 0x66, 0x66, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 0x66, 0x3c, 0x18, 0x18, 
0x3c, 0x66, 0xc3, 0xc3, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 
0xc3, 0x66, 0x3c, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xff, 0xc3, 0x86, 0x0c, 0x18, 0x30, 0x60, 0xc1, 0xc3, 0xff, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x30, 0x30, 0x30, 0x30, 0x30, 
0x30, 0x30, 0x30, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 
0xc0, 0xe0, 0x70, 0x38, 0x1c, 0x0e, 0x06, 0x02, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x3c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x3c, 
0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 
0x30, 0x30, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x78, 0x0c, 0x7c, 
0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe0, 0x60, 
0x60, 0x78, 0x6c, 0x66, 0x66, 0x66, 0x66, 0x7c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc0, 0xc0, 0xc0, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x0c, 0x0c, 0x3c, 0x6c, 0xcc, 
0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60, 0x60, 0x60, 0x60, 0xf0, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xcc, 0xcc, 
0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0xcc, 0x78, 0x00, 0x00, 0x00, 0xe0, 0x60, 
0x60, 0x6c, 0x76, 0x66, 0x66, 0x66, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x18, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x06, 0x00, 0x0e, 0x06, 0x06, 
0x06, 0x06, 0x06, 0x06, 0x66, 0x66, 0x3c, 0x00, 0x00, 0x00, 0xe0, 0x60, 
0x60, 0x66, 0x6c, 0x78, 0x78, 0x6c, 0x66, 0xe6, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe6, 0xff, 0xdb, 
0xdb, 0xdb, 0xdb, 0xdb, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0xdc, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x66, 0x66, 
0x66, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x76, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x7c, 0x0c, 0x0c, 0x1e, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0xdc, 0x76, 0x66, 0x60, 0x60, 0x60, 0xf0, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 0xc6, 0x60, 
0x38, 0x0c, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x30, 
0x30, 0xfc, 0x30, 0x30, 0x30, 0x30, 0x36, 0x1c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0xc3, 0xc3, 
0xc3, 0x66, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0xc3, 0xc3, 0xc3, 0xdb, 0xdb, 0xff, 0x66, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0x66, 0x3c, 0x18, 0x3c, 0x66, 0xc3, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0xc6, 0xc6, 
0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x0c, 0xf8, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0xfe, 0xcc, 0x18, 0x30, 0x60, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x0e, 0x18, 0x18, 0x18, 0x70, 0x18, 0x18, 0x18, 0x18, 0x0e, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x00, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x70, 0x18, 
0x18, 0x18, 0x0e, 0x18, 0x18, 0x18, 0x18, 0x70, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0xc6, 
0xc6, 0xc6, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 
0xc2, 0xc0, 0xc0, 0xc0, 0xc2, 0x66, 0x3c, 0x0c, 0x06, 0x7c, 0x00, 0x00, 
0x00, 0x00, 0xcc, 0x00, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x00, 0x7c, 0xc6, 0xfe, 
0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 
0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xcc, 0x00, 0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 0x00, 0x78, 0x0c, 0x7c, 
0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 0x38, 
0x00, 0x78, 0x0c, 0x7c, 0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x3c, 0x66, 0x60, 0x60, 0x66, 0x3c, 0x0c, 0x06, 
0x3c, 0x00, 0x00, 0x00, 0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xfe, 
0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x00, 
0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc0, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x60, 0x30, 0x18, 0x00, 0x7c, 0xc6, 0xfe, 0xc0, 0xc0, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x66, 0x00, 0x00, 0x38, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x3c, 0x66, 
0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x60, 0x30, 0x18, 0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x00, 0x10, 0x38, 0x6c, 0xc6, 0xc6, 
0xfe, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 0x38, 0x00, 
0x38, 0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00, 
0x18, 0x30, 0x60, 0x00, 0xfe, 0x66, 0x60, 0x7c, 0x60, 0x60, 0x66, 0xfe, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x6e, 0x3b, 0x1b, 
0x7e, 0xd8, 0xdc, 0x77, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3e, 0x6c, 
0xcc, 0xcc, 0xfe, 0xcc, 0xcc, 0xcc, 0xcc, 0xce, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x10, 0x38, 0x6c, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x00, 0x00, 0x7c, 0xc6, 0xc6, 
0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 
0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x30, 0x78, 0xcc, 0x00, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0xcc, 0x76, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x60, 0x30, 0x18, 0x00, 0xcc, 0xcc, 0xcc, 
0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x00, 
0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7e, 0x06, 0x0c, 0x78, 0x00, 
0x00, 0xc6, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0xc6, 0x00, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 
0xc6, 0xc6, 0xc6, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e, 
0xc3, 0xc0, 0xc0, 0xc0, 0xc3, 0x7e, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x38, 0x6c, 0x64, 0x60, 0xf0, 0x60, 0x60, 0x60, 0x60, 0xe6, 0xfc, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xc3, 0x66, 0x3c, 0x18, 0xff, 0x18, 
0xff, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfc, 0x66, 0x66, 
0x7c, 0x62, 0x66, 0x6f, 0x66, 0x66, 0x66, 0xf3, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x0e, 0x1b, 0x18, 0x18, 0x18, 0x7e, 0x18, 0x18, 0x18, 0x18, 0x18, 
0xd8, 0x70, 0x00, 0x00, 0x00, 0x18, 0x30, 0x60, 0x00, 0x78, 0x0c, 0x7c, 
0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 
0x00, 0x38, 0x18, 0x18, 0x18, 0x18, 0x18, 0x3c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x18, 0x30, 0x60, 0x00, 0x7c, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x30, 0x60, 0x00, 0xcc, 0xcc, 0xcc, 
0xcc, 0xcc, 0xcc, 0x76, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 
0x00, 0xdc, 0x66, 0x66, 0x66, 0x66, 0x66, 0x66, 0x00, 0x00, 0x00, 0x00, 
0x76, 0xdc, 0x00, 0xc6, 0xe6, 0xf6, 0xfe, 0xde, 0xce, 0xc6, 0xc6, 0xc6, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x6c, 0x6c, 0x3e, 0x00, 0x7e, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 0x6c, 
0x38, 0x00, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x60, 0xc0, 0xc6, 0xc6, 0x7c, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc0, 
0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0xfe, 0x06, 0x06, 0x06, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30, 0x60, 0xce, 0x9b, 0x06, 
0x0c, 0x1f, 0x00, 0x00, 0x00, 0xc0, 0xc0, 0xc2, 0xc6, 0xcc, 0x18, 0x30, 
0x66, 0xce, 0x96, 0x3e, 0x06, 0x06, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 
0x00, 0x18, 0x18, 0x18, 0x3c, 0x3c, 0x3c, 0x18, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x6c, 0xd8, 0x6c, 0x36, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xd8, 0x6c, 0x36, 
0x6c, 0xd8, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x11, 0x44, 0x11, 0x44, 
0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 0x11, 0x44, 
0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 0x55, 0xaa, 
0x55, 0xaa, 0x55, 0xaa, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 
0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0xdd, 0x77, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0xf6, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x18, 0xf8, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x36, 0x36, 0x36, 0x36, 
0x36, 0xf6, 0x06, 0xf6, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x06, 0xf6, 
0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x36, 0xf6, 0x06, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xfe, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 0x18, 0xf8, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0xf8, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0xff, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0xff, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 
0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x37, 0x30, 0x3f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x30, 0x37, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xf7, 0x00, 0xff, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0xff, 0x00, 0xf7, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0x36, 0x36, 0x37, 0x30, 0x37, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0x36, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0xff, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x36, 0x36, 0x36, 0x36, 
0x36, 0xf7, 0x00, 0xf7, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0xff, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0xff, 0x00, 0xff, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x3f, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x1f, 0x18, 0x1f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x1f, 0x18, 0x1f, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 
0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x36, 0x36, 0x36, 0xff, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 0x36, 
0x18, 0x18, 0x18, 0x18, 0x18, 0xff, 0x18, 0xff, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0xf8, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x1f, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 
0xff, 0xff, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 
0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xf0, 0xf0, 0xf0, 0xf0, 
0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 
0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 0x0f, 
0x0f, 0x0f, 0x0f, 0x0f, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x76, 0xdc, 0xd8, 0xd8, 0xd8, 0xdc, 0x76, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x78, 0xcc, 0xcc, 0xcc, 0xd8, 0xcc, 0xc6, 0xc6, 0xc6, 0xcc, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0xc6, 0xc6, 0xc0, 0xc0, 0xc0, 
0xc0, 0xc0, 0xc0, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0xfe, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0xfe, 0xc6, 0x60, 0x30, 0x18, 0x30, 0x60, 0xc6, 0xfe, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0xd8, 0xd8, 
0xd8, 0xd8, 0xd8, 0x70, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x66, 0x66, 0x66, 0x66, 0x66, 0x7c, 0x60, 0x60, 0xc0, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7e, 0x18, 0x3c, 0x66, 0x66, 
0x66, 0x3c, 0x18, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 
0x6c, 0xc6, 0xc6, 0xfe, 0xc6, 0xc6, 0x6c, 0x38, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x38, 0x6c, 0xc6, 0xc6, 0xc6, 0x6c, 0x6c, 0x6c, 0x6c, 0xee, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1e, 0x30, 0x18, 0x0c, 0x3e, 0x66, 
0x66, 0x66, 0x66, 0x3c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x7e, 0xdb, 0xdb, 0xdb, 0x7e, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x03, 0x06, 0x7e, 0xdb, 0xdb, 0xf3, 0x7e, 0x60, 0xc0, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x1c, 0x30, 0x60, 0x60, 0x7c, 0x60, 
0x60, 0x60, 0x30, 0x1c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x7c, 
0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0xc6, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x7e, 0x18, 
0x18, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 
0x18, 0x0c, 0x06, 0x0c, 0x18, 0x30, 0x00, 0x7e, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x0c, 0x18, 0x30, 0x60, 0x30, 0x18, 0x0c, 0x00, 0x7e, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0e, 0x1b, 0x1b, 0x1b, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 0x18, 
0x18, 0x18, 0x18, 0x18, 0xd8, 0xd8, 0xd8, 0x70, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x7e, 0x00, 0x18, 0x18, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x76, 0xdc, 0x00, 
0x76, 0xdc, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x38, 0x6c, 0x6c, 
0x38, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x18, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0f, 0x0c, 0x0c, 
0x0c, 0x0c, 0x0c, 0xec, 0x6c, 0x6c, 0x3c, 0x1c, 0x00, 0x00, 0x00, 0x00, 
0x00, 0xd8, 0x6c, 0x6c, 0x6c, 0x6c, 0x6c, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x70, 0xd8, 0x30, 0x60, 0xc8, 0xf8, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x7c, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 
0x00, 0x00, 0x00, 0x00, 
};
