      program RanfTest
c==============================================================================

c------------------------------------------------------------------------------
c
c  This program does test runs of the parallel pseudorandom number generator
c  Pranf.  It is designed to generate multiple streams of random 48-bit
c  floating point values between zero and one.  Each stream is autonomous,
c  independent, and requires its own seed.  it is based on the Cray RNG
c  known as "ranf".  The method for generating multiple streams by partitioning
c  the ranf sequence were provided by Dr. Mark J. Durst.  The original version
c  of Pranf was written in SISAL for the parallel simulated annealing program
c  PSA was written in SISAL.  This fortran version was written as part of a
c  comparative performance study, by T. M. DeBoni of Lawrence Livermore National
c  Laboratory, in conjunction with John Feo and David Cann.
c
c------------------------------------------------------------------------------

c==============================================================================
c
c  C O N S T A N T   D E F I N I T I O N S
c
c==============================================================================
c
c  Following are the contents of the file "ranftest.parm", which is included
c  in each routine.  It contains parameter definitions that are used to size the
c  data structures.  The include statement is of the form
c                     $include ranftest.parm
c

c**** Program control and storage sizing parameters:
c**** ----------------------------------------------
      parameter
c**** The upper limits possible in this compilation:
     1          ( MaxParallelism = 60,
c**** Some array sizing values:
     2            MaxRandNumStreams = MaxParallelism+1,
     3            IRandNumSize = 4, IBinarySize = 48,
     4            Mod4096DigitSize = 12,
     5            IAmountofParallelism = 1 )

c**** Data structuring parameters:
c**** ---------------------------

c**** Other control values:
c**** --------------------
      parameter ( IZero = 0, Inunit = 5, IOutunit = 6, MyTrue = 1, MyFalse = 0 )
      parameter ( MaxNumberofNumbers = 1000 )

c==============================================================================
c
c  S T O R A G E   D E F I N I T I O N S
c
c==============================================================================
c
c First, some discussion of the data sturctures we use in this program.
c Since Fortran has no data structuring abilities except arrays, all of these
c structures are represented by arrays. Special access and modificaation
c routines are provied to maintain the appropriate data abstractions. These
c routines are given elsewhere.
c
c**** This is one seed used in the random number generator; it is represented as
c**** an array of four modulo-4096 (12-bit) integers:
c     Seed( IRandNumSize )
c
c**** This is the array of seeds used in the parallel random number generator:
c     Seeds( IAmountofParallelism, IRandNumSize )
c
c**** Here, now, are the actual storage elements used in the main program.
c**** Others of those mentioned above will be found in other routines.
c
      integer 
c**** Scalars
     1        NumberofStreams, SeedIn, NumberofRandNums, Seed

c**** Arrays
     1        Seeds

c     real*8 
      double precision
c**** Scalars
     1       RandNum, AC, ACSQ,

c**** Arrays
     1       RandNums

      dimension 
     1          RandNums( MaxNumberofNumbers ),
     2          Seeds( IAmountofParallelism, IRandNumSize ),
     3          Seed( IRandNumSize )

c==============================================================================
c
c  M A I N   P R O G R A M
c
c==============================================================================

c**** Set up input and output files.
      call FileSetup

c**** Read the input data.
      call GetInput( NumberofStreams, SeedIn, NumberofRandNums )

      if( ( NumberofStreams .gt. IAmountofParallelism ) .or.
     1    ( IAmountofParallelism .gt. 1 ) .or.
     2    ( NumberofRandNums .gt. MaxNumberofNumbers ) ) stop

c**** Initialize the parallel random number generator to the number of 
c**** separate streams specified by the amount of parallelism needed.
      call Rans( IAmountofParallelism, SeedIn, Seeds )

c**** Generate a stream of numbers.
      AC = 0d0
      ACSQ = 0d0
      call GetSeed( Seeds, 1, Seed )
      do 100 I = 1, NumberofRandNums
         call ranf( Seed, RandNum )
         RandNums( I ) = RandNum
         AC = AC + RandNum
         ACSQ = ACSQ + RandNum * RandNum
 100  continue

c**** Write out the data.
      write( *, 125 ) NumberofRandNums
 125  format( ' Stream of ', i6, ' random numbers:' )
      write( *, 150 ) ( RandNums( I ), I = 1, NumberofRandNums )
 150  format( 3d22.15 )

      write( *, 160 ) AC, ACSQ
 160  format( ' AC = ', d22.15 '  ACSQ = ', d22.15 )

c**** Close all the files.
      call CloseAllFiles

      stop
      end
c     end Main

c==============================================================================
c
c  D A T A   S T R U C T U R E   M A N I P U L A T I O N   R O U T I N E S
c
c==============================================================================

      subroutine PutSeed( Seeds, Seed, SeedIndex )

c------------------------------------------------------------------------------

$include ranftest.parm

      integer Seeds, Seed, SeedIndex

      dimension Seeds( IAmountofParallelism, IRandNumSize ),
     2          Seed( IRandNumSize )

c------------------------------------------------------------------------------

c     write( *, 10 )
 10   format( ' PutSeed', /, '-------' )

      Seeds( SeedIndex, 1 ) = Seed( 1 )
      Seeds( SeedIndex, 2 ) = Seed( 2 )
      Seeds( SeedIndex, 3 ) = Seed( 3 )
      Seeds( SeedIndex, 4 ) = Seed( 4 )

c     write( *, 20 )
 20   format( '  leaving PutSeed' )

      return
      end
c     end PutSeed

c==============================================================================

      subroutine GetSeed( Seeds, SeedIndex, Seed )

c------------------------------------------------------------------------------

$include ranftest.parm

      integer Seeds, SeedIndex, Seed

      dimension Seeds( IAmountofParallelism, IRandNumSize ),
     2          Seed( IRandNumSize )

c------------------------------------------------------------------------------

c     write( *, 10 )
 10   format( ' GetSeed', /, '-------' )

      Seed( 1 ) = Seeds( SeedIndex, 1 )
      Seed( 2 ) = Seeds( SeedIndex, 2 )
      Seed( 3 ) = Seeds( SeedIndex, 3 )
      Seed( 4 ) = Seeds( SeedIndex, 4 )

c     write( *, 20 ) 
 20   format( '  leaving GetSeed' )

      return
      end
c     end GetSeed

c==============================================================================
c
c  A U X I L I A R Y   R O U T I N E S
c
c==============================================================================

      subroutine CloseAllFiles

c------------------------------------------------------------------------------

$include ranftest.parm

c------------------------------------------------------------------------------

c     write( *, 10 )
 10   format( ' CloseAllFiles ', /, '--------------' )

c     close( IOutunit )

      return
      end
c     end  CloseAllFiles

c==============================================================================

      subroutine FileSetup

c------------------------------------------------------------------------------

$include ranftest.parm

      character*23 InputFileName, OutputFileName

c------------------------------------------------------------------------------

c     write( *, 10 )
c10   format( ' Enter name of file containing data tuples: ' )
c     read( *, 20 ) InputFileName
c20   format( A )

c     write( *, 30 )
c30   format( ' Enter name of file to receive all text output: ' )
c     read( *, 40 ) OutputFileName
c40   format( A )

c     open( file=InputFileName,  unit=Inunit )
c     open( file=OutputFileName, unit=IOutunit )

c     write( *, 50 )
 50   format( ' FileSetup', /, '----------' )

      return
      end
c     end FileSetup

c==============================================================================

      subroutine GetInput( NumberofStreams, SeedIn, NumberofRandNums )

c------------------------------------------------------------------------------

$include ranftest.parm

      integer NumberofStreams, SeedIn, NumberofRandNums

c------------------------------------------------------------------------------

      write( *, 10 )
 10   format( ' GetInput', /, '---------' )

c**** Read the control values.
      read( *, 20 ) NumberofStreams, SeedIn, NumberofRandNums
 20   format( 3i6 )

c**** Echo the control values.
      write( *, 110 ) NumberofStreams, SeedIn, NumberofRandNums
 110  format( ' INPUT DATA:', /, '===========', /,
     1        '   NumberofStreams = ', i6, '  SeedIn = ', i6,
     2        '   NumberofRandNums = ', i6, / )

      write( *, 210 )
 210  format( 'Closing input file.', / )

c**** All done with the input file.
c     close( Inunit )

      return
      end
c     end GetInput

c==============================================================================

      subroutine PutOutput( )

c------------------------------------------------------------------------------

$include ranftest.parm

c------------------------------------------------------------------------------

      write( *, 100 )
 100  format( ' OUTPUT DATA:', /, '============' )

      return
      end
c     end PutOutput

c==============================================================================
c
c   R A N D O M   N U M B E R   G E N E R A T O R
c
c==============================================================================

      subroutine Ranf( Seed, RandNum )

c------------------------------------------------------------------------------
c
c  This is the one of the two user-callable routines of a linear congruential
c  random number generator, modeled after the RANF generator for the Cray.
c  This routine generates the next random number in the sequence and a new seed
c  for the remainder of the sequence.  The seed and the random number are the
c  same, but are returned in different form: the random number is a fortran
c  'real', but the seed is an array of four words, each containing an integer
c  that is used internally to the generator as one digit of a four-digit,
c  modulo-4096 integer.
c
c  It returns the new random number and a new seed.
c
c------------------------------------------------------------------------------

$include ranftest.parm

      integer Seed

      real*8 RandNum

c******************************************************************************
c**** Data common to the PRanf package.
      integer Multiplier, DefaultSeed

      real*8 Divisor

      dimension Multiplier( IRandNumSize ), DefaultSeed( IRandNumSize ),
     1          Divisor( IRandNumSize )

      dimension Seed( IRandNumSize )

      common / PRanf / Multiplier, DefaultSeed, Divisor 

      data Multiplier / 373, 3707, 1442, 647 /,
     1     DefaultSeed / 3281, 4041, 595, 2376 / ,
     2     Divisor / 281474976710656.0, 68719476736.0, 16777216.0, 4096.0 /
c**** End of PRanf common data
c******************************************************************************

c------------------------------------------------------------------------------

c     write( *, 10 )
 10   format( ' Ranf', /, '-----' )

      RandNum = float( Seed( 4 ) ) / Divisor( 4 ) +
     1          float( Seed( 3 ) ) / Divisor( 3 ) +
     2          float( Seed( 2 ) ) / Divisor( 2 ) +
     3          float( Seed( 1 ) ) / Divisor( 1 ) 

      call ranfmodmult( Multiplier, Seed, Seed )

c     write( *, 20 )
 20   format( '  leaving Ranf' )

      return
      end
c     end Ranf

c==============================================================================

      subroutine Rans( NIn, Seed1, SeedArray )

c------------------------------------------------------------------------------
c
c  This routine divides the sequence of random numbers into N subsequences,
c  each with its own seed.  The seeds for the independent subsequences are
c  returned in the seed array.  if Seed1 is zero, all zeroes will be returned.
c  To prevent this, Seed1 is set to [3281, 4041, 595, 2376], which is 
c  statistically the best starting seed.  The wheel is then divided into the
c  N pieces (where N is odd and >= NIn) by dividing its period (2**46) by N.
c
c  Then, seed(i) = seed(i-1) * (a**k mod 2**48), and 1<=k<=N.
c
c  Here, 'a' is the multiplier used by the linear congruential generator whose
c  wheel we are dividing up.
c
c  The number of streams must be odd; if NIn is even N will be NIn+1, and
c  n extra stream of random numbers will be available that will not get used.
c
c  It returns an array of seeds, each an array of 4 integers that are used as
c  the digits of a four-digit modulo-4096 integer.
c
c------------------------------------------------------------------------------

$include ranftest.parm

      integer NIn, Seed1, DefaultValues, SeedArray, N, atothek, K, KBinary,
     1        InSeed, OutSeed

c******************************************************************************
c**** Data common to the PRanf package.
      integer Multiplier, DefaultSeed

      real*8 Divisor

      dimension Multiplier( IRandNumSize ), DefaultSeed( IRandNumSize ),
     1          Divisor( IRandNumSize )

      common / PRanf / Multiplier, DefaultSeed, Divisor 
c**** End of PRanf common data
c******************************************************************************

      dimension DefaultValues( IRandNumSize ),
     1          SeedArray( 1, IRandNumSize ),
     2          atothek( IRandNumSize ),
     3          K( IRandNumSize ),
     4          KBinary( IBinarySize ),
     5          InSeed( IRandNumSize ),
     6          OutSeed( IRandNumSize )

c------------------------------------------------------------------------------

c**** Statement function the check for argument numbers being odd.
      NIsOdd( m ) = mod( m, 2 )

c     write( *, 10 ) NIn, Seed1
 10   format( ' Rans: NIn = ', i6, '  Seed1 = ', i6, /,
     1        '------' )

c**** Make sure we are generating an odd number of random number sequences.
      if( NisOdd( NIn ) .eq. 1 ) then
         N = NIN
      else
         N = NIn + 1
      endif

c**** Set up the initial seed to either a legal input value or its default
c**** values. The input integer, if nonzero, is used for the first of the
c**** four modulo-4096 digits of the actual initial seed.

c**** The First element of the returned SeedArray will be used here, since
c**** at least one seed will be returned, and the first seed of the set 
c**** returned will be the seed for the entire wheel.
      if( Seed1 .eq. IZero ) then
         SeedArray( 1, 1 ) = DefaultSeed( 1 )
         SeedArray( 1, 2 ) = DefaultSeed( 2 )
         SeedArray( 1, 3 ) = DefaultSeed( 3 )
         SeedArray( 1, 4 ) = DefaultSeed( 4 )
      else
         SeedArray( 1, 1 ) = abs( Seed1 )
         SeedArray( 1, 2 ) = IZero
         SeedArray( 1, 3 ) = IZero
         SeedArray( 1, 4 ) = IZero
      endif

c**** 'a' is the multiplier for the Ranf linear congruential generator.
      if( N .eq. 1 ) then
c******* If only one stream of random numbers is needed, do not bother to 
c******* raise 'a' to the first power.
         atothek( 1 ) = Multiplier( 1 )
         atothek( 2 ) = Multiplier( 2 )
         atothek( 3 ) = Multiplier( 3 )
         atothek( 4 ) = Multiplier( 4 )
      else
c******* more than one stream is needed; generate the Kth seed by multiplying
c******* the K-1st seed by the multiplier raised to the Nth power.
         call ranfk( N, K )
         call ranfkbinary( K, KBinary )
         call ranfatok( Multiplier, KBinary, atothek )
         do 100 I = 2, N
            InSeed( 1 ) = SeedArray( I-1, 1 )
            InSeed( 2 ) = SeedArray( I-1, 2 )
            InSeed( 3 ) = SeedArray( I-1, 3 )
            InSeed( 4 ) = SeedArray( I-1, 4 )
            call ranfmodmult( InSeed, atothek, OutSeed )
            SeedArray( I, 1 ) = OutSeed( 1 )
            SeedArray( I, 2 ) = OutSeed( 2 )
            SeedArray( I, 3 ) = OutSeed( 3 )
            SeedArray( I, 4 ) = OutSeed( 4 )
 100     continue
      endif

c     write( *, 110 )
 110  format( '   leaving Rans' )

      return
      end
c     end Rans

c==============================================================================

      subroutine ranfatok( a, Kbinary, atothek )

c------------------------------------------------------------------------------
c
c  This routine calculates a to the Kth power, mod 2**48. K is a binary number.
c
c  It returns the calculated value as an array of four modulo-4096 digits.
c
c------------------------------------------------------------------------------

$include ranftest.parm

      integer a, KBinary, atothek, asubi

      dimension a( IRandNumSize ),
     1          KBinary( IBinarySize ),
     2          atothek( IRandNumSize ),
     3          asubi( IRandNumSize )

c------------------------------------------------------------------------------

c**** The following amounts to the first iteration of a 46-loop.
      asubi( 1 ) = a( 1 )
      asubi( 2 ) = a( 2 )
      asubi( 3 ) = a( 3 )
      asubi( 4 ) = a( 4 )

      atothek( 1 ) = 1
      atothek( 2 ) = IZero
      atothek( 3 ) = IZero
      atothek( 4 ) = IZero

      do 100 I = 1, 45
         if( KBinary( I ) .ne. IZero ) then
            call ranfmodmult( atothek, asubi, atothek )
         endif
         call ranfmodmult( asubi, asubi, asubi )
 100  continue

      return
      end
c     ranfatok

c==============================================================================

      function iranfeven( N )

c------------------------------------------------------------------------------
c
c  This function checks the parity of the argument integer.
c
c  It returns one if the argument is even and zero otherwise.
c
c------------------------------------------------------------------------------

$include ranftest.parm

      integer N

c------------------------------------------------------------------------------

      if( mod( N, 2 ) .eq. 0 ) then
         iranfeven = 1
      else
         iranfeven = 0
      endif

      return
      end
c     end iranfeven

c==============================================================================

      subroutine ranfk( N, K )

c------------------------------------------------------------------------------
c
c  This routine calculates 2**46/N, which should be the period of each of the
c  subsequences of random numbers that are being created. Both the numerator
c  and the result of this calculation are represented as an array of four
c  integers, each of which is one digit of a four-digit moduo-4096 number.  The
c  numerator is represented as (1024, 0, 0, 0 ), using base ten digits.
c
c  It returns the result of the division.
c
c------------------------------------------------------------------------------

$include ranftest.parm

      integer N, K, nn, r4, r3, r2, r1, q4, q3, q2, q1

      dimension K( IRandNumSize )

c------------------------------------------------------------------------------

      nn = N + iranfeven( N )

      q4 = 1024 / nn
      r4 = 1024 - (nn * q4)
      q3 = (r4 * 4096) / nn
      r3 = (r4 * 4096) - (nn * q3)
      q2 = (r3 * 4096) / nn
      r2 = (r3 * 4096) - (nn * q2)
      q1 = (r2 * 4096) / nn

      K( 1 ) = q1
      K( 2 ) = q2
      K( 3 ) = q3
      K( 4 ) = q4

      return
      end
c     end ranfk

c==============================================================================

      subroutine ranfkbinary( K, KBinary )

c------------------------------------------------------------------------------
c
c  This routine calculates the binary expansion of the argument K, which is a
c  48-bit integer represented as an array of four 12-bit integers.
c
c  It returns an array of 48 binary values.
c
c------------------------------------------------------------------------------

$include ranftest.parm

      integer K, KBinary, X, Bits

      dimension K( IRandNumSize ),
     1          KBinary( IBinarySize ),
     2          Bits( Mod4096DigitSize )

c------------------------------------------------------------------------------

      do 300 I = 1, 4
         X = K( I ) / 2
         Bits( 1 ) = iranfodd( K( I ) )

         do 100 J = 2, Mod4096DigitSize 
            Bits( J ) = iranfodd( X )
            X = X / 2
 100     continue

         do 200 J = 1, Mod4096DigitSize
            KBinary( (I-1)*Mod4096DigitSize + J ) = Bits( J )
 200     continue

 300  continue

      return
      end
c     end ranfkbinary

c==============================================================================

      subroutine ranfmodmult( A, B, C )

c------------------------------------------------------------------------------
c
c  Ths routine calculates the product of the first two arguments.  All three
c  arguements are represented as arrays of 12-bit integers, each making up
c  the digits of one radix-4096 number.  The multiplication is done piecemeal.
c
c  It returns the product in the third argument.
c
c------------------------------------------------------------------------------

$include ranftest.parm

      integer A, B, C, j1, j2, j3, j4, k1, k2, k3, k4

      dimension A( IRandNumSize ),
     1          B( IRandNumSize ),
     2          C( IRandNumSize )

c------------------------------------------------------------------------------

c     write( *, 10 )
 10   format( ' ranfmodmult', /, '------------' )

      j1 = A( 1 ) * B( 1 )
      j2 = A( 1 ) * B( 2 ) + A( 2 ) * B( 1 )
      j3 = A( 1 ) * B( 3 ) + A( 2 ) * B( 2 ) + A( 3 ) * B( 1 )
      j4 = A( 1 ) * B( 4 ) + A( 2 ) * B( 3 ) + A( 3 ) * B( 2 ) + A( 4 ) * B( 1 )

      k1 = j1
      k2 = j2 + k1 / 4096
      k3 = j3 + k2 / 4096
      k4 = j4 + k3 / 4096

      C( 1 ) = mod( k1, 4096 )
      C( 2 ) = mod( k2, 4096 )
      C( 3 ) = mod( k3, 4096 )
      C( 4 ) = mod( k4, 4096 )

c     write( *, 20 )
 20   format( '  leaving ranfmodmult' )

      return
      end
c     end ranfmodmult

c==============================================================================

      function iranfodd( N )

c------------------------------------------------------------------------------
c
c  This function checks the parity of the argument integer.
c
c  It returns one if the argument is odd and zero otherwise.
c
c------------------------------------------------------------------------------

$include ranftest.parm

      integer N

c------------------------------------------------------------------------------

      if( mod( N, 2 ) .eq. 0 ) then
         iranfeven = 0
      else
         iranfeven = 1
      endif

      return
      end
c     end iranfodd

c==============================================================================

