signature MonoRefVector =
sig

(* VARIABLE VECTORS OF A NAMED TYPE

   Created by:	Dave Berry, LFCS, University of Edinburgh
		db@lfcs.ed.ac.uk
   Date:	12 Feb 1990

   Maintenance:	Author


   DESCRIPTION

   An MonoRefVector is a single-dimensional array of objects that can be
   updated in place.  The first element has the index 0.  Different
   MonoRefVectors are not equal, even if they contain the same elements.
   MonoRefVectors always admit equality.  There is one empty MonoRefVector
   for each application of the functor.

   To create a MonoRefVector of MonoRefVectors, use the "generate" functions.

   Example MonoRefVectors are ByteRefVectors and BoolVectors, which will often
   be implemented specially.  For example, a BoolRefVector can use one bit
   per element.


   SEE ALSO

   RefVector, MonoVector.
*)

  val version: real
   (* version = 0.1 *)


(* TYPES *)

  type Object
  type MonoRefVector


(* CREATORS *)

  val create: Nat -> Object -> MonoRefVector
   (* create n i; create a MonoRefVector of n locations, each containing i.
      Raise (Nat ("create", n)) if n < 0. *)

  val generate: Nat -> (int -> Object) -> MonoRefVector
   (* generate n f; create a MonoRefVector v of n locations, (v sub 0) to
      (v sub (n-1)) with (v sub i) initialised to (f i).
      Raise (Nat ("generate", n)) if n < 0. *)

  val generate': Nat -> ('b -> Object * 'b) -> 'b -> MonoRefVector
   (* generate' n f base; create a MonoRefVector of n locations, (v sub 0) to
      (v sub (n-1)) with (v sub 0) initialised to (# 1 (f base)) and
      (v sub i (i > 0)) initialised to (# 1 (f (# 2 f_i))), where f_i is
      the result of the i^th application of f.
      Raise (Nat ("generate'", n)) if n < 0. *)


(* CONVERTERS *)

  val stringSep: string -> string -> string ->
		 MonoRefVector -> string

  val string: MonoRefVector -> string

  val parseSepN': string -> string -> string -> Nat -> string list ->
                  (MonoRefVector * string list,
                   MonoRefVector Option * string list) Result

  val parseSep': string -> string -> string -> string list ->
                 (MonoRefVector * string list,
                  MonoRefVector Option * string list) Result

  val parseN': Nat -> string list ->
               (MonoRefVector * string list,
                MonoRefVector Option * string list) Result

  val parse': string list ->
	      (MonoRefVector * string list,
	       MonoRefVector Option * string list) Result

  val parseSepN: string -> string -> string -> Nat ->
                 string -> (MonoRefVector, MonoRefVector Option) Result

  val parseSep: string -> string -> string ->
                string -> (MonoRefVector, MonoRefVector Option) Result

  val parseN: Nat -> string -> (MonoRefVector, MonoRefVector Option) Result

  val parse: string -> (MonoRefVector, MonoRefVector Option) Result

  val readSep: string -> string -> string -> InStream ->
               (MonoRefVector, MonoRefVector Option) Result

  val readSepN: string -> string -> string -> Nat -> InStream ->
               (MonoRefVector, MonoRefVector Option) Result

  val readN: Nat -> InStream -> (MonoRefVector, MonoRefVector Option) Result

  val read: InStream -> (MonoRefVector, MonoRefVector Option) Result

  val list: MonoRefVector -> Object list
   (* list v; make a list containing (only) the elements of v, in
      the same order. *)

  val fromList: Object list -> MonoRefVector
   (* from_list l; make a Vector containing (only) the elements of l, in
      the same order. *)

  val fromFile: string -> MonoRefVector
   (* fromFile name; read the contents of the file called name into an
      MonoRefVector.  Stops reading the file as soon as a read of an Object
      returns Fail. Raises General.Io if something goes wrong. *)

  val file: string -> MonoRefVector -> unit
   (* file name v; write the contents of v to the new file called name.
      Raises General.Io if something goes wrong. *)


(* OBSERVERS *)

  val empty: MonoRefVector -> bool
   (* empty v; returns true if v is empty. *)

  val size: MonoRefVector -> Nat
   (* size v; return the number of elements in v. *)

  val same: MonoRefVector -> MonoRefVector -> bool
   (* same x y; returns true if x and y are the same MonoRefVector. *)

  val different: MonoRefVector -> MonoRefVector -> bool
   (* different x y; returns true if x and y are not the same MonoRefVector. *)

  val eq: MonoRefVector -> MonoRefVector -> bool
   (* eq x y; returns true if (size x = size y) and for all i,
         0 <= i <= size x, (Object.eq (x sub i) (y sub i)). *)

  val ne: MonoRefVector -> MonoRefVector -> bool
   (* ne x y; returns true if (size x = size y) and for all i,
         0 <= i <= size x, (Object.ne (x sub i) (y sub i)). *)

  val lt: MonoRefVector -> MonoRefVector -> bool
   (* lt x y; returns true if (Object.lt (x sub i) (y sub i)) for
         0 <= i <= size x or if x is empty and y is not empty. *)

  val le: MonoRefVector -> MonoRefVector -> bool
   (* le x y; returns true if (Object.le (x sub i) (y sub i)) for
         0 <= i <= size x or if x is empty. *)

  val gt: MonoRefVector -> MonoRefVector -> bool
   (* gt x y; returns true if (Object.gt (x sub i) (y sub i)) for
         0 <= i <= size y or if y is empty and x is not empty. *)

  val ge: MonoRefVector -> MonoRefVector -> bool
   (* ge x y; returns true if (Object.ge (x sub i) (y sub i)) for
         0 <= i <= size y or if y is empty. *)


(* SELECTORS *)

  exception Sub of string * int

  (* infix 9 sub *)
  val sub: MonoRefVector * int -> Object
   (* v sub n; return the n+1'th element of v.
      Raise (Sub ("sub", n)) if not (0 <= n <= size v). *)

  val nth: int -> MonoRefVector -> Object
   (* nth n v; return the n+1'th element of v.
      Raise (Sub ("nth", n)) if not (0 <= n <= size v). *)

  exception Extract of int * int
  val extract: int -> int -> MonoRefVector -> MonoRefVector
   (* extract start finish v; returns the sub-vector of v starting with
      (v sub start) and ending with (v sub (finish - 1)).
      Returns the empty vector if (start = finish).
      Raise (Extract (start, finish)) if not (0 <= start,finish <= size v). *)


(* MANIPULATORS *)

  val rev: MonoRefVector -> MonoRefVector
   (* rev v; builds a MonoRefVector containing the elements of v in
      reverse order. *)

  (* infix 6 ^ *)
  val ^ : MonoRefVector * MonoRefVector -> MonoRefVector
   (* v ^ v'; builds a new MonoRefVector containing the elements of v' appended
      to those of v. *)

  exception Update of int
  val update: int -> Object -> MonoRefVector -> unit
   (* update n i v; replace (v sub n) with i.
      Raise (Update n) if not (0 <= n <= size v). *)

  exception Copy of int * int * int
  val copy: int -> int -> MonoRefVector -> int -> MonoRefVector -> unit
   (* copy start finish v start' v'; copies the sub-vector of v starting with
      (v sub start) and ending with (v sub (finish - 1)) to the MonoRefVector
      v', starting with (v' sub start').  Has no effect if (start = finish).
      Raises (Copy (start, finish, start')) if
      not (0 <= start,finish <= size v) or if
      not (0 <= start',start'+finish-start <= size v'). *)

  exception UpdateRange of int * int
  val updateRange: int -> int -> Object -> MonoRefVector -> unit
   (* updateRange start finish i v; update the elements of v starting with
      (v sub start) and ending with (v sub (finish - 1)) with i.  Has no effect
      if (start = finish).  Raises (UpdateRange (start, finish)) if
      not (0 <= start,finish <= size v). *)

  val sort: MonoRefVector -> unit
     (* sort p v; sorts v in place by Object.lt. *)


(* REDUCERS *)

  val foldR: (Object -> 'b -> 'b) -> 'b -> MonoRefVector -> 'b
   (* foldR f base v; folds using f associating to the right over the
      base element.
      foldR f [a1,a2,...,an] base = f(a1,f(a2,...,f(an,base)...)). *)

  val foldL: (Object -> 'b -> 'b) -> 'b -> MonoRefVector -> 'b
   (* foldL f v base; folds using f associating to the left over the
      base element.
      foldL f [a1,a2,...,an] base = f(an,...,f(a2,f(a1,base))...). *)

  exception Empty of string

  val foldR': (Object -> Object -> Object) -> MonoRefVector -> Object
   (* foldR' f v; folds using f associating to the right over the
      last element of v.  Raises (Empty "foldR'") if v is empty. *)

  val foldL': (Object -> Object -> Object) -> MonoRefVector -> Object
   (* foldL' f v; folds using f associating to the right over the
      last element of v.  Raises (Empty "foldL'") if v is empty. *)

  val pairwise: (Object -> Object -> bool) -> MonoRefVector -> bool
   (* pairwise f v; true if (f (v sub i) (v sub (i + 1))) is true for all
      0 <= i < size v, or if v is empty. *)


(* ITERATORS *)

  val map: (Object -> Object) -> MonoRefVector -> MonoRefVector
   (* map f v; builds a new vector by applying f to each element of v. *)

  val apply: (Object -> unit) -> MonoRefVector -> unit
   (* apply f v; applies f to each element of v. *)

  val iterate: (Object * int -> Object) -> MonoRefVector -> MonoRefVector
   (* iterate f v; builds a new vector by applying f to each element of v
      paired with its index. *)

  val iterateApply: (Object * int -> unit) -> MonoRefVector -> unit
   (* iterate f v; applies f to each element of v paired with its index. *)
end
