(*$List : List' StringType String GeneralTypes InStreamType *)

loadSig "List";

structure List: List =

(* LIST CONVERSION FUNCTIONS

   Created by:  Dave Berry, LFCS, University of Edinburgh
                db@lfcs.ed.ac.uk
   Date:        4 Oct 1989

   Maintenance: Author


   DESCRIPTION

   These functions are separated from the other basic functions on lists
   so that the other functions can be used by Vector, which is used in turn
   by StringType, which is used by InStream, which is used here.
*)

struct

  open List'

  val version = 0.1


(* SEQUENCE *)

  type 'a T = 'a list

  exception Sep of string * string * string * string

  local
    (* checkSep checks that the starting, finishing and separating symbols
       are either the empty string or a single visible character. *)

    fun checkSep' "" = true
    |   checkSep' s = (String.size s = 1 andalso StringType.isVisible s)

    fun checkSep start finish sep error =
	  if checkSep' start andalso
	     checkSep' finish andalso
	     (checkSep' sep orelse sep = " ")
	  then ()
	  else raise Sep (error, start, finish, sep)
	 
    (* dec is used to decrement the count of the number of elements to be
       parsed.  When ~1 the count is to be ignored; dec ~1 = ~1 to
       avoid any possibility of overflow. *)
    fun dec ~1 = ~1
    |   dec  n = n - 1

    (* parseEntries parses the elements of a list in the case that
       finish <> sep and finish <> "".  It assumes that the starting symbol
       (if any) has been parsed and parseFirst has checked for an empty list.
       The list must end with a finishing symbol.  *)
    fun parseEntries 0 _ _ _ l = Fail ([], l)
    |   parseEntries n finish sep p l =
	  if sep = "" orelse sep = " " then
	    ( case p l of
	        Fail _ => Fail ([], l)
	      | OK (x, []) => Fail ([x], [])
	      | OK (x, l' as h::t) =>
	          if h = finish then
		    if n = 1 orelse n = ~1
		    then OK ([x], t)
		    else Fail ([x], t)
	          else
	            case parseEntries n finish "" p l' of
		      OK (x', l2) => OK (x :: x', l2)
		    | Fail (x', l2) => Fail (x :: x', l2)
	     )
	  else
	    ( case p l of 
	        Fail _ => Fail ([], l)
	      | OK (x, []) => Fail ([x], [])
	      | OK (x, l') =>
	          case dropPrefix (not o StringType.isVisible) l' of
		    [] => Fail ([x], l')
	          | l2 as h'::t' =>
		      if h' = sep then
		        case parseEntries n finish sep p t' of
		          OK (x', l3) => OK (x :: x', l3)
		        | Fail (x', l3) => Fail (x :: x', l3)
		      else if h' = finish andalso
		              (n = 1 orelse n = ~1)
		      then OK ([], t')
		      else Fail ([x], l2)
	     )

    (* parseEntries' parses elements in the cases that finish = sep or
       finish = "".  If the first argument is ~1, it parses until the end
       of the input; otherwise it parses that number of elements.  It assumes
       that if sep <> "" then leading whitespace has been skipped before the
       call. *)
    fun parseEntries' 0 _ _ _ l = OK ([], l)
    |   parseEntries' ~1 _ _ _ [] = OK ([], [])
    |   parseEntries' _ _ _ _ [] = Fail ([], [])
    |   parseEntries' n "" "" p l =
	  ( case p l of
	      Fail _ => Fail ([], l)
	    | OK (x, []) =>
	        if n = ~1 orelse n = 1 then OK ([x], [])
	        else Fail ([x], [])
	    | OK (x, l') =>
	        case parseEntries' (dec n) "" "" p l' of
		  Fail (x', l') => Fail (x :: x', l')
	        | OK (x', l') => OK (x :: x', l')
	  )
    |   parseEntries' n "" " " p l =
	  ( case p l of
	      Fail _ => Fail ([], l)
	    | OK (x, []) =>
	        if (n = ~1 orelse n = 1)
	        then OK ([x], [])
	        else Fail ([x], [])
	    | OK (x, l') =>
		  case parseEntries' (dec n) "" " " p
		 	     (dropPrefix (not o StringType.isVisible) l') of
	            Fail (x', l3) => Fail (x :: x', l3)
		  | OK (x', l3) => OK (x :: x', l3)
	  )
    |   parseEntries' n finish sep p l =
	 (case p l of
	    Fail _ => Fail ([], l)
	  | OK (x, l') =>
	      case dropPrefix (not o StringType.isVisible) l' of
	        [] =>
	          if finish = "" andalso (n = ~1 orelse n = 1)
	          then OK ([x], [])
	          else Fail ([x], [])
	      | l2 as h::t =>
		  if n = 1 andalso finish = "" then OK ([x], l2)
		  else if h = sep then
		    case parseEntries' (dec n) finish sep p
		 		 (dropPrefix (not o StringType.isVisible) t) of
		      Fail (x', l3) => Fail (x :: x', l3)
		    | OK (x', l3) => OK (x :: x', l3)
		  else Fail ([x], l2)
	 )
	   
    (* parseFirst is called to parse the first element in the list.  It is
       needed because the list might be empty, so it has to check for a
       finishing symbol.  parseFirst itself just skips leading whitespace
       if neccessary and calls parseFirst'. *)
    fun parseFirst' n finish sep p l =
	  case (l, finish) of
	    ([], "") => if n > 0 then Fail (Some [], l) else OK ([], [])
	  | ([], _) => Fail (Some [], l)
	  | (h::t, _) => 
	      if h = finish then
		if n > 0
		then Fail (Some [], l)
		else OK ([], t)
	      else if n = 0 then Fail (Some [], l)
	      else
		let val res =
		      if sep = finish orelse finish = ""
		      then parseEntries' n finish sep p l
	  	      else parseEntries n finish sep p l
		in case res of
		     Fail (l, l') => Fail (Some l, l')
		   | OK x => OK x
		end

    fun parseFirst 0 "" _ _ l = OK ([], l)
    |   parseFirst n finish "" p l =
	  parseFirst' n finish "" p l
    |   parseFirst n finish sep p l =
	  let val l' = dropPrefix (not o StringType.isVisible) l
	  in parseFirst' n finish sep p l'
	  end
    
    (* parseStart is called to parse a starting symbol after leading whitespace
       has been skipped.  The start symbol is known not to be "". *)
    fun parseStart n start finish sep p [] = Fail (Some [], [])
    |   parseStart n start finish sep p (l as c::s) =
	  if c <> start then Fail (Some [], l)
	  else parseFirst n finish sep p s

    (* parseList is the main parse function that the user visible functions
       call.  It takes the foillowing arguments:
	 n - the integer parameter to parseSepN or parseSepN'.  This is ~1
	     	when parseList is called from parse, parseSep, parse' etc.
	 start - the starting symbol.
	 finish - the finishing symbol.
	 sep - the separating symbol.
		These symbols may be "" or a single printable character.
	 p - the function to parse an element.
	 l - the string list to be parsed.
       
       parseList matches cases of the start symbol.  If start = "", it calls
       parseFirst, as if a start symbol had been parsed.  Otherwise it skips
       leading whitespace and calls parseStart to parse the starting symbol. *)

    fun parseList n "" finish sep p l =
	  parseFirst n finish sep p l
    |   parseList n start finish sep p l =
	  parseStart n start finish sep p
		         (dropPrefix (not o StringType.isVisible) l)

    (* readEntries reads the elements of a list in the case that finish <> sep
       and finish <> "".  It assumes that the starting symbol (if any) have
       been read and that readFirst has checked for EOF.  The list must
       end with a finishing symbol. *)
    fun readEntries 0 _ _ _ _ = Fail []
    |   readEntries n finish sep p i =
	  if sep = "" orelse sep = " " then
	  ( case p i of
	      Fail _ => Fail []
	    | OK x =>
		if InStream.eof i then Fail [x]
	        else if InStream.lookahead i = finish then 
		  if n = 1 orelse n = ~1
		  then (InStream.input1 i; OK [x])
		  else Fail [x]
	        else
		  case readEntries (dec n) finish "" p i of
		    OK l => OK (x :: l)
		  | Fail l => Fail (x :: l)
	  )
	  else
	    case p i of
	      Fail _ => Fail []
	    | OK x =>
	        if InStream.eof i then Fail []
	        else
	        ( InStream.skip (not o StringType.isVisible) i;
		  if InStream.eof i then Fail [x]
	          else
		    let val c = InStream.input1 i
		    in if c = sep then
		         case readEntries (dec n) finish sep p i of
		           OK l => OK (x :: l)
		         | Fail l => Fail (x :: l)
	              else if c = finish andalso (n = 1 orelse n = ~1)
		      then OK [x]
	              else Fail [x]
		    end
	        )


    (* readEntries' reads elements of a list in the cases that finish = sep or
       finish = "".  If the first argument is ~1, it parses until the end
       of the input; otherwise it reads that number of elements.  It assumes
       that leading whitespace has been skipped before the call. *)
    fun readEntries' 0 _ _ _ _ = OK []
    |   readEntries' n "" "" p i =
	( if InStream.eof i orelse InStream.lookahead i = "" then
	    if n = ~1 then OK [] else Fail []
	  else
	    case p i of
	      Fail _ =>
		if InStream.eof i andalso n = ~1 then OK [] else Fail []
	    | OK x =>
		if InStream.eof i then
		  if n = 1 orelse n = ~1 then OK [x] else Fail [x]
		else
		  case readEntries' (dec n) "" "" p i of
		    OK l => OK (x :: l)
		  | Fail l => Fail (x :: l)
	)
    |   readEntries' n "" " " p i =
	( if InStream.eof i orelse InStream.lookahead i = "" then
	    if n = ~1 then OK [] else Fail []
	  else
	    case p i of
	      Fail _ =>
		if InStream.eof i andalso n = ~1 then OK [] else Fail []
	    | OK x =>
		if InStream.eof i then
		  if n = 1 orelse n = ~1 then OK [x] else Fail [x]
		else
	        ( InStream.skip (not o StringType.isVisible) i;
		  case readEntries' (dec n) "" "" p i of
		    OK l => OK (x :: l)
		  | Fail l => Fail (x :: l)
		)
	)
    |   readEntries' n finish sep p i =
	  if InStream.eof i orelse InStream.lookahead i = "" then
	    if n = ~1 then OK [] else Fail []
	  else
	    case p i of
	      Fail _ =>
		if InStream.eof i andalso n = ~1 then OK [] else Fail []
	    | OK x =>
		if InStream.eof i then
		  if finish = "" andalso (n = 1 orelse n = ~1)
		  then OK [x]
		  else Fail [x]
		else if n = 1 andalso finish = "" then OK [x]
		else
	        ( InStream.skip (not o StringType.isVisible) i;
		  if not (InStream.eof i) then
	            if InStream.lookahead i = sep then
		      if n = 1 then (InStream.input1 i; OK [x])
		      else
	              ( InStream.input1 i;
			InStream.skip (not o StringType.isVisible) i;
			if InStream.eof i then
	    		  if n = ~1 orelse n = 1 then OK [x] else Fail [x]
			else
		          case readEntries' (dec n) finish sep p i of
		            OK l => OK (x :: l)
		          | Fail l => Fail (x :: l)
		      )
	            else Fail [x]
		  else if finish = "" andalso (n = 1 orelse n = ~1)
		  then OK [x]
		  else Fail [x]
		)

    (* readFirst is called to read the first element in the list.  It is
       needed because the list might be empty, so it has to check for a
       finishing symbol.  readFirst itself just skips leading whitespace
       if neccessary and calls readFirst'. *)
    fun readFirst' n finish sep p i =
	( if InStream.lookahead i = finish then
	    if n < 1 then
	      ( if finish <> "" then InStream.input1 i else "";
	        OK []
	      )
	    else Fail (Some [])
	  else if n = 0 then Fail (Some [])
	  else let
	    val res = if finish = "" orelse finish = sep
	  	      then readEntries' n finish sep p i
	  	      else readEntries n finish sep p i
	  in case res of
	       Fail x => Fail (Some x)
	     | OK x => OK x
	  end
	)

    fun readFirst 0 "" _ _ _ = OK []
    |   readFirst n finish "" p i =
          readFirst' n finish "" p i
    |   readFirst n finish sep p i =
	( InStream.skip (not o StringType.isVisible) i;
	  if InStream.eof i then
	    if finish = "" andalso n < 1 then OK [] else Fail (Some [])
	  else
            readFirst' n finish sep p i
	)

    (* start <> "" in readStart *)
    fun readStart n start finish sep p i =
	( InStream.skip (not o StringType.isVisible) i;
	  if InStream.eof i then
	    if finish = "" andalso n < 1 then OK [] else Fail (Some [])
	  else if InStream.input1 i = start
    	  then readFirst n finish sep p i
	  else Fail (Some [])
        )

    fun readList n "" finish sep p i =
	  readFirst n finish sep p i
    |   readList n start finish sep p i =
	  readStart n start finish sep p i
  in
    fun parseSep' start finish sep p l =
	  ( checkSep start finish sep "parseSep'";
    	    parseList ~1 start finish sep p l
	  )

    fun parseSepN' start finish sep p n l =
	  if n < 0 then raise General.Nat ("parseSepN'", n)
	  else
	  ( checkSep start finish sep "parseSepN'";
	    parseList n start finish sep p l
	  )

    fun parse' p l = parseSep' ("[") ("]") (",") p l

    fun parseN' p n l = parseSepN' ("[") ("]") (",") p n l

    fun parseSep start finish sep p s =
	( checkSep start finish sep "parseSep";
	  case parseList ~1 start finish sep p (explode s) of
	    OK (l, _) => OK l
	  | Fail (l, _) => Fail l
	)

    fun parseSepN start finish sep p n s =
	  if n < 0 then raise General.Nat ("parseSepN", n)
	  else
	  ( checkSep start finish sep "parseSepN";
	    case parseList n start finish sep p (explode s) of
	      OK (l, _) => OK l
	    | Fail (l, _) => Fail l
	  )

    fun parse p s = parseSep ("[") ("]") (",") p s

    fun parseN p n s = parseSepN ("[") ("]") (",") p n s

    fun readSep start finish sep p i =
	( checkSep start finish sep "readSep";
	  readList ~1 start finish sep p i
	)

    fun readSepN start finish sep p n i =
	  if n < 0 then raise General.Nat ("readSepN", n)
	  else
	  ( checkSep start finish sep "readSepN";
	    readList n start finish sep p i
	  )
	  
    fun read p i = readSep ("[") ("]") (",") p i
	  
    fun readN p n i = readSepN ("[") ("]") (",") p n i
  end

end
