
loadSig "InStream";

structure InStream: InStream =

(* INPUT STREAMS

   Created by:  Dave Berry, LFCS, University of Edinburgh
                db@lfcs.ed.ac.uk
   Date:        12 Nov 1989

   Maintenance: Author


   SEE ALSO

   Outstream, StreamPair
*)

struct

  val version = 0.1


(* TYPES *)

  datatype InStream = I of instream * (bool ref) * (bool ref)
   (* The second field is true iff the last call to input or lookahead
      returned "".  The last field is true iff the stream is closed. *)

  type instream = InStream


(* SYSTEM *)

  exception NotImplemented of string

  fun openString s = raise NotImplemented "openString"
  fun canInput i n = raise NotImplemented "canInput"
  fun reset i = raise NotImplemented "reset"
  fun interactive i = raise NotImplemented "interactive"
  fun input' i n = raise NotImplemented "input'"


(* MANIPULATORS *)

  val stdIn =
	I (std_in, ref false, ref false)
  val std_in = stdIn

  fun openIn s =
	I (open_in s, ref false, ref false)
	handle Io _ => raise General.Io ("openIn", s, "")
  val open_in = openIn

  fun closeIn (i as I (i', _, closed)) =
	(close_in i'; closed := true)
  val close_in = closeIn

  val old_input = input
  local
    fun inp (I (i', _, ref true)) n = ""
    |   inp (I (i', eof, _)) n =
	  let val s = old_input (i', n)
	  in if s = "" then eof := true else eof := false;
	     s
	  end
  in
    fun input (i, n) =
          if n < 0 then raise General.Nat ("input", n)
          else inp i n
  end

  fun input1 i = input (i, 1)

  val old_lookahead = lookahead
  fun lookahead (I (i', _, ref true)) = ""
  |   lookahead (I (i', eof, _)) =
	let val s = old_lookahead i'
	in if s = "" then eof := true else eof:= false;
	   s
	end

  fun end_of_stream (I (_, ref e, _)) = e
  val eof = end_of_stream

  local
    fun readString' _ [] = OK ()
    |   readString' i (h::t) =
	  case input1 i
	  of "" => Fail ""
	  |  c  =>
	      if h = c then 
		case readString' i t of
		  OK () => OK ()
		| Fail s => Fail (c ^ s)
	      else
		Fail c
  in
    fun readString i s = readString' i (explode s)
  end

  fun skip p i =
	case lookahead i
	of "" => ()
	|  s  => if p s then (input1 i; skip p i) else ()

  local
    fun line i s =
	case input1 i of
	   ""  => s
	| "\n" => s ^ "\n"
	|   c  => line i (s ^ c)
  in
    fun inputLine i = line i ""
  end
end
