;;; -*- Mode: LISP; Syntax: Common-Lisp; Package: DEMO-ARITH; Base: 10 -*-
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;The Garnet User Interface Development Environment
;;;Copyright (c) 1989, 1990 Carnegie Mellon University
;;;All rights reserved.  The CMU software License Agreement specifies
;;;the terms and conditions for use and redistribution.
;;;
;;;If you want to use this code or anything developed as part of the
;;;Garnet project, please contact Brad Myers (Brad.Myers@CS.CMU.EDU).
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;This file is a sample of a visual programming arithmetic expression
;;; editor created with Garnet.
;;;
;;;** Call (demo-arith:Do-Go) to start and (demo-arith:Do-Stop) to stop **
;;;
;;;Designed and implemented by Brad A. Myers

#|
==================================================================
Change log:
      9/4/90 Brad Myers - add load of scrolling-window
      8/21/90 Brad Myers - first character typed starts string over 
      7/04/90 Brad Myers - started based on demo-editor
==================================================================
|#

;;;  Load text-buttons-loader, graphics-loader, and arrow-line-loader unless
;;;  already loaded
;;;
(dolist (gadget '("text-buttons-loader" "arrow-line-loader"
		  "scrolling-window-loader"
		  ))
  (load (merge-pathnames gadget
			 #+cmu "gadgets:"
			 #+(not cmu) user::Garnet-Gadgets-PathName)))

(in-package "DEMO-ARITH" :use '("KR" "LISP"))

(export '(Do-Go Do-Stop))

;;;------------------------------------------------------------------------
;;;Global variables
;;;------------------------------------------------------------------------

(defparameter *Mode-Menu* NIL) ; menu of object types to create
(defparameter *Selection-Obj* NIL) ; the object that holds the selection
(defparameter *Objs-Agg* NIL) ; aggregate to hold the created objects
(defparameter *current-window* NIL) ; main, top-level window

;;;------------------------------------------------------------------------
;;;Utility Functions
;;;------------------------------------------------------------------------

(defun Init-Slot (obj slot new-val)
  (g-value obj slot) ; need to do this to set up the dependencies
  (s-value obj slot new-val))

; convert s to an integer or return NIL
(defun Make-Number (s)
  (let* ((sym (read-from-string s))
	 (number (when (numberp sym) sym)))
    number))

; check if a function
(defun My-Function-p (s)
  (if (symbolp s)
      ; need all 3 tests to do it right!
      (fboundp s)
      (functionp s)))

(defun Protected-Divide (&rest args)
  (if (null args)
      '**
      (progn
	(dolist (i args)
	  (when (zerop i)
	    (return-from Protected-Divide '**)))
	(apply '/ args))))

;;;------------------------------------------------------------------------
;;;First create the prototypes for the box and lines
;;;------------------------------------------------------------------------

(create-instance 'myarrowline garnet-gadgets:arrow-line
		 (:from-obj NIL) ;set this with the object this arrow is from
		 (:to-obj NIL)   ;set this with the object this arrow is from
		 (:x1 (o-formula (opal:gv-right (gvl :from-obj))))
		 (:y1 (o-formula (opal:gv-center-y (gvl :from-obj))))
		 (:x2 (o-formula (gvl :to-obj :left)))
		 (:y2 (o-formula (opal:gv-center-y (gvl :to-obj))))
		 (:value (o-formula (gvl :from-obj :value)))
		 (:open-p t) 
		 (:visible (o-formula (and (gvl :from-obj)(gvl :to-obj))))
		 (:line-p T) ;so that the selection object will know what
			     ; kind this is
		 )

(create-instance 'arith-box opal:aggregadget
		 (:box (list 20 20 NIL NIL)) ; this will be set by the
					     ; interactors with the position
					     ; of this object.

		 (:left (o-formula (first (gvl :box))))
		 (:top (o-formula (second (gvl :box))))
		 (:width 30)
		 (:height 30)

		 (:lines-to-me NIL)   ;Keep track of lines pointing
		 (:lines-from-me NIL) ;to me, in case I am deleted.

		 (:editable NIL)
		 (:line-p NIL) ;so that the selection object will know what
			       ; kind this is
		 (:func T) ; the function to execute or numerical value
		 (:func-to-execute (o-formula (gvl :func)))
		 (:string (o-formula (symbol-name (gvl :func))))
		 (:value (o-formula (let ((func (gvl :func-to-execute))
					  in-vals val final-val)
				      (dolist (i (gvl :lines-to-me))
					(setq val (gv i :value))
					(if (numberp val)
					    (push val in-vals)
					    (setq final-val '**)))
				      (or final-val
					  (if (my-function-p func)
					      (apply func in-vals)
					      '**)))))
		 (:parts
		  `((:frame ,opal:circle
			(:left ,(o-formula (first (gvl :parent :box))))
			(:top ,(o-formula (second (gvl :parent :box))))
			(:diam ,(o-formula (max (gvl :parent :width)
						(gvl :parent :height))))
			(:width ,(o-formula (gvl :diam)))
			(:height ,(o-formula (gvl :diam))))
		    (:label ,opal:text
			(:string ,(o-formula (gvl :parent :string) ""))
			(:actual-heightp T)
			(:font ,(create-instance NIL opal:font
						(:size :very-large)
						(:face :bold)))
			;;center me in the box
			(:left ,(o-formula (opal:gv-center-x-is-center-of
					    (gvl :parent))))
			(:top ,(o-formula (opal:gv-center-y-is-center-of
					   (gvl :parent))))))))

(create-instance 'plus-box arith-box
		 (:func '+))
(create-instance 'minus-box arith-box
		 (:func '-))
(create-instance 'times-box arith-box
		 (:func '*))
(create-instance 'divide-box arith-box
		 (:func '/)
		 (:func-to-execute 'Protected-Divide))



(create-instance 'number-box arith-box
		 (:editable (o-formula (null (gvl :lines-to-me))))
		 (:width (o-formula (+ 10 (gvl :label :width))))
		 (:height (o-formula (+ 10 (opal:string-height
					   (gvl :label :font)
					   "0" :actual-heightp T))))
		 (:string (o-formula (let ((val (gvl :value)))
				       (if (numberp val)
					   (if (integerp val)
					       (format NIL "~a" val)
					       ; else floating point
					       (if (zerop val)
						   "0.0"
						   (format NIL "~0,2F" val)))
					   ; else not a number
					   (symbol-name val)))))
		 (:func 0) ; set by text-interactor if string value edited
		 (:value (o-formula (let ((in-lines (gvl :lines-to-me)))
				      (cond ((null in-lines)
					     (gvl :func))
					    ((> (length in-lines) 1)
					     ">1")
					    (T (gv (car in-lines) :value))))))
		 (:parts `((:frame ,opal:roundtangle
			    (:width ,(o-formula (gvl :parent :width)))
			    (:height ,(o-formula (gvl :parent :height)))
			    :inherit (:left :top :line-style :filling-style))
			   (:label ,opal:cursor-text
			    (:cursor-index NIL)
			    :inherit (:string :font :left :top)))))

(defun Set-String-Value (inter obj event string x y)
  (declare (ignore event x y))
  (let (num)
  (if (and (g-value obj :parent :editable)
	   (setq num (Make-Number string)))
      (s-value (g-value obj :parent) :func num)
      ; else bad number or can't be edited
      (progn
	(inter:beep)
	(s-value obj :string (g-value inter :original-string))
	(inter:abort-interactor inter)))))
	

;;;------------------------------------------------------------------------
;;;Create main menu object
;;;------------------------------------------------------------------------

;; Create an arrow, a number-box and one of each kind of operator, and put
;; them in a menu, with an
;; interactor and feedback object to show which is selected.  
;; Agg is the top level aggregate to put the menu in, and window is the window.
(defun create-mode-menu (agg window)
    (setq *Mode-Menu*
	  (create-instance NIL opal:aggregadget
		   (:selected (o-formula (gvl :items :selected)))
		   (:line-p (o-formula (gvl :selected :line-p)))
		   (:parts
		    `((:items ,opal:aggregadget
			  (:parts
			   ((:plus ,plus-box (:box (10 10 NIL NIL)))
			    (:minus ,minus-box (:box (45 10 NIL NIL)))
			    (:times ,times-box (:box (80 10 NIL NIL)))
			    (:minus ,divide-box (:box (115 10 NIL NIL)))
			    (:number ,number-box (:box (10 60 NIL NIL)))
			    (:arrow ,garnet-gadgets:arrow-line
			       (:x1 10)(:y1 130)(:x2 130)(:y2 130)
			       (:line-p T)
			       (:open-p t)
			       (:point-in-gob
				,(g-value opal:aggregate :point-in-gob))))))
		      (:feedback ,opal:rectangle
			 (:line-style NIL)
			 (:obj-over NIL)
			 (:filling-style ,opal:black-fill)
			 (:left  ,(o-formula (- (gvl :obj-over :left) 4)))
			 (:top   ,(o-formula (- (gvl :obj-over :top) 4)))
			 (:width  ,(o-formula (+ (gvl :obj-over :width) 8)))
			 (:height ,(o-formula (+ (gvl :obj-over :height) 8)))
			 (:visible ,(o-formula (gvl :obj-over)))
			 (:draw-function :xor)
			 (:fast-redraw-p T))))
		   (:interactors
		    `((:select-it ,inter:button-interactor
			     (:continuous NIL)
			     (:window ,window)
			     (:how-set :set)
			     (:start-where ,(o-formula
					     (list :element-of
						   (gvl :operates-on :items))))
			     (:start-event :any-mousedown)
			     (:final-feedback-obj
			      ,(o-formula (gvl :operates-on :feedback))))))))
  (opal:add-component agg *Mode-Menu*)
  (let ((init-val (g-value *Mode-Menu* :items :number)))
    (Init-Slot *Mode-Menu* :selected init-val) 
    (Init-Slot (g-value *Mode-Menu* :feedback) :obj-over init-val)))

;;This creates the menu of commands.  
;;The menu is stored into the aggregate agg.  Returns the menu created.
(defun create-menu (agg)
  (opal:add-components agg demo-arith-menu)
#|
  (let ((menu (create-instance NIL Garnet-gadgets:Text-Button-Panel
			(:items '("Delete" "Clear-Workspace" "Quit"))
			(:selection-function 'DoIt)
			(:left 10)
			(:top 200)
			(:font Opal:Default-font)
			(:shadow-offset 5)
			(:final-feedback-p NIL))))
    (opal:add-components agg menu)
|#
    demo-arith-menu))
				 
;;;********************************************************************
;;;Create a selection object and the interactors to manipulate it.
;;; Also, allow objects to be moved
;;;********************************************************************

(defun Create-Selection-Obj (agg-to-put-it-in window)
  (setq *Selection-Obj*
	(create-instance NIL opal:aggregadget
		(:obj-over NIL)
		(:visible (o-formula (gvl :obj-over)))
		(:line-p (o-formula (gvl :obj-over :line-p)))
		(:parts
		 `((:rect ,opal:rectangle
			 (:line-style NIL)
			 (:obj-over ,(o-formula (if (gvl :parent :line-p)
						    NIL
 						    (gvl :parent :obj-over))))
			 (:filling-style ,opal:black-fill)
			 (:left ,(o-formula (- (gvl :obj-over :left) 2)))
			 (:top  ,(o-formula (- (gvl :obj-over :top) 2)))
			 (:width  ,(o-formula (+ (gvl :obj-over :width) 4)))
			 (:height ,(o-formula (+ (gvl :obj-over :height) 4)))
			 (:visible ,(o-formula (gvl :obj-over)))
			 (:draw-function :xor))
		   (:line ,opal:line
			 (:obj-over ,(o-formula (if (gvl :parent :line-p)
 						    (gvl :parent :obj-over)
						    NIL)))
			 (:line-style ,(create-instance NIL opal:line-0
					       (:line-thickness 11)))
			 (:x1 ,(o-formula (gvl :obj-over :x1)))
			 (:y1 ,(o-formula (gvl :obj-over :y1)))
			 (:x2 ,(o-formula (gvl :obj-over :x2)))
			 (:y2 ,(o-formula (gvl :obj-over :y2)))
			 (:visible ,(o-formula (gvl :obj-over)))
			 (:draw-function :xor))))))

  (opal:add-component agg-to-put-it-in *Selection-Obj*)
			  
  (create-instance 'Selector inter:move-grow-interactor
		 (:window window)
		 (:start-where `(:element-of ,*objs-agg* :type ,arith-box))
		 (:feedback-obj NIL)
		 (:final-function
		  #'(lambda (inter obj points)
		      (declare (ignore inter points))
		      (s-value *Selection-Obj* :obj-over obj))))
  (create-instance 'Help-Selection inter:button-interactor
		 (:window window)
		 (:start-where `(:element-of-or-none ,*objs-agg*))
		 (:feedback-obj NIL)
		 (:continuous NIL)
		 (:final-function
		  #'(lambda (inter obj)
		      (declare (ignore inter))
		      (unless (is-a-p obj arith-box) ; then taken care of
						     ; by other interactor
			(s-value *Selection-Obj* :obj-over (if (eq obj :none)
							       NIL
							       obj)))))))

;;;********************************************************************
;;;Procedures to do the work
;;;********************************************************************

;;;Delete-Line is called from delete object to delete lines
(defun Delete-Line(line-obj)
  (let ((from-obj (g-value line-obj :from-obj))
	(to-obj (g-value line-obj :to-obj)))
    ;;remove this line from the boxes' lists
    (s-value from-obj :lines-from-me
  	     (remove line-obj (g-value from-obj :lines-from-me)))
    (s-value to-obj :lines-to-me
  	     (remove line-obj (g-value to-obj :lines-to-me)))
    (opal:destroy line-obj)))
  
(proclaim '(special text-edit))

;;;Delete-object is called from the main menu routine
(defun Delete-Object ()
  (let ((selected-obj (g-value *Selection-Obj* :obj-over)))
    (if selected-obj
      (progn
	;;first turn off selection
	(s-value *Selection-Obj* :obj-over NIL)
	(inter:abort-interactor text-edit) ; just in case running
	;;now delete object
	(if (g-value selected-obj :line-p)
	    ;;then deleting a line
	    (Delete-Line selected-obj)
	    ;;else deleting a box
	    (progn
	      ;;first delete all lines to this box
	      (dolist (line-at-box (g-value selected-obj :lines-from-me))
		(delete-line line-at-box))
	      (dolist (line-at-box (g-value selected-obj :lines-to-me))
		(delete-line line-at-box))
	      ;;now delete the box
	      (opal:destroy selected-obj))))
	;;else nothing selected
	(inter:beep))))

(defun Delete-All ()
  (s-value *Selection-Obj* :obj-over NIL)
  (dolist (obj (copy-list (get-values *objs-agg* :components)))
    (opal:destroy obj)))

(defun Do-Quit ()
  (opal:destroy *Current-Window*)
  ;;if not CMU CommonLisp, then exit the main event loop
  #-cmu (inter:exit-main-event-loop))

(defun DoIt (gadget-obj objover)
  (declare (ignore gadget-obj))
  (let ((value (g-value objover :TEXT-24290 :string)))
  (cond ((string= value "Delete") (Delete-Object))
	((string= value "Quit") (Do-Quit))
	((string= value "Clear-Workspace") (Delete-All))
	(T (error "Bad selection: ~s" value)))))

;;;Create a new object.  Get the type of object to create from the *mode-menu*.
;;;This procedure is called as the final-function of the two-point interactor.
(defun Create-New-Obj (inter point-list)
  (declare (ignore inter))
  (let ((line-p (g-value *Mode-Menu* :line-p))) ;create a line or rectangle

    (if line-p
	;;then create a line, first have to find the objects where the line
	;; is drawn
	(let ((from-box (opal:point-to-component *objs-agg* (first point-list)
				      (second point-list) :type arith-box))
	      (to-box (opal:point-to-component *objs-agg* (third point-list)
				      (fourth point-list) :type arith-box))
	      new-line)
	  ;;If one end of the arrow is not inside a box, or is from and to
	  ;; the same box, or if more than one to a number box, then beep.
	  (if (or (null from-box)(null to-box)(eq from-box to-box)
		  (and (is-a-p to-box number-box)
		       (g-value to-box :lines-to-me))) ; if already exists a
						       ; line to that box
	      (inter:beep)
	      ;; else draw the arrow.
	      (progn
		(setf new-line (create-instance NIL myarrowline
						(:from-obj from-box)
						(:to-obj to-box)))
		;;keep track in case boxes are deleted so can delete this line.
		(push new-line (g-value from-box :lines-from-me))
		(push new-line (g-value to-box :lines-to-me))
		
		(opal:add-component *objs-agg* new-line))))
	;;else, create a new box
	(let* ((typ (g-value *Mode-Menu* :selected))
	       (new-obj (create-instance NIL typ 
		   (:box (copy-list point-list))))) ;have to copy this list
	  (opal:add-component *objs-agg* new-obj)))))

;;;********************************************************************
;;;Main procedures
;;;********************************************************************

(defun Do-Go (&key dont-enter-main-event-loop double-buffered-p)
  (let (top-win scroll-win work-win top-agg work-agg mylinefeedback)
    ;;;create top-level window
    (setf top-win (create-instance NIL inter:interactor-window
			      (:left 10) (:top 10)
                     (:double-buffered-p double-buffered-p)
		     (:width 700) (:height 400)
		     (:title "GARNET Arithmetic Editor")
		     (:icon-title "Arith")))
    (setf *current-window* top-win)

    ;;;create the top level aggregate in the window
    (setq top-agg (create-instance NIL opal:aggregate
		     (:left 0)(:top 0)
		     (:width (formula `(gv ',top-win :width)))
		     (:height (formula `(gv ',top-win :height)))))
    (s-value top-win :aggregate top-agg)

    (opal:update top-win)

    ;;;create window for the work area
    (setf scroll-win
	  (create-instance NIL garnet-gadgets:scrolling-window-with-bars
		     (:left 150)
		     (:top -2) ;no extra border at the top
		     (:width (formula `(- (gv ',top-win :width) 150)))
		     (:height (formula `(gv ',top-win :height)))
		     (:total-width 1000)
		     (:total-height 1000)
		     (:scroll-on-left-p NIL)
		     (:double-buffered-p double-buffered-p)
		     (:border-width 2)
		     (:parent-window top-win)))
    (opal:update scroll-win)
    
    (setq work-agg (g-value scroll-win :inner-aggregate))
    (setq work-win (g-value scroll-win :inner-window))

    ;;;create an aggregate to hold the user-created objects
    (setq *objs-agg* (create-instance NIL opal:aggregate
				      (:left 0)(:top 0)
				      (:width (o-formula (gvl :window :width)))
				      (:height (o-formula (gvl :window :height)))))
    (opal:add-component work-agg *objs-agg*)

    ;;;create menus
    (create-mode-menu top-agg top-win)
    (create-menu top-agg)

    ;;;create a graphics selection object
    (Create-Selection-Obj work-agg work-win)
				     
    ;;;Create an interactor to edit the text of the labels

    (create-instance 'text-edit inter:text-interactor
		 (:active
		  (o-formula (and (gv *selection-obj* :obj-over)
				  (gv *selection-obj* :obj-over :editable))))
		 (:start-event :any-keyboard)
		 (:start-where T)
		 (:obj-to-change (o-formula (gv *selection-obj* :obj-over :label)))
		 (:stop-event '(:any-mousedown #\return))
		 (:window work-win)
		 (:start-action
		  #'(lambda (inter obj event)
		      (call-prototype-method inter obj event)
		      (s-value obj :string (make-string 1 :initial-element
							(inter:event-char event)))
		      (s-value obj :cursor-index 1)))
		 (:final-function 'Set-String-Value))

    (setq mylinefeedback (create-instance NIL Opal:Line
				 (:points (list 0 0 10 10)) 
				 (:obj-over NIL)
				 (:visible (o-formula (gvl :obj-over)))
				 (:x1 (o-formula (first (gvl :points))))
				 (:y1 (o-formula (second (gvl :points))))
				 (:x2 (o-formula (third (gvl :points))))
				 (:y2 (o-formula (fourth (gvl :points))))
				 (:draw-function :xor)
				 (:fast-redraw-p T)
				 (:line-style opal:dashed-line)))

    (opal:add-component work-agg mylinefeedback) 
    ;;;create an interactor to create the new objects
    (create-instance 'creator Inter:Two-Point-Interactor
	     (:start-event :rightdown)
	     (:line-p (o-formula (gv *Mode-Menu* :line-p)))
	     (:continuous (o-formula (gvl :line-p))) ; if line-p then continous
	     (:feedback-obj mylinefeedback)
	     (:start-where T)
	     (:window work-win)
	     (:final-function #'Create-New-Obj))

    ;;;Now, add the aggregates to the window and update
    (opal:update top-win)  ;;will also update work-win

  ;;** Do-Go **
  (Format T "~%Demo-Arith: 
  Press with left button on top menu to change modes (box or line).
  Press with left button on bottom menu to execute a command.
  Press with right button in work window to create a new object
  	of the current mode.
  Boxes can be created anywhere, but lines must start and stop inside boxes.
  Press with left button on text string to start editing that string.
  	While editing a string, type RETURN or press a mouse button to stop.
  Press with left button in work window to select an object,
        continue to hold and move to move the object.
  While moving a box or typing a string, hit ^G or ^g to abort.
  ~%")

    ;;if not CMU CommonLisp, then start the main event loop to look for events
    (unless dont-enter-main-event-loop #-cmu (inter:main-event-loop))


    ;;return top window
    top-win))

(defun Do-Stop ()
  (opal:destroy *current-window*))

