-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: qualchecker.bs
-- Author: Andy Lowry
-- SCCS Info: @(#)qualchecker.bs	1.2 3/13/90

-- This process handles statement qualifiers for the type checker.
-- Mostly, all that is required is to add embedded clauses to the work
-- list, but in some cases assignments must also be made for values
-- computed by the embedded clause.  We always protect addition of
-- clauses to the worklist from duplicate key exceptions.  This would
-- occur if the same clause appeared in multiple places in the
-- program.  Such programs are not produced by the Hermes front-end,
-- but they might be constructed programmatically.

-- This code is adapted from code extracted from the 'type' module,
-- originally written by Dan Yellin

#include "typemark.h"

qualChecker: using (type, type_inference)

process (Q: qualCheckQueue)
  
declare
  args: qualCheckMessage;
begin
  receive args from Q;
  
  -- dispatch according to qualifier type
  select qualifier_types#(case of args.stmt.qualifier)

  where (qualifier_types#'selector')
    -- we need to add the embedded clause, and also that the selector
    -- result variable is predefined!boolean.  We do not check that
    -- the type of the element variable in the selector is element
    -- type of the selected table, as we assume the correct
    -- declaration is inserted by the resolution phase.
    reveal args.stmt.qualifier.selector;
    inspect scope in args.scopes [args.stmt.qualifier.selector.scope] begin
      block begin
	insert clauseid#(copy of scope.clause) into args.workList;
      on (DuplicateKey)
      end block;
    end inspect;
    call args.assignPort(type_assignment_function#'predefinedboolean',
      args.stmt.qualifier.selector.result,
      args.inferred, args.newlyInferred, args.backpatch,
      args.errors, args.position);
    
  where (qualifier_types#'constraintname')
    -- not yet implemented
    exit unsupported;
    
  where (qualifier_types#'block')
    -- We add the main clause and all the handler clauses to the work
    -- list
    reveal args.stmt.qualifier.block;
    inspect scope in args.scopes[args.stmt.qualifier.block.scope] begin
      block begin
	insert clauseid#(copy of scope.clause) into args.workList;
      on (DuplicateKey)
      end block;
    end inspect;
    for handler in args.stmt.qualifier.block.handlers[] inspect
      block begin
	insert clauseid#(copy of handler.clause) into args.workList;
      on (DuplicateKey)
      end block;
    end for;
    
  where (qualifier_types#'while')
    -- We need to add the test and body clauses to the worklist, and
    -- ensure that the test object is of type predefined!boolean
    reveal args.stmt.qualifier.while;
    block begin
      insert clauseid#(copy of args.stmt.qualifier.while.test_clause) 
	  into args.worklist;
    on (duplicateKey)
    end block;
    block begin
      insert clauseid#(copy of args.stmt.qualifier.while.repeated_clause)
	  into args.workList;
    on (duplicateKey)
    end block;
    call args.assignPort(type_assignment_function#'predefinedboolean',
      args.stmt.qualifier.while.result,
      args.inferred, args.newlyInferred, args.backpatch,
      args.errors, args.position);
    
  where (qualifier_types#'inspect table')
    -- We insert the body and selector clauses into the worklist, and
    -- ensure that the selector result is predefined!boolean.  We
    -- don't bother with the types of the inspect element or the
    -- selector element, as we assume they are declared by the
    -- resolution phase
    reveal args.stmt.qualifier.inspect_table;
    inspect scope in args.scopes[args.stmt.qualifier.inspect_table.scope] begin
      block begin
	insert clauseid#(copy of scope.clause) into args.workList;
      on (duplicateKey)
      end block;
    end inspect;
    inspect scope in args.scopes
	  [args.stmt.qualifier.inspect_table.selector.scope] begin
      block begin
	insert clauseid#(copy of scope.clause) into args.workList;
      end block;
    end inspect;
    call args.assignPort(type_assignment_function#'predefinedboolean',
      args.stmt.qualifier.inspect_table.selector.result,
      args.inferred, args.newlyInferred, args.backpatch,
      args.errors, args.position);

  where (qualifier_types#'inspect polymorph')
    -- We insert the body clause into the work list
    reveal args.stmt.qualifier.inspect_polymorph;
    inspect scope in args.scopes[args.stmt.qualifier.inspect_polymorph.scope]
    begin
      block begin
	insert clauseid#(copy of scope.clause) into args.workList;
      on (duplicateKey)
      end block;
    end inspect;
    
  where (qualifier_types#'for enumerate')
    -- We insert the body clause into the work list, and make sure
    -- that the enumerator is of an enumeration type 
    reveal args.stmt.qualifier.for_enumerate;
    block declare
      enumerator: objectname;
    begin
      inspect scope in args.scopes[args.stmt.qualifier.for_enumerate.scope]
      begin
	block begin
	  insert clauseid#(copy of scope.clause) into args.workList;
	on (duplicateKey)
	end block;
	-- build the enumerator objectname
	new enumerator;
	new enumerator.root;
	enumerator.root.scope := scope.id;
	enumerator.root.root := args.stmt.qualifier.for_enumerate.enumerator;
	new enumerator.components;
      end inspect;
      call args.classPort(type_class_function#'enumeration',
	enumerator, args.inferred, args.backpatch, 
	args.errors, args.position);
    end block;
    
  where (qualifier_types#'select')
    -- We add all the select clauses and the otherwise clause to the
    -- work list, as well as clauses computing boolean guards.  All
    -- event guard objects are checked for class 'inport'.  If there
    -- is an operand to the select statement, all the boolean guard
    -- objects are checked for a matching type.  Otherwise they are
    -- all checked to be of type predefined!boolean.
    reveal args.stmt.qualifier.select;

    -- add all clauses to worklist
    for s_clause in args.stmt.qualifier.select.clauses[] inspect
      select guard_type#(case of s_clause.info)
      where (guard_type#'boolean')
	reveal s_clause.info.boolean;
	block begin
	  insert clauseid#(copy of s_clause.info.boolean.clause)
	      into args.workList;
	on (duplicateKey)
	end block;
      where (guard_type#'both')
	reveal s_clause.info.both;
	block begin
	  insert clauseid#(copy of s_clause.info.both.boolean.clause)
	      into args.workList;
	on (duplicateKey)
	end block;
      otherwise
      end select;
      
      block begin
	insert clauseid#(copy of s_clause.clause) into args.workList;
      on (duplicateKey)
      end block;
    end for;

    block begin
      insert clauseid#(copy of args.stmt.qualifier.select.otherwise_clause)
	  into args.workList;
    on (duplicateKey)
    end block;
    
    -- check all event guards for class 'inport'
    for s_clause in args.stmt.qualifier.select.clauses[] inspect
      select guard_type#(case of s_clause.info)
      where (guard_type#'event')
	reveal s_clause.info.portname;
	call args.classPort(type_class_function#'inport',
	  s_clause.info.portname, args.inferred, args.backpatch, 
	  args.errors, args.position);
      where (guard_type#'both')
	reveal s_clause.info.both;
	call args.classPort(type_class_function#'inport',
	  s_clause.info.both.portname, args.inferred, args.backpatch,
	  args.errors, args.position);
      otherwise
      end select;
    end for;
    
    -- check which type of select statement this is
    if B(I(size of args.stmt.operands) <> ZERO) then
      block declare
	operand: objectname;
	boolObj: objectname;
      begin
	-- select statement with operand... all boolean guards must
	-- have the same type as operand
	operand := objectname#(args.stmt.operands[]);
	for s_clause in args.stmt.qualifier.select.clauses[] inspect
	  block begin
	    select guard_type#(case of s_clause.info)
	    where (guard_type#'boolean')
	      reveal s_clause.info.boolean;
	      boolObj := s_clause.info.boolean.result;
	    where (guard_type#'both')
	      reveal s_clause.info.both;
	      boolObj := s_clause.info.both.boolean.result;
	    otherwise
	      exit noBoolGuard;
	    end select;
	    call args.inferPort(type_inference_function#'sameas',
	      boolObj, operand, args.inferred, args.newlyInferred,
	      args.backpatch, args.errors, args.position);
	    call args.inferPort(type_inference_function#'sameas',
	      operand, boolObj, args.inferred, args.newlyInferred,
	      args.backpatch, args.errors, args.position);
	  on exit(noBoolGuard)
	  end block;
	end for;
      end block;
    else
      -- select statement without operand... all boolean guards must
      -- be of type predefined!boolean
      for s_clause in args.stmt.qualifier.select.clauses[] inspect
	block declare
	  boolObj: objectname;
	begin
	  select guard_type#(case of s_clause.info)
	  where (guard_type#'boolean')
	    reveal s_clause.info.boolean;
	    boolObj := s_clause.info.boolean.result;
	  where (guard_type#'both')
	    reveal s_clause.info.both;
	    boolObj := s_clause.info.both.boolean.result;
	  otherwise
	    exit noBoolGuard;
	  end select;
	  call args.assignPort(type_assignment_function#'predefinedboolean',
	    boolObj, args.inferred, args.newlyInferred, args.backpatch,
	    args.errors, args.position);
	on exit (noBoolGuard)
	end block;
      end for;
    end if;
    
  where (qualifier_types#'expression block')
    -- We add the embedded clause to the work list
    reveal args.stmt.qualifier.expression;
    inspect scope in args.scopes[args.stmt.qualifier.expression.scope] begin
      block begin
	insert clauseid#(copy of scope.clause) into args.workList;
      on (duplicateKey)
      end block;
    end inspect;
    
  where (qualifier_types#'if')
    -- We add the test clause, the then clause and, if present, the
    -- else clause, to the work list.  We also check that the test
    -- result is of type predefined!boolean
    reveal args.stmt.qualifier.if;
    block begin
      insert clauseid#(copy of args.stmt.qualifier.if.test_clause)
	  into args.workList;
    on (duplicateKey)
    end block;
    block begin
      insert clauseid#(copy of args.stmt.qualifier.if.then_clause)
	  into args.workList;
    on (duplicateKey)
    end block;
    if B(option#(case of args.stmt.qualifier.if.opt_else_clause) =
	    option#'present') then
      reveal args.stmt.qualifier.if.opt_else_clause.clauseid;
      block begin
	insert 
	    clauseid#(copy of args.stmt.qualifier.if.opt_else_clause.clauseid)
	    into args.workList;
      on (duplicateKey)
      end block;
    end if;
    call args.assignPort(type_assignment_function#'predefinedboolean',
      args.stmt.qualifier.if.test_result, args.inferred, 
      args.newlyInferred, args.backpatch,
      args.errors, args.position);
    
  where (qualifier_types#'literal')
    -- We need to test the strings appearing on named_literal and
    -- string_literal statements, but we can't do that until after we
    -- have typechecked the rest of the program, so we just add the
    -- position indicator to a list of literal statements to be
    -- rechecked later
#if FullChecking
    select args.stmt.operator
    where ('named_literal')
      insert copy of args.position into args.literalStmts;
    where ('string_literal')
      insert copy of args.position into args.literalStmts;
    otherwise
      -- no checking required for other literal statements
    end select;
#endif
    
  otherwise
    -- Either this qualifier requires no checking, or there is no
    -- qualifier
    
  end select;
  
  return args;
end process
    
