-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: getdep.p
-- Author: Daniel Yellin, major revisions by Andy Lowry
-- SCCS Info: "@(#)getdep.p	1.19 3/2/92"

-- getdep is a target name and a target type.  It computes:
--
-- 1.  The targets on which this target depends, as follows:
--       definitions module source: none
--       compiled definitions modele: a definitions module source, and
--         compiled definitions modules for all the definitions
--         modules on the "using (...)" list in the source
--       process module source: none
--       compiled process module: a process module source, compiled
--         definitions modules for all definitions modules on the
--         "using (...)" list in the source, and compiled process
--         modules for all process modules on the "linking(...)" list
--         in the source.
--       target list: all the targets listed in the file
-- 2.  A capability to an action routine that will "make" this node if
--     needed:  for source files and target files, the action routine
--     does nothing; for a compiled module, a process is created that
--     will invoke either dcom or pcom on the source file.
-- 3.  A timestamp, which will be compared with the timestamps of this
--     node's antecedents to determine whether or not this node needs
--     to be made.  A timestamp of -1 will force the node to be made.
--
-- When searching for files, the directory search paths are determined
-- as follows:
--
--   Source files and target files (.d, .p, and .target files): The
--   HSOURCEPATH environment variable is used if set.  Otherwise HPATH
--   is used.  If neither is set, ".:$HROOTDIR/bin" is used.
--
--   Definitions modules (.do files): The HPATH environment variable
--   is used if set.  Otherwise, ".:$HROOTDIR/bin" is used.
--
--   Process modules (.po files): The HLINKPATH environment variable
--   is used if set.  Otherwise, HLOADPATH is used.  If neither is
--   set, HPATH is used.  If even that is not set, ".:$HROOTDIR/bin"
--   is used.
--
-- Another path variable, MKSCOPE, can be used to limit the scope of
-- the make, as follows: If the first occurrence of an object module
-- in the search path is in a directory *not* included in MKSCOPE, the
-- object file will be considered up-to-date, and no dependencies will
-- be established for the corresponding node.  This is important when
-- you do not wish to disturb installed "system" modules, even
-- though they may be out of date with respet to their sources.
--
-- An exception to the above is made when the source file for an
-- object module exists in the current working directory.  In that
-- case, the search for the object module is limited to the current
-- directory as well, and MKSCOPE has no effect.

getdep: using (
  make, getdep, findfile, checkscope, unix, unixstat, getuses, load, 
  root, cload, cwd, rmanager, terminalIO, initloadfuncs, common
)
linking (initloadfuncs,initcheckscope,getuses,
  pprintshell,dprintshell,pcomshell,dcomshell, nullaction)

process(setupQ: setupGetDepQ)

declare
  setupM: setupGetDepIntf;
  findfilePO: findfile!findfile_func;  -- findfile for process modules
  findfileDO: findfile!findfile_func;  -- findfile for def modules
  findfilePsrc: findfile!findfile_func; -- findfile for proc mod sources
  findfileDsrc: findfile!findfile_func; -- findfile for def mod sources
  findfileTarget: findfile!findfile_func;-- findfile for target files
  checkscope: checkscope!checkscopeFn;  -- func to check whether a path is
					-- within the make "scope" or not
  stat: unixstat!statMtimeFunc;
  getuses: getusesFn;
  compileProcess: getdep!ActionInitFn;
  compileDefinition: getdep!ActionInitFn;
  nullAction: ActionFunc;
  pathLoad: load_func;
  environ: root!environ;
  stdio: stdio;
  CLoader: CLoad!CLoadFn;
  getCwd: getCwdFn;
  cwd: charstring;
  rm: rManager;
  terminal: terminalFunctions;
  getdepQ: GetDependenciesQ;
  getdepM: GetDependenciesIntf;
  
begin
  receive setupM from setupQ;
  unwrap pathLoad from setupM.rm.get("pathLoad", "") {init};
  unwrap environ from setupM.rm.get("environ", "") {init};
  unwrap stdio from setupM.rm.get("stdio", "") 
      {init, init(fopen), init(access)};
  unwrap CLoader from setupM.rm.get("CLoader", "") {init};
  unwrap getCwd from setupM.rm.get("getCwd", "") {init};
  cwd <- getCwd();
  insert '/' into cwd;
  rm := setupM.rm;
  terminal := setupM.terminal;
  
  -- initialize the findfile functions
  call (initloadfuncsFn#(create of process initloadfuncs))(
    stdio.access, getCwd, pathload, environ,
    findfilePO, findfileDO, findfilePsrc, findfileDsrc,findfileTarget);
  -- initialize the checkScope function, which checks if a path is in 
  -- MKSCOPE
  checkscope <- (initcheckscopeFn#(create of process initcheckscope))
      (getCwd(), environ);
  -- get file stat function
  unwrap stat from polymorph#(CLoader("File Modification Time")) {init};
  -- get capability to scan using() and linking() lists
  getuses <- (setupGetusesService#(create of process getuses))
      (pathload, stdio.fopen);
  -- load appropriate processes to serve as the "make" actions 
  if setupM.actionType = 'print' then
    -- the make functions are just print routines
    compileProcess <- procedure of process pprintshell;
    compileDefinition <- procedure of process dprintshell;
  else  -- setupM.actionType = 'compile'
    compileProcess <- procedure of process pcomshell;
    compileDefinition <- procedure of process dcomshell;
  end if;
  -- for .p and .d files, action is to do nothing.  
  -- This is what null action does (it always returns true) 
  nullAction <- create of process nullaction;
  
  -- return the getdep function
  new getdepQ;
  connect setupM.getdepFunc to getdepQ;
  return setupM;
  
  ---------------------------------------------------------------------
  ----------------------- main processing loop ------------------------
  ---------------------------------------------------------------------
  
  -- now wait and process requests
  while 'true' repeat
    block begin
      receive getdepM from getdepQ;
      new getdepM.uses;
      -- get dependencies for this node
      select 
      where (getdepM.id.type = "definitions source"
	      or getdepM.id.type = "process source") 
	-- source file must be available, but it has no dependencies and we
	-- don't need to do anything to it 
	-- to make it.
	block declare
	  filename: charstring;
	begin
	  if (getdepM.id.type = "definitions source") then
	    filename <- getdepM.id.name | ".d";
	    filename <- findfileDsrc(filename, 'read');
	  else
	    filename <- getdepM.id.name | ".p";
	    filename <- findfilePsrc(filename, 'read');
	  end if;
	  new getdepM.uses;
	  getdepM.action := nullAction;
	  getdepM.timestamp <- stat(filename);
	  getdepM.mustMake <- 'false';
	  if exists of getdepM.debug['dependencies'] then
	    call terminal.putLine("make: " | filename | ": no dependencies");
	  end if;
	on (findfile_intf.file_not_found)
	  call terminal.putLine("Could not find file: " | filename);
	  exit dependencyFailure;
	end block;
	
      where (getdepM.id.type = "definitions module")
	block declare
	  source: charstring;
	  object: charstring;
	  incwd: boolean;
	  inscope: boolean;
	begin
	  -- check whether there's a source file in the current working
	  -- directory
	  incwd <- stat(cwd | getdepM.id.name | ".d") <> -1;
	  if incwd then
	    -- timestamp the compiled module in cwd, regardless of whether
	    -- it's elsewhere in the path
	    object <- cwd | getdepM.id.name | ".do";
	    getdepM.timestamp <- stat(object);
	    getdepM.mustMake <- getdepM.timestamp = -1;
	    if getdepM.mustmake and exists of getdepM.debug['decisions'] then
	      call terminal.putline("make: " | object 
		    | " will be made because it does not exist");
	    end if;
	    inscope <- 'true';		-- always "in scope" in this case
	  else
	    -- otherwise search for it elsewhere and return the timestamp
	    block begin
	      object <- findfileDO(getdepM.id.name | ".do", 'read');
	      getdepM.timestamp <- stat(object);
	      inscope <- checkscope(object);
	      getdepM.mustMake <- 'false';
	    on (findfile_intf.file_not_found)
	      -- no object file anywhere 
	      object <- cwd | getdepM.id.name | ".do";
	      getdepM.timestamp <- -1;
	      getdepM.mustMake <- 'true';
	      if exists of getdepM.debug['decisions'] then
		call terminal.putline("make: " | object 
		      | " will be made because it does not exist");
	      end if;
	      inscope <- 'true';		-- consider it "in scope"
	    end block;
	  end if;
	  
	  -- scan source file for dependencies if we're in scope
	  if inscope then
	    -- add dependency on the source file
	    insert (evaluate d1: nodeID from
		new d1;
		d1.name := getdepM.id.name;
		d1.type <- "definitions source";
	      end) into getdepM.uses;
	    block begin
	      if incwd then
		source <- cwd | getdepM.id.name | ".d";
	      else
		source <- findfileDsrc(getdepM.id.name | ".d", 'read');
	      end if;
	      block begin
		merge getuses(source) into getdepM.uses;
	      on (getusesIntf.getusesfailure)
		call terminal.putLine
		    ("Unable to parse source file for dependencies: "
		      | source);
		exit dependencyFailure;
	      end block;
	      getdepM.action <- compileDefinition(rm, terminal, source);
	    on (findfile_intf.file_not_found)
	      -- no source file available
	      getdepM.action := nullAction;
	    end block;
	    if exists of getdepM.debug['dependencies'] then
	      for use in getdepM.uses[] inspect
		block declare
		  usefile: charstring;
		begin
		  select use.type
		  where ("definitions source")
		    usefile <- use.name | ".d";
		  where ("definitions module")
		    usefile <- use.name | ".do";
		  otherwise
		    usefile <- use.name | "(" | use.type | ")";
		  end select;
		  call terminal.putLine
		      ("make: " | object | ": depends on " | usefile);
		end block;
	      end for;
	    end if;
	  else
	    getdepM.action := nullAction;
	    if exists of getdepM.debug['dependencies'] then
	      call terminal.putLine
		  ("make: " | object | ": no dependencies (out of scope)");
	    end if;
	  end if;
	end block;
	
      where (getdepM.id.type = "process module")
	block declare
	  source: charstring;
	  object: charstring;
	  incwd: boolean;
	  inscope: boolean;
	begin
	  -- check whether there's a source file in the current working
	  -- directory
	  incwd <- stat(cwd | getdepM.id.name | ".p") <> -1;
	  if incwd then
	    -- timestamp the compiled module in cwd, regardless of whether
	    -- it's elsewhere in the path
	    object <- cwd | getdepM.id.name | ".po";
	    getdepM.timestamp <- stat(object);
	    getdepM.mustMake <- getdepM.timeStamp = -1;
	    if getdepM.mustmake and exists of getdepM.debug['decisions'] then
	      call terminal.putline("make: " | object 
		    | " will be made because it does not exist");
	    end if;
	    inscope <- 'true';		-- always "in scope" in this case
	  else
	    -- otherwise search for it elsewhere and return the timestamp
	    block begin
	      object <- findfilePO(getdepM.id.name | ".po", 'read');
	      getdepM.timestamp <- stat(object);
	      getdepM.mustMake <- 'false';
	      inscope <- checkscope(object);
	    on (findfile_intf.file_not_found)
	      -- no object file anywhere
	      object <- cwd | getdepM.id.name | ".po";
	      getdepM.timestamp <- -1;
	      getdepM.mustMake <- 'true';
	      if exists of getdepM.debug['decisions'] then
		call terminal.putline("make: " | object 
		      | " will be made because it does not exist");
	      end if;
	      inscope <- 'true';	-- consider it "in scope"
	    end block;
	  end if;
	  
	  -- scan source file for dependencies if we're in scope
	  if inscope then
	    -- add dependency on the source file
	    insert (evaluate d2: nodeID from
		new d2;
		d2.name := getdepM.id.name;
		d2.type <- "process source";
	      end) into getdepM.uses;
	    block begin
	      if incwd then
		source <- cwd | getdepM.id.name | ".p";
	      else
		source <- findfilePsrc(getdepM.id.name | ".p", 'read');
	      end if;
	      block begin
		merge getuses(source) into getdepM.uses;
	      on (getusesIntf.getusesfailure)
		call terminal.putLine
		    ("Unable to parse source file for dependencies: "
		      | source);
		exit dependencyFailure;
	      end block;
	      getdepM.action <- compileProcess(rm,terminal,source);
	    on (findfile_intf.file_not_found)
	      -- no source file available
	      getdepM.action := nullAction;
	    end block;
	    if exists of getdepM.debug['dependencies'] then
	      for use in getdepM.uses[] inspect
		block declare
		  usefile: charstring;
		begin
		  select use.type
		  where ("process source")
		    usefile <- use.name | ".p";
		  where ("process module")
		    usefile <- use.name | ".po";
		  where ("definitions module")
		    usefile <- use.name | ".do";
		  otherwise
		    usefile <- use.name | "(" | use.type | ")";
		  end select;
		  call terminal.putLine
		      ("make: " | object | ": depends on " | usefile);
		end block;
	      end for;
	    end if;
	  else
	    getdepM.action := nullAction;
	    if exists of getdepM.debug['dependencies'] then
	      call terminal.putLine
		  ("make: " | object | ": no dependencies (out of scope)");
	    end if;
	  end if;
	end block;
	
      where (getdepM.id.type = "target list")
	block declare
	  filename: charstring;
	  file: stream;
	begin
	  -- locate the target file
	  filename <- findfileTarget(getdepM.id.name | ".target", 'read'); 
	  file <- stdio.fopen(filename, 'read');
	  block declare
	    target: charstring;
	    extension: charstring;
	    junk: charstring;
	    dot: char;
	    modType: charstring;
	    targ: nodeID;
	  begin
	    while 'true' repeat
	      target <- file.gets();
	      extract junk from c in target where (c = ' ');
	      if size of target > 0 then
		if target[0] <> '#' then
		  extract extension from c in target where
		      (not exists of d in target where
			(d = '.' and position of d > position of c));
		  if target = "" then -- treat no extension as .target file
		    target <- extension;
		    extension <- ".target";
		  end if;
		  select extension
		  where (".d")
		    modType <- "definitions source";
		  where (".do")
		    modType <- "definitions module";
		  where (".p")
		    modType <- "process source";
		  where (".po")
		    modType <- "process module";
		  where (".target")
		    modType <- "target list";
		  otherwise
		    call terminal.putLine(
		      "I don't know how to build this target: "
			  | target | extension);
		    exit dependencyFailure;
		  end select;
		  new targ;
		  targ.name := target;
		  targ.type <- modType;
		  block begin
		    insert targ into getdepM.uses;
		  on (duplicateKey)
		  end block;
		  if exists of getdepM.debug['dependencies'] then
		    call terminal.putLine("make: " | filename
			  | ": depends on " | target | extension);
		  end if;
		end if;
	      end if;
	    end while;
	  on (getstringIntf.endOfInput)
	    call file.fclose();
	  end block;
	  getdepM.timestamp <- stat(filename);
	  getdepM.mustMake <- 'false';
	  getdepM.action := nullAction;
	  if size of getdepM.uses = 0 then
	    if exists of getdepM.debug['dependencies'] then
	      call terminal.putLine
		  ("make: " | filename
		    | ": no dependencies (empty target file)");
	    end if;
	  end if;
	on (findfile_intf.file_not_found)
	  call terminal.putLine("Could not find file: " 
		| getdepM.id.name | ".target");
	  exit dependencyFailure;
	end block;

      otherwise
	call terminal.putLine(
	  "Invalid target type for target " | getdepM.id.name
	      | ": " | getdepM.id.type); 
	exit dependencyFailure;	
      end select;
      
      return getdepM;
      
    on exit (dependencyFailure)
      return getdepM exception dependencyFailure;
    end block;
  end while;
    
on (disconnected)

end process
