-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: make.d
-- Author: Daniel Yellin
-- SCCS Info: "@(#)make.d	1.5 1/21/92"

make: using (predefined, terminalio)
definitions

-- Dependency Graph Structure

dependencyGraph: table of node { full } keys (id);

node: record (
  id: nodeID,			-- node name and type
  successors: nodeIDs,		-- nodes that depend on this one
  action: ActionFunc,		-- action to make this node up-to-date 
				--  (usually a compilation function).
  timeStamp: integer,		-- used to determine if node is out of
				-- date with respect to antecedent nodes
  mustMake: boolean,		-- must the action be invoked?
  status: status	        -- was GetDepdendencies and Make at the node 
                                -- tried? was it succesful? 
);

nodeIDs: table of nodeID {full} keys (*);
nodeID: record (
  name: charstring,
  type: charstring
);
    
status: enumeration(
   'DependenciesFailure',       -- GetDependecies failed at node
   'DependenciesSuccess',       -- GetDepend. succeeded & never tried action
   'ActionFailure',             -- action failed at node 
   'ActionSuccess'              -- action succeeded at node 
);

-- Language Dependent Functions

GetDependenciesIntf: callmessage (
  id: nodeID,			-- node whose dependencies we want to check
  debug: debugControls,		-- controls what debugging info is shown
  uses: nodeIDs,		-- nodes that it depends upon
  action: ActionFunc,		-- function to "make" this node.	
  timestamp: integer,		-- compared with antecedents to see if
  				-- this node must be made
  mustMake: boolean		-- true when we know this node must be made
) constant (id, debug)
  exit { full }
  exception dependencyFailure { full(id), init(debug) };

GetDependenciesQ: inport of GetDependenciesIntf { full(id) };
GetDependenciesFunc: outport of GetDependenciesQ;


ActionIntf: callmessage (
  made: boolean			-- out: were we able to make it?
) 
  exit { full };

ActionQ: inport of ActionIntf {};
ActionFunc: outport of ActionQ;

-- Language Independent: The Make Engine

ActionControl: enumeration ( 
  'None',			-- just build the graph, take no action.
  'Inconsistent',		-- do action to all out of date modules.
  'All'				-- do action to all modules (in order).
);

ErrorControl: enumeration ( 
  'Stop',			-- halt immediately on errors.
  'Ignore',			-- ignore all errors.
  'Continue Other Paths'	-- on errors stop processing successors
				--  of this node but process independent paths.
);

DebugControls: table of DebugControl {init} keys(*);
DebugControl: enumeration (
  'dependencies',		-- show dependencies as they're discovered
  'decisions',			-- explain decisions to make/not make nodes
  'actions',			-- announce when actions are being invoked
  'trace'			-- show general flow of control
);
  
MakeIntf: callmessage (
  root: nodeIDs,		-- root objects to be made
  actionControl: actionControl,	-- controls how much action to take.
  errorControl: errorControl,	-- what to do when there is an error.
  debugControl: debugControls,	-- how much debugging info to produce
  getDependencies: getDependenciesFunc,  -- function to get dependencies 
  terminal: terminalFunctions,	-- for messages to the terminal
  graph: dependencyGraph	-- resulting dependency graph
) constant 
    (actionControl, errorControl, debugControl, getDependencies, terminal)
  exit { full }
  exception Circular { full }	-- stopped due to circular graph
  exception NotMade  { full };	-- an error occured somewhere in make

MakeQ: inport of MakeIntf {
  init(root), init(actionControl), init(errorControl), init(debugControl), 
  init(GetDependencies), init(terminal) };
MakeFunc: outport of MakeQ;

end definitions
