-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: sysrmanager.d
-- Author: David F. Bacon, Andy Lowry, from Rob Strom
-- SCCS Info: @(#)sysrmanager.d	1.2 3/13/90

-- Interfaces between the resource manager and its clients, with user
-- identification supplied in every call from a client (user id
-- normally is supplied by a filter process created at login time and
-- interposed between the resource manager offering this interface,
-- and user processes which access the resource manager through the
-- filter via the rManager interface).

sysRManager: using (rManager, common) 

definitions

  -- initialization port for the resource manager
  rManagerInit: callmessage (
    rManager: sysRManager
  )
  exit {full};
  
  rManagerInitQ: inport of rManagerInit {};
  rManagerInitFn: outport of rManagerInitQ;

  -- ports to access resource manager    
  sysRManager: record (
    post: syspostResourceFn,	-- post a resource (by copying it)
    insert: sysinsertResourceFn,-- post a resource (by moving it)
    get: sysgetResourceFn,	-- get a copy of a posted resource
    remove: sysgetResourceFn,	-- remove a posted resource and return it
    delete: sysdeleteResourceFn	-- delete a resource 
  );

  -- Posting a resource causes a copy of the resource to be installed
  -- in the resource manager for later access by other processes
  sysPostResource: callmessage (
    userName: charString,	-- name of user posting resource
    resourceName: charString,	-- name being given to resource
    resource: polymorph,	-- the resource itself (usually)
    accessFunction: accessFn	-- access control function
  ) 
  constant(userName, resourceName, resource, accessFunction)
  exit {full}
  exception duplicateName {full};
    
  sysPostResourceQ: inport of sysPostResource {full};
  sysPostResourceFn: outport of sysPostResourceQ;
    
  -- Inserting a resource causes the resource itself to be installed
  -- (i.e. moved) into the resource manager for later access (via
  -- 'remove') by another process.  This is used instead of 'post'
  -- when the resource is uncopyable.
  sysInsertResource: callmessage (
    userName: charString,
    resourceName: charString,
    resource: polymorph,
    accessFunction: accessFn
  ) constant(userName, resourceName, accessFunction)
  exit {init(userName), init(resourceName), init(accessFunction)}
  exception duplicateName {full};
    
  sysInsertResourceQ: inport of sysInsertResource {full};
  sysInsertResourceFn: outport of sysInsertResourceQ;
    
    
  -- 'get' acquires a resource from the resource manager, provided
  -- access is granted by the access function supplied when the
  -- resource was posted.  Use 'remove' instead for uncopyable resources
  -- that were installed via 'insert'
  sysGetResource: callmessage (
    userName: charString,	-- name of user requesting resource
    resourceName: charString,	-- name of the resource
    parameter: charString,	-- parameter (e.g. passwd, type of access)
    resource: polymorph		-- the resource granted
  )
  constant(userName, resourceName, parameter)
  exit {full}
  exception notFound 
      {init(userName), init(resourceName), init(parameter)}
  exception accessDenied
      {init(userName), init(resourceName), init(parameter)};
    
  sysGetResourceQ: inport of sysGetResource
      {init(userName), init(resourceName), init(parameter)};
  sysGetResourceFn: outport of sysGetResourceQ;
    
    
  -- Use 'delete' to take a resource out of the repository.  Only the
  -- user who posted (inserted) a resource may delete it.
  sysDeleteResource: callmessage (
    userName: charString,	-- name of user deleting resource
    resourceName: charString	-- name of resource
  )
  constant(userName,resourceName)
  exit{full}
  exception notOwner{full};
    
  sysDeleteResourceQ: inport of sysDeleteResource {full};
  sysDeleteResourceFn: outport of sysDeleteResourceQ;
    
end definitions
