-- (C) Copyright International Business Machines Corporation 16 September
-- 1991.  All Rights Reserved.
--
-- See the file USERAGREEMENT distributed with this software for full
-- terms and conditions of use.
-- SCCS Info: @(#)chphase1.p	1.3 2/17/92

-- This process converts the li-code instructions into internal statements.
--
-- Input is a interpform!code table.
--
-- Output is a statements table and an operands table.
--
-- The process chdictionary is used to maintain a symbol table, associating
-- li-code operands (see interpform!operand) with entries in the internal
-- operands table.  The reference capability of the dictionary is used to
-- cumulate references to operands and to assign the Families of operands.
--
-- The opcode and qualifier of the statement are copied from the code.
-- Statement.operands is a created list of indicies to operands, associated
-- by the dictionary.  The flow and locus fields of the statement entry
-- come from the descriptors field of the argument ch, selected by opcode.
--
-- Operations with 'branch' flow are recorded for subsequent branch
-- redundancy analysis.
--
-- Special processing is provided for these opcodes:
--  * escape - Terminate processing and return exception  already .
--  * select - discriminate blocking from non-blocking select operations.
--    A blocking select has at least one inport operand.
--
-- Operations with 'ubiquitous' flow provide pairs of operand indices,
-- collected in equivalents, for subsequent operand family analysis.
--
-- Operand assignments are cumulated.
--
-- Following the construction of statements, chbranches is called for
-- branch redundancy analysis, and choperands is called for operand
-- family analysis.
--

chphase1: using(chdescriptors, chinternal, chphase1, chtransform,
    interpform, string)
  linking (chbranches, choperands, chassign, chdictionary)

  process(Q: phase1Q)
    
  declare
    
    cm: phase1;
    
    statements: ch_statements;
    operands: ch_operands;
    jump_targets: statement_indices;
    branch_targets: statement_indices;
    equivalents: equivalents;
    reveals: operandset;
    
    lookup: lookupFn;
    reference: lookupFn;
    assign: assignFn;
    chop: ch_operand;
    e: empty;
    
    st: ch_statement;
    ix: integer;
    index: integer;
    family: data_family;

  begin
    receive cm from Q;
    if cm.ch.options.verbose then
	call cm.ch.put("phase1 ");
      end if;
    
    call (dictionaryFn#(create of process chdictionary))
       (reference, lookup);
    assign <- procedure of process chassign;
    
    new statements;
    new operands;
    new jump_targets;
    new branch_targets;
    new equivalents;
    new reveals;
    
    -- collect statements and operands
--pragma "trace=99"        
    index <- 0;
    for operation in cm.code[]
      inspect
	new st;
	st.opcode := operation.opcode;
        st.qualifier := operation.qualifier;
	st.position := index;
	st.references <- 0;
	inspect desc in cm.ch.descriptors[operation.opcode]
	  begin
	    st.flow := desc.flow;
	    st.locus := desc.locus;
	    if desc.flow < 'emerge' then
		if desc.opcode = 'branch' then
		    insert copy of index into jump_targets;
		  else
		    insert copy of index into branch_targets;
		  end if;
	      end if;
	    
	    new st.operands;
	    select operation.opcode	-- special processing for some opcodes
		   
	      where('escape')
		exit already;
	      
	      where('select')		-- special recognition of operands
		for op in operation.operands[]
		  inspect
		    -- select operands are alternating: inport,guard
		    --         (either may be an empty list)
		    -- If all inports are omitted (empty) and no guard is
		    -- omitted, the select can be included in a cblock;
		    -- otherwise the operation may block, and it must
		    -- occupy a cblock of its own (at least until the
		    -- interpreter is abandoned).
		    if (position of op) mod 2 = 0 then
			family <- 'inport';
			if size of op <> 0 then
			    st.locus <- 'block';
			    st.flow <- 'frame';
			  end if;
		      else
			family <- 'boolean';
		      end if;
		    insert reference(op, family, operands)
		       into st.operands;
		  end for;
		
	      otherwise
		for op in operation.operands[]
		  inspect
		    ix <- position of op;
		    if ix >= size of desc.families then
			ix <- size of desc.families - 1;
		      end if;
		    insert reference(op, desc.families[ix], operands)
		       into st.operands;
		  end for;
	      
	      end select;
	    
	    if st.flow = 'ubiquitous' then
		block
		  declare
		    eqv: equivalent;
		  begin
		    new eqv;
		    eqv.x2 := st.operands[1];
		    if st.opcode = 'copy' or st.opcode = 'move' then
			eqv.x1 := st.operands[0];
		      else
			eqv.x1 := st.operands[2];
		      end if;
		    insert copy of eqv into equivalents;
		    ix <- eqv.x1;
		    eqv.x1 <- eqv.x2;
		    eqv.x2 <- ix;
		    insert eqv into equivalents;
		  on(DuplicateKey)
		  end block;
		st.flow <- 'continue';
	      end if;
	    
	    -- handle assignment
	    if size of st.operands <> 0 then
		call assign(desc.destination, st, operands, reveals);
	      end if;
		
	  end inspect;
	insert st into statements;

	index <- index + 1;
	
      end for;

    -- branch processing
    call (branchesFn#(create of process chbranches))
       (statements, jump_targets, branch_targets, cm.references);
    
    -- operand processing
    call (operandsFn#(create of process choperands))
       (operands, equivalents);
    
    cm.statements <- statements;
    cm.operands <- operands;
    return cm;
    
  on exit(already)
    return cm exception already;
  
--on (others)
--pragma "trace=0" print index;  
    
  end process
