-- (C) Copyright International Business Machines Corporation 23 January 
-- 1990.  All Rights Reserved. 
--  
-- See the file USERAGREEMENT distributed with this software for full 
-- terms and conditions of use. 
-- File: literalcheck.p
-- Author: Andy Lowry
-- SCCS Info: @(#)literalcheck.p	1.2 3/13/90

-- This process checks to make sure that qualifiers for string_literal
-- and named_literal statements are all valid.  This is done after all
-- other type checking has been performed for the program, so we'll be
-- sure of knowing the necessary object types if they can be inferred.

-- This code is adapted from code extracted from the 'type' module,
-- originally written by Dan Yellin.

literalCheck: using (type, errors, positions)

process (Q: literalCheckQueue)
  
declare
  args: literalCheckMessage;
  operand: objectname;
  type: typename;
  def: type_definition;
  error: error;
  errObj: errorObject;
  errorpos: aposition;
begin
  receive args from Q;

  reveal args.stmt.qualifier.literal;

  -- extract the destination operand (the only operand), and find its
  -- type and type definition
  operand := args.stmt.operands[];
  block begin
    type <- args.findType(operand, args.scopes, args.definitions, 
      args.inferred);
  on (findTypeMessage.unknown_type)
    -- we'll get an 'uninferred' error message about this one, so
    -- we'll just exit silently here
    exit done;
  end block;
  def <- args.findDef(type, args.definitions);

  -- proceed according to statement operator
  select args.stmt.operator
  where ('named_literal')
    -- qualifier must be the name of one of the enumeration or boolean
    -- values appearing in the type definition
    select case of def.specification
    where ('enumerationtype')
      reveal def.specification.enumeration;
      if not exists of val in def.specification.enumeration.values
	    where (val = args.stmt.qualifier.literal) then
	exit litNotFound;
      end if;
    where ('booleantype')
      reveal def.specification.boolean;
      if ((args.stmt.qualifier.literal <> 
		def.specification.boolean.true_name)
	      and (args.stmt.qualifier.literal <> 
		def.specification.boolean.false_name))
      then
	exit litNotFound;
      end if;
    otherwise
      -- the type is not an enumeration or boolean... an error will
      -- already have been issued for a class rule violation
      exit done;
    end select;
    
  where ('string_literal')
    -- every element of the qualifier must be the name of one of the
    -- enumeration values appearing in the enumeration type which is
    -- the element type of the table operand
    block declare
      valname: charstring;
    begin
      -- find the element type of the table operand
      block begin
	reveal def.specification.table_info;
      on (caseError)
	-- operand is not a table... a class rule error will already
	-- have been generated
	exit done;
      end block;
      -- we only check ordered tables, since for a non-ordered table
      -- a class error has already appeared
      if def.specification.table_info.ordered_table then
	type := def.specification.table_info.element_type;
	def := args.findDef(type, args.definitions);
      else
	exit done;
      end if;
      block begin
	reveal def.specification.enumeration;
      on (caseError)
	-- element is not an enumeration... a class error will already
	-- have been generated
	exit done;
      end block;
      -- make sure each character names one of the enumeration values
      for element in args.stmt.qualifier.literal[] inspect
	new valname;
	insert copy of element into valname;
	if not exists of val in def.specification.enumeration.values
	      where (val = valname) then
	  -- issue an error message (we don't just exit to litNotDone,
	  -- because we may need to issue errors for other table
	  -- elements too)
	  new error;
	  error.code <- 'named literal not found';
	  new errorpos;
	  errorpos.clause := args.clause_id;
	  errorpos.statement := args.stmt.id;
	  unite error.position.apos from errorpos;
	  -- error objects are the enumeration type, and the literal string
	  new error.objects;
	  unite errObj.typename from copy of type;
	  insert errObj into error.objects;
	  unite errObj.charstring from valname;
	  insert errObj into error.objects;
	  insert error into args.errors;
	end if;
      end for;
    end block;
    
  otherwise
    -- we should not be getting called for any other statements
    exit cantHappen;
  end select;
  
  return args;
  
on exit(litNotFound)
  -- here when there's literal value that doesn't appear in the
  -- appropriate type definition... this exit is not used for
  -- string_literal, since several characters might be invalid.. we
  -- get a message for each (is this what we really want?)
  new error;
  error.code <- 'named literal not found';
  new errorpos;
  errorpos.clause := args.clause_id;
  errorpos.statement := args.stmt.id;
  unite error.position.apos from errorpos;
  -- error objects are: object type and name string
  new error.objects;
  unite errObj.typename from type;
  insert errObj into error.objects;
  unite errObj.charstring from copy of args.stmt.qualifier.literal;
  insert errObj into error.objects;
  insert error into args.errors;
  return args;
  
on exit(done)
  -- here when we skip the test for some reason or another... just
  -- return
  return args;
  
on exit(cantHappen)
  print charstring#"CantHappen happened in literalcheck!";
end process

    
