/*
 * helpindex.c --- read help files and create an index
 * (duz 13Sep94)
 */

#define _XOPEN_SOURCE 1

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "help.h"


/*
 * Data structures.
 */

typedef struct node Node;	/* record in binary tree in memory */
struct node
{
  Item rec;
  Node *left, *right;
};

static Head head = { "HELP" };
static char (*file) [PATH_LENGTH];
static Node *root = NULL;


static void
insert (Item *new)
/*
 * Insert *new into a binary tree starting at root.
 */
{
  Node **p = &root, *q;
  while (*p != NULL)
    {
      int cmp = strcmp (new->name, (*p)->rec.name);
      if (cmp == 0)
	{
	  fprintf (stderr, "\"%s\" is redefined\n", new->name);
	  return;
	}
      if (cmp < 0)
	p = &(*p)->left;
      else if (cmp > 0)
	p = &(*p)->right;
    }
  NEW (q, 1);
  q->rec = *new;
  q->left = q->right = NULL;
  *p = q;
}


/*
 * Read records from file and insert them in the tree.
 */

static fpos_t
scan_record (FILE *f, char *name)
/*
 * Reads the file until a `:' is found in column 0. Returns the name in *name.
 * Returns position of `:' or -1 at end of file.
 */
{
  for (;;)
    {
      fpos_t pos = ftell (f);
      char line [0x80];
      char *p = fgets (line, sizeof line, f);

      if (p == NULL)
	{
	  if (ferror (f))
	    sys_error ();
	  else
	    return -1;
	}
      if (line [0] == ':')
	{
	  sscanf (line, ":%s", name);
	  return pos;
	}
    }
}

static void
skip_record (FILE *f)
/*
 * Skip over all lines in file until either two empty lines
 * or a `:' in column 0 is found.
 */
{
  int empty = 0;
  for (;;)
    {
      fpos_t pos = ftell (f);
      char line [0x80];
      char *p = fgets (line, sizeof line, f);

      if (p == NULL)
	{
	  if (ferror (f))
	    sys_error ();
	  else
	    return;
	}
      if (line [0] == '\n' || line [0] == '#')
	{
	  if (++empty == 2)
	    return;
	  else
	    empty = 0;
	}
      if (line [0] == ':')
	{
	  fseek (f, pos, SEEK_SET);
	  return;
	}
    }
}

static void
read_file (FILE *f, int fidx)
{
  Item x;

  for (;;)
    {
      x.pos = scan_record (f, x.name);
      x.fidx = fidx;
      if (x.pos == -1) return;
      skip_record (f);
      insert (&x);
      head.nitems++;
    }
}


/*
 * Write file.
 */

static void
write_tree (Node *p, FILE *f)
{
  if (p == NULL) return;
  write_tree (p->left, f);
  if (fwrite (&p->rec, sizeof p->rec, 1, f) != 1)
    sys_error ();
  write_tree (p->right, f);
}

static void
write_file (FILE *f)
{
  fwrite (&head, 1, sizeof head, f);
  fwrite (file, head.nfiles, sizeof *file, f);
  if (ferror (f)) sys_error ();
  write_tree (root, f);
}


/*
 * Main program, process command line.
 */

static void
usage (void)
{
  fatal ("usage:\t%s [-o file] file ..."
	 "\n    -o\toutput index to file (default: standard output)",
    	 progname);
}

int
main (int argc, char *argv [])
{
  char *outfile = NULL;
  FILE *in, *out;
  int c, i, n;

  /*
   * get program name for error messages
   */
  progname = strrchr (argv [0], '/');
  if (progname)
    progname++;
  else
    progname = argv [0];

  /*
   * parse command line options
   */
  while ((c = getopt (argc, argv, ":ho:")) != EOF)
    {
      switch (c)
	{
	case '?': fatal ("illegal option -%c", optopt);
	case ':': fatal ("option %c requires an argument", optopt);
	case 'h': usage ();
	case 'o': outfile = optarg; break;
	}
    }
  n = argc - optind;
  if (n <= 0) usage ();

  /*
   * process
   */
  head.nfiles = 0;
  head.nitems = 0;
  NEW (file, n);
  for (i = 0; i < n; i++)
    {
      strcpy (file [i], argv [optind + i]);
      in = fopen (file [i], "r");
      if (in == NULL)
	file_error (argv [i]);
      read_file (in, i);
      head.nfiles++;
      fclose (in);
    }

  if (outfile)
    {
      out = fopen (outfile, "wb");
      if (out == NULL)
	file_error (outfile);
    }
  else
    {
      out = stdout;
    }
  write_file (out);
  return 0;
}
