(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Created by Jorge Stolfi on Sat Oct  7  2:14:47 PDT 1989     *)
(* Last modified on Tue Jun  2 13:09:35 PDT 1992 by stolfi     *)
(*      modified on Tue Feb 11 21:39:38 PST 1992 by muller     *)


INTERFACE YPQ;

(*  A perceptual color model.

  This interface defines an alternative representation of colors
  in a coordinate system (linearly related to RGB) whose
  axes correpond roughly to orthogonal perceptual quantities.

  Index: colors; graphics, colors; display, color.
*)

IMPORT Color, RGB, RGBCIE;

TYPE
  T = Color.T; (* ARRAY [0..2] OF REAL; *)
  (* 
    An YPQ.T value describes a color by the three attributes, Brightness
    (Y), linear redness (P), and linear yellowness (Q), all real
    numbers.  
    
    Y is the same as RGB.Brightness, that is,  CIE Y linearly scaled
    to [0.0_1.0] for [Black_White].  P and Q are linear chroma
    coordinates orthogonal to Y, and roughly orthogonal among themselves
    in the RGB.Dist metric; The P component is roughly the Red-Green
    chroma signal, and the Q compoment is roughly the Yellow-Blue chroma
    signal.  Their ranges are certain sub-intervals of [-1.0_+1.0].
    *)

CONST
  Primary = ARRAY [0..2] OF T{
    T{1.0, 0.0, 0.0}, 
    T{0.0, 1.0, 0.0}, 
    T{0.0, 0.0, 1.0}
  };

(**********************************************************)
(*                                                        *)
(* CONNECTION WITH RGB SYSTEM                             *)
(*                                                        *)
(**********************************************************)

CONST
  rY = RGBCIE.rY;  rP =  0.0470;  rQ =  0.0170;
  gY = RGBCIE.gY;  gP = -0.0500;  gQ =  0.0180;
  bY = RGBCIE.bY;  bP =  0.0030;  bQ = -0.0350;
    (*
      YPQ coordinates of the RGB primaries. Must satisfy
|           rY + gY + bY = 1.0
|           rP + gP + bP = 0.0
|           rQ + gQ + bQ = 0.0
    *)
  
  Yr =   1.000000000;  Yg =   1.000000000;  Yb =   1.000000000;
  Pr =  14.952830190;  Pg =  -5.683962264;  Pb =   4.339622642;
  Qr =   3.424528302;  Qg =   1.655660377;  Qb = -26.056603770;
    (*
      RGB coordinates of the YPQ primaries. *)

PROCEDURE RGBFromYPQ (ypq: T): RGB.T;
PROCEDURE YPQFromRGB (rgb: RGB.T): T;
  (*
    Conversion between YPQ and RGB coordinates, using the 
    matrices above. *)

CONST Undefined = Color.Undefined;
  (*
    The "undefined" YPQ triple *)
    
CONST Default = Color.Default;

TYPE Bytes = Color.Bytes;

CONST (*PROCEDURE*) ToBytes = Color.ToBytes;
(* A safe LOOPHOLE of /x/ into ARRAY OF BYTE. *)

END YPQ.
