(* Copyright (C) 1992, Digital Equipment Corporation           *)
(* All rights reserved.                                        *)
(* See the file COPYRIGHT for a full description.              *)

(* File: EnumExpr.m3                                           *)
(* Last modified on Mon Mar  2 10:43:24 PST 1992 by kalsow     *)
(*      modified on Fri Nov 30 21:02:21 1990 by muller         *)

MODULE EnumExpr;

IMPORT Expr, ExprRep, Type, Emit, Temp, MBuf;

TYPE
  P = Expr.T BRANDED "EnumExpr.T" OBJECT
        value: INTEGER;
      OVERRIDES
        typeOf       := ExprRep.NoType;
        check        := ExprRep.NoCheck;
        compile      := Compile;
        evaluate     := ExprRep.Self;
        fprint       := FPrinter;
        write        := Writer;
        isEqual      := EqCheck;
        getBounds    := Bounder;
        isWritable   := ExprRep.IsNever;
        isDesignator := ExprRep.IsNever;
	isZeroes     := IsZeroes;
	note_write   := ExprRep.NotWritable;
	genLiteral   := GenLiteral;
      END;

PROCEDURE New (type: Type.T;  value: INTEGER): Expr.T =
  VAR p: P;
  BEGIN
    p := NEW (P);
    ExprRep.Init (p);
    p.value   := value;
    p.type    := type;
    p.checked := TRUE;
    RETURN p;
  END New;

PROCEDURE EqCheck (a: P;  e: Expr.T): BOOLEAN =
  BEGIN
    TYPECASE e OF
    | NULL => RETURN FALSE;
    | P(b) => RETURN (a.value = b.value);
    ELSE      RETURN FALSE;
    END;
  END EqCheck;

PROCEDURE Compile (p: P): Temp.T =
  BEGIN
    RETURN Temp.FromExpr (p);
  END Compile;

PROCEDURE Writer (p: P;  <*UNUSED*> t1, t2: Temp.T) =
  BEGIN
    Emit.OpI (" @ ", p.value);
  END Writer;

PROCEDURE Bounder (p: P;  VAR min, max: INTEGER) =
  BEGIN
    min := p.value;
    max := p.value;
  END Bounder;

PROCEDURE Compare (a, b: Expr.T;  VAR sign: INTEGER): BOOLEAN =
  VAR x, y: INTEGER;
  BEGIN
    IF NOT SplitPair (a, b, x, y) THEN RETURN FALSE END;
    IF    (x < y) THEN  sign := -1
    ELSIF (x > y) THEN  sign := 1
    ELSE                sign := 0
    END;
    RETURN TRUE;
  END Compare;

PROCEDURE Split (e: Expr.T;  VAR i: INTEGER;  VAR t: Type.T): BOOLEAN =
  BEGIN
    TYPECASE e OF
    | NULL => RETURN FALSE;
    | P(p) => i := p.value;  t := Type.Base (p.type);  RETURN TRUE;
    ELSE      RETURN FALSE;
    END;
  END Split;

PROCEDURE SplitPair (a, b: Expr.T;  VAR x, y: INTEGER): BOOLEAN =
  BEGIN
    TYPECASE a OF
    | NULL => RETURN FALSE;
    | P(p) => x := p.value;
    ELSE      RETURN FALSE;
    END;
    TYPECASE b OF
    | NULL => RETURN FALSE;
    | P(p) => y := p.value;
    ELSE      RETURN FALSE;
    END;
    RETURN TRUE;
  END SplitPair;

PROCEDURE FPrinter (p: P;  map: Type.FPMap;  wr: MBuf.T) =
  BEGIN
    MBuf.PutText (wr, "enum ");
    MBuf.PutInt  (wr, p.value);
    MBuf.PutText (wr, " ");
    Type.Fingerprint (p.type, map, wr);
  END FPrinter;

PROCEDURE IsZeroes (p: P): BOOLEAN =
  BEGIN
    RETURN (p.value = 0);
  END IsZeroes;

PROCEDURE GenLiteral (p: P) =
  BEGIN
    Emit.OpI ("@", p.value);
  END GenLiteral;

BEGIN
END EnumExpr.
