(* Copyright (C) 1992, Digital Equipment Corporation                         *)
(* All rights reserved.                                                      *)
(* See the file COPYRIGHT for a full description.                            *)

(* Created 1990-01-16 by Jorge Stolfi                          *)
(* Last modified on Tue Feb 11 21:40:45 PST 1992 by muller     *)
(*      modified on Wed Nov 20 19:32:09 PST 1991 by stolfi     *)

MODULE RGBTest EXPORTS Main;

IMPORT 
  Fmt,  RGB, LDW, RGBDist, RGBScale, 
  ColorMatch, IntensityScale, Wr, Rd, Thread, Text, Intensity, Random;
FROM Stdio IMPORT stderr;
FROM LongRealTime IMPORT Now, Millisecond;

PROCEDURE Main () =
  <* FATAL Wr.Failure, Rd.Failure, Thread.Alerted *>
  BEGIN
    (* RGBPrivate.PrintMatrix (stderr); *)
    TestCoords ();
    TestDist ();
    TestNearestQuad ();
    TimeDist (30000);
    Wr.PutText (stderr, "Done.\n");
    Wr.Close(stderr);
  END Main;

PROCEDURE TestCoords () =

  PROCEDURE Tst (na: Text.T; a: RGB.T) =
    <* FATAL Wr.Failure, Rd.Failure, Thread.Alerted *>
    BEGIN
      Wr.PutText (stderr, "  " 
        & Fmt.Pad (na, 8) & " = ("
        & Fmt.Pad (Fmt.Real (a[0], 3, Fmt.Style.Flo), 5) & " "
        & Fmt.Pad (Fmt.Real (a[1], 3, Fmt.Style.Flo), 5) & " "
        & Fmt.Pad (Fmt.Real (a[2], 3, Fmt.Style.Flo), 5)
        & ") = ("
        & Fmt.Pad (Fmt.Real (LDW.LFromRGB (a), 3, Fmt.Style.Flo), 5) & " "
        & Fmt.Pad (Fmt.Real (LDW.DFromRGB (a), 3, Fmt.Style.Flo), 5) & " "
        & Fmt.Pad (Fmt.Real (LDW.WFromRGB (a), 3, Fmt.Style.Flo), 5) & ")\n")
    END Tst;

  <* FATAL Wr.Failure, Rd.Failure, Thread.Alerted *>
  BEGIN
    Wr.PutText (stderr, "Testing Lightness, Redness, Yellowness...\n");
    Tst ("black", RGB.Black);
    Tst ("grey", RGB.Grey (0.5));
    Tst ("white", RGB.White);
    Tst ("red", RGB.Red);
    Tst ("green", RGB.Green);
    Tst ("blue", RGB.Blue);
    Tst ("yellow", RGB.Yellow);
    Tst ("cyan", RGB.Cyan);
    Tst ("magenta", RGB.Magenta);
  END TestCoords;

PROCEDURE TestDist () =

  PROCEDURE Tst (na, nb: Text.T; a, b: RGB.T) =
    <* FATAL Wr.Failure, Rd.Failure, Thread.Alerted *>
    BEGIN
      Wr.PutText (stderr, "  " 
        & Fmt.Pad (na, 7) & "=("
        & Fmt.Pad (Fmt.Real (a[0], 3, Fmt.Style.Flo), 5) & " "
        & Fmt.Pad (Fmt.Real (a[1], 3, Fmt.Style.Flo), 5) & " "
        & Fmt.Pad (Fmt.Real (a[2], 3, Fmt.Style.Flo), 5)
        & ")");
      Wr.PutText (stderr, " " 
        & Fmt.Pad (nb, 7) & "=("
        & Fmt.Pad (Fmt.Real (b[0], 3, Fmt.Style.Flo), 5) & " "
        & Fmt.Pad (Fmt.Real (b[1], 3, Fmt.Style.Flo), 5) & " "
        & Fmt.Pad (Fmt.Real (b[2], 3, Fmt.Style.Flo), 5)
        & ")");
      Wr.PutText (stderr, " d = "
        & Fmt.Real (RGBDist.Perceptual (a, b), 5, Fmt.Style.Flo));
      Wr.PutText (stderr, " qd = "
        & Fmt.Real (RGBDist.Quick (a, b), 5, Fmt.Style.Flo));
      Wr.PutText (stderr, "\n")
    END Tst;

  VAR a, b: RGB.T;
  <* FATAL Wr.Failure, Rd.Failure, Thread.Alerted *>
  BEGIN
    Wr.PutText (stderr, "Testing distance function...\n");
    Tst ("black", "white",   RGB.Black, RGB.White);
    Tst ("grey", "white",    RGB.Grey (0.5), RGB.White);
    Tst ("grey", "black",    RGB.Grey (0.5), RGB.Black);
    Tst ("black", "red",     RGB.Black, RGB.Red);
    Tst ("black", "green",   RGB.Black, RGB.Green);
    Tst ("black", "blue",    RGB.Black, RGB.Blue);
    Tst ("black", "yellow",  RGB.Black, RGB.Yellow);
    Tst ("black", "cyan",    RGB.Black, RGB.Cyan);
    Tst ("black", "magenta", RGB.Black, RGB.Magenta);
    Tst ("white", "red",     RGB.White, RGB.Red);
    Tst ("white", "green",   RGB.White, RGB.Green);
    Tst ("white", "blue",    RGB.White, RGB.Blue);
    Tst ("white", "yellow",  RGB.White, RGB.Yellow);
    Tst ("white", "cyan",    RGB.White, RGB.Cyan);
    Tst ("white", "magenta", RGB.White, RGB.Magenta);
    Tst ("grey", "red",      RGB.Grey (RGB.Brightness (RGB.Red)), RGB.Red);
    Tst ("grey", "green",    RGB.Grey (RGB.Brightness (RGB.Green)), RGB.Green);
    Tst ("grey", "blue",     RGB.Grey (RGB.Brightness (RGB.Blue)), RGB.Blue);
    Tst ("grey", "yellow",   RGB.Grey (RGB.Brightness (RGB.Yellow)), RGB.Yellow);
    Tst ("grey", "cyan",     RGB.Grey (RGB.Brightness (RGB.Cyan)), RGB.Cyan);
    Tst ("grey", "magenta",  RGB.Grey (RGB.Brightness (RGB.Magenta)), RGB.Magenta);
    FOR i := 1 TO 10 DO
      a[0] := Random.Real (NIL);
      b[0] := Random.Real (NIL);
      a[1] := Random.Real (NIL);
      b[1] := Random.Real (NIL);
      a[2] := Random.Real (NIL);
      b[2] := Random.Real (NIL);
      Tst ("random", "random", a, b);
    END;
  END TestDist;

PROCEDURE TestNearestQuad () =

  PROCEDURE Tst (READONLY rgb, c0, c1, c2, c3: RGB.T) =
    VAR c: RGB.T;
    <* FATAL Wr.Failure, Rd.Failure, Thread.Alerted *>
    BEGIN
      Wr.PutText (stderr, "  ("
        & Fmt.Pad (Fmt.Real (rgb[0], 3, Fmt.Style.Flo), 5)
        & " "
        & Fmt.Pad (Fmt.Real (rgb[1], 3, Fmt.Style.Flo), 5)
        & " "
        & Fmt.Pad (Fmt.Real (rgb[2], 3, Fmt.Style.Flo), 5)
        & ") --> ("
        & Fmt.Pad (Fmt.Real (c0[0], 3, Fmt.Style.Flo), 5)
        & " "
        & Fmt.Pad (Fmt.Real (c0[1], 3, Fmt.Style.Flo), 5)
        & " "
        & Fmt.Pad (Fmt.Real (c0[2], 3, Fmt.Style.Flo), 5)
        & ")  dist =  "
        & Fmt.Real (RGBDist.Perceptual (rgb, c0), 6, Fmt.Style.Flo)
        & "\n");
      Wr.PutText (stderr, "                          ("
        & Fmt.Pad (Fmt.Real (c1[0], 3, Fmt.Style.Flo), 5)
        & " "
        & Fmt.Pad (Fmt.Real (c1[1], 3, Fmt.Style.Flo), 5)
        & " "
        & Fmt.Pad (Fmt.Real (c1[2], 3, Fmt.Style.Flo), 5)
        & ")  dist =  "
        & Fmt.Real (RGBDist.Perceptual (rgb, c1), 6, Fmt.Style.Flo)
        & "\n");
      Wr.PutText (stderr, "                          ("
        & Fmt.Pad (Fmt.Real (c2[0], 3, Fmt.Style.Flo), 5)
        & " "
        & Fmt.Pad (Fmt.Real (c2[1], 3, Fmt.Style.Flo), 5)
        & " "
        & Fmt.Pad (Fmt.Real (c2[2], 3, Fmt.Style.Flo), 5)
        & ")  dist =  "
        & Fmt.Real (RGBDist.Perceptual (rgb, c2), 6, Fmt.Style.Flo)
        & "\n");
      Wr.PutText (stderr, "                          ("
        & Fmt.Pad (Fmt.Real (c3[0], 3, Fmt.Style.Flo), 5)
        & " "
        & Fmt.Pad (Fmt.Real (c3[1], 3, Fmt.Style.Flo), 5)
        & " "
        & Fmt.Pad (Fmt.Real (c3[2], 3, Fmt.Style.Flo), 5)
        & ")  dist =  "
        & Fmt.Real (RGBDist.Perceptual (rgb, c3), 6, Fmt.Style.Flo)
        & "\n");
      c := RGB.Scale (0.25, RGB.Add (RGB.Add (c0, c1), RGB.Add (c2, c3)));
      Wr.PutText (stderr, "                average = ("
        & Fmt.Pad (Fmt.Real (c[0], 3, Fmt.Style.Flo), 5) & " "
        & Fmt.Pad (Fmt.Real (c[1], 3, Fmt.Style.Flo), 5) & " "
        & Fmt.Pad (Fmt.Real (c[2], 3, Fmt.Style.Flo), 5)
        & ")  dist =  "
        & Fmt.Real (RGBDist.Perceptual (rgb, c), 6, Fmt.Style.Flo)
        & "\n");
      Wr.PutText (stderr, "\n");
    END Tst;

  VAR
    rgb: RGB.T;
    quad: ColorMatch.IndexQuad;
    table: REF ARRAY OF RGB.T;
    sc: REF ARRAY OF Intensity.T;
  <* FATAL Wr.Failure, Rd.Failure, Thread.Alerted *>
  BEGIN
    Wr.PutText (stderr, "Testing ColorMatch.NearestQuad...\n");
    sc := IntensityScale.New (6, 0.0, 1.0, 1.4);
    table := RGBScale.Grid (sc^, sc^, sc^);
    FOR i := 1 TO 10 DO
      rgb[0] := Random.Real (NIL);
      rgb[1] := Random.Real (NIL);
      rgb[2] := Random.Real (NIL);
      quad := ColorMatch.NearestQuad (rgb, table^, RGBDist.Quick, 1.0);
      Tst (rgb, table[quad[0]], table[quad[1]], table[quad[2]], table[quad[3]]);
    END;

  END TestNearestQuad;

PROCEDURE TimeDist (nCalls: CARDINAL) =
  VAR t: LONGREAL; 
      d: REAL; 
      a, b: RGB.T;
  <* FATAL Wr.Failure, Rd.Failure, Thread.Alerted *>
  BEGIN
    Wr.PutText (stderr, "Timing distance function...\n");
    a[0] := Random.Real (NIL);
    a[1] := Random.Real (NIL);
    a[2] := Random.Real (NIL);
    b[0] := Random.Real (NIL);
    b[1] := Random.Real (NIL);
    b[2] := Random.Real (NIL);
    t := Now ();
    FOR i := 1 TO nCalls DO d := RGBDist.Perceptual (a, b);  END;
    t := Now () - t;
    Wr.PutText(stderr, "RGBDist.Perceptual: "
      & Fmt.Int(nCalls) & " calls at "
      & Fmt.Pad (Fmt.Real (FLOAT(t/Millisecond)/FLOAT(nCalls), 3, Fmt.Style.Flo), 5)
      & "ms per call\n"
    );
    t := Now ();
    FOR i := 1 TO nCalls DO d := RGBDist.Quick (a, b);  END;
    t := Now () - t;
    Wr.PutText(stderr, "RGBDist.Quick: "
      & Fmt.Int(nCalls) & " calls at "
      & Fmt.Pad (Fmt.Real (FLOAT(t/Millisecond)/FLOAT(nCalls), 3, Fmt.Style.Flo), 5)
      & "ms per call\n"
    );
  END TimeDist;

BEGIN
  Main ();
END RGBTest.

