/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/


#include "include.h"

/**************************************************************
*
*  Function: vertex_average()
*
*  Purpose:  For soapfilm model, move unconstrained vertices
*            to average position of neighbors.
*
*/

static  struct averages { REAL x[MAXCOORD];
                    REAL normal[MAXCOORD]; 
                    REAL area;  /* weighting */
                    int status;   /* <0 for don't move */
                  } *average;

void vertex_average(mode)
int mode;   /* VOLKEEP to keep volumes on both sides same */
{
  facet_id f_id;
  edge_id e_id;
  vertex_id v_id;
  REAL *x;
  struct averages *ave;
  int i;
  REAL xbar[MAXCOORD];
  double lambda;

  if ( web.skel[VERTEX].count == 0 ) return;

  average = (struct averages *)temp_calloc(web.skel[VERTEX].max_ord,
                                      sizeof(struct averages));

  if ( web.simplex_flag )
   { if ( mode == VOLKEEP )
       error("Vertex averaging not implemented for simplex model.\n",
	     RECOVERABLE);
     FOR_ALL_FACETS(f_id)
       simplex_facet_average(f_id);
   } 
  else if ( web.dimension == SOAPFILM )
   {
     FOR_ALL_FACETS(f_id)
       facet_average(f_id);
   } 
  else if ( web.dimension == STRING )
   {
     FOR_ALL_EDGES(e_id)
       edge_average(e_id);
   } 

  if ( web.simplex_flag || (web.dimension == SOAPFILM) )
   FOR_ALL_VERTICES(v_id)
   {
     if ( get_vattr(v_id) & (FIXED|BOUNDARY) ) continue;
     ave = average + ordinal(v_id);
     if (ave->status <= 0 ) continue;
/*
     if (ave->status == 0 )
       { sprintf(errmsg,"vertex %d has status 0\n",ordinal(v_id)+1);
	 error(errmsg,WARNING);
         continue;
       }
*/
     if (ave->area == 0.0 ) continue;
     x = get_coord(v_id);
     for ( i = 0 ; i < web.sdim ; i++ )
        xbar[i] = ave->x[i]/ave->area;

     if ( mode == VOLKEEP )
     {
       /* volume preserving projection */
       lambda = (dot(ave->normal,xbar,web.sdim) 
	       - dot(ave->normal,x,web.sdim))
                  /dot(ave->normal,ave->normal,web.sdim);
       for ( i = 0 ; i < web.sdim ; i++ )
         x[i] = xbar[i] - lambda*ave->normal[i];
     }
     else  /* tends to round off sharp vertices, I hope */
      for ( i = 0 ; i < web.sdim ; i++ )
       x[i] = xbar[i];

     if ( get_vattr(v_id) & CONSTRAINT )
       project_v_constr(v_id);
   }
  else /* string */
   FOR_ALL_VERTICES(v_id)
   {
     ave = average + ordinal(v_id);
     if (ave->status != 2 ) continue; /* too many edges or something */
     x = get_coord(v_id);

     if ( mode == VOLKEEP )
     {
       /* area preserving projection */
       lambda = 
	       - dot(ave->normal,ave->x,web.sdim)/2
                  /dot(ave->normal,ave->normal,web.sdim);
       for ( i = 0 ; i < web.sdim ; i++ )
         x[i] -=  lambda*ave->normal[i];
     }
     else
     {
       for ( i = 0 ; i < web.sdim ; i++ )
         x[i] +=  ave->x[i]/ave->status;
     }
   }

  temp_free((char *)average);

  /* recalculate stuff */
  calc_content();
  calc_pressure();
  calc_energy();

}

/***********************************************************************
*
* function: facet_average()
*
* purpose: find one facet's contributions to averaging of its vertices.
*
*/     

void facet_average(f_id)
facet_id f_id;
{
  int i,j;
  REAL side[FACET_EDGES][MAXCOORD];
  REAL normal[MAXCOORD];
  REAL area;
  REAL *x[FACET_VERTS];
  REAL centroid[MAXCOORD];
  facetedge_id fe_id;
  int sign;

  /* initialize centroid */
  memset((char *)centroid,0,sizeof(centroid));

  /* get side vectors */
  generate_facet_fe_init();
  for ( i = 0 ; i < FACET_EDGES ; i++ )
    { 
      generate_facet_fe(f_id,&fe_id);
      get_fe_side(fe_id,side[i]);
      x[i] = get_coord(get_fe_headv(fe_id));
      for ( j = 0 ; j < web.sdim ; j++ ) centroid[j] += x[i][j]/3;
    }
    
  /* calculate normal */ 
  cross_prod(side[0],side[1],normal);
    
  /* calculate surface tension energy */
  area = sqrt(dot(normal,normal,web.sdim))/2;
/*  if ( get_fattr(f_id) & DENSITY )
       area *= get_facet_density(f_id); */

  generate_facet_fe_init();
  while ( generate_facet_fe(f_id,&fe_id) )
   {
     vertex_id headv = get_fe_headv(fe_id);
     vertex_id tailv = get_fe_tailv(fe_id);
     struct averages *head,*tail;
     MAP econmap = get_f_constraint_map(f_id);

     head = average + ordinal(headv);
     tail = average + ordinal(tailv);
     if ( !equal_id(get_prev_facet(fe_id),get_next_facet(fe_id)) )
       /* max 2 facets */
        { head->status = -1; tail->status = -1; continue; }

#ifdef XXX
     if ( get_v_constraint_map(headv) == econmap )
       {
         /* make sure signs consistent around vertex */
	 
         if ( dot(normal,head->normal,web.sdim) < 0.0 ) sign = -1;
         else sign = 1;
         head->area += area;
         for ( i = 0 ; i < web.sdim ; i++ )
           { 
             head->normal[i] += sign*normal[i];
             head->x[i] += centroid[i]*area;
           }
       }
#endif

     if ( (tail->status >= 0) && (get_v_constraint_map(tailv) == econmap) )
       {
         /* make sure signs consistent around vertex */
	 facetedge_id vfe = get_vertex_fe(tailv);  /* reference orientation */
         facetedge_id thisfe = fe_id;

         for(;;) 
	  { if ( equal_id(vfe,thisfe) ) { sign = 1; break; }
	    thisfe = get_next_facet(thisfe);
	    if ( equal_id(vfe,thisfe) ) { sign = -1; break; }
	    thisfe = inverse_id(get_prev_edge(thisfe));
	    if ( equal_id(fe_id,thisfe) )
	      { sign = 0; 
		goto afterlost; /* lost */ 
              } 
          } 
	    
#ifdef XXX
         if ( dot(normal,tail->normal,web.sdim) < 0.0 ) sign = -1;
         else sign = 1;
#endif
         tail->area += area;
         for ( i = 0 ; i < web.sdim ; i++ )
           { 
             tail->normal[i] += sign*normal[i];
             tail->x[i] += centroid[i]*area;
           }
         tail->status++;
afterlost:;
       }

   }

}
 

/***********************************************************************
*
* function: edge_average()
*
* purpose: find one edge's contributions to averaging of its vertices.
*
*/     

void edge_average(e_id)
edge_id e_id;
{
  int i;
  struct averages *head,*tail;
  vertex_id headv = get_edge_headv(e_id);
  vertex_id tailv = get_edge_tailv(e_id);
  REAL side[MAXCOORD];

  get_edge_side(e_id,side);  /* in case of wraps */

  /* head */
  head = average + ordinal(headv);
  if ( (get_vattr(headv)&(BOUNDARY|FIXED)) || get_v_constraint_state(headv) )
    { head->status += 10; /* signal don't move */
      goto dotail;
    }
  for ( i = 0 ; i < web.sdim ; i++ )
    head->x[i] -= side[i];
   
  if ( head->status )
    for ( i = 0 ; i < web.sdim ; i++ )
      head->normal[i] += side[i];
  else
    for ( i = 0 ; i < web.sdim ; i++ )
      head->normal[i] -= side[i];
    
  head->status++;  /* mark one more edge done */

dotail:
  tail = average + ordinal(tailv);
  if ( (get_vattr(tailv)&(BOUNDARY|FIXED)) || get_v_constraint_state(tailv) )
    { tail->status += 10; /* signal don't move */
      return;
    }
  for ( i = 0 ; i < web.sdim ; i++ )
    tail->x[i] += side[i];
   
  if ( tail->status )
    for ( i = 0 ; i < web.sdim ; i++ )
      tail->normal[i] -= side[i];
  else
    for ( i = 0 ; i < web.sdim ; i++ )
      tail->normal[i] += side[i];
    
  tail->status++;  /* mark one more edge done */
    
  
}

/***********************************************************************
*
*  function: simplex_facet_average()
*
*  purpose:  add neighboring vertex coordinates for one facet
*
*/

void simplex_facet_average(f_id)
facet_id f_id;
{
  struct averages *head,*tail;
  vertex_id *v = get_facet_vertices(f_id);
  int i,j,k;

  for ( i = 0 ; i <= web.dimension ; i++ )
    { MAP conmapi = get_v_constraint_map(v[i]);
      REAL *x = get_coord(v[i]);

      for ( j = 0 ; j <= web.dimension ; j++ )
	{ head = average + ordinal(v[j]);
	  if ( get_v_constraint_map(v[j]) & ~conmapi ) continue;
	  for ( k = 0 ; k < web.sdim ; k++ )
	    head->x[k] += x[k];
          head->area += 1.0;
	  head->status++;
        }
    }
}

