/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@geom.umn.edu              *
*************************************************************/

/*****************************************************************
*
*  File: skeleton.c
*
*  Purpose: Variables and functions for skeleton element handling
*/

#include "include.h"

int sizes[NUMELEMENTS] = { sizeof(struct vertex),
                           sizeof(struct edge),
                           sizeof(struct facet),
                           sizeof(struct body),
                           sizeof(struct facetedge)
                         };

/* quadratic interpolation coefficients */
/* partials of interpolation polynomials at midpoints of patch edges */
/* control points are numbered 0 to 5 counterclockwise */
/* midpoints are numbered 0 to 2 counterclockwise after control pt 0 */
/* dip[midpoint][control point][partial]  */

REAL dip[FACET_EDGES][FACET_CTRL][2] = {
   { { -0.5, -0.5 }, { 0.0, -1.0 }, { 0.5, 0.0 }, { 0.0, 1.0 },
     { 0.0, -0.5 }, { 0.0, 1.0 } },
   { { 0.5, 0.5 }, { -1.0, -1.0 }, { 0.5, 0.0  }, { 1.0, 1.0 },
     { 0.0, 0.5 }, { -1.0, -1.0 } },
   { { -0.5, -0.5 }, { 1.0, 0.0 }, { -0.5, 0.0 }, { 1.0, 0.0 },
     { 0.0, 0.5 }, { -1.0, 0.0 } }
  };

/* this made into macro */
/* int get_attr(id)
 element_id id;
 {
   return elptr(id)->attr;
 }
*/

void set_attr(id,attrib)
element_id id;
ATTR attrib;
{
  elptr(id)->attr |= attrib;
}

void unset_attr(id,attrib)
element_id id;
ATTR attrib;
{
  elptr(id)->attr &= ~attrib;
}

void set_fe_edge(fe_id,e_id)
facetedge_id fe_id;
edge_id e_id;
{
  if ( inverted(fe_id) ) invert(e_id);
  feptr(fe_id)->fe_edge_id = e_id;
}

#ifndef get_fe_edge
edge_id get_fe_edge(fe_id)
facetedge_id fe_id;
{
  edge_id e_id;
   
  e_id = feptr(fe_id)->fe_edge_id;
  if ( inverted(fe_id) ) invert(e_id);
  return e_id;
}
#endif


void set_fe_facet(fe_id,e_id)
facetedge_id fe_id;
edge_id e_id;
{
  if ( inverted(fe_id) ) invert(e_id);
  feptr(fe_id)->fe_facet_id = e_id;
}

facet_id get_fe_facet(fe_id)
facetedge_id fe_id;
{
  facet_id f_id;
  
  f_id = feptr(fe_id)->fe_facet_id;
  if ( inverted(fe_id) ) invert(f_id);
  return f_id;
}

#ifndef get_prev_edge
facetedge_id get_prev_edge(fe_id)
facetedge_id fe_id;
{
  if ( inverted(fe_id) ) return inverse_id(feptr(fe_id)->nextedge[1]);
  else return feptr(fe_id)->nextedge[0];
}
#endif

#ifndef get_next_edge
facetedge_id get_next_edge(fe_id)
facetedge_id fe_id;
{
  if ( inverted(fe_id) ) return inverse_id(feptr(fe_id)->nextedge[0]);
  else return feptr(fe_id)->nextedge[1];
}
#endif

facetedge_id get_prev_facet(fe_id)
facetedge_id fe_id;
{
  if ( inverted(fe_id) ) return inverse_id(feptr(fe_id)->nextfacet[1]);
  else return feptr(fe_id)->nextfacet[0];
}

facetedge_id get_next_facet(fe_id)
facetedge_id fe_id;
{
  if ( inverted(fe_id) ) return inverse_id(feptr(fe_id)->nextfacet[0]);
  else return feptr(fe_id)->nextfacet[1];
}

void set_prev_edge(fe_id,fe)
facetedge_id fe_id;
facet_id fe;
{
  if ( !valid_id(fe_id) ) return;
  if ( inverted(fe_id) )
   { invert(fe);
     feptr(fe_id)->nextedge[1] = fe;
   }
  else
     feptr(fe_id)->nextedge[0] = fe;
}


void set_next_edge(fe_id,fe)
facetedge_id fe_id,fe;
{
  if ( !valid_id(fe_id) ) return;
  if ( inverted(fe_id) )
   { invert(fe);
     feptr(fe_id)->nextedge[0] = fe;
   }
  else
     feptr(fe_id)->nextedge[1] = fe;
}


void set_prev_facet(fe_id,fe)
facetedge_id fe_id,fe;
{
  if ( !valid_id(fe_id) ) return;
  if ( inverted(fe_id) )
   { invert(fe);
     feptr(fe_id)->nextfacet[1] = fe;
   }
  else
     feptr(fe_id)->nextfacet[0] = fe;
}


void set_next_facet(fe_id,fe)
facetedge_id fe_id,fe;
{
  if ( !valid_id(fe_id) ) return;
  if ( inverted(fe_id) )
   { invert(fe);
     feptr(fe_id)->nextfacet[0] = fe;
   }
  else
     feptr(fe_id)->nextfacet[1] = fe;
}


void set_edge_wrap(e_id,wrap)
edge_id e_id;
WRAPTYPE  wrap;
{
   eptr(e_id)->wrap =  inverted(e_id)  ? (*sym_inverse)(wrap) : wrap ;
}

WRAPTYPE get_edge_wrap(e_id)
edge_id e_id;
{
  WRAPTYPE wrap = eptr(e_id)->wrap;
  return   ( inverted(e_id) ? (*sym_inverse)(wrap) : wrap );
}

void set_edge_fe(e_id,fe)
edge_id e_id;
facetedge_id fe;
{
   if ( inverted(e_id) ) invert(fe);
   eptr(e_id)->fe_id = fe;
}

facetedge_id get_edge_fe(e_id)
edge_id e_id;
{
   facetedge_id fe;
   
   fe = eptr(e_id)->fe_id;
   if ( inverted(e_id) ) invert(fe);
   return fe;
}

#ifndef get_edge_tailv
vertex_id get_edge_tailv(e_id)
edge_id e_id;
{
  if ( inverted(e_id) )
    return eptr(e_id)->vertices[1];
  else
    return eptr(e_id)->vertices[0];
}
#endif

#ifndef get_edge_headv
vertex_id get_edge_headv(e_id)
edge_id e_id;
{
  if ( inverted(e_id) )
    return eptr(e_id)->vertices[0];
  else
    return eptr(e_id)->vertices[1];
}
#endif

void set_edge_tailv(e_id,v_id)
edge_id e_id;
vertex_id v_id;
{
  if ( inverted(e_id) )
    eptr(e_id)->vertices[1] = v_id;
  else
    eptr(e_id)->vertices[0] = v_id;
}


void set_edge_headv(e_id,v_id)
edge_id e_id;
vertex_id v_id;
{
  if ( inverted(e_id) )
    eptr(e_id)->vertices[0] = v_id;
  else
    eptr(e_id)->vertices[1] = v_id;
}

body_id get_facet_body(f_id)
facet_id f_id;
{
  if ( !valid_id(f_id) ) return NULLID;
  if ( inverted(f_id) ) return fptr(f_id)->body[1];
  else  return fptr(f_id)->body[0];
}

void set_facet_body(f_id,b_id)
facet_id f_id;
body_id  b_id;
{
  if ( !valid_id(f_id) ) return;
  if ( inverted(f_id) )  fptr(f_id)->body[1] = b_id;
  else  fptr(f_id)->body[0] = b_id;
}

facetedge_id get_facet_fe(f_id)
facet_id f_id;
{
  facetedge_id fe;
  
  fe = fptr(f_id)->fe_id;
  if ( inverted(f_id) ) invert(fe);
  return fe;
}

void set_facet_fe(f_id,fe)
facet_id f_id;
facetedge_id fe;
{
  
  if ( inverted(f_id) ) invert(fe);
  fptr(f_id)->fe_id = fe;
}

vertex_id new_vertex(x)
REAL *x;
{ 
  int i;
  vertex_id v_id;
  REAL *y;

  v_id = new_element(VERTEX);
  if ( x )
    { y = get_coord(v_id);
      for ( i = 0 ; i < web.sdim ; i++ ) y[i] = x[i];
    }
  set_vertex_fe(v_id,NULLFACETEDGE);
  if ( web.con_global_map )
   {
     vptr(v_id)->constraint_map |= web.con_global_map;
     set_attr(v_id,CONSTRAINT);
   }

  set_v_global(v_id);

  /* interrupt conjugate gradient */
  if ( cg_hvector ) { free((char *)cg_hvector); cg_hvector = NULL; }

  return v_id;
}

vertex_id dup_vertex(old_v)
vertex_id old_v;
{ 
  vertex_id v_id;
  int ord;

  v_id = new_element(VERTEX);
  ord = ordinal(v_id);
  memcpy((char *)&(elptr(v_id)->attr),(char *)&(elptr(old_v)->attr),
           sizeof(struct vertex) - 2*sizeof(element_id));
  elptr(v_id)->ordinal = ord;  /* restore new ordinal */
  return v_id;
}

edge_id   new_edge(tail_id,head_id)
vertex_id tail_id,head_id;
{ 
  edge_id e_id;
  vertex_id v_id;
  REAL *x,*h,*t;
  int i;

  e_id = new_element(EDGE);
  set_edge_tailv(e_id,tail_id);
  set_edge_headv(e_id,head_id);
  set_edge_fe(e_id,NULLFACETEDGE);
  set_edge_color(e_id,DEFAULT_COLOR);
  if ( web.modeltype == QUADRATIC )
    {
      /* quadratic version; linear interpolation of midpoint */
      v_id = new_element(VERTEX);
      set_edge_midv(e_id,v_id);
      set_vertex_fe(v_id,NULLFACETEDGE);
      h = get_coord(head_id);
      t = get_coord(tail_id);
      x = get_coord(v_id);
      for ( i = 0 ; i < web.sdim ; i++ )
        x[i] = (h[i] + t[i])/2.0;
    }
  return e_id;
}

edge_id dup_edge(old_e)
edge_id old_e;
{ 
  edge_id e_id;
  int ord;

  e_id = new_element(EDGE);
  ord = ordinal(e_id);
  memcpy((char *)&(elptr(e_id)->attr),(char *)&(elptr(old_e)->attr),
           sizeof(struct edge) - 2*sizeof(element_id));
  elptr(e_id)->ordinal = ord; /* restore new ordinal */
  if ( inverted(old_e) ) return inverse_id(e_id);
  return e_id;
}

facet_id  new_facet ()
{ 
  facet_id f_id;

  f_id = new_element(FACET);
  set_facet_body(f_id,NULLBODY);
  set_facet_body(inverse_id(f_id),NULLBODY);
  set_facet_fe(f_id,NULLFACETEDGE);
  set_facet_color(f_id,DEFAULT_COLOR);
  if  ( web.surf_global_map  )
    { set_attr(f_id,SURF_ENERGY);
      fptr(f_id)->surfen_map |= web.surf_global_map;
    }
  if  ( web.quant_global_map  )
    { set_attr(f_id,SURF_QUANTITY);
      fptr(f_id)->quantity_map |= web.quant_global_map;
    }
  return f_id;
}

facet_id dup_facet(old_f)
facet_id old_f;
{ 
  facet_id f_id;
  int ord;

  f_id = new_element(FACET);
  ord = ordinal(f_id);
  memcpy((char *)&(elptr(f_id)->attr),(char *)&(elptr(old_f)->attr),
           sizeof(struct facet) - 2*sizeof(element_id));
  elptr(f_id)->ordinal = ord; /* restore new ordinal */
  if ( inverted(old_f ) ) return inverse_id(f_id);
  return f_id;
}

body_id new_body()
{ 
  body_id b_id;

  b_id = new_element(BODY);
  set_body_fe(b_id,NULLFACETEDGE);
  web.bodycount++;
  return b_id;
}

body_id dup_body(old_b)
body_id old_b;
{ 
  body_id b_id;
  int ord;

  b_id = new_element(BODY);
  ord = ordinal(b_id);
  memcpy((char *)&(elptr(b_id)->attr),(char *)&(elptr(old_b)->attr),
           sizeof(struct body) - 2*sizeof(element_id));
  elptr(b_id)->ordinal = ord; /* restore new ordinal */
  web.bodycount++;
  return b_id;
}

facetedge_id new_facetedge(f_id,e_id)
facet_id f_id;
edge_id e_id;
{ 
  facet_id fe_id;

  fe_id = new_element(FACETEDGE);
  set_fe_edge(fe_id,e_id);
  set_fe_facet(fe_id,f_id);
  set_prev_edge(fe_id,NULLFACETEDGE);
  set_next_edge(fe_id,NULLFACETEDGE);
  set_prev_facet(fe_id,NULLFACETEDGE);
  set_prev_facet(fe_id,NULLFACETEDGE);
  return fe_id;
}

REAL get_edge_length(e_id)
edge_id e_id;
{
  return   (eptr(e_id)->length);
}


REAL get_facet_pressure(f_id)
facet_id f_id;
{ 
  return  (get_body_pressure(MSCBUG get_facet_body(f_id)) - 
      get_body_pressure(MSCBUG get_facet_body(MSCBUG facet_inverse(f_id))));
}


static facet_id cur_facet;
void generate_facet_fe_init()  { cur_facet = NULLFACET; }

int generate_facet_fe(f_id,fe_id)
facet_id f_id;
facetedge_id *fe_id;
{
  static facetedge_id first_fe,gfe;

  if ( !equal_id(f_id,cur_facet) )  /* first call */
    { cur_facet = f_id;
      first_fe = gfe = *fe_id = get_facet_fe(f_id);
      return 1;
    }
  gfe = *fe_id = get_next_edge(gfe);
  if ( !equal_id(gfe,first_fe) && valid_id(gfe) ) return 1;
  cur_facet = NULLFACET;
  *fe_id = NULLFACETEDGE;
  return 0;
}


static edge_id cur_edge;
void generate_edge_fe_init() { cur_edge = NULLEDGE; }

int generate_edge_fe(e_id,fe_id)
edge_id e_id;
facetedge_id *fe_id;
{
  static facetedge_id first_fe,fe;

  if ( !equal_id(e_id,cur_edge) )  /* first call */
    { 
      first_fe = get_edge_fe(e_id);
      if ( !valid_id(first_fe) ) return 0;
      cur_edge = e_id;
      fe = *fe_id = first_fe;
      return 1;
    }
  if ( !equal_id((fe = *fe_id = get_next_facet(fe)),first_fe) ) return 1;
  *fe_id = NULLFACETEDGE;
  cur_edge = NULLEDGE;
  return 0;
}

/**************************************************************
*
*  Function:  equal_constr()
*
*  Purpose:   See if two elements have the same set of constraints.
*
*/

int equal_constr(id1,id2)
element_id id1,id2;
{
  int con1=0,con2=0;

  switch ( id_type(id1) )
    {
      case VERTEX: 
                   con1      = get_v_constraint_map(id1);
                   break;

      case EDGE  : 
                   con1      = get_e_constraint_map(id1) | web.con_global_map;
                   break;

      case FACET : 
                   con1      = get_f_constraint_map(id1) | web.con_global_map;
                   break;
    }


  switch ( id_type(id2) )
    {
      case VERTEX: 
                   con2      = get_v_constraint_map(id2);
                   break;

      case EDGE  :
                   con2      = get_e_constraint_map(id2) | web.con_global_map;
                   break;

      case FACET :
                   con2      = get_f_constraint_map(id2) | web.con_global_map;
                   break;
    }

  return (con1 == con2);
}


  

  
