/*
 * shepherd.c
 *
 * x-kernel v3.2
 *
 * Copyright (c) 1991  Arizona Board of Regents
 *
 *
 * $Revision: 1.6 $
 * $Date: 1992/01/30 20:08:16 $
 */

/* this cannot be used with the XKMACHKERNEL option because of the
   reliance on machIPC
 */

#include    <mach/message.h>

#ifndef XKMACHKERNEL
#include    <mach.h>
#include    <cthreads.h>
#else
#include    <kern/thread.h>
#endif ! XKMACHKERNEL

#include    "shepherd.h"


typedef struct xkShepMsg_s {

  mach_msg_header_t    h;
  mach_msg_type_t    t;

  void            (*func)();    /* 32 bits */
  char            *arg;    /* 32 bits */

} xkShepMsg_t;


mach_port_t    mgShepBlockingPort;
mach_port_t    mgShepNonblockingPort;
mach_port_t    mgShepReceivePortSet;



/*
 * Entry code for each shepherd thread
 *
 * Waits on port set mgShepReceivePortSet; 
 * Interprets messages and performs proper action
 *
 */
void
xkShepherd( )
{
  xkShepMsg_t        msg;

  msg.t.msgt_name       = MACH_MSG_TYPE_INTEGER_32;
  msg.t.msgt_size       = 32;
  msg.t.msgt_number     = 2;
  msg.t.msgt_inline     = 1;
  msg.t.msgt_longform     = 0;
  msg.t.msgt_deallocate   = 0;
  msg.t.msgt_unused     = 0;

  while( TRUE ) {

    if( mach_msg( &msg, MACH_RCV_MSG, 0, sizeof( msg )
             , mgShepReceivePortSet
             , MACH_MSG_TIMEOUT_NONE, MACH_PORT_NULL ) 
      == KERN_SUCCESS ) {

      (*msg.func)( msg.arg );

    }
  }
}



/*
 * Invoke a shepherd thread
 *
 * Invoke a shepherd thread of priority pri requesting it to
 *  run (*func)( arg )
 *
 * type == XK_SHEP_NONBLOCKING
 *   non-blocking, non-queuing, failure implies no available threads
 *     
 * type == XK_SHEP_BLOCKING
 *   blocking, queuing, failure implies kernel problems
 *     
 * The Mach ccom dies if arg is void *
 *
 */
kern_return_t
xInvokeShepherd( type, func, arg  )
     xkShepType_t type;
     void (*func)();
     char *arg;
{
  xkShepMsg_t        msg;
  mach_msg_option_t    option;


  msg.h.msgh_bits = MACH_MSGH_BITS( MACH_MSG_TYPE_COPY_SEND, 0 );
  msg.h.msgh_size = sizeof( msg );
  msg.h.msgh_remote_port  
    = type == XK_SHEP_NONBLOCKING 
      ? mgShepNonblockingPort : mgShepBlockingPort;
  msg.h.msgh_local_port   = MACH_PORT_NULL;

  msg.t.msgt_name       = MACH_MSG_TYPE_INTEGER_32;
  msg.t.msgt_size       = 32;
  msg.t.msgt_number     = 2;
  msg.t.msgt_inline     = 1;
  msg.t.msgt_longform     = 0;
  msg.t.msgt_deallocate   = 0;
  msg.t.msgt_unused     = 0;

  msg.func = func;
  msg.arg  = arg;

  option = MACH_SEND_MSG
    |  ( type == XK_SHEP_NONBLOCKING ? MACH_SEND_TIMEOUT : 0 );

  return( mach_msg( &msg, option, sizeof( msg )
              , 0, MACH_PORT_NULL
              , type == XK_SHEP_NONBLOCKING ? 0 : MACH_MSG_TIMEOUT_NONE
              , MACH_PORT_NULL ) );

}


/*
 * Create shepherd message ports and single port set
 *
 * *bPort   -- msg requests are queuing
 * *nbPort  -- msg requests are nonqueuing
 * *portSet -- port set containing both *bPort and *nbPort receive rights
 *
 */
kern_return_t
xkCreateShepPorts( bPort, nbPort, portSet )
     mach_port_t *bPort;
     mach_port_t *nbPort;
     mach_port_t *portSet;
{
  register kern_return_t      r;
  register task_t          self;

  if( (r = mach_port_allocate(  ( self = mach_task_self() )
                     , MACH_PORT_RIGHT_RECEIVE
                     , bPort ))  !=  KERN_SUCCESS ) {

    return( r );
  }

  if( (r = mach_port_insert_right(  self
                        , *bPort, *bPort
                        , MACH_MSG_TYPE_MAKE_SEND )) 
    != KERN_SUCCESS ) {

    return( r );
  }

  if( (r = mach_port_set_qlimit(  self, *bPort
                      , MACH_PORT_QLIMIT_MAX  )) != KERN_SUCCESS ) {

    return( r );
  }

  if( (r = mach_port_allocate(  self
                     , MACH_PORT_RIGHT_RECEIVE
                     , nbPort ))  !=  KERN_SUCCESS ) {

    return( r );
  }

  if( (r = mach_port_insert_right(  self
                        , *nbPort, *nbPort
                        , MACH_MSG_TYPE_MAKE_SEND )) 
    != KERN_SUCCESS ) {

    return( r );
  }

  if( (r = mach_port_set_qlimit( self, *nbPort, 1 )) != KERN_SUCCESS ) {

    return( r );
  }

  if( (r = mach_port_allocate(  self
                     , MACH_PORT_RIGHT_PORT_SET
                     , portSet ))  !=  KERN_SUCCESS ) {

    return( r );
  }

  if( (r = mach_port_move_member(  self
                       , *bPort
                       , *portSet ))  !=  KERN_SUCCESS ) {

    return( r );
  }

  if( (r = mach_port_move_member(  self
                       , *nbPort
                       , *portSet ))  !=  KERN_SUCCESS ) {

    return( r );
  }


  return( KERN_SUCCESS );
}


/*
 * Initialize a pool of shepherd threads
 * 
 * Start up count threads
 *
 */
kern_return_t
shepInit( count )
     int count;
{
  kern_return_t      r;
  task_t          self;

  if( (r = xkCreateShepPorts(  &mgShepBlockingPort
                    , &mgShepNonblockingPort
                    , &mgShepReceivePortSet )) != KERN_SUCCESS ) {

    return( r );
  }


#ifndef XKMACHKERNEL

  cthread_init();

  while( count-- )

    cthread_detach( cthread_fork( xkShepherd, 0 ) );

#else  XKMACHKERNEL

  self = mach_task_self();

  while( count-- )

    kernel_thread( self, xkShepherd, 0 );

#endif ! XKMACHKERNEL

  return( KERN_SUCCESS );
}

/**************************************************************************/



