/* 
 * z8530scc.c
 *
 * x-kernel v3.1	12/10/90
 *
 * Copyright (C) 1990  Larry L. Peterson and Norman C. Hutchinson
 */

#include "xkernel.h"
#include "z8530scc_internal.h"

int tracesccp;

static XObj SCC;

/* define DO_SERIAL_LINES if you want the kernel to mess with the serial
   ports.  This makes kdbx unusable, however */
#undef DO_SERIAL_LINES

/* Private */
void SccInterruptHandler();
static void SccInitChannel();
static void SccSetDefaults();
static SccBellKludge2();

/* Chip needs 1.6 usec settling time between commands */
#ifdef SUN3
SccSnooze() { int i = 5; while(i) i--; }
#else
#define SccSnooze() { int i, j; i = j; j = i; }
#endif SUN3


unsigned char ReadRegister0(scc) 
SccChannel *scc;
{
  unsigned char value;
  int x = spl7();
  SccSnooze();
  value = (scc)->control;
  splx(x);
  return value;
}

WriteRegister0(value, scc)
unsigned char value;
SccChannel *scc;
{
  int x = spl7();
  SccSnooze();
  (scc)->control = (value);
  splx(x);
}

unsigned char ReadRegister(reg, scc)
int reg;
register SccChannel *scc;
{
  register unsigned char value;

  int x = spl7();
  SccSnooze(); 
  (scc)->control = (reg);
  SccSnooze();
  value = (scc)->control;
  splx(x);
  return value;
}

unsigned char ReadDataRegister(scc)
SccChannel *scc;
{
  register unsigned char value;

  int x = spl7();
  SccSnooze();
  value = (scc)->data;
  splx(x);
  return value;
}

WriteRegister(reg, value, scc)
int reg;
char value;
SccChannel *scc;
{
  int x = spl7();
  SccSnooze();
  (scc)->control = (reg);
  SccSnooze();
  (scc)->control = (value);
  splx(x);
}

WriteDataRegister(value, scc)
char value;
SccChannel *scc;
{
  int x = spl7();
  SccSnooze();
  (scc)->data = (value);
  splx(x);
}

/* List of channel status structures */
#define KeyboardSccStatus	(AllSccStatus[0])
#define MouseSccStatus		(AllSccStatus[1])
#define SerialSccStatus		(&AllSccStatus[2])

XObj SccSessions[4] = { NULL, NULL, NULL, NULL };

SccChannelStatus AllSccStatus[4] =
{
  {
    /* link 	*/	NULL,
    /* channel 	*/	&( ((SccChip *) V_KBD_MOUSE_UART)->a ),
    /* ab 	*/	SCC_CHANNEL_A,
    /* index    */	0,
  },
  {
    /* link 	*/	&AllSccStatus[0],
    /* channel 	*/	&( ((SccChip *) V_KBD_MOUSE_UART)->b ),
    /* ab 	*/	SCC_CHANNEL_B,
    /* index    */	1,
  },
  {
    /* link */		&AllSccStatus[1],
    /* channel */	&( ((SccChip *) V_SERIAL_PORT)->a ),
    /* ab */		SCC_CHANNEL_A,
    /* index    */	2,
  },
  {
    /* link */		&AllSccStatus[2],
    /* channel */	&( ((SccChip *) V_SERIAL_PORT)->b ),
    /* ab */		SCC_CHANNEL_B,
    /* index    */	3,
  }
};

SccChannelStatus *SccChannelList = 
#ifdef DO_SERIAL_LINES
				&SerialSccStatus[1];
#else
				&MouseSccStatus;
#endif

/*
 * Powerup and initialize the Sun-3 SCC's.
 */
scc_init(self)
XObj self;
{
  register SccChannelStatus *s;
  extern int Asm_SccInterruptHandler();

  TRACE0(sccp, 1, "scc init");
  assert(!SCC);
  SCC = self;

  /* Plug interrupt vector */
  setexvec(Asm_SccInterruptHandler, INT6);

  TRACE0(sccp, 3, "scc resetting controllers");
  /* Reset SCCs */
  WriteRegister0(SccNoOperation, KeyboardSccStatus.channel);
  WriteRegister(9, SccForceHardwareReset, KeyboardSccStatus.channel);

#ifdef DO_SERIAL_LINES
  WriteRegister0(SccNoOperation, SerialSccStatus[0].channel);
  WriteRegister(9, SccForceHardwareReset, SerialSccStatus[0].channel);
#endif

  TRACE0(sccp, 3, "scc init status structures");
  /* Initialize status structures */
  SccSetDefaults(&KeyboardSccStatus);
  SccSetDefaults(&MouseSccStatus);
#ifdef DO_SERIAL_LINES
  SccSetDefaults(&SerialSccStatus[0]);
  SccSetDefaults(&SerialSccStatus[1]);
#endif

  /* The Sun-3 uses vectored interrupts instead of autovectored. We
   * set the scc vector number to point at the level-6 autovector
   * location for easier compatibility.
   */
#ifdef DO_SERIAL_LINES
  WriteRegister(2, (INT6/4), SerialSccStatus[0].channel);
#endif

  WriteRegister(2, (INT6/4), MouseSccStatus.channel);


  TRACE0(sccp, 3, "scc init channels");
  /* Initialize channels from status structures */
  for (s = SccChannelList; s != NULL; s = s->link) {
    SccInitChannel(s);
    InitSemaphore(&s->mutex, 1);
  }

  /* Enable chip interrupts */
#ifdef SUN3
  WriteRegister(9,  SccMasterIntEnable, KeyboardSccStatus.channel);
#else
  WriteRegister(9,  SccMasterIntEnable|
		    SccNoVector, KeyboardSccStatus.channel);
#endif

#ifdef DO_SERIAL_LINES
#ifdef SUN3
  WriteRegister(9,  SccMasterIntEnable, SerialSccStatus[0].channel);
#else
  WriteRegister(9,  SccMasterIntEnable|
		    SccNoVector, SerialSccStatus[0].channel);
#endif
#endif
}

/* Initialize a status structure with default values */
static void SccSetDefaults(s)
SccChannelStatus *s;
{
  s->dataRate = (s == &MouseSccStatus || s == &KeyboardSccStatus ? 1200 : 9600);
  s->charWidth = 8;
  s->parity = PARITY_NONE;
  s->stopBits = STOP_BITS_1;
  s->dtr = 1;
  s->rts = 0;
  s->brk = 0;
}

/* Initialize a channel from its status structure */
static void SccInitChannel(s)
SccChannelStatus *s;
{
  register int timeconst;

  /* Set clock divisor, parity, and stop bits.  The chip seems to
   *  work much better in X16 mode than in X1 mode.  Fortunately,
   *  the baud rate generator has no trouble generating high enough
   *  frequencies. */
  WriteRegister(4,  SccX16Clock|
		    SccStopBits(s->stopBits)|
		    SccParity(s->parity), s->channel);
		
  /* Set data rate */
  WriteRegister(14, 0, s->channel);  /* temp disable baud rate gen */
  timeconst = (int) ((SCC_CLOCK_RATE*10/2/16) / s->dataRate - 20 + 5)/10;
  TRACE1(sccp, 5, "Time const = %d", timeconst);
  WriteRegister(12, timeconst, s->channel);
  WriteRegister(13, timeconst>>8, s->channel);
  WriteRegister(14, SccBrGenEnable|
		    SccBrGenSourcePclk, s->channel);	/*?*/
		
  /* Establish baud rate generator as clock source */
  WriteRegister(11, SccRxClockBrGen|
		    SccTxClockBrGen|
		    SccTRxCOutput|
		    SccTRxCOutputTxClock, s->channel);
		
  /* Disable all external/status interrupts */
  WriteRegister(15, 0, s->channel);

  /* Establish Tx parameters and enable transmitter */
  WriteRegister(5,  (s->dtr ? SccDtr : 0)|
		    (s->rts ? SccRts : 0)|
		    SccTxBitsPerChar(s->charWidth)|
		    (s->brk ? SccSendBreak : 0)|
		    SccTxEnable, s->channel);
		
  /* Establish Rx parameters and enable receiver */
  WriteRegister(3,  SccRxBitsPerChar(s->charWidth)|
		    SccRxEnable, s->channel);
			
  /* Enable Rx and Tx interrupts */
  WriteRegister(1,  SccRxIntAllChars|
		    SccTxIntEnable, s->channel);
}

/*ARGSUSED*/
XObj scc_open( self, hlp, parts )
XObj self, hlp;
Part *parts;
/*
 * Create an session for a serial line.  parts[0] is the line number (0 =
 * keyboard,  1 = mouse, 2 = serial0, 3 = serial1).  We allow only 1 opens per 
 * device.
 */
{
  register SccChannelStatus *scc;
  int channel;
  register XObj s;

  channel = *(int *) parts[0].address;
  TRACE1(sccp, 3, "scc_open channel = %d", channel);
  if (channel < 0 || channel > 3) {
    TRACE0(sccp, 3, "invalid channel number, return ERR_SESSN");
    return(ERR_SESSN);
  }
  /* Find SccStatus structure for the requested unit number */
  scc = &AllSccStatus[channel];

  /* These are single-user devices; check for conflict */
  if (SccSessions[channel] != NULL) {
    TRACE0(sccp, 1, "Open of busy device, return ERR_SESSN");
    return (ERR_SESSN);
  }
  s = x_createsession(hlp, SCC, 0);
  s->state = (char *) &AllSccStatus[channel];
  SccSessions[channel] = s;

  /* Initialize the device */
  SccSetDefaults(scc);
  SccInitChannel(scc);

  TRACE1(sccp, 3, "scc open returns %x", SccSessions[channel]);
  return( SccSessions[channel] );
}

void scc_writeAChar(scc)
register SccChannelStatus *scc;
{
  Msg msg;
  register SccChannel *channel = scc->channel;
  char theChar;
  
  msg = scc->outputMsg;
  msg_peek(msg, 0, 1, &theChar);
  TRACE3(sccp, 5, "scc_writeAChar (%S) to %d", 1, &theChar, channel);
  WriteDataRegister(theChar, channel);
  if (--(scc->outputCount) == 0) {
    msg_free(msg);
    msg_clear(scc->outputMsg);
    TRACE0(sccp, 4, "Done writing characters, Ving");
    V(&scc->mutex);
  } else {
    msg_pop(msg, 1);
  }
}

/*ARGSUSED*/
scc_push( s, msg, rmsg_ptr )
XObj s;
Msg msg;
Msg *rmsg_ptr;
{
  register SccChannelStatus *scc;
  register SccChannel *channel;
  int x, len;

  if ((len = msg_len(msg)) == 0) return;

  TRACE1(sccp, 3, "scc push, len = %d", len);
  scc = (SccChannelStatus *) s->state;
  P(&scc->mutex);
  scc->outputMsg = msg;
  scc->outputCount = len;
  channel = scc->channel;

  /* Output a byte if we can */
  x = spl7();
  if (ReadRegister0(channel) & SccTxBufferEmpty) scc_writeAChar(scc);
  (void) splx(x);
  return;
}

/*ARGSUSED*/
scc_demux(self, s, c)
XObj self, s;
char c;
{
  Msg msg;
  TRACE3(sccp, 5, "scc demux s = %x, c = %S", s, 1, &c);
  msg_make_allstack(msg, 128, (char *)0, 0);
  msg_push(msg,1)[0] = c;
  x_demux(s, msg);
}

void SccInterruptHandler()
{
  register SccChannelStatus *scc;
  register XObj s;
  char rr0, c;

  TRACE0(sccp, 7, "scc interrupt");
  /* Examine each channel to see which interrupted */
  for (scc = SccChannelList; scc != NULL; scc = scc->link) {
    rr0 = ReadRegister0(scc->channel);

    /* First check receiver interrupt */
    if (rr0 & SccRxCharacterAvailable) {
      /* Read the received character */
      TRACE1(sccp, 7, "scc rx on %d", scc->index);
      c = ReadDataRegister(scc->channel);
#ifdef CATCHINFINITELOOPS
      localInfiniteLoopCatcher(c);
#endif
      /* Is there an instance for this channel? */
      s = SccSessions[scc->index];
      if (s) {
	/* pop something */
	CreateKernelProcess(scc_demux, 4, 3, 0, s, c);
      }
    }

    /*
     * Next check if transmitter buffer is empty.  If it is,
     * even though it may not have caused the interrupt, there is no harm
     * in executing this code anyway.
     */
    if (rr0 & SccTxBufferEmpty) {
      /* Is there a writer waiting? */
      if (!msg_isnull(scc->outputMsg)) {
	scc_writeAChar(scc);
      } else {
	/* No writer waiting -- reset the interrupt */
	WriteRegister0(SccResetTxIntPending, scc->channel);
      }
    }
    WriteRegister0(SccResetHighestIus, scc->channel);
  }
}

#ifdef undef
/*
 * Query the device-dependent parameters of an Scc channel
 */
SystemResult SccQuery(pd, inst, dirIndex)
    Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    register QuerySerialDeviceReply *reply = (QuerySerialDeviceReply *)&(pd->msg);
    register SccChannelStatus *s;

    s = &SerialSccStatus[inst->unit];
    reply->charWidth = s->charWidth;
    reply->parity = s->parity;
    reply->dtr = s->dtr;
    reply->rts = s->rts;
    reply->brk = s->brk;
    reply->dataRate = s->dataRate;
    reply->stopBits = s->stopBits;

    return (OK);
  }
#endif

#ifdef undef
/*
 * Modify the device-dependent parameters of an Scc channel
 */
SystemResult SccModify(pd, inst, dirIndex)
    Process *pd;
    DeviceInstance *inst;
    unsigned short dirIndex;
  {
    ModifySerialDeviceRequest *req = (ModifySerialDeviceRequest *)&(pd->msg);
    register SccChannelStatus *s;

    s = &SerialSccStatus[inst->unit];
    s->charWidth = req->charWidth;
    s->parity = req->parity;
    s->dtr = req->dtr;
    s->rts = req->rts;
    s->brk = req->brk;
    s->dataRate = req->dataRate;
    s->stopBits = req->stopBits;

    SccInitChannel(s);

    return (OK);
  }
#endif

int scc_close(s)
XObj s;
{
  register SccChannelStatus *scc = (SccChannelStatus *)s->state;
  TRACE1(sccp, 3, "scc close %x", s);
  SccSessions[scc->index] = NULL;
  s->state = NULL;
  x_destroysession(s);
}

/* Sound the bell on a Sun-2 keyboard.  This requires sending out a
 * "bell-on" character, delaying a while, and then sending a "bell-off."
 */
int SccBellKludge()
{
  /* If it's ready, turn on the bell, else wait and try again */
  while (ReadRegister0(KeyboardSccStatus.channel) & SccTxBufferEmpty) {
    Delay(10);
  }
  WriteDataRegister('\2', KeyboardSccStatus.channel);
  event_register(SccBellKludge2, 0, 100, EV_ONCE);
}

static SccBellKludge2( )
{
/* Now turn off the bell.  We punt if it's busy, since that means
 * either someone is turning it off, in which case we are happy,
 * or else they are turning it on, in which case they can worry
 * about turning it off, and we are still happy.
 */
  if (ReadRegister0(KeyboardSccStatus.channel) & SccTxBufferEmpty) {
    WriteDataRegister('\3', KeyboardSccStatus.channel);
  }
}

static noop() {}

scc_getproc(p,type)
XObj p;
XObjType type;
{
  if (type == Protocol) {
    p->instantiateprotl = noop;
    p->init = scc_init;
    p->close = noop;
    p->push = noop;
    p->pop = noop;
    p->control = noop;
  } else {
    p->push = scc_push;
    p->pop = noop;
    p->instantiateprotl = noop;
    p->init = noop;
    p->close = scc_close;
    p->control = noop;
  }
  p->open = (Pfi) scc_open;
  p->openenable = noop;
  p->opendone = noop;
  p->closedone = noop;
  p->opendisable = noop;
  p->demux = scc_demux;
  p->getproc = scc_getproc;
}
