;;;
;;; Postgres/CommonLISP interface
;;;
;;; Copyright (c) 1986 Regents of the University of California
;;; 
;;; Permission to use, copy, modify, and distribute this software and its
;;; documentation for any purpose and without fee is hereby granted,
;;; provided that the above copyright notice appear in all copies and
;;; that both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of the University of
;;; California not be used in advertising or publicity pertaining to
;;; distribution of the software without specific, written prior
;;; permission.  The University of California makes no representations
;;; about the suitability of this software for any purpose.  It is
;;; provided "as is" without express or implied warranty.
;;; 
;;; $Author: picasso $
;;; $Source: RCS/pqexec.cl,v $
;;; $Revision: 1.1 $
;;; $Date: 90/07/23 13:18:03 $
;;;

(in-package 'libpq :use '(lisp excl))


;;; 
;;; The value of the first four variables can be changed by setting
;;; the appropriate environment variables.
;;; *pqhost*: PGHOST
;;; *pqport*: PGPORT
;;; *pqtty*:  PGTTY
;;; *pqoption*: PGOPTION
;;;
;;; Their default values are defined in the file defaults.cl.
;;;

(defvar *pqhost* nil
  "The host on which the POSTGRES backend is running")

(defvar *pqport* nil
  "The communication port with the POSTGRES backend")

(defvar *pqtty* nil
  "The tty on *pqhost* to display backend messages")

(defvar *pqoption* nil
  "Optional arguments passed to the POSTGRES backend")

(defvar *pqportset* nil
  "T if the communication port with the POSTGRES backend is set") 

(defvar *pqxactid* 0
  "The POSTGRES transaction id")

(defvar *pqdatabase* nil
  "The POSTGRES database to access")

(defvar *initstr* nil
  "The initialization string passed to the POSTGRES backend")

(defvar *pqtrace* nil
  "Print out the trace message if the value of this variable is T")

;;;
;;; Set up the POSTGRES database for access
;;;

(defun pqdatabase ()
  "Return the current POSTGRES database being accessed"
  *pqdatabase*)

(defun setf-pqdatabase (dbname)
  "Set up the POSTGRES database to \fBdbname\fP, return \fBdbname\fP"
  (if (null dbname)
      (error "Can't set the database name to NIL"))
  (setq *pqdatabase* dbname))

(defsetf pqdatabase setf-pqdatabase)

;;;
;;; Synonym for setf-pqdatabase
;;;
(setf (symbol-function 'pqsetdb) (symbol-function 'setf-pqdatabase))

;;;
;;; Reset the communication port with the POSTGRES backend
;;;
(defun pqreset ()
  "Reset the communication port with the POSTGRES backend and return T"
  (unless (null *pqportset*)
          (cpclose)
          (setq *pqportset* nil)
          (setq *initstr* nil))
  (close-all-portals)
  T)

;;;
;;; Close communication ports with the POSTGRES backend
;;;
(defun pqfinish ()
  "Close the communication with the POSTGRES backend, and return T"
  (cpclose)
  (setq *pqportset* nil)
  (close-all-portals)
  T)

;;;
;;; Set the trace flag
;;;
(defun pqtrace ()
  "Set the trace flag"
  (setq *pqtrace* t))

;;;
;;; Reset the trace flag
;;;
(defun pquntrace ()
  "Reset the trace flag"
  (setq *pqtrace* nil))

;;;
;;; Trace
;;;
(defmacro pqdebug (msg)
  `(if *pqtrace*
      (format t "~A~%" ,msg)))

;;;
;;; Read in the init-str to be passed to the POSTGRES backend.
;;; The initstr has the format of
;;; USER,DATABASE,TTY,OPTION #\newline
;;; There are two ways to set up the arguments:
;;;   1) use special variables
;;;   2) set up environment variables.
;;; Special variables would overwrite environment variables.
;;;
;;; Since these special variables (except *pqdatabase*) are used for
;;; debugging by the programmers, they are not exported from the LIBPQ
;;; package.
;;;
(defun read-initstr (&optional no-error-p)
  "Read in the *initstr* to be passed to the POSTGRES backend"
  (if (and (null *pqdatabase*) 
	   (null (setq *pqdatabase* (system:getenv "PGDATABASE"))))
      (cond (no-error-p (pqdebug "ERROR: No database specified")
			(return-from read-initstr nil))
	    (t (error "Fatal -- No database is specified"))))
  (if (and (null *pqhost*) (null (setq *pqhost* (system:getenv "PGHOST"))))
      (setq *pqhost* *default-server-hostname*))
  (if (and (null *pqtty*) (null (setq *pqtty* (system:getenv "PGTTY"))))
      (setq *pqtty* *default-tty*))
  (if (and (null *pqoption*) 
	   (null (setq *pqoption* (system:getenv "PGOPTION"))))
      (setq *pqoption* *default-option*))
  (if (and (null *pqport*) (null (setq *pqport* (system:getenv "PGPORT"))))
      (setq *pqport* *default-port*))
  (setq *initstr* 
	(format nil "~A,~A,~A,~A~%"
		(system:getenv "USER")
		*pqdatabase*
		*pqtty*
		*pqoption*)))

;;;
;;; This function takes a query from user application program and passes it
;;; to the backend.  If there are any values returned from backend, it will
;;; either return a stream to the caller or dump the values in the libpq
;;; buffer depending on whether the portal is a 'blank portal'.
;;; Refer to the Communication Protocol Document for more information
;;; about the communication format between LIBPQ and the POSTGRES backend.
;;;
(defun pqexec (query &optional no-error-p)
  "Return T if \fBquery\fP succeeds"
  ;; if the communication is not established
  (when (null *pqportset*) 
	(if (null *initstr*)
	    (if (null (read-initstr no-error-p))
		(return-from pqexec nil)))
	;; if the backend cannot be reached, signals an error
	;; cpconnect will take a port number in the future
	(pqdebug (format nil "~%"))
	(pqdebug (format nil "Backend Connection, host: ~S, port: ~S"
			 *pqhost* *pqport*))
	(if (equal (cpconnect *pqhost* *pqport*) -1)
	    (if no-error-p
		(return-from pqexec nil)
	      (error "Fatal -- No POSTGRES backend to connect to")))
	;; send the initstr to the backend
	(pqdebug (format nil "~%"))
	(pqdebug (format nil "Initstr sent to the backend:  ~A" *initstr*))
	(Cputstr *initstr*) 
	(cpflush) 
	(setq *pqportset* t))
  ;; send the query to backend
  (cputnchar "Q" 1)
  (cputint *pqxactid* 4)
  (cputstr query)
  (pqdebug (format nil "Query Sent: ~A" query))
  (cpflush)   
  (let ((id "?")			;id holds the identifier of each block
	(errormsg (make-string error-msg-length :initial-element '#\a)))  
    ;; get the identifier 
    (cgetpid id 0 1)
    ;; read remarks
    (setq id (read-remark id))
    ;; read the transaction id
    ;;(pqdebug (format nil "Identifier is ~A" id))
    (setq *pqxactid* (cgetpint 4))
    ;;(pqdebug (format nil "Transaction id = ~D" *pqxactid*))
    (cond 
     ;; "E" indicates an error message, return nil and error message
     ((equal id "E") 
      (let ((strlen (cgetpstr errormsg error-msg-length)))
	(if no-error-p 
	    (values nil (list "E" (subseq errormsg 0 strlen)))
	  (error "POSTQUEL ERROR: ~A" (subseq errormsg 0 strlen)))))
     ;; "A" for asynchronized portal, "P" for synchronized portal
     ;; call process-portal to process portal queries
     ((or (equal id "A") (equal id "P"))  
      (process-portal no-error-p))
     ;; "C" indicates a query command, which comes at the end of transmission
     ((equal id "C") 
      (let  ((strlen (cgetpstr errormsg command-length)))
	;;(pqdebug (format nil "Query command: ~A" (subseq errormsg 0 strlen)))
	(values t (list "C" (subseq errormsg 0 strlen)))))
     ;; otherwise, it is an error, reset the communication port
     (t 
      (pqreset)
      (if no-error-p
	  (values nil (list "E" "Weird error from the POSTGRES backend"))
	(error "Cannot understand the information from the POSTGRES backend"))))))


;;;
;;; Process portal queries
;;; 
(defun process-portal (&optional no-error-p)
  "Process portal queries, return T if successful, NIL otherwise"
  (let ((pname (make-string portal-name-length :initial-element '#\p))
	(id "?")
	(command (make-string command-length :initial-element '#\c))
	(strlen 0))
    ;; read the portal name
    (setq strlen (cgetpstr pname portal-name-length))
    (setq pname (subseq pname 0 strlen))
    ;;(pqdebug (format nil "portal name = ~A" pname))
    ;; read the identifier following the portal name
    (Cgetpid id 0 1)
    ;; read remarks
    (setq id (read-remark id))
    ;;(pqdebug (format nil "Identifier is ~A" id))
    (cond 
     ;; "T" following the portal name indicates returned tuples
     ((string= id "T")
      ;; dump data into the buffer, whether it is a blank portal or data portal
      (dump-data pname)
      (values t (list "P" pname)))
     ;; "C" indicates query command
     ((string= id "C")
      (setq *pqxactid* (cgetpint 4))
      ;;(pqdebug (format nil "Transaction id = ~D" *pqxactid*))
      (setq strlen (Cgetpstr command command-length))
      (setq command (subseq command 0 strlen))
      ;;(pqdebug (format nil "Portal command: ~A" command))
      ;; process portal commands
      (cond
       ;; set up a portal
       ((string= command "retrieve") 
	(portal-setup pname) 
	(values t '("C" "retrieve")))
       ;; close a portal
       ((string= command "close") 
	(portal-close pname) 
	(values t '("C" "close")))
       (t (values t '("C" command)))))
     (t (pqreset)
	(if no-error-p
	    (values nil '("E" "Weird error from the POSTGRES backend"))
	  (error "illegal identifier following portal name"))))))

;;;
;;; Read remarks
;;;
(defun read-remark (id)
  "Return the first id that is not \fBR\fP"
  (do ((remarks (make-string remark-length :initial-element '#\r)))
      ;; loop until we get an id that is not "R", then returns id
      ((not (equal id "R")) id)
      ;; read and drop the remarks
      (Cgetpstr remarks remark-length) 
      ;; get next id
      (Cgetpchar id 0 1)))

;;;
;;; Exports from the Postgres/CommonLISP library package
;;;
(export '(pqdatabase
	  pqsetdb
	  pqfinish
	  pqreset
	  pqtrace
	  pquntrace
	  pqexec)
	(find-package 'libpq))

