{
package DBI;

require 5.001;

$VERSION = substr(q$Revision: 1.39 $, 10);

# $Id: DBI.pm,v 1.39 1995/06/22 00:02:19 timbo Rel $
#
# Copyright (c) 1995, Tim Bunce
#
# You may distribute under the terms of either the GNU General Public
# License or the Artistic License, as specified in the Perl README file.

require DynaLoader;
@ISA = qw(DynaLoader);

use strict;
use Carp;

sub import { }	# workaround Perl5.001i AutoLoader problems :-(

$DBI::dbi_debug = $ENV{PERL_DBI_DEBUG} || 0;
warn "IN DBI.pm\n" if $DBI::dbi_debug;

bootstrap DBI;

DBI->_debug_dispatch($DBI::dbi_debug) if $DBI::dbi_debug;

%DBI::installed_drh = ();  # maps driver names to installed driver handles


# Setup special DBI dynamic variables. See DBI::var::FETCH for details.
# These are dynamically associated with the last handle used.
tie $DBI::err,    'DBI::var', '*err';    # special case: referenced via IHA list
tie $DBI::lasth,  'DBI::var', '!lasth';  # special case: return boolean
tie $DBI::errstr, 'DBI::var', '&errstr'; # call &errstr in last used pkg
tie $DBI::rows,   'DBI::var', '&rows';   # call &rows   in last used pkg
sub DBI::var::TIESCALAR{ my($var) = $_[1]; bless \$var, 'DBI::var'; }
sub DBI::var::STORE{ Carp::carp "Can't modify \$DBI::${$_[0]} special variable" }


# --- Dynamically create the DBI Standard Interface

my($std) = undef;

my(@TieHash_IF)=(	# Generic Tied Hash Interface
	'STORE'   => $std, 'FETCH'   => $std,
	'FIRSTKEY'=> $std, 'NEXTKEY' => $std,
	'EXISTS'  => $std, 'CLEAR'   => $std,
	'DESTROY' => $std,
);
my(@Common_IF)=(	# Interface functions common to all DBI classes
	'event' =>	{'U'=>[2,3,'$message, $retvalue']},
	'debug'   =>	{'U'=>[1,2,'[$debug_level]']},
	'private_data'=>{'U'=>[1,1]},
	'errstr'  =>	$std,
	'rows'    =>	$std,
);

my(%DBI_IF)= (	# Define the DBI Interface:

    'dr' => {		# Database Driver Interface
	'connect' =>    {'U'=>[1,5,'[$db [,$user [,$passwd [,\%attr]]]]']},
	'disconnect_all' => {'U'=>[1,1]},
	@Common_IF,
	@TieHash_IF,
    },
    'db' => {		# Database Session Class Interface
	'commit'     => {'U'=>[1,1]},
	'rollback'   =>	{'U'=>[1,1]},
	'prepare'    => {'U'=>[2,3,'$statement [, \%attribs]']},
	'handler'    => {'U'=>[2,2,'\&handler']},
	'errstate'   =>	{'U'=>[1,1]},
	'errmsg'     => {'U'=>[1,1]},
	'disconnect' =>	{'U'=>[1,1]},
	@Common_IF,
	@TieHash_IF,
    },
    'st' => {		# Statement Class Interface
	'execute' =>	{'U'=>[1,0,'[@args]']},
	'fetchrow' =>	undef, # no checks, no args, max speed
	'readblob' =>   {'U'=>[4,5,'$field, $offset, $len [, \\$buf]']},
	'finish' => 	{'U'=>[1,1]},
	@Common_IF,
	@TieHash_IF,
    },
);

my($class, $method);
foreach $class (keys %DBI_IF){
    my(%pkgif) = %{$DBI_IF{$class}};
    foreach $method (keys %pkgif){
	DBI->install_method("DBI::${class}::$method", 'DBI.pm',
			$pkgif{$method});
    }
}

# End of init code


END {
    warn "DBI::END\n" if $DBI::dbi_debug;
    # Let drivers know why we are calling disconnect_all:
    $DBI::ENDING = 1;	# Perl is END'ing
    DBI->disconnect_all();
    warn "DBI::END complete\n" if $DBI::dbi_debug;
}


sub install_method{
    my($dbi, $name, $file, $attr) = @_;
    croak "install_method: invalid name '$name'"
	    unless $name=~m/^DBI::\w+::\w+$/;
    # convert 'loose' $attr hash into fixed Internal Method Attributes
    # some of this code might get moved down into XS
    my(@ima, $a);
    if ($attr && ($a = $attr->{'U'})){
	$ima[0] = $a->[0];	# MINARGS
	$ima[1] = $a->[1];	# MAXARGS
	$ima[2] = $a->[2] || '';	# ARGS
    }
    _add_dispatch($name, $file, ($attr) ? \@ima : undef);
}



# --- The DBI->connect Front Door function

sub connect{
    my($class, $database, $user, $passwd, $driver, $attr) = @_;

    $database = $ENV{'DBI_DBNAME'} unless $database;
    $driver   = $ENV{'DBI_DRIVER'} unless $driver;

    warn "DBI->connect($database, $user, $passwd, $driver, $attr)\n"
	    if $DBI::dbi_debug;
    die 'Usage: DBI->connect([$db [,$user [,$passwd [, $driver [,\%attr]]]]])'
	    unless ($class eq 'DBI' && @_ <= 6);

    confess "DBI->connect() currently needs a driver" unless $driver;

    my($drh) = $DBI::installed_drh{$driver};
    unless (defined($drh)){
	$drh = DBI->install_driver($driver, $attr)
		or confess "install_driver($driver) failed";
    }
    warn "DBI->connect using $driver driver $drh\n" if $DBI::dbi_debug;

    my($dbh) = $drh->connect($database, $user, $passwd);
    warn "DBI->connect = $dbh\n" if $DBI::dbi_debug;

    $dbh;
}


sub disconnect_all{
    warn "DBI::disconnect_all @_\n" if $DBI::dbi_debug;
    foreach(values %DBI::installed_drh){
	warn "DBI::disconnect_all for '$_'\n" if $DBI::dbi_debug;
	next unless ref $_;	# avoid problems on premature death
	$_->disconnect_all();
    }
}


sub install_driver{
    my($class, $driver_name, $install_attributes) = @_;

    Carp::carp "DBI->install_driver @_\n" if $DBI::dbi_debug;
    die 'usage DBI->install_driver($driver_name [, \%attribs])'
	unless ($class eq 'DBI' and $driver_name and @_<=3);

    # --- load the code
    eval "package DBI::_firesafe; require DBD::$driver_name";
    confess "install_driver($driver_name) failed: $@" if $@;
    warn "DBI->install_driver($driver_name) loaded\n" if $DBI::dbi_debug;

    # --- do some behind-the-scenes checks and setups on the driver
    foreach(qw(dr db st)){
	no strict 'refs';
	my($class) = "DBD::${driver_name}::$_";
	push(@{"${class}::ISA"}, "DBD::_::$_"); # always update @ISA
	Carp::carp "install_driver($driver_name): setup \@ISA for $class\n"
	    if ($DBI::dbi_debug>=3);
    }

    # --- run the driver function
    $install_attributes = {} unless $install_attributes;
    my($drh) = eval "DBD::${driver_name}->driver(\$install_attributes)";
    croak "DBD::$driver_name initialisation failed: $@"
	unless $drh && ref $drh && !$@;

    warn "DBI->install_driver($driver_name) = $drh\n" if $DBI::dbi_debug;
    $DBI::installed_drh{$driver_name} = $drh;
    $drh;
}


sub internal{
    &DBD::Switch::dr::driver;	# redirect with args
}


sub available_drivers{
    my(@drivers, $d, $f);
    local(*DBI::DIR);
    my(%seen_dir, %seen_dbd);
    foreach $d (@INC){
	chomp($d); # perl 5 beta 3 bug in #!./perl -Ilib from Test::Harness
	next unless -d "$d/DBD";
	next if $seen_dir{$d};
	$seen_dir{$d} = 1;
	opendir(DBI::DIR,"$d/DBD") || Carp::carp "opendir $d/DBD: $!\n";
	foreach $f (sort readdir(DBI::DIR)){
	    next unless $f =~ s/\.pm$//;
	    if ($seen_dbd{$f}){
		Carp::carp "DBD::$f in $d is hidden by DBD::$f in $seen_dbd{$f}\n";
            } else {
		push(@drivers, $f);
	    }
	    $seen_dbd{$f} = $d;
	}
	closedir(DBI::DIR);
    }
    @drivers;
}


sub default_handler{	# will migrate to xs later
    my($h, $type, @args) = @_;
    Carp::carp "$type EVENT: $h->DBI::default_handler:\n\t@args\n";
#   DBI::set_err($h, $args[0]||1);	# HACK, NOW CLASHES WITH Oracle/Oraperl
    undef;
}


sub MakeMakerAttribs{
    # return extra attributes for DBD Makefile.PL WriteMakefile()
    ();
}


# --- Private Internal Function for Creating New DBI Handles

sub _new_handle{
    my($class, $parent, $attr, $imp_data) = @_;
    $parent = '' unless $parent;

    confess 'Usage: DBI::_new_handle'
	.'($class_name, parent_handle, \%attribs, $imp_data)'."\n"
	.'got: ('.join(", ",$class, $parent, $attr, $imp_data).")\n"
	unless (@_ == 4
		and (!$parent or ref $parent)
		and ref $attr eq 'HASH'
		);

    my($imp_class) = $attr->{'ImplementorClass'} or
	croak "_new_handle($class): 'ImplementorClass' attribute not given";

    printf(STDERR "    New $class (for $imp_class, parent=$parent, id=%s)\n",
	    ($imp_data||''))
	if ($DBI::dbi_debug >= 2);

    Carp::carp "_new_handle($class): "
		."invalid implementor class '$imp_class' given\n"
	    unless $imp_class =~ m/::(dr|db|st)$/;

    # This is how we create a DBI style Object:
    my(%hash, $i, $h);
    $i = tie    %hash, $class, $attr;  # ref to inner hash (for driver)
    $h = bless \%hash, $class;         # ref to outer hash (for application)
    # The above tie and bless may migrate down into _setup_handle()...
    # Now add magic so DBI method dispatch works
    DBI::_setup_handle($h, $imp_class, $parent, $imp_data||0);

    warn "    New $class => $h (inner=$i) for $imp_class\n"
	if ($DBI::dbi_debug >= 2);
    return $h unless wantarray;
    ($h, $i);
}
{   # implement minimum constructors for the tie's (could be moved to xs)
    package DBI::dr; sub TIEHASH { bless $_[1] };
    package DBI::db; sub TIEHASH { bless $_[1] };
    package DBI::st; sub TIEHASH { bless $_[1] };
}


# These three constructors are called by the drivers

sub _new_drh{	# called by DBD::<drivername>::driver()
    my($class, $initial_attr, $imp_data) = @_;
    my($h_err_store, $h_errstr_store) = (0, '');
    my($attr) = {
	'ImplementorClass' => $class,
	# these attributes get copied down to child handles
	'Handlers'	=> [],
	'Err'		=> \$h_err_store,    # Holder for DBI::err
	'Errstr'	=> \$h_errstr_store, # Holder for DBI::errstr
	'Debug' 	=> 0,
	%$initial_attr,
	'Type'=>'dr',
    };
    _new_handle('DBI::dr', undef, $attr, $imp_data);
}

sub _new_dbh{	# called by DBD::<drivername>::dr::connect()
    my($drh, $initial_attr, $imp_data) = @_;
    my($imp_class) = $drh->{'ImplementorClass'};
    $imp_class =~ s/::dr$/::db/;
    confess "new db($drh, $imp_class): not given an driver handle"
	    unless $drh->{'Type'} eq 'dr';
    my($attr) = {
	'ImplementorClass' => $imp_class,
	%$initial_attr,
	'Type'   => 'db',
	'Driver' => $drh,
    };
    _new_handle('DBI::db', $drh, $attr, $imp_data);
}

sub _new_sth{	# called by DBD::<drivername>::db::prepare()
    my($dbh, $initial_attr, $imp_data) = @_;
    my($imp_class) = $dbh->{'ImplementorClass'};
    $imp_class =~ s/::db$/::st/;
    confess "new st($dbh, $imp_class): not given a database handle"
	unless (ref $dbh eq 'DBI::db' and $dbh->{'Type'} eq 'db');
    my($attr) = {
	'ImplementorClass' => $imp_class,
	%$initial_attr,
	'Type'     => 'st',
	'Database' => $dbh,
    };
    _new_handle('DBI::st', $dbh, $attr, $imp_data);
}

} # end of DBI package scope



# --------------------------------------------------------------------
# === The internal DBI Switch pseudo 'driver' class ===

{   package DBD::Switch::dr;
    @ISA = qw(DBD::_::dr);	# inherit from generic driver
    use Carp;

    $err = 0;

    sub driver{
	return $drh if $drh;	# a package global

	my($inner);
	($drh, $inner) = DBI::_new_drh('DBD::Switch::dr', {
		'Name'    => 'Switch',
		'Version' => $DBI::VERSION,
		# the Attribution is defined as a sub as an example
		'Attribution' => sub { "DBI Switch by Tim Bunce" },
	    }, \$err);
	confess("DBD::Switch init failed!") unless ($drh && $inner);
	$DBD::Switch::dr::drh;
    }

    sub FETCH{
	my($drh, $key) = @_;
	return DBI->_debug_dispatch if $key eq 'DebugDispatch';
	Carp::carp("Unknown $drh attribute '$key'");
	undef;
    }
    sub STORE{
	my($drh, $key, $value) = @_;
	if ($key eq 'DebugDispatch') {
	    DBI->_debug_dispatch($value);
	} elsif ($key eq 'DebugLog') {
	    DBI->_debug_dispatch(-1, $value);
	} else {
	    Carp::carp("Unknown $drh attribute '$key'");
	    undef;
	}
    }
}



# --------------------------------------------------------------------
# === OPTIONAL MINIMAL BASE CLASSES FOR DBI SUBCLASSES ===

# We only define default methods for harmless functions.
# We don't, for example, define a DBD::_::st::prepare()

{   package DBD::_::common; # ====== Common base class methods ======
    use strict;

    # methods common to all handle types:

    *debug = \&DBI::_debug_handle;
    sub errstr	{ $DBI::err }

    # generic TIEHASH default methods:
    sub FETCH    { undef }  # dispatch may have dealt with this
    sub STORE    { Carp::carp "Can't set any attributes of $_[0] (DBI)" }
    sub FIRSTKEY { undef }
    sub NEXTKEY  { undef }
    sub EXISTS   { defined($_[0]->FETCH($_[1])) } # to be sure
    sub CLEAR    { Carp::carp "Can't CLEAR $_[0] (DBI)" }
}


{   package DBD::_::dr;  # ====== DRIVER ======
    @ISA = qw(DBD::_::common);
    use strict;
    use Carp;

    sub connect { # normally overridden, but a handy default
	my($drh, $dbname, $user, $auth)= @_;
	my($this) = DBI::_new_dbh($drh, {
	    'Name' => $dbname,
	    'User' => $user,
	    });
	$this;
    }
    sub disconnect_all {	# Driver must take responsibility for this
	Carp::confess "Driver has not implemented disconnect_all for @_";
    }
    sub _untied		{ undef }
}


{   package DBD::_::db;  # ====== DATABASE ======
    @ISA = qw(DBD::_::common);
    use strict;

    sub quote	{ my $str=shift; $str=~s/'/''/g; "'$str'"; } # ISO SQL2

    sub rows	{ -1 }

    sub commit	{
	Carp::carp "commit: not supported by $_[0]\n" if $DBI::dbi_debug;
	undef;
    }
    sub rollback{
	Carp::carp "rollback: not supported by $_[0]\n" if $DBI::dbi_debug;
	undef;
    }
    sub disconnect  { undef }
    sub _untied     { undef }
}


{   package DBD::_::st;  # ====== STATEMENT ======
    @ISA = qw(DBD::_::common);
    use strict;

    sub execute { 1 }
    sub finish  { undef }
    sub _untied { undef }
    sub rows	{ -1 }

    # Drivers are required to implement *::st::DESTROY to encourage tidy-up
    sub DESTROY  { Carp::confess "Driver has not implemented DESTROY for @_" }
}

1;
__END__

DBI for Perl 5  -  Function Summary  (Sep 29 1994)
---------------------------------------------------------------

NOTATION

Object Handles:

  DBI static 'top-level' class name
  $drh   Driver Handle (rarely seen or used)
  $dbh   Database Handle
  $sth   Statement Handle

note that Perl 5 will automatically destroy database and statement
objects if all references to them are deleted.

Object attributes are shown as:

  $handle->{'attribute_name'}  (type)

where (type) indicates the type of the value of the attribute,
if it's not a simple scalar:

  \@   reference to a list:  $h->{a}->[0]  or  @a = @{$h->{a}}
  \%   reference to a hash:  $h->{a}->{a}  or  %a = %{$h->{a}}


---------------------------------------------------------------
DBI OBJECTS

$drh = DBI->internal; # return $drh for internal Switch 'driver'
$drh = DBI->install_driver($driver_name [, \%attributes ] );
$rv  = DBI->install_method($class_method, $filename [, \%attribs]);

$dbh = DBI->connect([$database [, $username [, $auth [, \%attribs]]]]);
$rc  = DBI->disconnect_all;  # disconnect all database sessions

$DBI::db_error   same as DBI->internal->{LastDbh}->{Error}
$DBI::db_errstr  same as DBI->internal->{LastDbh}->{ErrorStr}
$DBI::db_rows    same as DBI->internal->{LastSth}->{ROW_COUNT}


---------------------------------------------------------------
DRIVER OBJECTS (not normally used by an application)

$dbh = $drh->connect([$database [, $username [, $auth [ ...]]]])

$drh->{Type}       "dr"
$drh->{Name}       (name of driver, e.g., Oracle)
$drh->{Version}
$drh->{Attribution}

Additional Attributes for internal DBI Switch 'driver'

$drh->{DebugDispatch}
$drh->{InstalledDrivers} (@)
$drh->{LastAdh}
$drh->{LastDbh}
$drh->{LastSth}


---------------------------------------------------------------
DATABASE OBJECTS

$rc  = $dbh->disconnect;			undef or 1
$rc  = $dbh->commit;				undef or 1
$rc  = $dbh->rollback;				undef or 1
$sth = $dbh->prepare($statement [, \%attr]);

$rv  = $dbh->handler($handler_function);
$rv  = $dbh->errstate;
@ary = $dbh->errmsg;

$sql = $dbh->quote($str);
$udt = $dbh->ndt2udt($ndt [, $local]);
$ndt = $dbh->udt2ndt($udt [, $local]);

$dbh->{Type}       "db"
$dbh->{Name}       (name of database the handle is connected to)
$dbh->{Driver}     (\%)

$dbh->{Error}      normally use $db_error
$dbh->{ErrorStr}   normally use $db_errstr
$dbh->{ROW_COUNT}  normally use $db_rows


---------------------------------------------------------------
STATEMENT OBJECTS

$rc  = $sth->execute(@bind_values);     	undef or 1
@ary = $sth->fetchrow;
$rc  = $sth->finish;				undef or 1

$sth->{Type}       "st"
$sth->{Name}
$sth->{Database}   (\%)  # eg $sth->{Database}->{Driver}->{Name} !

$sth->{NAME}       (\@)
$sth->{NULLABLE}   (\@)
$sth->{TYPE}       (\@)
$sth->{PRECISION}  (\@)
$sth->{SCALE}      (\@)

$sth->{NumParams}  (\@)

---------------------------------------------------------------
