/*
 * dirent.c
 *
 * emulation of
 *
 *   - closedir()
 *   - opendir()
 *   - readdir()
 *
 * (c) 1996 by Dirk Ohme
 */

/*---| includes |------------------------------------------------------------*/
#define INCL_DOSFILEMGR
#ifdef OS2
#include <os2.h>
#endif
#ifdef WINNT
#include <windows.h>
#define DosFindFirst FindFirstFile
#define DosFindNext  FindNextFile
#define FILEFINDBUF3 WIN32_FIND_DATA
#define DosFindClose FindClose
#define HDIR HANDLE
#endif
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include "errno.h"
#include "dirent.h"

/*---| structures |----------------------------------------------------------*/
typedef struct DIR_s {                          /* directory search struct   */
        int              ciNumber;              /*   - number of record      */
        char            *pszPath;               /*   - dir. search string    */
        HDIR             hDirSearch;            /*   - search handle         */
#ifdef OS2
        FILEFINDBUF3     sBuffer;               /*   - search buffer         */
#endif
#ifdef WINNT
        WIN32_FIND_DATA  sBuffer;               /*   - search buffer         */
#endif
        struct dirent    sDirEnt;               /*   - current entry         */
} DIR_t;                                        /*---------------------------*/

/*---| local prototypes |----------------------------------------------------*/

/*---------------------------------------------------------------------------
 * name:        closedir()
 * input:       pDIR            pointer to directory search structure
 * output:      int     == 0    success
 *                      != 0    on error, error code set in errno
 *                      EBADF           bad pointer
 * function:    ends access to an open directory structure
 *---------------------------------------------------------------------------
 */
int closedir( DIR *pDir )
{
        /*
        ** check for correct pointer
        */
        if( NULL == pDir )
        {
                errno = EBADF;
                return -1;
        }

        /*
        ** free memory
        */
#ifdef OS2
        if( HDIR_CREATE != ((DIR_t *) pDir)->hDirSearch )
                DosFindClose( ((DIR_t *) pDir)->hDirSearch );
#endif        
#ifdef WINNT
        if( INVALID_HANDLE_VALUE != ((DIR_t *) pDir)->hDirSearch )
                FindClose( ((DIR_t *) pDir)->hDirSearch );
#endif        
        if( NULL != ((DIR_t *) pDir)->pszPath )
                free( ((DIR_t *) pDir)->pszPath );
		free( pDir );

        /*
        ** return success
        */
        return (errno = 0);

} /* closedir() */

/*---------------------------------------------------------------------------
 * name:        opendir()
 * input:       pszPath         path and directory to be accessed
 * output:      DIR *   != NULL success, pointer to dir access structure
 *                      == NULL error, error code set in errno
 *                      EACCES          no permission to read directory
 *                      EBADF           bad pointer
 *                      EMFILE          too many file descriptors open
 *                      ENAMETOOLONG    path in 'pszPath' too long
 *                      ENFILE          too many file descriptors open
 *                      ENOENT          directory does not exist
 *                      ENOTDIR         invalid path specifier
 * function:    starts access to a directory
 *---------------------------------------------------------------------------
 */
DIR *opendir( char *pszPath )
{
        DIR_t           *pDir;
        int              cb;

        /*
        ** check pointers
        */
        if( NULL == pszPath || '\0' == *pszPath )
        {
                errno = EBADF;
                return NULL;
        }
        if( MAXNAMLEN < (cb=strlen(pszPath)) )
        {
                errno = ENAMETOOLONG;
                return NULL;
        }

        /*
        ** allocate memory for structure
        */
        if( NULL == (pDir=malloc(sizeof(DIR_t))) ||
            NULL == (pDir->pszPath=malloc(cb + 3)) )
        {
                if( NULL != pDir )
                        free( pDir );
                errno = EACCES;
                return NULL;
        }

        /*
        ** translate path:
        **   - convert '/' to '\'
        **   - add terminating '\' to end of path
        **   - add '*' for search mask (DosFind...)
        */
        cb = 0;
        while( '\0' != *pszPath )
        {
                switch( *pszPath )
                {
                        case '/':
                                pDir->pszPath[cb++] = '\\';
                                break;
                        default:
                                pDir->pszPath[cb++] = *pszPath;
                } /* switch */

                pszPath++;

        } /* while */
        if( '\\' != pDir->pszPath[cb - 1] )
                pDir->pszPath[cb++] = '\\';
        pDir->pszPath[cb++] = '*';
        pDir->pszPath[cb]   = '\0';

        /*
        ** initiate search
        */
        cb               = 1;
        pDir->ciNumber   = 0;
        
		memset( &(pDir->sDirEnt), 0, sizeof(struct dirent) );
#ifdef OS2
        pDir->hDirSearch = HDIR_CREATE;
		memset( &(pDir->sBuffer), 0, sizeof(FILEFINDBUF3) );
        if( 0 != DosFindFirst( pDir->pszPath, &(pDir->hDirSearch),
                               FILE_NORMAL | FILE_READONLY  | FILE_ARCHIVED |
                               FILE_HIDDEN | FILE_DIRECTORY | FILE_SYSTEM,
                               (PVOID) &(pDir->sBuffer), sizeof(FILEFINDBUF3),
                               (ULONG *) &cb, FIL_STANDARD ) )
#endif
#ifdef WINNT
		memset( &(pDir->sBuffer), 0, sizeof(WIN32_FIND_DATA) );
        pDir->hDirSearch = FindFirstFile( pDir->pszPath,
			(LPWIN32_FIND_DATA) &(pDir->sBuffer));
		if (pDir->hDirSearch == INVALID_HANDLE_VALUE)
#endif
        {
                free( pDir->pszPath );
                free( pDir );
                errno = ENOTDIR;
                return NULL;
        }

        /*
        ** return success
        */
        return (void *) pDir;

} /* opendir() */

/*---------------------------------------------------------------------------
 * name:        readdir()
 * input:       pDIR            pointer to directory search structure
 * output:      struct dirent   pointer to current entry
 *                      EACCES          no permission to read directory
 *                      EBADF           bad pointer
 *                      EMFILE          too many file descriptors open
 *                      ENAMETOOLONG    path in 'pszPath' too long
 *                      ENFILE          too many file descriptors open
 *                      ENOENT          directory does not exist
 *                      ENOTDIR         invalid path specifier
 * function:    returns the next entry of the directory
 *---------------------------------------------------------------------------
 */
struct dirent *readdir( DIR *pDir )
{
        /*
        ** check for correct pointer
        */
        if( NULL == pDir )
        {
                errno = EBADF;
                return NULL;
        }

        /*
        ** check for search status
        */
#ifdef OS2
        if( HDIR_CREATE == ((DIR_t *) pDir)->hDirSearch )
#endif
#ifdef WINNT
        if( INVALID_HANDLE_VALUE == ((DIR_t *) pDir)->hDirSearch )
#endif
        {
                errno = ENOENT;
                return NULL;
        }

        /*
        ** check for '.' and '..'
        */
        ((DIR_t *) pDir)->ciNumber++;
        if( 1 == ((DIR_t *) pDir)->ciNumber &&
#ifdef OS2
            0 != strcmp(".", ((DIR_t *) pDir)->sBuffer.achName) )
#endif
#ifdef WINNT
            0 != strcmp(".", ((DIR_t *) pDir)->sBuffer.cFileName) )
#endif
        {
                /*---| simulate an '.' entry |---*/
                ((DIR_t *) pDir)->sDirEnt.d_off     = 0;
                ((DIR_t *) pDir)->sDirEnt.d_ino     = 0;
                ((DIR_t *) pDir)->sDirEnt.d_fileno  =
                                                   ((DIR_t *) pDir)->ciNumber;
                ((DIR_t *) pDir)->sDirEnt.d_reclen  = sizeof( struct dirent );
                ((DIR_t *) pDir)->sDirEnt.d_namlen  = 1;
                ((DIR_t *) pDir)->sDirEnt.d_name[0] = '.';
                ((DIR_t *) pDir)->sDirEnt.d_name[1] = '\0';
        }
        else if( 2 == ((DIR_t *) pDir)->ciNumber &&
#ifdef OS2
                 0 != strcmp("..", ((DIR_t *) pDir)->sBuffer.achName) )
#endif
#ifdef WINNT
				 0 != strcmp("..", ((DIR_t *) pDir)->sBuffer.cFileName) )
#endif
        {
                /*---| simulate an '..' entry |---*/
                ((DIR_t *) pDir)->sDirEnt.d_off     = 0;
                ((DIR_t *) pDir)->sDirEnt.d_ino     = 0;
                ((DIR_t *) pDir)->sDirEnt.d_fileno  =
                                                   ((DIR_t *) pDir)->ciNumber;
                ((DIR_t *) pDir)->sDirEnt.d_reclen  = sizeof( struct dirent );
                ((DIR_t *) pDir)->sDirEnt.d_namlen  = 2;
                ((DIR_t *) pDir)->sDirEnt.d_name[0] = '.';
                ((DIR_t *) pDir)->sDirEnt.d_name[1] = '.';
                ((DIR_t *) pDir)->sDirEnt.d_name[2] = '\0';
        }
        else
        {
                int             cb;

                /*---| copy entry |---*/
                ((DIR_t *) pDir)->sDirEnt.d_off     = 0;
                ((DIR_t *) pDir)->sDirEnt.d_ino     = 0;
                ((DIR_t *) pDir)->sDirEnt.d_fileno  =
                                                   ((DIR_t *) pDir)->ciNumber;
                ((DIR_t *) pDir)->sDirEnt.d_reclen  = sizeof( struct dirent );
#ifdef OS2
                ((DIR_t *) pDir)->sDirEnt.d_namlen  =
                                            ((DIR_t *) pDir)->sBuffer.cchName;
                strcpy( ((DIR_t *) pDir)->sDirEnt.d_name,
                        ((DIR_t *) pDir)->sBuffer.achName );
#endif
#ifdef WINNT
                ((DIR_t *) pDir)->sDirEnt.d_namlen  =
                                     strlen(((DIR_t *) pDir)->sBuffer.cFileName);
                strcpy( ((DIR_t *) pDir)->sDirEnt.d_name,
                        ((DIR_t *) pDir)->sBuffer.cFileName );
#endif

                /*---| find next entry |---*/
                cb = 1;
#ifdef OS2
                if( 0 != DosFindNext( ((DIR_t *) pDir)->hDirSearch,
                                      (PVOID) &(((DIR_t *) pDir)->sBuffer),
                                      sizeof(FILEFINDBUF3), (ULONG *) &cb ) )
                {
                        DosFindClose( ((DIR_t *) pDir)->hDirSearch );
                        ((DIR_t *) pDir)->hDirSearch = HDIR_CREATE;
                }
#endif
#ifdef WINNT
                if(!FindNextFile( ((DIR_t *) pDir)->hDirSearch,
								  (LPWIN32_FIND_DATA) &(((DIR_t *) pDir)->sBuffer)))
                {
                        FindClose( ((DIR_t *)pDir)->hDirSearch);
                        ((DIR_t *) pDir)->hDirSearch = INVALID_HANDLE_VALUE;
                }
#endif

        }

        /*
        ** return success
        */
        errno = 0;
        return &(((DIR_t *) pDir)->sDirEnt);

} /* readdir() */

/*===| end of file |=========================================================*/
