(*************************************************************)
(* Elaboration of top-level declarations, modules included,  *)
(*   Section 5 of the Definition, v3                         *)
(*************************************************************)

(*
$File: Common/ElabTopdec.sml $
$Date: 1993/03/05 14:38:23 $
$Revision: 1.41 $
$Locker: birkedal $
*)

(*$ElabTopdec:
	TOPDEC_GRAMMAR ELABDEC ENVIRONMENTS_PROP 
	MODULE_STATOBJECT MODULE_ENVIRONMENTS
	MODULE_UNIFY STRID SIGID ERROR_INFO GRAMMAR_INFO BASIC_IO CRASH
	ELABTOPDEC 
 *)

functor ElabTopdec(structure IG: TOPDEC_GRAMMAR

		   structure OG: TOPDEC_GRAMMAR
		     sharing type OG.funid = IG.funid
			 and type OG.strid = IG.strid
			 and type OG.sigid = IG.sigid
			 and type OG.longstrid = IG.longstrid
			 and type OG.longtycon = IG.longtycon
			 and type OG.tyvar = IG.tyvar
			 and type OG.tycon = IG.tycon
			 and type OG.excon = IG.excon
			 and type OG.con = IG.con

		   structure ElabDec: ELABDEC
		     sharing type ElabDec.PreElabDec = IG.dec
			 and type ElabDec.PostElabDec = OG.dec
			 and type ElabDec.PreElabTy = IG.ty
			 and type ElabDec.PostElabTy = OG.ty

		   structure ModuleStatObject: MODULE_STATOBJECT
		     sharing type ModuleStatObject.SyntaxTyVar = IG.tyvar
		         and type ModuleStatObject.ty = IG.ty
			 and type ModuleStatObject.longstrid = IG.longstrid
			 and type ModuleStatObject.Env = ElabDec.Env
			 and type ModuleStatObject.Type = ElabDec.Type
			 and type ModuleStatObject.tycon = IG.tycon
			 and type ModuleStatObject.longtycon = IG.longtycon
			 and type ModuleStatObject.id = IG.id
					(* Identifiers in the topdec grammar
					   are always converted into vars *)
		         and type ModuleStatObject.var = OG.id

		   structure ModuleEnvironments: MODULE_ENVIRONMENTS
		     sharing type ModuleEnvironments.Realisation
				  = ModuleStatObject.Realisation
			 and type ModuleEnvironments.TypeScheme
				  = ModuleStatObject.TypeScheme
			 and type ModuleEnvironments.longstrid = IG.longstrid
			 and type ModuleEnvironments.longtycon = IG.longtycon
			 and type ModuleEnvironments.TypeFcn
				  = ModuleStatObject.TypeFcn
			 and type ModuleEnvironments.TypeScheme
				  = ModuleStatObject.TypeScheme
			 and type ModuleEnvironments.NameSet
				  = ModuleStatObject.NameSet
			 and type ModuleEnvironments.FunSig
				  = ModuleStatObject.FunSig
			 and type ModuleEnvironments.TyStr
				  = ModuleStatObject.TyStr
			 and type ModuleEnvironments.Context = ElabDec.Context
			 and type ModuleEnvironments.Sig = ModuleStatObject.Sig
			 and type ModuleEnvironments.Str = ModuleStatObject.Str
			 and type ModuleEnvironments.Type
				  = ModuleStatObject.Type
			 and type ModuleEnvironments.Env = ElabDec.Env
			 and type ModuleEnvironments.strid = IG.strid
			 and type ModuleEnvironments.sigid = IG.sigid
			 and type ModuleEnvironments.funid = IG.funid
			 and type ModuleEnvironments.tycon = IG.tycon
			 and type ModuleEnvironments.excon = IG.excon
			 and type ModuleEnvironments.con = IG.con
			 and type ModuleEnvironments.id = IG.id

		   structure U: MODULE_UNIFY
		     sharing type U.StrName = ModuleStatObject.StrName
			 and type U.NameSet = ModuleStatObject.NameSet
			 and type U.Assembly = ModuleEnvironments.Assembly 
			 and type U.Realisation = ModuleStatObject.Realisation
			 and type U.TyStr = ModuleEnvironments.TyStr 
			 and type U.longstrid = IG.longstrid
			 and type U.longtycon = IG.longtycon

		   structure StrId: STRID
		     sharing type StrId.strid = IG.strid
			 and type StrId.longstrid = IG.longstrid

		   structure SigId: SIGID
		     sharing type SigId.sigid = IG.sigid
		       
		   structure ErrorInfo: ERROR_INFO
		     sharing type ModuleStatObject.ErrorInfo
				  = U.ErrorInfo = ErrorInfo.info
		         and type ErrorInfo.Tail = ModuleEnvironments.Tail
			 and type ErrorInfo.id = IG.id
			 and type ErrorInfo.tycon = IG.tycon
			 and type ErrorInfo.con = IG.con
                         and type ErrorInfo.excon = IG.excon
		         and type ErrorInfo.TyVar = ModuleEnvironments.TyVar
			 and type ErrorInfo.TyVar = ModuleStatObject.TyVar
		         and type ErrorInfo.strid = StrId.strid
			 and type ErrorInfo.sigid = SigId.sigid
			 and type ErrorInfo.funid = IG.funid
		         and type ErrorInfo.longtycon = IG.longtycon
			 and type ErrorInfo.longstrid = StrId.longstrid

		   structure GrammarInfo: GRAMMAR_INFO
		     sharing type IG.info = GrammarInfo.PreElabGrammarInfo
			 and type OG.info = GrammarInfo.PostElabGrammarInfo
			 and type GrammarInfo.ErrorInfo = ErrorInfo.info

		   structure BasicIO: BASIC_IO
		   structure Crash: CRASH
                  ): ELABTOPDEC  =
  struct
    structure M = ModuleStatObject	(* Shorthand. *)

    infixr onS           val op onS = ModuleStatObject.onS
    infixr onE           val op onE = ModuleStatObject.onE
    infixr onA           val op onA = ModuleEnvironments.onA
    infixr onB           val op onB = ModuleEnvironments.onB
    infixr oo            val op oo  = ModuleStatObject.oo
    infixr B_cplus_E     val op B_cplus_E  = ModuleEnvironments.B_cplus_E
    infixr B_plus_E      val op B_plus_E   = ModuleEnvironments.B_plus_E
    infixr B_plus_N      val op B_plus_N   = ModuleEnvironments.B_plus_N
    infixr G_plus_G      val op G_plus_G   = ModuleEnvironments.G_plus_G
    infixr B_plus_G      val op B_plus_G   = ModuleEnvironments.B_plus_G
    infixr E_plus_E      val op E_plus_E   = ModuleEnvironments.E_plus_E
    infixr F_plus_F      val op F_plus_F   = ModuleEnvironments.F_plus_F
    infixr B_plus_F      val op B_plus_F   = ModuleEnvironments.B_plus_F
    infixr SE_plus_SE    val op SE_plus_SE = ModuleEnvironments.SE_plus_SE
    infix union          val op union      = ModuleEnvironments.union

    type PreElabTopdec  = IG.topdec
     and PostElabTopdec = OG.topdec

    type StaticBasis = ModuleEnvironments.Basis

   (* Error handling stuff. *)

    type PreElabGrammarInfo  = GrammarInfo.PreElabGrammarInfo
     and PostElabGrammarInfo = GrammarInfo.PostElabGrammarInfo

    val okConv = GrammarInfo.convertGrammarInfo

    fun errorConv(i: PreElabGrammarInfo, e: ErrorInfo.info)
          : PostElabGrammarInfo =
      GrammarInfo.addPostElabErrorInfo (okConv i) e

    fun repeatedIdsError(i: PreElabGrammarInfo,
			 rids: ErrorInfo.RepeatedId list): PostElabGrammarInfo =
      errorConv(i, ErrorInfo.REPEATED_IDS rids)


    (* Functions used to check syntactic restrictions *)
    fun getRepeatedElements ls =
      let
	fun NoOfOccurences x [] = 0
	  | NoOfOccurences x (y::ys) = 
	    if x = y then 1 + NoOfOccurences x ys
	    else NoOfOccurences x ys
      in
	List.all (fn e => (NoOfOccurences e ls) > 1) ls
      end


   (* We occasionally need to add error info to existing "out" phrases
      (currently, this only occurs for SigExp). *)

    fun addSigExpEI(ei, sigexp) =
      case sigexp
	of OG.SIGsigexp(i, spec) =>
	     OG.SIGsigexp(GrammarInfo.addPostElabErrorInfo i ei, spec)

	 | OG.SIGIDsigexp(i, sigid) =>
	     OG.SIGIDsigexp(GrammarInfo.addPostElabErrorInfo i ei, sigid)

    (********
    Shorten structure names
    ********)

    structure Env  = ModuleEnvironments
    structure Stat = ModuleStatObject

    (********
    Generate a dummy structure
    ********)

    fun trivStr() = Stat.mkStr(Stat.freshStrName(), Env.emptyE)

    local
      (********
      Handle signature matching errors.
      ********)

      fun dealWith(i, x, bogus) =
	case x
	  of M.OK Sig => (okConv i, Sig)
	   | M.ERROR ei => (errorConv(i, ei), bogus)
    in
      (********
      Functor signature matching.
      *********
      We bind the argument names in error_result so that the argument signature
      returned in the event of an error is as general as possible.
      ********)

      fun local_funsigMatchStr(i, funsig', S0) =
	let
	  val (N, S, N'S') = Stat.unFunSig funsig'
	  val (N', S') = Stat.unSig N'S'
	  val bogus = Stat.closeStr(Env.NameSetUnion(N, N'), S')
	in
	  dealWith(i, Stat.funsigMatchStr(funsig', S0), bogus)
	end

      (********
      Signature matching.
      *********
      We rename flexible names in error_result so that the result structure
      returned in the event of an error is as general as possible.
      ********)

      fun sigMatchStr(i, Sigma, S) =
	let
	  val bogus = Stat.instanceSig Sigma
	in
	  dealWith(i, Stat.sigMatchStr(Sigma, S), bogus)
	end
    end

    local
      (********
      Handle unification errors.
      ********)

      fun dealWith(i, x) =
	case x
	  of U.OK rea => (okConv i, rea)
	   | U.ERROR ei => (errorConv(i, ei), Stat.bogus_Realisation)
    in
      (********
      Unification of structures.
      ********)

      fun unifyStr(i, N_of_B, A, pairs) =
	dealWith(i, U.unifyStr(N_of_B, A, pairs))
      (********
      Unification of types.
      ********)

      fun unifyTy(i, N_of_B, A, pairs) =
	dealWith(i, U.unifyTy(N_of_B, A, pairs))
    end

    (********
    Generate a dummy functor signature
    ********)

    val trivFunSig =
      let
	val m = Stat.freshStrName()
	val N = Stat.mkNameSet_Str(Stat.singleM m)
	val S = Stat.mkStr(m, Env.emptyE)
      in
	Stat.mkFunSig(N, S, Stat.trivSig)
      end

    (*********
    Find the TE to be used initially in elaborating a datbind
    **********
    We determine the correct equality attributes when we maximise equality
    *********)

    local
      fun make_TE tyvar_list tycon =
      let
	val arity =
	  List.size tyvar_list

	val tyname =
	  Stat.freshTyName {name = tycon, arity = arity, equality = false}

	val typefcn =
	  Stat.TyName_in_TypeFcn tyname

	val tystr =
	  Env.mkTyStr(typefcn, Env.emptyCE)
      in
	Env.singleTE(tycon, tystr)
      end
    in
      fun initial_TE (IG.DATDESC(_, tyvar_list, tycon, _, None)) =
	  make_TE tyvar_list tycon
	| initial_TE (IG.DATDESC(_, tyvar_list, tycon, _, Some datdesc)) =
	  Env.TE_plus_TE(make_TE tyvar_list tycon, initial_TE datdesc)
    end

    (*****************************************************)
    (* Structure Expressions - Definition v3 pages 36-37 *)
    (*****************************************************)

    fun elab_strexp (B: Env.Basis, strexp: IG.strexp): 
      (Stat.Str * OG.strexp) =

      case strexp of

	(* Generative *)
	IG.STRUCTstrexp(i, strdec) =>
	  let
	    val (E, out_strdec) = elab_strdec(B, strdec)
	    val m = Stat.freshStrName()
	  in
	    (Stat.mkStr(m,E), OG.STRUCTstrexp(okConv i, out_strdec))
	  end

	(* Structure identifier *)
      | IG.LONGSTRIDstrexp(i, lstrid) =>
	  (case Env.lookup_lstrid(B, lstrid)
	     of Some str =>
	          (str, OG.LONGSTRIDstrexp(okConv i, lstrid))

	      | None =>
		  let
		    val ei = ErrorInfo.LOOKUP_LONGSTRID lstrid
		  in
		    (Env.bogus_Str,
		     OG.LONGSTRIDstrexp(errorConv(i, ei), lstrid)
		    )
		  end
	  )

	(* Functor application *)
      | IG.APPstrexp(i, funid, strexp) =>  
	  let
	    val (S, out_strexp) = elab_strexp(B, strexp)
	  in
	    case Env.lookup_funid(B, funid)
	      of Some funsig' =>
		   let
		     val (i', N'S') = local_funsigMatchStr(i, funsig', S)
		     val S'   = Stat.instanceSig N'S'
		   in
		     (S', OG.APPstrexp(i', funid, out_strexp))
		   end

	       | None =>
		   let
		     val ei = ErrorInfo.LOOKUP_FUNID funid
		   in
		     (Env.bogus_Str,
		      OG.APPstrexp(errorConv(i, ei), funid, out_strexp)
		     )
		   end
	  end

	(* Local declaration *)
      | IG.LETstrexp(i, strdec, strexp) =>
	  let
	    val (E, out_strdec) = elab_strdec(B, strdec)
	    val (S, out_strexp) = elab_strexp(B B_cplus_E E, strexp)
	  in
	    (S, OG.LETstrexp(okConv i, out_strdec, out_strexp))
	  end

    (********************************************************)
    (* Structure-level Declarations - Definition v3 page 37 *)
    (********************************************************)

    and elab_strdec (B: Env.Basis,strdec: IG.strdec):
	(Env.Env * OG.strdec) =

      case strdec of

	(* Core declaration *)
	IG.DECstrdec(i, dec) => 
	  let
	    (* Note that E is always principal for dec *)
	    val (E, out_dec) = ElabDec.elab_dec(Env.C_of_B B, dec)
	  in
	    (E, OG.DECstrdec(okConv i, out_dec))
	  end

	(* Structure declaration *)
      | IG.STRUCTUREstrdec(i, strbind) =>
	  let
	    val (SE, out_strbind) = elab_strbind(B, strbind)
	  in
	    (Env.SE_in_E SE, OG.STRUCTUREstrdec(okConv i, out_strbind))
	  end

	(* Local declaration *)
      | IG.LOCALstrdec(i, strdec1, strdec2) =>
	  let
	    val (E1, out_strdec1) = elab_strdec(B, strdec1)
	    val (E2, out_strdec2) = elab_strdec(B B_cplus_E E1, strdec2)
	  in
	    (E2, OG.LOCALstrdec(okConv i, out_strdec1, out_strdec2))
	  end

	(* Empty declaration *)
      | IG.EMPTYstrdec i =>
	  (Env.emptyE, OG.EMPTYstrdec(okConv i))

	(* Sequential declaration *)
      | IG.SEQstrdec(i, strdec1, strdec2) =>
	  let
	    val (E1, out_strdec1) = elab_strdec(B, strdec1)
	    val (E2, out_strdec2) = elab_strdec(B B_cplus_E E1, strdec2)
	  in
	    (E1 E_plus_E E2, OG.SEQstrdec(okConv i, out_strdec1, out_strdec2))
	  end

    (**********************************************)
    (* Structure Bindings - Definition v3 page 38 *)
    (**********************************************)

    and elab_strbind (B: Env.Basis, strbind: IG.strbind):
      (Env.StrEnv * OG.strbind) =

      case strbind of 

       (* Structure bindings *)
       IG.STRBIND(i, strid, None, strexp, strbind_opt) =>
	 let
	   val (S , out_strexp) = elab_strexp(B, strexp)
	   val SE1 = Env.singleSE(strid, S)
	   val B'  = B B_plus_N (Stat.namesS S)
	   val (SE2, out_strbind_opt) = elab_strbind_opt(B', strbind_opt)
	 in
	   if (List.member strid (Env.SEdom SE2)) then
	     (SE2,
	      OG.STRBIND(repeatedIdsError(i, [ErrorInfo.STRID_RID strid]),
			 strid, None, out_strexp, out_strbind_opt))
	   else
	     (SE1 SE_plus_SE SE2,
	      OG.STRBIND(okConv i, strid, None, out_strexp, out_strbind_opt))
	 end

       (* Structure bindings *)
     | IG.STRBIND(i, strid, Some sigexp, strexp, strbind_opt) =>
	let
	  val (S , out_strexp) = elab_strexp(B, strexp)
	  val (Sigma, out_sigexp) = elab_sigexp(B, sigexp)
	  val (i', S')  = sigMatchStr(i, Sigma, S)
	  val SE1 = Env.singleSE(strid, S')
	  val B'  = B B_plus_N (Stat.namesS S)
	  val (SE2, out_strbind_opt) = elab_strbind_opt(B', strbind_opt)
	in
	  if (List.member strid (Env.SEdom SE2)) then
	    (SE2,
	     OG.STRBIND(repeatedIdsError(i, [ErrorInfo.STRID_RID strid]),
			 strid, Some out_sigexp, out_strexp, out_strbind_opt))
	   else
	     (SE1 SE_plus_SE SE2,
	      OG.STRBIND(i', strid, Some out_sigexp,
			 out_strexp, out_strbind_opt))
	end

    and elab_strbind_opt
      (B: Env.Basis, strbind_opt: IG.strbind Option):
      (Env.StrEnv * OG.strbind Option) =

      case strbind_opt of 

	None =>
	  (Env.emptySE, None)

      | Some strbind =>
	  let
	    val (SE, out_strbind) = elab_strbind(B, strbind)
	  in
	    (SE, Some out_strbind)
	  end

    (*********************************************************)
    (* Signature Expressions, Definition v3 page 38, rule 65 *)
    (*********************************************************)

    and elab_sigexp (B: Env.Basis, sigexp: IG.sigexp)
          : (Stat.Sig * OG.sigexp) =
      let
	val A = Env.mkAssembly B
	val (_, _, S, out_sigexp) = elab_sigexp'(B, sigexp, A)
	val NofB = Env.N_of_B B
      in

	case Env.covers(NofB, A, S)
	  of Env.NOT_OK(strids, tail) =>
	       let
		 val ei = ErrorInfo.ASSEMBLY_COVER(strids, tail)
	       in
		 (Stat.bogus_Sig, addSigExpEI(ei, out_sigexp))
	       end

	   | Env.OK =>
	       case (Stat.type_explicit(NofB, S),
		     Stat.wellformedsig(NofB, S),
		     Stat.equality_principal(NofB, S)
		    )
		 of (false, _, _) =>
		      let
			val ei = ErrorInfo.NOTTYPEEXPLICIT
		      in
			(Stat.bogus_Sig, addSigExpEI(ei, out_sigexp))
		      end

		  | (true, false, _) =>
		      let 
			val ei = ErrorInfo.NOTWELLFORMEDSIG
		      in
			(Stat.bogus_Sig, addSigExpEI(ei, out_sigexp))
		      end
		  | (true, true, Stat.FAIL longtycons) =>
		      let
			val ei = ErrorInfo.NOTEQPRINCIPAL longtycons
		      in
			(Stat.bogus_Sig, addSigExpEI(ei, out_sigexp))
		      end

		  | (true, true, Stat.PRINCIPAL S) =>
		      let
			val Sigma = Stat.closeStr(NofB, S)
		      in
			(Sigma, out_sigexp)
		      end
      end

    (*****************************************************************)
    (* Signature Expressions, Definition v3 page 38, rules 63 and 64 *)
    (*****************************************************************)

    and elab_sigexp' (B: Env.Basis, sigexp: IG.sigexp, A: Env.Assembly):
      (Stat.Realisation * Env.Assembly * Stat.Str * OG.sigexp) =

       case sigexp of

	 (* Generative *)
	 IG.SIGsigexp(i, spec) =>
	   let
	     val (rea, A1, E, out_spec) = elab_spec(B, spec, A)
	     val S  = Stat.mkStr(Stat.freshStrName(), E)
	     val A2 = A1 union Env.singleA_Str(S)
	   in
	     (rea, A2, S, OG.SIGsigexp(okConv i, out_spec))
	   end

	 (* Signature identifier *)
       | IG.SIGIDsigexp(i, sigid) =>
	   case Env.lookup_sigid(B, sigid)
	     of Some sigma =>
	          let
		    val (S, A') = Env.Sig_instance sigma
		  in
		    (Stat.Id, A', S, OG.SIGIDsigexp(okConv i, sigid))
		  end

	      | None =>
		  let
		    val ei = ErrorInfo.LOOKUP_SIGID sigid
		  in
		    (Stat.bogus_Realisation,
		     Env.bogus_Assembly, Env.bogus_Str,
		     OG.SIGIDsigexp(errorConv(i, ei), sigid)
		    )
		  end

    (*********************************************************)
    (* Signature Declarations - Definition v3 page  39       *)
    (*********************************************************)

    and elab_sigdec (B: Env.Basis, sigdec: IG.sigdec):
      (Env.SigEnv * OG.sigdec) =

      case sigdec of

	(* Single declaration *)
	IG.SIGNATUREsigdec(i, sigbind) =>
	  let
	    val (G, out_sigbind) = elab_sigbind(B, sigbind)
	  in
	    (G, OG.SIGNATUREsigdec(okConv i, out_sigbind))
	  end

	(* Empty declaration *)
      | IG.EMPTYsigdec i =>
	  (Env.emptyG, OG.EMPTYsigdec(okConv i))

	(* Sequential declaration *)
      | IG.SEQsigdec(i, sigdec1, sigdec2) =>
	  let
	    val (G1, out_sigdec1) = elab_sigdec(B, sigdec1)
	    val (G2, out_sigdec2) = elab_sigdec(B B_plus_G G1, sigdec2)
	  in
	    (G1 G_plus_G G2, OG.SEQsigdec(okConv i, out_sigdec1, out_sigdec2))
	  end

    (**********************************************)
    (* Signature Bindings - Definition v3 page 39 *)
    (**********************************************)

    and elab_sigbind (B: Env.Basis, sigbind: IG.sigbind):
      (Env.SigEnv * OG.sigbind) =

      case sigbind of

	(* Signature bindings *)
	IG.SIGBIND(i, sigid, sigexp, None) =>
	  let
	    val (sigma, out_sigexp) = elab_sigexp(B, sigexp)
	    val G = Env.singleG(sigid, sigma)
	  in
	    (G, OG.SIGBIND(okConv i, sigid, out_sigexp, None))
	  end

	(* Signature bindings *)
      | IG.SIGBIND(i, sigid, sigexp, Some sigbind) =>
	  let
	    val (sigma, out_sigexp) = elab_sigexp(B, sigexp)
	    val G1 = Env.singleG(sigid, sigma)
	    val (G2, out_sigbind) = elab_sigbind(B, sigbind)
	  in
	    if (EqSet.member sigid (Env.Gdom G2)) then
	      (G2,
	       OG.SIGBIND(repeatedIdsError(i, [ErrorInfo.SIGID_RID sigid]),
			  sigid, out_sigexp, Some out_sigbind))
	    else
	      (G1 G_plus_G G2, 
	       OG.SIGBIND(okConv i, sigid, out_sigexp, Some out_sigbind))
	  end

    (**********************************************)
    (* Specifications - Definition v3 pages 39-40 *)
    (**********************************************)

    and elab_spec (B: Env.Basis, spec: IG.spec, A: Env.Assembly): 
      (Stat.Realisation * Env.Assembly * Env.Env * OG.spec) =

      case spec of

	(* Value specification *)
	IG.VALspec(i, valdesc) =>
	  let
	    val (VE, out_valdesc) = elab_valdesc(Env.C_of_B B, valdesc)
	    val ClosVE = Env.ClosVE VE
	  in
	    (Stat.Id, Env.emptyA, Env.VE_in_E ClosVE,
	     OG.VALspec(okConv i, out_valdesc))
	  end

	(* Type specification *)
      | IG.TYPEspec(i, typdesc) =>
	  let
	    val (TE, A, out_typdesc) = elab_typdesc false (Env.C_of_B B, typdesc)
	  in
	    (Stat.Id, A, Env.TE_in_E TE,
	     OG.TYPEspec(okConv i, out_typdesc))
	  end

	(* Equality type specification *)
      | IG.EQTYPEspec(i, typdesc) =>
	  let
	    val (TE, A, out_typdesc) = elab_typdesc true (Env.C_of_B B, typdesc)
	  in
	    (Stat.Id, A, Env.TE_in_E TE,
	     OG.EQTYPEspec(okConv i, out_typdesc))
	  end

	(* Datatype specification *)
      | IG.DATATYPEspec(i, datdesc) =>
	  let
	    val TE = initial_TE datdesc
	    val C_and_TE = Env.C_cplus_TE(Env.C_of_B B, TE)
	    val ((VE, TE), A, out_datdesc) = elab_datdesc(C_and_TE, datdesc)
	  in
	    (Stat.Id, A, Env.VE_and_TE_in_E(VE,TE),
	     OG.DATATYPEspec(okConv i, out_datdesc))
	  end

	(* Exception specification *)
      | IG.EXCEPTIONspec(i, exdesc) =>
	  let
	    val (EE, out_exdesc) = elab_exdesc(Env.C_of_B B, exdesc)
	    val VE = Env.VE_of_EE EE
	  in
	    (Stat.Id, Env.emptyA, Env.VE_and_EE_in_E(VE, EE),
	     OG.EXCEPTIONspec(okConv i, out_exdesc))
	  end

	(* Structure specification *)
      | IG.STRUCTUREspec(i, strdesc) =>
	  let
	    val (rea, A', SE, out_strdesc) = elab_strdesc(B, strdesc, A)
	  in
	    (rea, A', Env.SE_in_E SE,
	     OG.STRUCTUREspec(okConv i, out_strdesc))
	  end

	(* Sharing specification *)
      | IG.SHARINGspec(i, shareq) =>
	  let
	    val (rea, out_shareq) = elab_shareq(B, shareq, A)
	  in
	    (rea, Env.emptyA, Env.emptyE,
	     OG.SHARINGspec(okConv i, out_shareq))
	  end

	(* Local specification *)
      | IG.LOCALspec(i, spec1, spec2) =>
	  let
	    val (rea1, A1, E1, out_spec1) = elab_spec(B, spec1, A)
	    val (B', A') = ((rea1 onB B) B_plus_E E1, (rea1 onA A) union A1)
	    val (rea2, A2, E2, out_spec2) = elab_spec(B', spec2, A')
	  in
	    (rea2 oo rea1, (rea2 onA A1) union A2, E2, 
	     OG.LOCALspec(okConv i, out_spec1, out_spec2))
	  end

	(* Open specification *)
      | IG.OPENspec(i, list) =>
	  let
	    fun process list =
	      case list
		of IG.WITH_INFO(i, longstrid) :: rest =>
		     (case Env.lookup_lstrid(B, longstrid)
			of Some S =>
			     let
			       val (m, E) = Stat.unStr S
			       val (E', rest') = process rest
			     in
			       (E E_plus_E E',
				OG.WITH_INFO(okConv i, longstrid) :: rest'
			       )
			     end

			 | None  =>	(* lookup(longstrid) fails *)
			     let
			       val (E', rest') = process rest
			       val ei = ErrorInfo.LOOKUP_LONGSTRID longstrid
			     in
			       (E', OG.WITH_INFO(errorConv(i, ei), longstrid)
				    :: rest'
			       )
			     end
		     )

		 | nil => (Env.emptyE, nil)

	    val (E, list') = process list
	  in
	    (Stat.Id, Env.emptyA, E, OG.OPENspec(okConv i, list'))
	  end

	(* Include specification *)

      | IG.INCLUDEspec(i, list) =>
	  let
	    fun process list =
	      case list
		of IG.WITH_INFO(i, sigid) :: rest =>
		     (case Env.lookup_sigid(B, sigid)
			of Some sigma =>
			     let
			       val (S, A) = Env.Sig_instance sigma
			       val (m, E) = Stat.unStr S
			       val (E', A', rest') = process rest
			     in
			       (E E_plus_E E', A union A',
				OG.WITH_INFO(okConv i, sigid) :: rest'
			       )
			     end

			 | None =>	(* lookup(sigid) fails *)
			     let
			       val (E', A', rest') = process rest
			       val err = ErrorInfo.LOOKUP_SIGID sigid
			     in
			       (E', A',
				OG.WITH_INFO(errorConv(i, err), sigid) :: rest'
			       )
			     end
		     )

		 | nil =>
		     (Env.emptyE, Env.emptyA, nil)

	    val (E, A, list') = process list
	  in
	    (Stat.Id, A, E, OG.INCLUDEspec(okConv i, list'))
	  end

	(* Empty specification *)
      | IG.EMPTYspec i =>
	  (Stat.Id, Env.emptyA, Env.emptyE, OG.EMPTYspec(okConv i))

	(* Sequential specification *)
      | IG.SEQspec(i, spec1, spec2) =>
	  let
	    val (rea1, A1, E1, out_spec1) = elab_spec(B, spec1, A)
	    val (B', A') = ((rea1 onB B) B_plus_E E1, (rea1 onA A) union A1)
	    val (rea2, A2, E2, out_spec2) = elab_spec(B', spec2, A')
	  in
	    (rea2 oo rea1, (rea2 onA A1) union A2, (rea2 onE E1) E_plus_E E2, 
	     OG.SEQspec(okConv i, out_spec1, out_spec2))
	  end

    (**********************************************)
    (* Value Descriptions - Definition v3 page 41 *)
    (**********************************************)

    and elab_valdesc (C: Env.Context, valdesc: IG.valdesc):
      (Env.VarEnv * OG.valdesc) =

      case valdesc of
	IG.VALDESC(i, id, ty, None) =>
	  let
	    val (tau, out_ty) = ElabDec.elab_ty(C, ty)
	    val ts  = Stat.Type_in_TypeScheme(tau)
	    val var = Stat.mk_var id
	  in
	    (Env.singleVarVE(id, ts),
	     OG.VALDESC(okConv i, var, out_ty, None))
	  end

       | IG.VALDESC(i, id, ty, Some valdesc) =>
	  let
	    val (tau, out_ty) = ElabDec.elab_ty(C, ty)
	    val ts  = Stat.Type_in_TypeScheme(tau)
	    val VE  = Env.singleVarVE(id, ts)
	    val var = Stat.mk_var id
	    val (VE', out_valdesc) = elab_valdesc(C, valdesc)
	  in
	    if (EqSet.member id (Env.VEdom VE')) then
	      (VE',
	       OG.VALDESC(repeatedIdsError(i, [ErrorInfo.ID_RID id]),
			  var, out_ty, Some out_valdesc))
	    else
	      (Env.VE_plus_VE(VE, VE'),
	       OG.VALDESC(okConv i, var, out_ty, Some out_valdesc))
	  end

    (*********************************************)
    (* Type Descriptions - Definition v3 page 41 *)
    (*********************************************)

    and elab_typdesc (equality: bool) (C: Env.Context, typdesc: IG.typdesc):
      (Env.TyEnv * Env.Assembly * OG.typdesc) =

       case typdesc of
	 IG.TYPDESC(i, tyvar_list, tycon, typdesc_opt) =>
	   let
	      val TyVar_list =
		map (fn tv => Stat.mkExplicitTyVar tv) tyvar_list
	     val tyvarsRepeated = getRepeatedElements TyVar_list

	     val arity =
	       List.size tyvar_list
	     val tyname =
	       Stat.freshTyName {name=tycon, arity=arity, equality=equality}

	     val typefcn = Stat.TyName_in_TypeFcn tyname
	     val tystr = Env.mkTyStr(typefcn, Env.emptyCE)
	     val TE = Env.singleTE(tycon, tystr)
	     val A  = Env.singleA_TyStr(tystr)

	     val (TE', A', out_typdesc_opt) =
	       elab_typdesc_opt equality (C, typdesc_opt)
	   in
	     if (List.member tycon (Env.TEdom TE')) then
	       (TE', A', 
		OG.TYPDESC(repeatedIdsError(i, [ErrorInfo.TYCON_RID tycon]),
			   tyvar_list, tycon, out_typdesc_opt))
	     else
	       if tyvarsRepeated <> [] then
		 (TE', A', 
		  OG.TYPDESC(repeatedIdsError(i, 
			      map ErrorInfo.TYVAR_RID tyvarsRepeated),
			     tyvar_list, tycon, out_typdesc_opt))
	       else
		 (Env.TE_plus_TE(TE, TE'), A union A',
		  OG.TYPDESC(okConv i, tyvar_list, tycon, out_typdesc_opt))
	   end

    and elab_typdesc_opt (equality: bool)
      (C: Env.Context, typdesc_opt: IG.typdesc Option):
      (Env.TyEnv * Env.Assembly * OG.typdesc Option) =

       case typdesc_opt of
	 None =>
	   (Env.emptyTE, Env.emptyA, None)

	| Some(typdesc) =>
	    let
	      val (TE, A, out_typdesc) = elab_typdesc equality (C, typdesc)
	    in
	      (TE, A, Some out_typdesc)
	    end

    (*************************************************)
    (* Datatype Descriptions - Definition v3 page 42 *)
    (*************************************************)

    and elab_datdesc (C: Env.Context, datdesc: IG.datdesc):
      ((Env.VarEnv * Env.TyEnv) * Env.Assembly * OG.datdesc) =

	case datdesc of
	  IG.DATDESC(i, tyvar_list, tycon, condesc, datdesc_opt) =>
	    let
	      val TyVar_list =
		map (fn tv => Stat.mkExplicitTyVar tv) tyvar_list

	      val (typefcn, _) = 
		case Env.Lookup_tycon(C, tycon) of
		  Some(tystr) => Env.unTyStr(tystr)
		| None => Crash.impossible "ElabTopdec.datdesc(1)"

	      val tyname =
		case Stat.unTyName_TypeFcn(typefcn) of
		  Some(tyname) => tyname
		| None => Crash.impossible "ElabTopdec.datdesc(2)"

	      val tau_list =
		map Stat.mkTypeTyVar TyVar_list

	      val tau =
		Stat.mkTypeConsType
		(Stat.mkConsType(tau_list, tyname))

	      val (CE, out_condesc) = elab_condesc (C, tau, condesc)

	      val ClosCE  = Env.ClosCE CE
	      val typefcn = Stat.TyName_in_TypeFcn tyname
	      val tystr   = Env.mkTyStr(typefcn, ClosCE)
	      val A       = Env.singleA_TyStr(tystr)
	      val CEasVE  = Env.ClosCE_to_VE CE

	      val ((VE, TE), A', out_datdesc_opt) =
		elab_datdesc_opt(C, datdesc_opt)

	      val singleTE = Env.singleTE(tycon, tystr)
	      val TE' = Env.TE_plus_TE(singleTE, TE)
		
	      val intCEVEdom = EqSet.intersect (Env.VEdom CEasVE) (Env.VEdom VE)
	      val tyvarsRepeated = getRepeatedElements TyVar_list
	      val tyvarsNotInTyVarList =
		List.all 
	        (fn tv => not (List.member tv TyVar_list)) 
		(map (fn tv => Stat.mkExplicitTyVar tv) 
		     (IG.getExplicitTyVarsCondesc condesc))
	    in
	      if (List.member tycon (Env.TEdom TE)) then
		((VE, TE), A',
		 OG.DATDESC(repeatedIdsError(i, [ErrorInfo.TYCON_RID tycon]),
			    tyvar_list, tycon, out_condesc, out_datdesc_opt))
	      else
		if not (EqSet.isEmpty intCEVEdom) then
		  ((VE, TE), A',
		   OG.DATDESC(repeatedIdsError(i, 
			       map ErrorInfo.ID_RID (EqSet.list intCEVEdom)),
			      tyvar_list, tycon, out_condesc, out_datdesc_opt))
		else
		  if tyvarsRepeated <> [] then
		    ((VE, TE), A',
		     OG.DATDESC(repeatedIdsError(i,
                                map ErrorInfo.TYVAR_RID tyvarsRepeated),
				tyvar_list, tycon, out_condesc, out_datdesc_opt))
		  else
		    if tyvarsNotInTyVarList <> [] then
		      ((VE, TE), A',
		       OG.DATDESC(repeatedIdsError(i,
				  map ErrorInfo.TYVAR_RID tyvarsNotInTyVarList),
				  tyvar_list, tycon, out_condesc, out_datdesc_opt))
		    else  
		      ((Env.VE_plus_VE(CEasVE, VE), TE'), A union A',
		       OG.DATDESC(okConv i, tyvar_list, tycon,
				  out_condesc, out_datdesc_opt))
	  end

    and elab_datdesc_opt (C: Env.Context, datdesc_opt: IG.datdesc Option):
      ((Env.VarEnv * Env.TyEnv) * Env.Assembly * OG.datdesc Option) =

      case datdesc_opt of

	Some(datdesc) =>
	  let
	    val ((VE, TE), A, out_datdesc) = elab_datdesc(C, datdesc)
	  in
	    ((VE, TE), A, Some out_datdesc)
	  end

       | None =>
	  ((Env.emptyVE, Env.emptyTE), Env.emptyA, None)


    (****************************************************)
    (* Constructor Descriptions - Definition v3 page 42 *)
    (****************************************************)

    and elab_condesc (C: Env.Context, tau: Stat.Type, condesc: IG.condesc):
      (Env.ConEnv * OG.condesc) =

       case condesc of
	 IG.CONDESC(i, con, None, condesc_opt) =>
	   let
	     val ts = Stat.Type_in_TypeScheme tau
	     val CE = Env.singleCE(con, ts)
	     val (CE', out_condesc_opt) = elab_condesc_opt(C, tau, condesc_opt)
	   in
	     if (List.member con (Env.domCE CE')) then
	       (CE',
		OG.CONDESC(repeatedIdsError(i, [ErrorInfo.CON_RID con]),
			   con, None, out_condesc_opt))
	     else
	       (Env.CE_plus_CE(CE, CE'),
		OG.CONDESC(okConv i, con, None, out_condesc_opt))
	   end

	| IG.CONDESC(i, con, Some ty, condesc_opt) =>
	   let
	     val (tau', out_ty) = ElabDec.elab_ty(C, ty)
	     val arrow = Stat.mkTypeArrow(tau', tau)
	     val ts    = Stat.Type_in_TypeScheme arrow
	     val CE    = Env.singleCE(con, ts)
	     val (CE', out_condesc_opt) = elab_condesc_opt(C, tau, condesc_opt)
	   in
	     if (List.member con (Env.domCE CE')) then
	       (CE',
		OG.CONDESC(repeatedIdsError(i, [ErrorInfo.CON_RID con]),
			   con, Some out_ty, out_condesc_opt))
	     else
	       (Env.CE_plus_CE(CE, CE'),
		OG.CONDESC(okConv i, con, Some out_ty, out_condesc_opt))
	   end

    and elab_condesc_opt
      (C: Env.Context, tau: Stat.Type, condesc_opt: IG.condesc Option):
      (Env.ConEnv * OG.condesc Option) =

       case condesc_opt of
	 None =>
	   (Env.emptyCE, None)

	| Some(condesc) =>
	    let
	      val (CE, out_condesc) = elab_condesc(C, tau, condesc)
	    in
	      (CE, Some out_condesc)
	    end

    (**************************************************)
    (* Exception Descriptions - Definition v3 page 42 *)
    (**************************************************)

    and elab_exdesc (C: Env.Context, exdesc: IG.exdesc):
      (Env.ExConEnv * OG.exdesc) =

       case exdesc of
	 IG.EXDESC(i, excon, None, exdesc_opt) =>
	   let
	     val EE = Env.singleEE(excon, Stat.TypeExn)
	     val (EE', out_exdesc_opt) = elab_exdesc_opt(C, exdesc_opt)
	   in
	     if (EqSet.member excon (Env.EEdom EE')) then
	       (EE',
		OG.EXDESC(repeatedIdsError(i, [ErrorInfo.EXCON_RID excon]),
			  excon, None, out_exdesc_opt))
	     else
	       (Env.EE_plus_EE(EE, EE'),
		OG.EXDESC(okConv i, excon, None, out_exdesc_opt))
	   end

	| IG.EXDESC(i, excon, Some ty, exdesc_opt) =>
	   let
	     val tvset = Stat.syntaxtyvarsTy ty
	     val (tau, out_ty) = ElabDec.elab_ty(C, ty)
	     val arrow = Stat.mkTypeArrow(tau, Stat.TypeExn)
	     val EE    = Env.singleEE(excon, arrow)
	     val (EE', out_exdesc_opt) = elab_exdesc_opt(C, exdesc_opt)
	   in
	     if EqSet.isEmpty tvset then
	       if (EqSet.member excon (Env.EEdom EE')) then
		 (EE',
		  OG.EXDESC(repeatedIdsError(i, [ErrorInfo.EXCON_RID excon]),
			    excon, Some out_ty, out_exdesc_opt))
	       else
		 (Env.EE_plus_EE(EE, EE'),
		  OG.EXDESC(okConv i, excon, Some out_ty, out_exdesc_opt))
	     else
	       (EE',
		OG.EXDESC(errorConv(i, ErrorInfo.EXDESC_SIDECONDITION),
			  excon, Some out_ty, out_exdesc_opt))
	   end

    and elab_exdesc_opt (C: Env.Context, exdesc_opt: IG.exdesc Option):
      (Env.ExConEnv * OG.exdesc Option) =

       case exdesc_opt of
	 None =>
	   (Env.emptyEE, None)

	| Some(exdesc) =>
	    let
	      val (EE, out_exdesc) = elab_exdesc(C, exdesc)
	    in
	      (EE, Some out_exdesc)
	    end

    (***************************************************)
    (* Structure Desctriptions - Definition v3 page 41 *)
    (***************************************************)

    and elab_strdesc (B: Env.Basis, strdesc: IG.strdesc, A: Env.Assembly):
      (Stat.Realisation * Env.Assembly * Env.StrEnv * OG.strdesc) =

      case strdesc of

	 (* Structure description *)
	 IG.STRDESC(i, strid, sigexp, None) =>
	   let
	     val (rea, A', S, out_sigexp) = elab_sigexp' (B, sigexp, A)
	   in
	     (rea, A', Env.singleSE(strid, S),
	      OG.STRDESC(okConv i, strid, out_sigexp, None))
	   end

	 (* Structure description *)
       | IG.STRDESC(i, strid, sigexp, Some strdesc) =>
	   let
	     val (rea1, A1, S1, out_sigexp) = elab_sigexp'(B, sigexp, A)
	     val (B', A') = (rea1 onB B, (rea1 onA A) union A1)
	     val (rea2, A2, SE2, out_strdesc) = elab_strdesc(B', strdesc, A')
	    in
	      if (List.member strid (Env.SEdom SE2)) then
		(rea2, A2, SE2,
		 OG.STRDESC(repeatedIdsError(i, [ErrorInfo.STRID_RID strid]),
			    strid, out_sigexp, Some out_strdesc))
	      else
		(rea2 oo rea1, (rea2 onA A1) union A2, 
		 Env.singleSE(strid, rea2 onS S1) SE_plus_SE SE2, 
		 OG.STRDESC(okConv i, strid, out_sigexp, Some out_strdesc))
	   end

    (*********************************************)
    (* Sharing Equations - Definition v3 page 41 *)
    (*********************************************)

    and elab_shareq (B: Env.Basis, shareq: IG.shareq, A: Env.Assembly)
        : (Stat.Realisation * OG.shareq) =
      case shareq
	of IG.STRUCTUREshareq(i, list) =>	(* Structure sharing *)
	     let
	       fun process list =
		 case list
		   of IG.WITH_INFO(i, longstrid) :: rest =>
			(case Env.lookup_lstrid(B, longstrid)
			   of Some S =>
				((#1(Stat.unStr S), longstrid),
				 OG.WITH_INFO(okConv i, longstrid)
				) :: process rest

			    | None =>
				let
				  val ei = ErrorInfo.LOOKUP_LONGSTRID longstrid
				in
				  ((Stat.bogus_StrName, longstrid),
				   OG.WITH_INFO(errorConv(i, ei), longstrid)
				  ) :: process rest
				end
			)

		    | nil => nil

	       val (pairs, list') = ListPair.unzip(process list)
	       val (i', rea) = unifyStr(i, Env.N_of_B B, A, pairs)
	     in
	       (rea, OG.STRUCTUREshareq(i', list'))
	     end

	 | IG.TYPEshareq(i, list) =>		(* Type sharing *)
	     let
	       fun process list =
		 case list
		   of IG.WITH_INFO(i, longtycon) :: rest =>
		        (case Env.lookup_ltycon(B, longtycon)
			   of Some TyStr =>
			        ((TyStr, longtycon),
				 OG.WITH_INFO(okConv i, longtycon)
				) :: process rest

			    | None =>
				let
				  val ei = ErrorInfo.LOOKUP_LONGTYCON longtycon
				in
				  ((Stat.bogus_TyStr, longtycon),
				   OG.WITH_INFO(errorConv(i, ei), longtycon)
				  ) :: process rest
				end
			)

		    | nil => nil

	       val (pairs, list') = ListPair.unzip(process list)
	       val (i', rea) = unifyTy(i, Env.N_of_B B, A, pairs)
	     in
	       (rea, OG.TYPEshareq(i', list'))
	     end

	 | IG.ANDshareq(i, shareq1, shareq2) =>		(* Multiple sharing *)
	     let
	       val (rea1, out_shareq1) = elab_shareq(B, shareq1, A)
	       val (B', A') = (rea1 onB B, rea1 onA A)
	       val (rea2, out_shareq2) = elab_shareq(B', shareq2, A')
	     in
	       (rea2 oo rea1, OG.ANDshareq(okConv i, out_shareq1, out_shareq2))
	     end

    (****************************************************)
    (* Functor Declarations - Definition v3 pages 42-43 *)
    (****************************************************)

    and elab_fundec (B: Env.Basis, fundec: IG.fundec):
      (Env.FunEnv * OG.fundec) =

      case fundec of

	IG.FUNCTORfundec(i, funbind) =>
	  let
	    val (F, out_funbind) = elab_funbind(B, funbind)
	  in
	    (F, OG.FUNCTORfundec(okConv i, out_funbind))
	  end

      | IG.EMPTYfundec i =>
	  (Env.emptyF, OG.EMPTYfundec(okConv i))

      | IG.SEQfundec(i, fundec1, fundec2) =>
	  let
	    val (F1, out_fundec1) = elab_fundec(B, fundec1)
	    val (F2, out_fundec2) = elab_fundec(B B_plus_F F1, fundec2)
	  in
	    (F1 F_plus_F F2, OG.SEQfundec(okConv i, out_fundec1, out_fundec2))
	  end

    (********************************************)
    (* Functor Bindings - Definition v3 page 43 *)
    (********************************************)

    and elab_funbind (B: Env.Basis, funbind: IG.funbind): 
      (Env.FunEnv * OG.funbind) =

      case funbind of

	IG.FUNBIND(i, funid, strid, sigexp, sigexp_opt, strexp, funbind_opt) =>
	  let
	    val (Sig, out_sigexp) = elab_sigexp(B, sigexp)
	    val (N, S) = Stat.unSig Sig

	    val E1 = Env.SE_in_E(Env.singleSE(strid, S))
	    val B1 = B B_cplus_E E1

	    val (S' , out_strexp) = elab_strexp(B1, strexp)
	    val (i', S'', out_sigexp_opt) =
	      case sigexp_opt of
		None =>
		  (okConv i, S', None)
	      | Some sigexp' =>
		  let
		    val (Sig', out_sigexp') = elab_sigexp(B1, sigexp')
		    val (i', S'') = sigMatchStr(i, Sig', S')
		  in
		    (i', S'', Some out_sigexp')
		  end

	    val N'S''  = Stat.closeStr(Env.NameSetUnion(Env.N_of_B B, N), S'')
	    val funsig' = Stat.mkFunSig(N, S, N'S'')

	    val (F, out_funbind_opt) = elab_funbind_opt(B, funbind_opt)
       in
	 if (EqSet.member funid (Env.Fdom F)) then
	   (F,
	    OG.FUNBIND(repeatedIdsError(i, [ErrorInfo.FUNID_RID funid]),
		       funid, strid, out_sigexp, out_sigexp_opt,
		       out_strexp, out_funbind_opt))
	 else
	   (Env.singleF(funid, funsig') F_plus_F F,
	    OG.FUNBIND(i', funid, strid, out_sigexp, out_sigexp_opt,
		       out_strexp, out_funbind_opt))
       end

    and elab_funbind_opt (B: Env.Basis, funbind_opt: IG.funbind Option):
      (Env.FunEnv * OG.funbind Option) =

       case funbind_opt of
	 None =>
	   (Env.emptyF, None)

       | Some funbind => 
	   let
	     val (F, out_funbind) = elab_funbind(B, funbind)
	   in
	     (F, Some out_funbind)
	   end

    (**************************************************)
    (* Top-level Declarations - Definition v3 page 43 *)
    (**************************************************)

    and elab_topdec (B: Env.Basis, topdec: IG.topdec)
          : (Env.Basis * OG.topdec) =
      let
	fun complain(i, tvs, con, arg) =
	  let
	    val ei = ErrorInfo.FREE_TYVARS tvs
	  in
	    con(errorConv(i, ei), arg)
	  end
      in
	case topdec
	  of IG.STRtopdec(i, strdec) =>
	       let
		 val (E, out_strdec) = elab_strdec(B, strdec)
	       in
		 case Env.impTyVarsE E
		   of Env.TVResult.IMP_OK =>
		       (Env.E_in_B(Env.namesE E, E),
			   OG.STRtopdec(okConv i, out_strdec)
			   )
		    | Env.TVResult.FAULT tvs =>
			(Env.bogus_Basis,
			 complain(i, tvs, OG.STRtopdec, out_strdec)
			)
	       end

	   | IG.SIGtopdec(i, sigdec) =>
	       let
		 val (G, out_sigdec) = elab_sigdec(B, sigdec)
	       in
		 case Env.impTyVarsG G
		   of Env.TVResult.IMP_OK =>
			(Env.G_in_B(Env.namesG G, G),
			 OG.SIGtopdec(okConv i, out_sigdec)
			)

		    | Env.TVResult.FAULT tvs =>
			(Env.bogus_Basis,
			 complain(i, tvs, OG.SIGtopdec, out_sigdec)
			)
	       end

	   | IG.FUNtopdec(i, fundec) =>
	       let
		 val (F, out_fundec) = elab_fundec(B, fundec)
	       in
		 case Env.impTyVarsF F
		   of Env.TVResult.IMP_OK =>
			(Env.F_in_B(Env.namesF F, F),
			 OG.FUNtopdec(okConv i, out_fundec)
			)

		    | Env.TVResult.FAULT tvs =>
			(Env.bogus_Basis,
			 complain(i, tvs, OG.FUNtopdec, out_fundec)
			)
	       end
      end

    (********
    Printing functions
    ********)

    type StringTree = Env.StringTree
    val layoutStaticBasis = Env.layoutBasis
  end;
