(* Environments of the static semantics - Definition pages 16-22 *)

(*
$File: Common/ENVIRONMENTS.sml $
$Date: 1992/12/18 08:57:00 $
$Revision: 1.9 $
$Locker: birkedal $
*)

(*$ENVIRONMENTS *)

signature ENVIRONMENTS = 
  sig
    (********
    Syntax of value bindings
    ********)

    type valbind

    (********
    Type variables
    ********)

    type TyVar

    (********
    Explicit TyVars
    ********)

    type TyVarSet
    val Scoped_TyVars : valbind  * TyVarSet -> TyVarSet
    val intersect     : TyVarSet * TyVarSet -> TyVar list

    (********
    Types and TypeSchemes
    ********)

    type Type and TypeScheme


    (********
    The type of items associated with identifiers in a VarEnv
    ********)

    datatype VarEnvRange =
       LONGVAR   of TypeScheme
     | LONGCON   of TypeScheme
     | LONGEXCON of Type

    (********
    Substitutions
    ********)

    type Substitution

    (********
    Unqualified identifiers
    ********)

    eqtype id

    (********
    Variable environments
    ********)

    type VarEnv
    val emptyVE     : VarEnv
    and singleVarVE : id  * TypeScheme -> VarEnv
    and VE_plus_VE  : VarEnv  * VarEnv -> VarEnv
    and lookupVE    : VarEnv  * id     -> VarEnvRange Option
    and onVE        : Substitution * VarEnv -> VarEnv
    and tyvarsVE    : VarEnv -> TyVarSet
    and VEdom       : VarEnv -> id EqSet.Set
    val bogus_VE: VarEnv

    (********
    Constructors
    ********)

    type con

    (********
    Constructor environments
    ********)

    type ConEnv
    val emptyCE    : ConEnv
    val domCE      : ConEnv -> con list
    and singleCE   : con * TypeScheme -> ConEnv
    and CE_plus_CE : ConEnv * ConEnv -> ConEnv

    (********
    Type functions
    ********)

    type TypeFcn

    (********
    Type structures
    ********)

    type TyStr
    val mkTyStr : TypeFcn * ConEnv -> TyStr
    and unTyStr : TyStr -> TypeFcn * ConEnv

    (********
    Type constructors
    ********)

    type tycon

    (********
    Type environments
    ********)

    type TyEnv
    val emptyTE    : TyEnv
    and singleTE   : tycon * TyStr -> TyEnv
    and TE_plus_TE : TyEnv * TyEnv -> TyEnv
    and TEdom      : TyEnv -> tycon list

    (********
    Exception constructors
    ********)

    type excon

    (********
    Exception constructor environments
    ********)

    type ExConEnv
    val EEdom      : ExConEnv -> excon EqSet.Set
    val EE_plus_EE : ExConEnv * ExConEnv -> ExConEnv
    and singleEE   : excon * Type -> ExConEnv
    and VE_of_EE   : ExConEnv -> VarEnv
    and emptyEE    : ExConEnv

    (********
    Environments
    ********)

    type Env
    val VE_and_TE_in_E : (VarEnv * TyEnv   ) -> Env
    and VE_and_EE_in_E : (VarEnv * ExConEnv) -> Env
    and E_plus_E : Env   * Env -> Env
    and VE_in_E  : VarEnv -> Env
    and TE_in_E  : TyEnv  -> Env
    and onE : Substitution * Env -> Env
    and ABS : TyEnv * Env -> Env
    and emptyE : Env

    val bogus_Env: Env

    (********
    Contexts
    ********)

    type Context
    val C_cplus_VE_and_TE : Context * (VarEnv * TyEnv) -> Context
    and C_plus_VE  : Context * VarEnv   -> Context
    and C_plus_U   : Context * TyVarSet -> Context
    and C_cplus_TE : Context * TyEnv    -> Context
    and C_cplus_E  : Context * Env      -> Context
    and onC    : Substitution * Context  -> Context
    and U_of_C : Context -> TyVarSet
    and E_in_C : Env -> Context

    (********
    The `Clos' function for valbinds
    ********)

    val Clos : Context * valbind * VarEnv -> VarEnv

    (********
    The `Clos' function for constructor environments
    ********)

    val ClosCE : ConEnv -> ConEnv

    (********
    The `Clos' function for constructor environments
    *********
    This function converts the resulting CE to a VE
    ********)

    val ClosCE_to_VE : ConEnv -> VarEnv

    (********
    Function which maximises equality in a type environment
    *********
    The VarEnv passed is also modified so that it contains the correct equality attributes
    ********)

    val maximise_equality : VarEnv * TyEnv -> VarEnv * TyEnv

    (*******
    Patterns and domains of patterns
    *******)

    type pat 
    val dom_pat : Context * pat -> id list

    (********
    Qualified identifiers
    ********)

    type longid

    (********
    Function which looks up a long identifier in the context
    ********)

    val Lookup_longid : Context * longid -> VarEnvRange Option

    (* Additional function for taking a longid (which must represent a
       constructor) to a complete list of its fellow constructors. Needed
       for compilation and exhaustiveness checking. Returns the cons in
       a canonical order. Note: assumed to succeed (i.e. you must do the
       Lookup_longid first). *)

    val lookupFellowCons: Context * longid -> con list

    (********
    Qualified type constructors
    ********)

    type longtycon

    (********
    Function which looks up a long type constructor in the context
    ********)

    val Lookup_longtycon : Context * longtycon -> TyStr Option

    (********
    Function which looks up a type constructor in the context
    ********)

    val Lookup_tycon : Context * tycon -> TyStr Option

    (*******
     Function which returns the domain of a pattern
    *******)

    (********
    Structure names
    ********)

    type StrName

    (********
    Structures
    ********)

    type Str
    val unStr : Str -> StrName * Env

    (********
    Qualified structure identifiers
    ********)

    type longstrid

    (********
    Function which looks up a structure identifier in the context
    ********)

    val Lookup_longstrid : Context * longstrid -> Str Option

    (********
    Printing functions for longtycon's (for error reporting)
    ********)

    val pr_LongTyCon : longtycon -> string

    (********
    Printing function for environments
    ********)

    type StringTree
    val layoutVE: VarEnv -> StringTree
    val layoutEnv : Env -> StringTree
  end;
