#ifndef DGEMATRIX_H
#define DGEMATRIX_H
#pragma once 

/*
 *	Declarations for double precision general matricies.
 *
 *	Copyright (C) 1988, 1989.
 *
 *	Dr. Thomas Keffer
 *	Rogue Wave Associates
 *	P.O. Box 85341
 *	Seattle WA 98145-1341
 *
 *	Permission to use, copy, modify, and distribute this
 *	software and its documentation for any purpose and
 *	without fee is hereby granted, provided that the
 *	above copyright notice appear in all copies and that
 *	both that copyright notice and this permission notice
 *	appear in supporting documentation.
 *	
 *	This software is provided "as is" without any
 *	expressed or implied warranty.
 *
 *
 *	@(#)DGEMatrix.h	2.2	9/26/89
 */

/*
 *	This class is derived from class DoubleVec.  Data is stored
 *	FORTRAN style: by columns.
 *
 *	Defining the preprocessor directive "BOUNDS_CHECK" will invoke
 *	bounds checking.
 */

#include "DoubleVec.h"

class DGEMatrix : public DoubleVec {
  int ncols;			// Number of columns
  int nrows;			// Number of rows
protected:
  void 			assertColRange(int);
  void 			assertRowRange(int);
  void			assertRowCol(const DGEMatrix&);
  void			assertLength(const DoubleVec&);
  void			assertSquare();
  void			assertProduct(const DGEMatrix&);
  void			assertProduct(const DoubleVec&);
public:
  DGEMatrix();
  DGEMatrix(int rows, int cols);
  DGEMatrix(int rows, int cols, double initval);
  DGEMatrix(const double* dat, int, int);  // Copy of dat will be made
  DGEMatrix(const DoubleVec& v, int, int); // Reference to v will be made
  DGEMatrix(const DGEMatrix& m);	   // Reference to m will be made

  double*		data()		{return DoubleVec::data();}
  int			cols();
  int			rows();

  DGEMatrix&		reference(DGEMatrix& m); // Reference self to m
  DGEMatrix		deepCopy();	// copy of self with distinct instance variables 
  DGEMatrix		copy()		{return deepCopy();} // Synonym for deepCopy()
  void			deepenShallowCopy();	// Guarantee that references==1:
  void			resize(unsigned,unsigned);

  void			scanFrom(istream& s);	// Format 2 x 2 [ 3 2 1 4 ]
  void			printOn(ostream& s);
  void			readFrom(istream&);
  void			storeOn(ostream&);
  void			readFrom(fileDescTy&);
  void			storeOn(fileDescTy&);

  DoubleVec		operator[](int j);	// Return a col as a slice
  DoubleVec		col(int j);		// Return a col as a slice
  DoubleVec		row(int i);		// Return a row as a slice
  DoubleVec		diagonal(int idiag=0);	// Return a diagonal as a slice
  double&		operator()(int i, int j); // Subscripting

// Math functions
  DGEMatrix		product(const DGEMatrix&); // Inner product
  DoubleVec		product(const DoubleVec&);

// Assignment operators --- self must be same size as m
  DGEMatrix&		operator=(const DGEMatrix& m);
  DGEMatrix&		operator=(double);
  DGEMatrix&		operator+=(const DGEMatrix& m);
  DGEMatrix&		operator+=(double);
  DGEMatrix&		operator-=(const DGEMatrix& m);
  DGEMatrix&		operator-=(double);
  DGEMatrix&		operator*=(const DGEMatrix& m);
  DGEMatrix&		operator*=(double);
  DGEMatrix&		operator/=(const DGEMatrix& m);
  DGEMatrix&		operator/=(double);

// Increment/decrement operators
  DGEMatrix&		operator++();
  DGEMatrix&		operator--();

// Friendly arithmetic operators; Notice that operator* is an element-by-
// element multiply, NOT a matrix multiply.
  friend DGEMatrix	operator-(const DGEMatrix&);	// Unary minus
  friend DGEMatrix	operator+(const DGEMatrix&);	// Unary plus
  friend DGEMatrix	operator*(const DGEMatrix&, const DGEMatrix&);
  friend DGEMatrix	operator/(const DGEMatrix&, const DGEMatrix&);
  friend DGEMatrix	operator+(const DGEMatrix&, const DGEMatrix&);
  friend DGEMatrix	operator-(const DGEMatrix&, const DGEMatrix&);
  friend DGEMatrix	operator*(const DGEMatrix&, double);
  friend DGEMatrix	operator*(double, const DGEMatrix&);
  friend DGEMatrix	operator/(const DGEMatrix&, double);
  friend DGEMatrix	operator/(double, const DGEMatrix&);
  friend DGEMatrix	operator+(const DGEMatrix&, double);
  friend DGEMatrix	operator+(double, const DGEMatrix&);
  friend DGEMatrix	operator-(const DGEMatrix&, double);
  friend DGEMatrix	operator-(double, const DGEMatrix&);

};

// Other (related) declarations:
ostream&		operator<<(ostream& s, const DGEMatrix& m);
istream& 		operator>>(istream& s, DGEMatrix& m);
DGEMatrix		transpose(const DGEMatrix&);

/******************* I N L I N E S **************************/

Inline int DGEMatrix::cols() { return ncols;}
Inline int DGEMatrix::rows() { return nrows;}
Inline void DGEMatrix::deepenShallowCopy(){DoubleVec::deepenShallowCopy();}
Inline DGEMatrix operator+(const DGEMatrix& m)		{ return m; }
Inline DGEMatrix operator*(double d, const DGEMatrix& m){ return m*d; }
Inline DGEMatrix operator+(double d, const DGEMatrix& m){ return m+d; }

// Return a column
Inline DoubleVec DGEMatrix::operator[](int j){
#if BOUNDS_CHECK
  assertColRange(j);
#endif
  return DoubleVec::slice(j*nrows,nrows,1);
}

Inline DoubleVec DGEMatrix::col(int j){	// Same as above
#if BOUNDS_CHECK
  assertColRange(j);
#endif
  return DoubleVec::slice(j*nrows,nrows,1);
}

Inline DoubleVec DGEMatrix::row(int i){
#if BOUNDS_CHECK
  assertRowRange(i);
#endif
  return DoubleVec::slice(i, ncols, nrows);
}

Inline DoubleVec DGEMatrix::diagonal(int i){
  register int iabs=ABSOLUTE(i);
#if BOUNDS_CHECK
  assertSquare();
  assertRowRange(iabs);
#endif
  return DoubleVec::slice(i>0 ? i*nrows : iabs, nrows-iabs, nrows+1);
}

Inline double& DGEMatrix::operator()(int i, int j){
#if BOUNDS_CHECK
  assertRowRange(i); assertColRange(j);
#endif
  return DoubleVec::operator()(j*nrows+i);
}

#endif DGEMATRIX_HXX
