#ifndef DCOMPLEXVEC_H
#define DCOMPLEXVEC_H
#pragma once

/*
 *	Declarations for DComplex Precision Vectors
 *
 *	Copyright (C) 1988, 1989.
 *
 *	Dr. Thomas Keffer
 *	Rogue Wave Associates
 *	P.O. Box 85341
 *	Seattle WA 98145-1341
 *
 *	Permission to use, copy, modify, and distribute this
 *	software and its documentation for any purpose and
 *	without fee is hereby granted, provided that the
 *	above copyright notice appear in all copies and that
 *	both that copyright notice and this permission notice
 *	appear in supporting documentation.
 *	
 *	This software is provided "as is" without any
 *	expressed or implied warranty.
 *
 *
 *	@(#)DComplexVec.h	2.2	9/18/89
 */

/*	This code is designed to be as compatible as possible with the
 *	NIH Vector classes, while preserving efficiency.  These Vectors
 *	are NOT based on the NIH "Object" class, making them much
 *	smaller.  They also implement reference counting, making them
 *	faster. 
 */


#ifdef NO_VECTOR_MATHFUN
#include "vdefs.h"
#include <math.h>
#else
#include "DoubleVec.h"
#endif
#include "RComplex.h"

class istream;
class ostream;
class DComplexVec;
class DoubleVec;

class DComplexBlock {
  unsigned short  	refs;		// Number of references
  unsigned  		npts;		// Number of elements
  DComplex    		array[1];	// The data
  friend		DComplexVec;
public:
  DComplexBlock(unsigned n);
  DComplexBlock(unsigned n, DComplex val);
  DComplexBlock(unsigned n, DComplex val, DComplex by);
  ~DComplexBlock();

  void			add_reference()	{refs++;}
  unsigned		references()	{return refs;}
  DComplex*		data()		{return array;}
};

class DComplexVec {
  DComplexBlock*	block;
  DComplex*		begin;
  unsigned		npts;
  int			step;

  static int		numberPerLine; // For printing
  DComplexVec(DComplexVec&, int, unsigned, int); // For slices
protected:
  void			boundsErr(int);
  void			boundsCheck(int);
  void			lengthErr(int);
  void			lengthCheck(int i)	{if(npts!=i) lengthErr(i);}
  void			emptyErr(const char* fname);
  void			sliceErr(unsigned, int, unsigned, int);
public:
  DComplexVec();
  DComplexVec(unsigned n);
  DComplexVec(unsigned n, DComplex val);
  DComplexVec(unsigned n, DComplex val, DComplex by);
  DComplexVec(const DComplexVec& a);
  DComplexVec(const DoubleVec& re); // Conversion
  DComplexVec(const DoubleVec& re, const DoubleVec& im);
  DComplexVec(const DComplex* dat, unsigned n);  // Copy of dat will be made
  ~DComplexVec();
  
  DComplexVec		slice(int start, unsigned lgt, int strider=1);
  
  DComplex*		data()		{return begin;}
  unsigned		length()	{return npts;}
  int			stride()	{return step;}

  DComplexVec&		reference(DComplexVec& v);	// Reference self to v
  DComplexVec		deepCopy();	// copy of self with distinct instance variables 
  DComplexVec		copy()		{return deepCopy();} // Synonym for deepCopy()
  void			deepenShallowCopy();	// Insures only 1 reference to data
  void			resize(unsigned);	// Will pad with zeroes if necessary

  void			scanFrom(istream& s); // Read to eof or delimit with []
  void			printOn(ostream& s);  // Pretty print
  void			setFormatting(int);   // Change # items per line
  void			readFrom(istream&);   // Internal ASCII formatting
  void			storeOn(ostream&);
  void			readFrom(fileDescTy&);// Internal binary formatting
  void			storeOn(fileDescTy&);
  
  // Indexing:
  DComplex&		operator[](int i);	// With bounds checking
  DComplex&		operator()(int i);	// With optional bounds checking
  
  // Assignment:
  DComplexVec&		operator=(const DComplexVec& v); // Must be same length as v
  DComplexVec&		operator=(DComplex);
  
  // Arithmetic operators:
//DComplexVec&		operator++();
//DComplexVec&		operator--();
  DComplexVec&		operator+=(const DComplexVec&);
  DComplexVec&		operator+=(DComplex);
  DComplexVec&		operator-=(const DComplexVec&);
  DComplexVec&		operator-=(DComplex);
  DComplexVec&		operator*=(const DComplexVec&);
  DComplexVec&		operator*=(DComplex);
  DComplexVec&		operator/=(const DComplexVec&);
  DComplexVec&		operator/=(DComplex);
  
  // Friendly arithmetic operators:
  friend DComplexVec	operator-(const DComplexVec&);
  friend DComplexVec	operator+(const DComplexVec&);
  friend DComplexVec	operator*(const DComplexVec&,const DComplexVec&);
  friend DComplexVec	operator/(const DComplexVec&,const DComplexVec&);
  friend DComplexVec	operator+(const DComplexVec&,const DComplexVec&);
  friend DComplexVec	operator-(const DComplexVec&,const DComplexVec&);
  friend DComplexVec	operator*(const DComplexVec&,DComplex);
  friend DComplexVec	operator*(DComplex,const DComplexVec&);
  friend DComplexVec	operator/(const DComplexVec&,DComplex);
  friend DComplexVec	operator/(DComplex,const DComplexVec&);
  friend DComplexVec	operator+(const DComplexVec&,DComplex);
  friend DComplexVec	operator+(DComplex,const DComplexVec&);
  friend DComplexVec	operator-(const DComplexVec&,DComplex);
  friend DComplexVec	operator-(DComplex,const DComplexVec&);
  
  
#ifndef NO_VECTOR_MATHFUN
  // Math functions (not all implemented by complex):
  DComplexVec	apply(CmathFunTy);
  DoubleVec	apply2(CmathFunTy2);
  friend	DoubleVec	abs(const DComplexVec&);
//friend	DComplexVec	acos(const DComplexVec&);
//friend	DComplexVec	asin(const DComplexVec&);
//friend	DComplexVec	atan(const DComplexVec&);
//friend	DComplexVec	atan2(const DComplexVec&,const DComplexVec&);
//friend	DComplexVec	ceil(const DComplexVec&);
  friend	DComplexVec	cos(const DComplexVec&);
  friend	DComplexVec	cosh(const DComplexVec&);
  friend	DComplexVec	cumsum(const DComplexVec&);
  friend	DComplexVec	delta(const DComplexVec&);
  friend	DComplex	dot(const DComplexVec&,const DComplexVec&);
  friend	DComplexVec	exp(const DComplexVec&); 
//friend	DComplexVec	floor(const DComplexVec&);
  friend	DComplexVec	log(const DComplexVec&);
//friend	int		max(const DComplexVec&);
//friend	int		min(const DComplexVec&);
  friend	DComplex	mean(const DComplexVec&);
  friend	DComplex	prod(const DComplexVec&);
  friend	DComplexVec	pow(const DComplexVec&,const DComplexVec&);
  friend	DComplexVec	reverse(const DComplexVec&);
//friend	DComplexVec	rint(const DComplexVec&);
  friend	DComplexVec	sin(const DComplexVec&);
  friend	DComplexVec	sinh(const DComplexVec&);
  friend	DComplexVec	sqrt(const DComplexVec&);
  friend	DComplex	sum(const DComplexVec&);
//friend	DComplexVec	tan(const DComplexVec&);
//friend	DComplexVec	tanh(const DComplexVec&);
  friend	double		variance(const DComplexVec&);

// Complex specific functions:
  friend	DoubleVec	arg(const DComplexVec& V);
  friend	DComplexVec	conj(const DComplexVec& V);
  friend	DoubleVec	imag(const DComplexVec& V);
  friend	DoubleVec	norm(const DComplexVec& V);
  friend	DoubleVec	real(const DComplexVec& V);

#endif
  
};

// Other (related) declarations:
DComplexVec		rootsOfOne(int N, unsigned nterms);
Inline DComplexVec	rootsOfOne(int N) {return rootsOfOne(N,N);}
DComplexVec		expandConjugateEven(const DComplexVec&);
DComplexVec		expandConjugateOdd(const DComplexVec&);
ostream&		operator<<(ostream&, const DComplexVec&);
istream&		operator>>(istream&, DComplexVec&);

/******************* I N L I N E S **************************/

Inline void		DComplexVec::setFormatting(int i){numberPerLine = i;}

Inline void		DComplexVec::boundsCheck(int i){
  if(i<0 || i>npts) boundsErr(i);
}
Inline DComplex&	DComplexVec::operator[](int i){
  boundsCheck(i); return begin[i*step];
}
Inline DComplex&	DComplexVec::operator()(int i) {
#if BOUNDS_CHECK    
  boundsCheck(i);
#endif
  return begin[i*step];
}

Inline DComplexVec	operator+(const DComplexVec& a)			{return a;}
Inline DComplexVec	operator*(DComplex a, const DComplexVec& b)	{return b*a;}
Inline DComplexVec	operator+(DComplex a, const DComplexVec& b)	{return b+a;}

#ifndef NO_VECTOR_MATHFUN
Inline DComplexVec cos(const DComplexVec& V)	{ return V.apply(::cos); }
Inline DComplexVec cosh(const DComplexVec& V)	{ return V.apply(::cosh); }
Inline DComplexVec exp(const DComplexVec& V)	{ return V.apply(::exp); }
Inline DComplexVec log(const DComplexVec& V)	{ return V.apply(::log); }
Inline DComplexVec sin(const DComplexVec& V)	{ return V.apply(::sin); }
Inline DComplexVec sinh(const DComplexVec& V)	{ return V.apply(::sinh); }
Inline DComplexVec sqrt(const DComplexVec& V)	{ return V.apply(::sqrt); }
Inline DComplex mean(const DComplexVec& V)	{ return sum(V)/DComplex(double(V.length())); }

// Complex specific:
Inline DoubleVec arg(const DComplexVec& V)	{ return V.apply2(::arg); }
Inline DComplexVec conj(const DComplexVec& V)	{ return V.apply(::conj); }
Inline DoubleVec norm(const DComplexVec& V)	{ return V.apply2(::norm); }
#endif
 
#endif
