/*
 * Copyright (C) 1995, 1996 Systemics Ltd (http://www.systemics.com/)
 * All rights reserved.
 *
 */

package cryptix.pgp;

import java.io.DataInput;
import java.io.DataOutput;
import java.io.IOException;

import java.util.Date;
import cryptix.math.MPI;
import cryptix.math.BigInteger;
import cryptix.crypt.MessageHash;
import cryptix.crypt.MD5;
import cryptix.crypt.rsa.SecretKey;
import cryptix.crypt.rsa.PublicKey;

public final class Signature extends Packet
{
	private static final byte PADDING[] =
	{
		(byte)0x00, (byte)0x30, (byte)0x20, (byte)0x30, (byte)0x0C, (byte)0x06, (byte)0x08, (byte)0x2A,
		(byte)0x86, (byte)0x48, (byte)0x86, (byte)0xF7, (byte)0x0D, (byte)0x02, (byte)0x05, (byte)0x05, 
		(byte)0x00, (byte)0x04, (byte)0x10
	};

	private BigInteger number;
	public byte extra[];
	private byte keyId[];
	private byte md_check[];
	
	public Signature( SecretKey key, MD5 md )
	{
		int unixtime = (int)(System.currentTimeMillis()/1000);
		extra = new byte[5];
		extra[0] = 0x0; // binary data.
		extra[1] = (byte)( (unixtime >>> 24 ) & 0xFF );
		extra[2] = (byte)( (unixtime >>> 16 ) & 0xFF );
		extra[3] = (byte)( (unixtime >>> 8 ) & 0xFF );
		extra[4] = (byte)( unixtime & 0xFF );
		md.add( extra );
		byte hash[] = md.digest();
		BigInteger tmp = BigIntFromHash( key, hash );
		number = key.decrypt( tmp ); // signing is encrypting with secret therefore normal decryption.
		md_check = new byte[2];
		md_check[0] = hash[0];
		md_check[1] = hash[1];
		keyId = key.id();
	}

	public Signature( DataInput in, int length )
	throws IOException
	{
		super( in, length );
	}

	public void 
	read( DataInput in, int length )
	throws IOException
	{
		in.readByte(); // version
		int extraLen = in.readByte() & 0xFF;

		if ( extraLen != 5 && extraLen != 7 )
			throw new FormatException( "The length of the extra data field is incorrect." );

		in.readFully( extra = new byte[extraLen] );
		in.readFully( keyId = new byte[8] );

		if ( in.readByte() != 0x01 )
			throw new FormatException( "The algorthim byte is not RSA." );

		if ( in.readByte() != 0x01 )
			throw new FormatException( "The Message Digest byte is not MD5." );

		in.readFully(md_check = new byte[2]);

		number = MPI.read(in);
	}

	public int 
	write( DataOutput out )
	throws IOException
	{
		out.write(0x02);
		out.write(extra.length);
		out.write(extra);
		out.write(keyId);
		out.write(0x01);
		out.write(0x01);
		out.write(md_check);
		byte tmp[] = MPI.save(number);
		out.write(tmp);
		return extra.length + tmp.length + 14;
	}
	
	public String
	toString()
	{
		return "Signature packet - number: " + number.toString();
	}

	private static BigInteger BigIntFromHash( PublicKey key, byte hash[] )
	{
		if ( hash.length != 16 )
			throw new Error( "The code for hashes of lengths other than 16 is not in place yet." );
		int byteLength = key.bitLength() / 8;
		int fillLength = byteLength - ( 19 + hash.length ); // note that fillLength is 2 longer that the required number of FF's.
		byte buffer[] = new byte[byteLength];
		buffer[0] =0;
		buffer[1] =1;
		for (int i = 2; i < fillLength; i++)
			buffer[i] = (byte)0xFF;
		System.arraycopy( PADDING, 0, buffer, fillLength, 19 );
		System.arraycopy( hash, 0, buffer, fillLength + 19, hash.length );
		return new BigInteger( buffer );
	}

	private boolean
	check(PublicKey key, byte filehash[])
	{
		if ((filehash[0] != md_check[0]) || (filehash[1] != md_check[1]))
			return false;

		BigInteger hashFromPacket = key.encrypt( number );
		BigInteger hashFromFile = BigIntFromHash( key, filehash );
		if (hashFromPacket.cmp(hashFromFile) != 0)
			return false;

		byte id[] = key.id();
		int len = id.length;
		if ( len == keyId.length )
		{
			int i;
			for ( i=0; i < len; i++ )
				if ( id[i] != keyId[i] )
					return false;
		}
		return true;
	}
	
	public boolean
	check( PublicKey key, MD5 md )
	{
		md.add( extra );
		return check( key, md.digest() );
	}
	
	public byte[]
	checkAndGetHash( PublicKey key, MD5 md )
	{
		md.add( extra );
		byte hash[] = md.digest();
		if ( !check( key, hash ) )
			return null;
		return hash;
	}

	// use with caution.
	public void
	addExtasToHash( MD5 md )
	{
		md.add( extra );
	}

	public boolean
	check( PublicKey key, MessageHash completeHash )
	{
		return check( key, completeHash.toByteArray() );
	}

	public boolean
	verify(PublicKey key, byte[] data)
	{
		MD5 md = new MD5();
		md.add(data);
		md.add(extra);
		byte[] b = md.digest();
		return check(key, b);
	}
	
	public byte[]
	getKeyId()
	{
 		byte tmp[];
 		int len = keyId.length;
		System.arraycopy( keyId, 0, tmp = new byte[len], 0, len );
		return tmp;
	}
}
