/*
 *  nwfsctrl.c
 *
 *  Down file server, open bindery, close bindery
 *
 *  Copyright (C) 1998 by Petr Vandrovec <vandrove@vc.cvut.cz>
 *
 */

#include <stdio.h>
#include <unistd.h>
#include <ctype.h>
#include <stdlib.h>
#include "nwcalls.h"

static char *progname;

static void
usage(void)
{
	fprintf(stderr, "usage: %s [options]\n", progname);
	return;
}

static void
help(void)
{
	printf("\n");
	printf("usage: %s [options] command\n", progname);
	printf("\n"
	       "-h             Print this help text\n"
	       "-S server      Server name to be used\n"
	       "-U username    Username sent to server\n"
	       "-P password    Use this password\n"
	       "-n             Do not use any password\n"
	       "-C             Don't convert password to uppercase\n"
	       "-p path        Permanent connection to server to be used\n"
	       "\n"
	       "command is one of:\n"
	       "load XXX       load module XXX\n"
	       "unload XXX     unload module XXX\n"
	       "mount XXX      mount volume XXX\n"
	       "dismount XXX   dismount volume XXX\n"
	       "down [force]   down fileserver\n"
	       "open bindery   open bindery\n"
	       "close bindery  close bindery\n"
	       "disable login  disable login to fileserver\n"
	       "enable login   enable login to fileserver\n"
	       "disable tts    disable TTS\n"
	       "enable tts     enable TTS\n"
	       "set XXX=YYY    set server variable XXX to value YYY\n"
	       "XXX            try execute XXX as .NCF file\n"
	       "\n"
	       "Warning: If you do not have sufficient priviledge level,\n"
	       "         some of commands (load, unload, mount, dismount,\n"
	       "         set, run .NCF) will emit server console broadcast\n"
	       "         to all conected users!\n"
	       "\n");
}

static char* skipspaces(char* str) {
	while (*str && isspace(*str)) str++;
	return str;
}

struct cmdlist {
	const char* cmd;
	int (*proc)(char*);
	int opt;
};

int cmdexec(const struct cmdlist* cmds, char* cmd, size_t cmdl,
		char* param) {
	while (cmds->cmd) {
		size_t l;

		l = strlen(cmds->cmd);
		if ((l == cmdl) && !memcmp(cmd, cmds->cmd, l)) {
			if ((!*param) && (cmds->opt&1)) {
				fprintf(stderr, "%s: \"%s\" what?\n", progname, cmds->cmd);
				return 120;
			}
			return cmds->proc(param);
		}
		cmds++;
	}
	return cmds->proc(cmd);
}

static struct ncp_conn *conn;

static int cmd_load(char* module) {
	NWCCODE nwerr;
	
	nwerr = NWSMLoadNLM(conn, module);
	if (nwerr) {
		fprintf(stderr, "%s: Unable to load `%s', error 0x%04X (%u)\n",
			progname, module, nwerr, nwerr);
		return 100;
	}
	return 0;
}

static int cmd_unload(char* module) {
	NWCCODE nwerr;

	nwerr = NWSMUnloadNLM(conn, module);
	if (nwerr) {
		fprintf(stderr, "%s: Unable to unload `%s', error 0x%04X (%u)\n",
			progname, module, nwerr, nwerr);
		return 100;
	}
	return 0;
}

static int cmd_mount(char* volume) {
	NWCCODE nwerr;
	nuint32 volID;
	
	nwerr = NWSMMountVolume(conn, volume, &volID);
	if (nwerr) {
		fprintf(stderr, "%s: Unable to mount `%s', error 0x%04X (%u)\n",
			progname, volume, nwerr, nwerr);
		return 100;
	}
	printf("Volume `%s' was mounted as #%u\n", volume, volID);
	return 0;
}

static int cmd_dismount(char* volume) {
	NWCCODE nwerr;

	nwerr = NWSMDismountVolumeByName(conn, volume);
	if (nwerr) {
		fprintf(stderr, "%s: Unable to dismount `%s', error 0x%04X (%u)\n",
			progname, volume, nwerr, nwerr);
		return 100;
	}
	return 0;
}

static int cmd_down(char* cmd) {
	NWCCODE nwerr;
	
	nwerr = NWDownFileServer(conn, *cmd == 'f');
	if (nwerr) {
		fprintf(stderr, "%s: Unable to down file server, error 0x%04X (%u)\n",
			progname, nwerr, nwerr);
		return 100;
	}
	return 0;
}

static int cmd_run(char* cmd) {
	NWCCODE nwerr;

	nwerr = NWSMExecuteNCFFile(conn, cmd);
	if (nwerr) {
		fprintf(stderr, "%s: Unable to execute `%s', error 0x%04X (%u)\n",
			progname, cmd, nwerr, nwerr);
		return 100;
	}
	return 0;
}

static int cmd_set(char* cmd) {
	char* begin = cmd;
	char* val;
	int numeric;
	u_int32_t value;
	NWCCODE nwerr;

	while (*cmd && (*cmd != '=')) cmd++;
	if (*cmd != '=') {
		fprintf(stderr, "%s: Syntax error in `set' command\n",
			progname);
		return 99;
	}
	val = skipspaces(cmd + 1);
	do {
		--cmd;
	} while ((cmd >= begin) && isspace(*cmd));
	cmd[1] = 0;

	numeric = 1;
	if (!strcmp(val, "on") || !strcmp(val, "enabled")) {
		value = 1;
	} else if (!strcmp(val, "off") || !strcmp(val, "disabled")) {
		value = 0;
	} else {
		char* eptr;
		
		value = strtoul(val, &eptr, 0);
		if (eptr) eptr = skipspaces(eptr);
		if (eptr && *eptr) numeric = 0;
	}
	if (numeric) {
		nwerr = NWSMSetDynamicCmdIntValue(conn, begin, value);
		if (!nwerr) return 0;
		if (nwerr != 0x0206) {
			fprintf(stderr, "%s: Unable to set variable `%s' to value `%s', error 0x%04X (%u)\n",
			progname, begin, val, nwerr, nwerr);
			return 100;
		}
	}
	nwerr = NWSMSetDynamicCmdStrValue(conn, begin, val);
	if (nwerr) {
		fprintf(stderr, "%s: Unable to set variable `%s' to value `%s', error 0x%04X (%u)\n",
			progname, begin, val, nwerr, nwerr);
		return 100;
	}
	return 0;
}

static int opclo;

static int cmd_login(char* cmd) {
	NWCCODE nwerr;

	nwerr = opclo ? NWEnableFileServerLogin(conn) : NWDisableFileServerLogin(conn);
	if (nwerr) {
		fprintf(stderr, "%s: Unable to %s file server login, error 0x%04X (%u)\n",
			progname, opclo?"enable":"disable", nwerr, nwerr);
		return 100;
	}
	return 0;
}

static int cmd_tts(char* cmd) {
	NWCCODE nwerr;

	nwerr = opclo ? NWEnableTTS(conn) : NWDisableTTS(conn);
	if (nwerr) {
		fprintf(stderr, "%s: Unable to %s TTS, error 0x%04X (%u)\n",
			progname, opclo?"enable":"disable", nwerr, nwerr);
		return 100;
	}
	return 0;
}

static int cmd_bindery(char* cmd) {
	NWCCODE nwerr;

	if ((!opclo) && (ncp_get_conn_type(conn) == NCP_CONN_TEMPORARY)) {
		fprintf(stderr, "%s: Warning: Bindery cannot be closed by temporary connection\n", progname);
	}
	nwerr = opclo ? NWOpenBindery(conn) : NWCloseBindery(conn);
	if (nwerr) {
		fprintf(stderr, "%s: Unable to %s bindery, error 0x%04X (%u)\n",
			progname, opclo?"open":"close", nwerr, nwerr);
		return 100;
	}
	return 0;
}

static int cmd_what(char* what) {
	fprintf(stderr, "%s: What is `%s'?\n", progname, what);
	return 98;
}

static struct cmdlist cmds_endis[] = {
	{ "login", cmd_login, 2 },
	{ "tts", cmd_tts, 2 },
	{ NULL, cmd_what, 0 }};

static struct cmdlist cmds_opclo[] = {
	{ "bindery", cmd_bindery, 2 },
	{ NULL, cmd_what, 0 }};

static int cmd_open(char* cmd) {
	opclo = 0;
	return cmdexec(cmds_opclo, cmd, strlen(cmd), "");
}

static int cmd_close(char* cmd) {
	opclo = 1;
	return cmdexec(cmds_opclo, cmd, strlen(cmd), "");
}

static int cmd_disable(char* cmd) {
	opclo = 0;
	return cmdexec(cmds_endis, cmd, strlen(cmd), "");
}

static int cmd_enable(char* cmd) {
	opclo = 1;
	return cmdexec(cmds_endis, cmd, strlen(cmd), "");
}

static struct cmdlist cmds[] = {
	{ "load", cmd_load, 1 },
	{ "unload", cmd_unload, 1 },
	{ "mount", cmd_mount, 1 },
	{ "dismount", cmd_dismount, 1 },
	{ "down", cmd_down, 0 },
	{ "open", cmd_open, 1 },
	{ "close", cmd_close, 1 },
	{ "disable", cmd_disable, 1 },
	{ "enable", cmd_enable, 1 },
	{ "set", cmd_set, 1 },
	{ NULL, cmd_run, 0 }};

int
main(int argc, char **argv)
{
	long err;
	int opt;
	int forceDown = 1;
	int useConn = 0;
	int act = 0;
	
	progname = argv[0];

	if ((conn = ncp_initialize_2(&argc, argv, 1, NCP_BINDERY_USER, &err, 0)) == NULL)
	{
		useConn = 1;
	}
	while ((opt = getopt(argc, argv, "h?fp:cod")) != EOF)
	{
		switch (opt)
		{
		case 'h':
		case '?':
			help();
			exit(1);
		case 'f':
			forceDown = 0;
			act = 3;
			break;
		case 'c':
			act = 1;
			break;
		case 'o':
			act = 2;
			break;
		case 'd':
			act = 3;
			break;
		case 'p':
			if (!useConn) {
				fprintf(stderr, "-p and -S are incompatible\n");
				return 123;
			}
			if (NWParsePath(optarg, NULL, &conn, NULL, NULL) || !conn) {
				fprintf(stderr, "%s: %s is not Netware directory/file\n",
					progname, optarg);
				return 122;
			}
			break;
		default:
			usage();
			exit(1);
		}
	}
	if (!conn) {
		fprintf(stderr, "%s: You must specify connection to server\n", progname);
		return 121;
	}
	if (act) {
		if (optind < argc) {
			fprintf(stderr, "Compatibility option entered, rest of commandline ignored\n");
		}
		switch (act) {
			case 1:	opclo = 0;
				err = cmd_bindery("");
				break;
			case 2:	opclo = 1;
				err = cmd_bindery("");
				break;
			default:
				err = cmd_down(forceDown?"force":"");
				break;
		}
	} else {
		int i;
		int l;
		char* cmdl;
		char* cmd2;
		char* begin;
		char* end;

		l = 1;
		for (i = optind; i < argc; i++)
			l += strlen(argv[i]) + 1;
		cmdl = (char*)malloc(l);
		if (!cmdl) {
			fprintf(stderr, "%s: Out of memory!\n", progname);
			return 123;
		}
		cmd2 = cmdl;
		for (i = optind; i < argc; i++) {
			if (cmd2 != cmdl)
				*cmd2++ = ' ';
			l = strlen(argv[i]);
			memcpy(cmd2, argv[i], l);
			cmd2 += l;
		}
		*cmd2 = 0;

		cmd2 = skipspaces(cmdl);
		begin = cmd2;
		while (*cmd2 && !isspace(*cmd2)) cmd2++;
		end = cmd2;
		if (begin == end) {
			fprintf(stderr, "%s: Missing command\n", progname);
			return 122;
		}
		err = cmdexec(cmds, begin, end-begin, skipspaces(cmd2));
	}
	ncp_close(conn);
	return err;
}
